/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: pps.c,v 1.13 2002/02/06 20:17:59 vlm Exp $
 */


#include "ipcad.h"
#include "got.h"
#include "opt.h"

void
adjust_sorting() {
	static time_t oldtime;
	time_t curtime;
	register int tactive;

	time(&curtime);
	if(!(curtime - oldtime))
		return;

	if(replacements_requested == 0) {
		saved_top_direction = -1;
	} else {
		saved_top_direction = 1;
	};

	replacements_made_saved = replacements_made;
	replacements_made = replacements_requested = 0;

	tactive = top_active + saved_top_direction;

	/* Atomicaly adjust top_active */
	if(tactive < 0)
		top_active = 0;
	else
		top_active = tactive;

	oldtime = curtime;

}


static struct timeval pst = { 0 };
static struct timeval npst = { 0 };

long
time_difference() {
	long tmpms;

	if(!pst.tv_sec) {
		gettimeofday(&pst, NULL);
		return 0;
	}

	gettimeofday(&npst, NULL);

	tmpms = (npst.tv_sec - pst.tv_sec) * 1000 +
			(npst.tv_usec - pst.tv_usec) / 1000;

	if(tmpms < 100)
		return 0;	/* Too short period */

	pst = npst;

	return (tmpms);

};

int
adjust_stats(packet_source *pss) {
	packet_source *ps;
	unsigned long tmpb;
	unsigned long tmpp;
	long tdif;

	tdif = time_difference();
	if(!tdif)	/* Too short period */
		return -1;

	for(ps = pss; ps; ps = ps->next) {
		/* Semi-atomic */
		tmpb = ps->bytes_cur; ps->bytes_cur = 0;
		tmpp = ps->packets_cur; ps->packets_cur= 0;

		/*
		 * Bytes
		 */

		ps->bytes_prev = tmpb;
		if(ps->bytes_lp == -1) {
			ps->bytes_lp = tmpb * (ps->period / tdif);
		} else {
			ps->bytes_lp = (float)(1 - (float)tdif / ps->period)
				* ps->bytes_lp + tmpb;
		}

		/* Estimate bps */
		ps->bps_lp = ps->bytes_lp * 1000 / ps->period;

		/*
		 * Packets
		 */

		ps->packets_prev = tmpp;
		if(ps->packets_lp == -1) {
			ps->packets_lp = tmpp * (ps->period / tdif);
		} else {
			ps->packets_lp = (float)(1 - (float)tdif / ps->period)
				* ps->packets_lp + tmpp;
		}

		/* Estimate bps */
		ps->pps_lp = ps->packets_lp * 1000 / ps->period;

	}

	return 0;
}


int
process_packet_sources(packet_source *pss) {
	packet_source *ps;
	sigset_t set, oset;
	int unslept;

	/*
	 * Block all signals.
	 * The signal mask is inherited from the creating thread.
	 */
	sigfillset(&set);
	sigprocmask(SIG_BLOCK, &set, &oset);

	/* Run threads */
	for(ps = pss; ps; ps = ps->next)
		if( pthread_create(&ps->thid, NULL, pps_loop, ps) ) {
			fprintf(stderr, "Can't start thread for %s!\n", ps->ifname);
			signoff_now = 1;
			sigprocmask(SIG_SETMASK, &oset, NULL);
			return -1;
		} else {
			printf("Thread %ld processing %s started.\n",
				(long)ps->thid, ps->ifname);
		}

	sigprocmask(SIG_SETMASK, &oset, NULL);

	/* Signal-processing loop */

#define	SLEEP_TIME	2	/* Seconds */

	/* Preinitialize */
	time_difference();

	while(1) {
		unslept = sleep(SLEEP_TIME);

		if(signoff_now)
			break;

		if(display_now) {
			if(daemon_mode) {   
				FILE *f = fopen("/dev/console", "w");
				if(f) {
					display(f, 0);
					fclose(f);
				}
			} else {
				display(stdout, 0);
			}
			display_now = 0;
		}

		if(unslept != SLEEP_TIME) {
			adjust_sorting();
		}

		adjust_stats(pss);
	}

	return 0;
}

