/*-
 * Copyright (c) 2001, 2002 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: main.c,v 1.21 2002/03/25 04:04:03 vlm Exp $
 */

#include "ipcad.h"
#include "cfgvar.h"
#include "servers.h"
#include "opt.h"

int draw_chroot();
void terminate_threads();

void
sigalarm(int z) {
	fprintf(stderr, "Thread %ld alarmed.\n", (long)pthread_self());
	signal(z, sigalarm);
}

void
set_display_now(int z) {

	display_now = 1;

	signal(z, set_display_now);
}

void
sigquit(int z) {

	signoff_now = 1;
	
	signal(z, sigquit);
};

int save_on_exit = 0;

struct timeval self_started;

int
main(int ac, char **av) {
	int c;
	char *config_file = CONFIG_FILE;
	int restore = 0;
	int disable_servers = 0;

	gettimeofday(&self_started, NULL);

	while((c = getopt(ac, av, "dhc:rsS")) != -1)
	switch(c) {
		case 'c':
			config_file = optarg;
			if(!config_file)
				usage();
			break;
		case 'd':
			daemon_mode = 1;
			break;
		case 'h':
			usage();
			break;
		case 'r':
			restore = 1;
			break;
		case 's':
			save_on_exit = 1;
			break;
		case 'S':
			disable_servers = 1;
			break;
		default:
			usage();
	};

	if(init_pthread_options()) {
		fprintf(stderr, "Can't initialize thread options.\n");
		exit(EX_OSERR);
	}

	if(cfgread(config_file))
		exit(EX_NOINPUT);


	/******************************/
	/* Process configuration data */
	/******************************/

	if(!packet_sources_head) {
		fprintf(stderr, "No interfaces initialized.\n");
		exit(EX_NOINPUT);
	}

	if(save_on_exit || restore) {
		if(!cfg_dump_file) {
			fprintf(stderr, "Dump file is not defined.\n");
			exit(EX_NOINPUT);
		};
	};

	/* Security */
	if( draw_chroot() ) {
		fprintf(stderr, "Can't chroot to %s: Invalid pathname\n",
			chroot_to);
		exit(EX_DATAERR);
	};

	/* Restore previously saved table */
	if(restore)
		import_table(cfg_dump_file, stderr, 1);

	time(&ipstream_time);

	/* Daemon mode should be entered BEFORE threads created. */
	if(daemon_mode) {
		if(daemon(0, 1) == -1) {
			perror("ipcad");
			exit(EX_OSERR);
		}
	}


	/***********************************/
	/* Set appropriate signal handlers */
	/***********************************/

	/* Ignore signals, by default */
	for(c = 1; c < 32; c++)
		signal(c, SIG_IGN);

	signal(SIGALRM, sigalarm);
	signal(SIGINT, set_display_now);
	signal(SIGHUP, set_display_now);
	signal(SIGQUIT, sigquit);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGTERM, sigquit);
	signal(SIGTTIN, SIG_IGN);
	signal(SIGSEGV, SIG_DFL);

	siginterrupt(SIGALRM, 1);

	/*******************************************/
	/* Start servers to serve clients requests */
	/*******************************************/

	if(!disable_servers) {
		if( start_servers() != 0 ) {
			fprintf(stderr, "Failed to start one or more servers.\n");
			exit(EX_OSERR);
		}
	}

	/*******************/
	/* Main processing */
	/*******************/

#ifdef	HAVE_SETPRIORITY
	/* Set nice, but safe priority */
	setpriority(PRIO_PROCESS, 0, -15);
#endif

	/* Work */
	process_packet_sources(packet_sources_head);

	/* Termination */
	terminate_threads();

	if(save_on_exit)
		make_dump(cfg_dump_file, stderr);

	printf("Quit.\n");

	return 0;
};

int
draw_chroot() {
	if(!chroot_to)
		return 0;

	if(*chroot_to != '/')
		return -1;

	/*
	 * Pre-open needed files before doing chroot().
	 */
	ifst_preopen();

	if(chroot(chroot_to) != 0)
		return -1;

	if(chdir("/") != 0)
		return -1;

	return 0;
};


void
terminate_threads() {
	packet_source *ps;

	end_servers();

	printf("Waiting for interface processing threads to terminate...\n");

	for(ps = packet_sources_head; ps; ps = ps->next) {
		if(!ps->thid)
			continue;

#ifdef	HAVE_PTHREAD_CANCEL
		if(pthread_cancel(ps->thid)) {
			printf("%s processing thread is already gone.\n",
				ps->ifname);
			ps->thid = 0;
			continue;
		}
#endif

		/* Kick it */
		pthread_kill(ps->thid, SIGALRM);

		if(pthread_join(ps->thid, NULL)) {
			if(errno == EINVAL)
				printf("Thread processing %s is not a joinable thread.\n",
					ps->ifname);
			if(errno == ESRCH)
				printf("No thread running for processing %s\n",
					ps->ifname);
			if(errno == EDEADLK)	/* Impossible state */
				printf("Deadlock avoided for thread processing %s\n",
					ps->ifname);
		} else {
				printf("Thread processing %s terminated.\n", ps->ifname);
		}
	}
}


/*
 * Code for people

: 
-     . 
: 
- . 
-   . 
-    ! 
- ,       . 
-   ,        . 

 */
