/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ips-pcap.c,v 1.10 2001/08/28 08:46:20 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"


#warning "Ignore sent" interface feature is not supported by PCAP. Minor warning.

int apply_ip_filter(pcap_t *dev);

int
reopen_packet_source(packet_source *ps, int loop) {
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *dev;

	ps->dev = NULL;

	while(1) {
		dev = pcap_open_live(ps->ifname, 68,
			(ps->iflags & 2)?1:0, 0, errbuf);
		if(!dev) {
			if(loop) {
				sleep(10);
				continue;
			} else {
				fprintf(stderr, "[%s] ", errbuf);
				break;
			}
		}

		/* Get device type */
		ps->dlt = pcap_datalink(dev);

		if( apply_ip_filter(dev) ) {
			pcap_close(dev);

			errno = ENODEV;
			if(loop) {
				sleep(10);
				continue;
			} else {
				fprintf(stderr, "[Warning: Can't initialize filter!] ");
				break;
			}
		}

		ps->dev = dev;
		return 0;
	}

	return -1;
}

packet_source *
init_packet_source(char *ifname, int iflags) {
	packet_source *ps;

	if(iflags & 1) {
		printf("%s: Input-only feature not supported by PCAP.\n", ifname);
		return NULL;
	}

	/* Allocate new packet source */
	ps = init_packet_source_common(ifname, iflags);
	if(!ps)
		return NULL;

	if( reopen_packet_source(ps, 0) ) {
		free(ps);
		return NULL;
	}

	/* Complain about unknown devices. */
	switch(ps->dlt) {
	case DLT_NULL:    /* Loopback */
	case DLT_RAW:     /* Some PPP implementations, etc. */
	case DLT_EN10MB:  /* Generic Ethernet-compatible */
	case DLT_PPP:     /* Point-to point interface */
#ifdef	DLT_LINUX_SLL
	case DLT_LINUX_SLL:	/* fake header for Linux cooked socket */
#endif
		break;
	default:
		fprintf(stderr, "[Unknown interface type] ");
		pcap_close(ps->dev);
		free(ps);
		errno = ENODEV;
		return NULL;
	};

	return ps;
}

int
apply_ip_filter(pcap_t *dev) {
	struct bpf_program fp;

	if( pcap_compile(dev, &fp, "ip", 1, -1) )
		return -1;

	if( pcap_setfilter(dev, &fp) )
		return -1;

	return 0;
}

