/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ifst_bsd.c,v 1.1 2001/09/28 15:11:14 vlm Exp $
 */

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "ipcad.h"

#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/sysctl.h>
#include <sys/time.h>

#ifndef __OpenBSD__
#include <sys/module.h>
#include <sys/linker.h>
 
#include <net/ethernet.h>
#include <net/if.h>
#include <net/if_var.h>
#endif

#include <net/if_dl.h>
#include <net/if_types.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_var.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

void	rt_xaddrs __P((caddr_t, caddr_t, struct rt_addrinfo *));
void	status __P((FILE *f, const char *ifname,
		    struct sockaddr_dl *sdl, struct if_msghdr *ifm));

void in_status(FILE *, struct rt_addrinfo *, int flags);
void ether_status(FILE *, struct rt_addrinfo *);

/*
 * Expand the compacted form of addresses as returned via the
 * configuration read via sysctl().
 */

#define ROUNDUP(a) \
	((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))
#define ADVANCE(x, n) (x += ROUNDUP((n)->sa_len))

void
rt_xaddrs(cp, cplim, rtinfo)
	caddr_t cp, cplim;
	struct rt_addrinfo *rtinfo;
{
	struct sockaddr *sa;
	int i;

	memset(rtinfo->rti_info, 0, sizeof(rtinfo->rti_info));
	for (i = 0; (i < RTAX_MAX) && (cp < cplim); i++) {
		if ((rtinfo->rti_addrs & (1 << i)) == 0)
			continue;
		rtinfo->rti_info[i] = sa = (struct sockaddr *)cp;
		ADVANCE(cp, sa);
	}
}


char *
get_encaps(int encaps) {
	static char buf[32];

	switch(encaps) {
		case IFT_ETHER:
			return "Ethernet";
		case IFT_PPP:
			return "PPP";
		case IFT_LOOP:
			return "Loopback";
		case IFT_SLIP:
			return "SLIP";
		case IFT_RS232:
			return "RS232";
#ifdef	IFT_GIF
		case IFT_GIF:
			return "GIF";
#endif
#ifdef	IFT_FAITH
		case IFT_FAITH:
			return "FAITH";
#endif
	}

	snprintf(buf, sizeof(buf), "Unknown/%d", encaps);
	return buf;
}

int
if_stat(FILE *f, char *ifname) {
	struct	if_msghdr *ifm;
	struct	sockaddr_dl *sdl;
	char	*buf, *lim, *next;


	size_t needed;
	int mib[6];

	mib[0] = CTL_NET;
	mib[1] = PF_ROUTE;
	mib[2] = 0;
	mib[3] = 0;	/* address family */
	mib[4] = NET_RT_IFLIST;
	mib[5] = 0;

	/* if particular family specified, only ask about it */

	if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)  {
		fprintf(f, "System error.\n");
		return -1;
	}
	if ((buf = malloc(needed)) == NULL) {
		fprintf(f, "Resource shortage.\n");
		return -1;
	}
	if (sysctl(mib, 6, buf, &needed, NULL, 0) < 0) {
		fprintf(f, "System error.\n");
		free(buf);
		return -1;
	}

	for(next = buf, lim = buf + needed;
			next < lim;
				next += ((struct if_msghdr *)next)->ifm_msglen) {

		ifm = (struct if_msghdr *)next;
		
		if (ifm->ifm_type == RTM_IFINFO) {
			sdl = (struct sockaddr_dl *)(ifm + 1);
		} else if(ifm->ifm_type == RTM_NEWADDR){
			continue;
		} else {
			fprintf(f, "No such interface %s\n", ifname);
			next = lim;	/* For the last check */
			break;
		}

		if (strlen(ifname) != sdl->sdl_nlen)
			continue; /* not same len */
		if (strncmp(ifname, sdl->sdl_data, sdl->sdl_nlen) != 0)
			continue; /* not same name */

		status(f, ifname, sdl, ifm);

		break;
	}

	free(buf);

	if(next >= lim) {
		fprintf(f, "No such interface %s\n", ifname);
		return -1;
	}

	return 0;
}


/*
 * Print the status of the interface.  If an address family was
 * specified, show it and it only; otherwise, show them all.
 */
void
status(f, ifname, sdl, ifm)
	FILE *f;
	const char *ifname;
	struct	sockaddr_dl *sdl;
	struct if_msghdr *ifm;
{
	struct ifa_msghdr *ifam;
	struct	rt_addrinfo info;

	fprintf(f, "%s is %s, line protocol is %s\n",
		ifname,
		((ifm->ifm_flags & IFF_RUNNING) ? "up" : "down"),
		((ifm->ifm_flags & IFF_UP) ? "up" : "down")
	);

	ether_status(f, (struct rt_addrinfo *)sdl);

	ifam = (struct ifa_msghdr *)((char *)ifm + ifm->ifm_msglen);
	while(ifam->ifam_type == RTM_NEWADDR) {
		
		info.rti_addrs = ifam->ifam_addrs;

		/* Expand the compacted addresses */
		rt_xaddrs((char *)(ifam + 1), ifam->ifam_msglen + (char *)ifam,
			  &info);

		if(info.rti_info[RTAX_IFA]->sa_family == AF_INET)
			in_status(f, &info, ifm->ifm_flags);

		ifam = (struct ifa_msghdr *)((char *)ifam + ifam->ifam_msglen);
	}

	fprintf(f, "  Encapsulation %s, loopback %s\n",
		get_encaps(ifm->ifm_data.ifi_type),
		((ifm->ifm_flags & IFF_LOOPBACK) ? "set": "not set")
	);

	fprintf(f, "  MTU %lu bytes, BW %lu Kbit\n",
		ifm->ifm_data.ifi_mtu,
		(long)(ifm->ifm_data.ifi_baudrate / 1000)
	);

	fprintf(f, "  Input queue: %lu drops\n",
		ifm->ifm_data.ifi_iqdrops );
	if(ifm->ifm_data.ifi_lastchange.tv_sec)
		fprintf(f, "  Last administrative status change at %s",
			ctime(&ifm->ifm_data.ifi_lastchange.tv_sec));

	display_internal_averages(f, ifname);

	fprintf(f, "     %lu packets input, ", ifm->ifm_data.ifi_ipackets);
		fprintf(f, "%lu bytes, ", ifm->ifm_data.ifi_ibytes);
		fprintf(f, "%d no buffer\n", 0);
	fprintf(f, "     %lu input errors, ", ifm->ifm_data.ifi_ierrors);
		fprintf(f, "%d CRC, %d frame, %d overrun, %lu ignored, %d abort\n",
			0, 0, 0, ifm->ifm_data.ifi_ierrors, 0 );

	fprintf(f, "     %lu packets output, ", ifm->ifm_data.ifi_opackets);
		fprintf(f, "%lu bytes, ", ifm->ifm_data.ifi_obytes);
		fprintf(f, "%d underruns\n", 0);
	fprintf(f, "     %lu output errors, ", ifm->ifm_data.ifi_oerrors);
		fprintf(f, "%lu collisions, %d interface resets, %d restarts\n",
			ifm->ifm_data.ifi_collisions,
			0, 0 );

	return;
}

void
in_status(f, info, flags)
	FILE *f;
	struct rt_addrinfo * info;
	int flags;
{
	struct sockaddr_in *sin, null_sin;
	
	memset(&null_sin, 0, sizeof(null_sin));

	sin = (struct sockaddr_in *)info->rti_info[RTAX_IFA];
	fprintf(f, "  Internet address is ");
	print_ip(f, sin->sin_addr);

	sin = (struct sockaddr_in *)info->rti_info[RTAX_NETMASK];
	if (!sin)
		sin = &null_sin;
	if(sin->sin_addr.s_addr != 0xffffffff) {
		fprintf(f, " ");
		print_ip(f, sin->sin_addr);
	}
	fprintf(f, "\n");

	if (flags & IFF_POINTOPOINT) {
		/* note RTAX_BRD overlap with IFF_BROADCAST */
		sin = (struct sockaddr_in *)info->rti_info[RTAX_BRD];
		if (!sin)
			sin = &null_sin;
		fprintf(f, "  Peer IP address is ");
		print_ip(f, sin->sin_addr);
		fprintf(f, "\n");
	}

	if (flags & IFF_BROADCAST) {
		/* note RTAX_BRD overlap with IFF_POINTOPOINT */
		sin = (struct sockaddr_in *)info->rti_info[RTAX_BRD];
		if (sin && sin->sin_addr.s_addr != 0)
			fprintf(f, "  IP broadcast address is ");
			print_ip(f, sin->sin_addr);
			fprintf(f, "\n");
	}

}

void
ether_status(f, info)
	FILE *f;
	struct rt_addrinfo *info;
{
	char *cp;
	int n;
	struct sockaddr_dl *sdl = (struct sockaddr_dl *)info;

	/* Assume address % 2 = 0 */

	cp = (char *)LLADDR(sdl);
	if ((n = sdl->sdl_alen) > 0) {
		fprintf(f, "  Hardware is %s, address is ",
			get_encaps(sdl->sdl_type));

		for(; n > 0; n-=2, cp += 2)
			fprintf(f, "%02x%02x%s",
				cp[0] & 0xff, cp[1] & 0xff,
				n>2 ? "." : "" );

#ifdef	ENABLE_BIA_OUTPUT
		fprintf(f, " (bia ");
		n = sdl->sdl_alen;
		cp = (char *)LLADDR(sdl);
		for(; n > 0; n-=2, cp += 2)
			fprintf(f, "%02x%02x%s",
				cp[0] & 0xff, cp[1] & 0xff,
				n>2 ? "." : "" );

		fprintf(f, ")");
#else
		fprintf(f, "\n");
#endif

	}
}

int
ifst_preopen() {
	/* Nothing to preopen */
	return 0;
}


void
system_uptime(FILE *f) {
	struct timeval boottime;
	int mib[2] = { CTL_KERN, KERN_BOOTTIME };
	size_t size;
	char buf[MAXHOSTNAMELEN];
	
	size = sizeof(boottime);
	if(sysctl(mib, 2, &boottime, &size, NULL, 0) == -1)
		return;

	gethostname(buf, sizeof(buf));
	buf[MAXHOSTNAMELEN - 1] = '\0';

	fprintf(f, "%s uptime is", buf);
	display_uptime(f, time(NULL) - boottime.tv_sec + 30);
}

