/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: disp.c,v 1.14 2001/08/30 08:40:24 vlm Exp $
 */

#include "ipcad.h"
#include "cfgvar.h"
#include "got.h"
#include "opt.h"

void show_stats(FILE *f);

#define ROOM_FOR_IPADDR	17
#define	C(f)	(unsigned char)((f) & 0xff)

void
print_ip(FILE *f, struct in_addr ip) {
	fprintf(f, "%d.%d.%d.%d",
		C(ip.s_addr),
		C(ip.s_addr >> 8),
		C(ip.s_addr >> 16),
		C(ip.s_addr >> 24)
	);
}

void
print_aligned_ip(FILE *f, struct in_addr ip) {
	int z;

	z = fprintf(f, " %d.%d.%d.%d",
		C(ip.s_addr),
		C(ip.s_addr >> 8),
		C(ip.s_addr >> 16),
		C(ip.s_addr >> 24)
	);

	while(z++ < ROOM_FOR_IPADDR)
		putc(' ', f);
}



/*
 * fast_get_table
 * Makes a plain array copy of linked list for efficiency.
 */

struct ipstream *
fast_get_table(struct ipstream *stream, size_t *size) {
	struct ipstream *ips_new;
	struct ipstream *ips;
	size_t entries;

	if(!stream || !size)
		return NULL;

	entries = *size;

	ips_new = (struct ipstream *)malloc(
		sizeof(struct ipstream) * (entries?entries:1));

	if(!ips_new) {
		*size = -1;
		return NULL;
	};

	for(ips = ips_new;
		stream && entries;
			stream=stream->next, ips++, entries--){
		*ips = *stream;
	};

	*size -= entries;

	return ips_new;
};

int
display(FILE *f, int ischeckpoint) {
	struct ipstream *ips;
	struct ipstream *table;
	time_t stalled;
	size_t ex_packets;
	size_t ex_bytes;
	int entries, entr;
	time_t saved_time;
	pthread_mutex_t *lmutex = ischeckpoint?&checkpoint_mutex:&ipstream_mutex;

	pthread_mutex_lock(lmutex);

		if(ischeckpoint) {
			ips = checkpoint;
		entr=entries = checkpoint_entries;
			stalled = ex_checkpoint_stalled;
			ex_packets = ex_checkpoint_packets;
			ex_bytes = ex_checkpoint_bytes;
			saved_time = checkpoint_time;
		} else {
			ips = ipstream;
		entr=entries = ipstream_entries;
			stalled = ex_current_stalled;
			ex_packets = ex_current_packets;
			ex_bytes = ex_current_bytes;
			saved_time = ipstream_time;
		};

		table = fast_get_table(ips, &entries);

	pthread_mutex_unlock(lmutex);

	if(!table) {
		if(entries == -1) {
			fprintf(f, "Memory allocation error.\n");
			return -1;
		}
	};

	fprintf(f, "\n");
	fprintf(f, "   Source           Destination              Packets               Bytes\n");

	if(table) {
		for(ips=table, entr = 0;
		  entr < entries; entr++, ips++) {
			print_aligned_ip(f, ips->src);
			print_aligned_ip(f, ips->dst);
			fprintf(f, "%18u %19qu\n",
				ips->packets, ips->bytes);
		}
	}

	fprintf(f, "\n");

	if(table)
		free(table);

	{
		time_t age = 0;
		if(saved_time)
			age = (time(NULL) - saved_time) / 60;
		fprintf(f, "Accounting data age is %5ld\n", (long)age);
		if(saved_time)
			fprintf(f, "Accounting data age exact %ld\n",
				time(NULL) - saved_time);
	};

	if(ex_packets)
	fprintf(f, "Accounting threshold exceeded for %u packets and %u bytes\n",
		ex_packets, ex_bytes);

	if(stalled)
		fprintf(f, "Information incomplete since %s", ctime(&stalled));

	if(!ischeckpoint)
		show_stats(f);

	return 0;
};

void display_capture_stats(FILE *);

void
show_stats(FILE *f) {
	register int ipecnt = ipstream_entries;
	register int tactive = top_active;
	register size_t used_memory = ipecnt * sizeof(struct ipstream);

	display_capture_stats(f);

	fprintf(f, "Entries made: %d\n", ipecnt);

	if(memsize)   
		fprintf(f, "Memory usage: %d%% (%u from %u)\n",
			used_memory * 100 / memsize, used_memory, memsize);
	else
		fprintf(f, "Memory usage: %u kbytes.\n", used_memory >> 10);


	if(ipecnt) {
		if(tactive > ipecnt)
			tactive = ipecnt;

		fprintf(f, "Top active entries: %d%% (%d)\n",
			(ipecnt?(tactive * 100 / ipecnt):0),
			tactive);
	}

	fprintf(f, "Resorts per period: %u\n", replacements_made_saved);
	fprintf(f, "Free slots for clients: %d\n", max_clients);

	ipcad_uptime(f);
	system_uptime(f);
  
};

int
display_internal_averages(FILE *f, const char *ifname) {
	packet_source *ps;

	/* Find the interface */
	for(ps = packet_sources_head; ps; ps = ps->next) {
		if(!strcmp(ps->ifname, ifname))
			break;
	}

	if(!ps)
		return -1;

	fprintf(f, "  %lu minute average rate %lu bits/sec, %lu packets/sec\n",  
		ps->period / 60000,
		ps->bps_lp * 8,
		ps->pps_lp 
	);

	return 0;
}

void
show_version(FILE *f) {
	fprintf(f,
		IPCAD_VERSION "\n"
		IPCAD_COPYRIGHT "\n"
		"\n"
	);

	ipcad_uptime(f);
	system_uptime(f);
}


void
ipcad_uptime(FILE *f) {
	fprintf(f, "IPCAD uptime is");
	display_uptime(f, time(NULL) - self_started.tv_sec);
}


void
display_uptime(FILE *f, time_t uptime) {
	int days, hrs, mins, secs;

	days = uptime / 86400;
	uptime %= 86400;
	hrs = uptime / 3600;
	uptime %= 3600;
	mins = uptime / 60;
	secs = uptime % 60;

	if(days > 0)
		fprintf(f, " %d day%s", days, days>1?"s":"");
	if(hrs > 0 && mins > 0)
		fprintf(f, " %2d:%02d", hrs, mins);
	else if(hrs > 0)
		fprintf(f, " %d hour%s,", hrs, hrs>1?"s":"");
	else if(mins > 0)
		fprintf(f, " %d minute%s", mins, mins>1?"s":"");
	else
		fprintf(f, " %d second%s", secs, secs>1?"s":"");
	fprintf(f, "\n");

}

