/*
 * ion/ioncore/screen.c
 *
 * Copyright (c) Tuomo Valkonen 1999-2007. 
 *
 * Ion is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 */

#include <string.h>

#include <libtu/objp.h>
#include <libtu/minmax.h>
#include <libmainloop/defer.h>

#include "common.h"
#include "global.h"
#include "screen.h"
#include "region.h"
#include "attach.h"
#include "manage.h"
#include "focus.h"
#include "property.h"
#include "names.h"
#include "reginfo.h"
#include "saveload.h"
#include "resize.h"
#include "event.h"
#include "bindmaps.h"
#include "regbind.h"
#include "frame-pointer.h"
#include "rectangle.h"
#include "infowin.h"
#include "activity.h"
#include "extlconv.h"
#include "llist.h"
#include "group-ws.h"
#include "mplex.h"
#include "tags.h"
#include "gr.h"
#include "gr-util.h"
#include "conf.h"


WHook *screen_managed_changed_hook=NULL;


static void screen_update_infowin(WScreen *scr);



/*{{{ Init/deinit */


bool screen_init(WScreen *scr, WRootWin *parent,
                 const WFitParams *fp, int id, Window rootwin)
{
    Window win;
    XSetWindowAttributes attr;
    ulong attrflags=0;
    bool is_root=FALSE;
    
    scr->id=id;
    scr->atom_workspace=None;
    scr->managed_off.x=0;
    scr->managed_off.y=0;
    scr->managed_off.w=0;
    scr->managed_off.h=0;
    scr->next_scr=NULL;
    scr->prev_scr=NULL;
    scr->rotation=SCREEN_ROTATION_0;
    
    watch_init(&(scr->notifywin_watch));
    watch_init(&(scr->infowin_watch));

    if(parent==NULL){
        win=rootwin;
        is_root=TRUE;
    }else{
        attr.background_pixmap=ParentRelative;
        attrflags=CWBackPixmap;
        
        win=XCreateWindow(ioncore_g.dpy, WROOTWIN_ROOT(parent),
                          fp->g.x, fp->g.y, fp->g.w, fp->g.h, 0, 
                          DefaultDepth(ioncore_g.dpy, parent->xscr),
                          InputOutput,
                          DefaultVisual(ioncore_g.dpy, parent->xscr),
                          attrflags, &attr);
        if(win==None)
            return FALSE;
            
    }

    if(!mplex_do_init((WMPlex*)scr, (WWindow*)parent, fp, win)){
        if(!is_root)
            XDestroyWindow(ioncore_g.dpy, win);
        return FALSE;
    }

    /*scr->mplex.win.region.rootwin=rootwin;
    region_set_parent((WRegion*)scr, (WRegion*)rootwin);*/
    scr->mplex.flags|=MPLEX_ADD_TO_END;
    scr->mplex.win.region.flags|=REGION_BINDINGS_ARE_GRABBED;
    
    if(!is_root){
        scr->mplex.win.region.flags|=REGION_MAPPED;
        window_select_input((WWindow*)scr, IONCORE_EVENTMASK_SCREEN);
    }
    
    if(id==0){
        scr->atom_workspace=XInternAtom(ioncore_g.dpy, 
                                        "_ION_WORKSPACE", False);
    }else if(id>=0){
        char *str;
        libtu_asprintf(&str, "_ION_WORKSPACE%d", id);
        if(str!=NULL){
            scr->atom_workspace=XInternAtom(ioncore_g.dpy, str, False);
            free(str);
        }
    }

    /* Add all the needed bindings here; mplex does nothing so that
     * frames don't have to remove extra bindings.
     */
    region_add_bindmap((WRegion*)scr, ioncore_screen_bindmap);
    region_add_bindmap((WRegion*)scr, ioncore_mplex_bindmap);
    region_add_bindmap((WRegion*)scr, ioncore_mplex_toplevel_bindmap);

    LINK_ITEM(ioncore_g.screens, scr, next_scr, prev_scr);
    
    return TRUE;
}


WScreen *create_screen(WRootWin *parent, const WFitParams *fp, int id)
{
    CREATEOBJ_IMPL(WScreen, screen, (p, parent, fp, id, None));
}


void screen_deinit(WScreen *scr)
{
    UNLINK_ITEM(ioncore_g.screens, scr, next_scr, prev_scr);
    
    mplex_deinit((WMPlex*)scr);
}


/*}}}*/


/*{{{ Attach/detach */


void screen_managed_geom(WScreen *scr, WRectangle *geom)
{
    geom->x=scr->managed_off.x;
    geom->y=scr->managed_off.y;
    geom->w=REGION_GEOM(scr).w+scr->managed_off.w;
    geom->h=REGION_GEOM(scr).h+scr->managed_off.h;
    geom->w=maxof(geom->w, 0);
    geom->h=maxof(geom->h, 0);
}


static bool screen_handle_drop(WScreen *scr, int x, int y, WRegion *dropped)
{
    WRegion *curr=mplex_mx_current(&(scr->mplex));

    /* This code should handle dropping tabs on floating workspaces. */
    if(curr && HAS_DYN(curr, region_handle_drop)){
        int rx, ry;
        region_rootpos(curr, &rx, &ry);
        if(rectangle_contains(&REGION_GEOM(curr), x-rx, y-ry)){
            if(region_handle_drop(curr, x, y, dropped))
                return TRUE;
        }
    }
    
    /* Do not attach to ourselves unlike generic WMPlex. */
    return FALSE;
}


/*}}}*/


/*{{{ Region dynfun implementations */


static bool screen_fitrep(WScreen *scr, WWindow *par, const WFitParams *fp)
{
    WRegion *sub;
    
    if(par==NULL)
        return FALSE;
    
    if(scr->uses_root)
        return FALSE;

    return mplex_fitrep((WMPlex*)scr, NULL, fp);
}




static void screen_managed_changed(WScreen *scr, int mode, bool sw, 
                                   WRegion *reg_)
{
    if(ioncore_g.opmode==IONCORE_OPMODE_DEINIT)
        return;
    
    if(sw && scr->atom_workspace!=None){
        WRegion *reg=mplex_mx_current(&(scr->mplex));
        const char *n=NULL;
        
        if(reg!=NULL)
            n=region_displayname(reg);
        
        xwindow_set_string_property(region_root_of((WRegion*)scr),
                                    scr->atom_workspace, 
                                    n==NULL ? "" : n);
    }
    
    screen_update_infowin(scr);
    
    mplex_call_changed_hook((WMPlex*)scr,
                            screen_managed_changed_hook,
                            mode, sw, reg_);
}


static void screen_map(WScreen *scr)
{
    if(scr->uses_root)
        return;
    mplex_map((WMPlex*)scr);
}


static void screen_unmap(WScreen *scr)
{
    if(scr->uses_root)
        return;
    mplex_unmap((WMPlex*)scr);
}

void screen_inactivated(WScreen *scr)
{
    screen_update_infowin(scr);
}


void screen_activated(WScreen *scr)
{
    screen_update_infowin(scr);
}


/*}}}*/


/*}}}*/


/*{{{ Notifications */


static void do_notify(WScreen *scr, Watch *watch, bool right,
                      const char *str, char *style)
{

    WInfoWin *iw=(WInfoWin*)(watch->obj);
    WFitParams fp;
    
    if(iw==NULL){
        WMPlexAttachParams param=MPLEXATTACHPARAMS_INIT;
        
        param.flags=(MPLEX_ATTACH_UNNUMBERED|
                     MPLEX_ATTACH_SIZEPOLICY|
                     MPLEX_ATTACH_GEOM|
                     MPLEX_ATTACH_LEVEL);
        param.level=STACKING_LEVEL_ON_TOP;
        
        if(!right){
            param.szplcy=SIZEPOLICY_GRAVITY_NORTHWEST;
            param.geom.x=0;
        }else{
            param.szplcy=SIZEPOLICY_GRAVITY_NORTHEAST;
            param.geom.x=REGION_GEOM(scr).w-1;
        }
        
        param.geom.y=0;
        param.geom.w=1;
        param.geom.h=1;
        
        iw=(WInfoWin*)mplex_do_attach_new(&scr->mplex, &param,
                                          (WRegionCreateFn*)create_infowin, 
                                          style);
        
        if(iw==NULL)
            return;

        watch_setup(watch, (Obj*)iw, NULL);
    }

    infowin_set_text(iw, str);
}


void screen_notify(WScreen *scr, const char *str)
{
    do_notify(scr, &scr->notifywin_watch, FALSE, str, "actnotify");
}


void screen_windowinfo(WScreen *scr, const char *str)
{
    do_notify(scr, &scr->infowin_watch, TRUE, str, "tab-info");
}


void screen_unnotify(WScreen *scr)
{
    Obj *iw=scr->notifywin_watch.obj;
    if(iw!=NULL){
        watch_reset(&(scr->notifywin_watch));
        region_dispose((WRegion*)iw, FALSE);
    }
}


void screen_nowindowinfo(WScreen *scr)
{
    Obj *iw=scr->infowin_watch.obj;
    if(iw!=NULL){
        watch_reset(&(scr->infowin_watch));
        region_dispose((WRegion*)iw, FALSE);
    }
}


static char *addnot(char *str, WRegion *reg)
{
    const char *nm=region_name(reg);
    char *nstr=NULL;
    
    if(nm==NULL)
        return str;
    
    if(str==NULL)
        return scat(TR("act: "), nm);

    nstr=scat3(str, ", ", nm);
    if(nstr!=NULL)
        free(str);
    return nstr;
}


static char *screen_managed_activity(WScreen *scr)
{
    char *notstr=NULL;
    WMPlexIterTmp tmp;
    WRegion *reg;
    
    FOR_ALL_MANAGED_BY_MPLEX(&scr->mplex, reg, tmp){
        if(region_is_activity_r(reg) && !REGION_IS_MAPPED(reg))
            notstr=addnot(notstr, reg);
    }
    
    return notstr;
}


static void screen_notify_activity(WScreen *scr)
{
    if(ioncore_g.screen_notify){
        char *notstr=screen_managed_activity(scr);
        if(notstr!=NULL){
            screen_notify(scr, notstr);
            free(notstr);
            return;
        }
    }

    screen_unnotify(scr);
    
    screen_update_infowin(scr);
}


static void screen_notify_tag(WScreen *scr)
{
    screen_update_infowin(scr);
}


GR_DEFATTR(active);
GR_DEFATTR(inactive);
GR_DEFATTR(selected);
GR_DEFATTR(tagged);
GR_DEFATTR(not_tagged);
GR_DEFATTR(not_dragged);
GR_DEFATTR(activity);
GR_DEFATTR(no_activity);


static void init_attr()
{
    GR_ALLOCATTR_BEGIN;
    GR_ALLOCATTR(active);
    GR_ALLOCATTR(inactive);
    GR_ALLOCATTR(selected);
    GR_ALLOCATTR(tagged);
    GR_ALLOCATTR(not_tagged);
    GR_ALLOCATTR(not_dragged);
    GR_ALLOCATTR(no_activity);
    GR_ALLOCATTR(activity);
    GR_ALLOCATTR_END;
}


static void screen_update_infowin(WScreen *scr)
{
    WRegion *reg=mplex_mx_current(&(scr->mplex));
    bool tag=(reg!=NULL && region_is_tagged(reg));
    bool act=(reg!=NULL && region_is_activity_r(reg) && !REGION_IS_ACTIVE(scr));
    bool sac=REGION_IS_ACTIVE(scr);
    
    if(tag || act){
        const char *n=region_displayname(reg);
        WInfoWin *iw;
                
        screen_windowinfo(scr, n);
        
        iw=(WInfoWin*)scr->infowin_watch.obj;
        
        if(iw!=NULL){
            GrStyleSpec *spec=infowin_stylespec(iw);
            
            init_attr();
            
            gr_stylespec_unalloc(spec);
            
            gr_stylespec_set(spec, GR_ATTR(selected));
            gr_stylespec_set(spec, GR_ATTR(not_dragged));
            gr_stylespec_set(spec, sac ? GR_ATTR(active) : GR_ATTR(inactive));
            gr_stylespec_set(spec, tag ? GR_ATTR(tagged) : GR_ATTR(not_tagged));
            gr_stylespec_set(spec, act ? GR_ATTR(activity) : GR_ATTR(no_activity));
        }
            
    }else{
        screen_nowindowinfo(scr);
    }
}


static void screen_managed_notify(WScreen *scr, WRegion *reg, WRegionNotify how)
{
    if(how==ioncore_g.notifies.sub_activity){
        /* TODO: multiple calls */
        mainloop_defer_action((Obj*)scr, 
                              (WDeferredAction*)screen_notify_activity);
    }else if(how==ioncore_g.notifies.tag){
        mainloop_defer_action((Obj*)scr, 
                              (WDeferredAction*)screen_notify_tag);
    }
}


/*}}}*/


/*{{{ Misc. */


/*EXTL_DOC
 * Find the screen with numerical id \var{id}. 
 */
EXTL_SAFE
EXTL_EXPORT
WScreen *ioncore_find_screen_id(int id)
{
    WScreen *scr, *maxscr=NULL;
    
    FOR_ALL_SCREENS(scr){
        if(id==-1){
            if(maxscr==NULL || scr->id>maxscr->id)
                maxscr=scr;
        }
        if(scr->id==id)
            return scr;
    }
    
    return maxscr;
}


/*EXTL_DOC
 * Switch focus to the screen with id \var{id} and return it.
 * 
 * Note that this function is asynchronous; the screen will not
 * actually have received the focus when this function returns.
 */
EXTL_EXPORT
WScreen *ioncore_goto_nth_screen(int id)
{
    WScreen *scr=ioncore_find_screen_id(id);
    if(scr!=NULL){
        if(!region_goto((WRegion*)scr))
            return NULL;
    }
    return scr;
}


static WScreen *current_screen()
{
    if(ioncore_g.focus_current==NULL)
        return ioncore_g.screens;
    else
        return region_screen_of(ioncore_g.focus_current);
}

       
/*EXTL_DOC
 * Switch focus to the next screen and return it.
 * 
 * Note that this function is asynchronous; the screen will not
 * actually have received the focus when this function returns.
 */
EXTL_EXPORT
WScreen *ioncore_goto_next_screen()
{
    WScreen *scr=current_screen();
    
    if(scr!=NULL)
        scr=scr->next_scr;
    if(scr==NULL)
        scr=ioncore_g.screens;
    if(scr!=NULL){
        if(!region_goto((WRegion*)scr))
            return NULL;
    }
    return scr;
}


/*EXTL_DOC
 * Switch focus to the previous screen and return it.
 * 
 * Note that this function is asynchronous; the screen will not
 * actually have received the focus when this function returns.
 */
EXTL_EXPORT
WScreen *ioncore_goto_prev_screen()
{
    WScreen *scr=current_screen();

    if(scr!=NULL)
        scr=scr->prev_scr;
    else
        scr=ioncore_g.screens;
    if(scr!=NULL){
        if(!region_goto((WRegion*)scr))
            return NULL;
    }
    return scr;
}


/*EXTL_DOC
 * Return the numerical id for screen \var{scr}.
 */
EXTL_SAFE
EXTL_EXPORT_MEMBER
int screen_id(WScreen *scr)
{
    return scr->id;
}


static bool screen_managed_may_destroy(WScreen *scr, WRegion *reg)
{
    bool onmxlist=FALSE;
    WLListNode *lnode;
    WLListIterTmp tmp;

    if(OBJ_IS(reg, WClientWin))
        return TRUE;
    
    FOR_ALL_NODES_ON_LLIST(lnode, scr->mplex.mx_list, tmp){
        if(lnode->st->reg==reg)
            onmxlist=TRUE;
        else /*if(OBJ_IS(node->reg, WGenWS))*/
            return TRUE;
    }
    
    if(!onmxlist)
        return TRUE;
    
    warn(TR("Only workspace may not be destroyed."));
    
    return FALSE;
}


static bool screen_may_destroy(WScreen *scr)
{
    warn(TR("Screens may not be destroyed."));
    return FALSE;
}



void screen_set_managed_offset(WScreen *scr, const WRectangle *off)
{
    scr->managed_off=*off;
    mplex_fit_managed((WMPlex*)scr);
}


/*EXTL_DOC
 * Set offset of objects managed by the screen from actual screen geometry.
 * The table \var{offset} should contain the entries \code{x}, \code{y}, 
 * \code{w} and \code{h} indicating offsets of that component of screen 
 * geometry.
 */
EXTL_EXPORT_AS(WScreen, set_managed_offset)
bool screen_set_managed_offset_extl(WScreen *scr, ExtlTab offset)
{
    WRectangle g;
    
    if(!extl_table_to_rectangle(offset, &g))
        goto err;
    
    if(-g.w>=REGION_GEOM(scr).w)
        goto err;
    if(-g.h>=REGION_GEOM(scr).h)
        goto err;
    
    screen_set_managed_offset(scr, &g);
    
    return TRUE;
err:
    warn(TR("Invalid offset."));
    return FALSE;
}


WPHolder *screen_get_rescue_pholder_for(WScreen *scr, WRegion *mgd)
{
#warning "TODO: better special case handling for groups"
    
    return (WPHolder*)mplex_get_rescue_pholder_for(&(scr->mplex), mgd);
}

/*}}}*/


/*{{{ Save/load */


ExtlTab screen_get_configuration(WScreen *scr)
{
    return mplex_get_configuration(&scr->mplex);
}


static WRegion *do_create_initial(WWindow *parent, const WFitParams *fp, 
                                  WRegionLoadCreateFn *fn)
{
    return fn(parent, fp, extl_table_none());
}


static bool create_initial_ws(WScreen *scr)
{
    WRegion *reg=NULL;
    WMPlexAttachParams par=MPLEXATTACHPARAMS_INIT;
    ExtlTab lo=ioncore_get_layout("default");
    
    if(lo==extl_table_none()){
        reg=mplex_do_attach_new(&scr->mplex, &par,
                                (WRegionCreateFn*)create_groupws, NULL);
    }else{
        reg=mplex_attach_new_(&scr->mplex, &par, 0, lo);
        extl_unref_table(lo);
    }
    
    if(reg==NULL){
        warn(TR("Unable to create a workspace on screen %d."), scr->id);
        return FALSE;
    }
    
    return TRUE;
}


bool screen_init_layout(WScreen *scr, ExtlTab tab)
{
    char *name;
    ExtlTab substab, subtab;
    int n, i;

    if(tab==extl_table_none())
        return create_initial_ws(scr);
    
    mplex_load_contents(&scr->mplex, tab);
    
    return TRUE;
}

/*}}}*/


/*{{{ Dynamic function table and class implementation */


static DynFunTab screen_dynfuntab[]={
    {region_map, 
     screen_map},
    
    {region_unmap, 
     screen_unmap},
     
    {region_activated, 
     screen_activated},
     
    {region_inactivated, 
     screen_inactivated},
    
    {(DynFun*)region_managed_may_destroy,
     (DynFun*)screen_managed_may_destroy},

    {(DynFun*)region_may_destroy,
     (DynFun*)screen_may_destroy},

    {mplex_managed_changed, 
     screen_managed_changed},
    
    {region_managed_notify, 
     screen_managed_notify},
    
    {mplex_managed_geom, 
     screen_managed_geom},

    {(DynFun*)region_get_configuration,
     (DynFun*)screen_get_configuration},

    {(DynFun*)region_handle_drop, 
     (DynFun*)screen_handle_drop},

    {(DynFun*)region_fitrep,
     (DynFun*)screen_fitrep},

    {(DynFun*)region_get_rescue_pholder_for,
     (DynFun*)screen_get_rescue_pholder_for},
    
    END_DYNFUNTAB
};


EXTL_EXPORT
IMPLCLASS(WScreen, WMPlex, screen_deinit, screen_dynfuntab);


/*}}}*/
