/* string.c
 * Utilities to manipulate character data
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"

#include <stdio.h>

#include "definitions.h"

#ifdef HAVE_ASSERT_H
#include <assert.h>
#endif
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <ctype.h>

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#else
#include <winsock.h>
#endif

#include "avl.h"
#include "threads.h"
#include "icecast.h"
#include "icetypes.h"
#include "utility.h"
#include "ice_string.h"
#include "sock.h"
#include "log.h"
#include "logtime.h"
#include "memory.h"
#include "ice_string.h"

extern server_info_t info;
extern int running;
extern mutex_t authentication_mutex, library_mutex, sock_mutex;

const char const_null[] = "(null)";

char *
splitc (char *first, char *rest, const char divider)
{
	char *p;

	if (!rest)
	{
		write_log (LOG_DEFAULT, "WARNING: splitc called with NULL pointers");
		return NULL;
	}

	p = strchr(rest, divider);
	if (p == NULL) {
		if ((first != rest) && (first != NULL))
			first[0] = 0;

		return NULL;
	}

	*p = 0;
	if (first != NULL) strcpy(first, rest);
	if (first != rest) strcpy(rest, p + 1);

	return rest;
}

char *
clean_string(char *string)
{
	register unsigned i;

	i = 0;
	while (string[i] == ' ' && string[i] != '\0')
		i++;
	
	return &string[i];
}

void
flags2string(admin_t *adm, void *param)
{
	char fls[5] = "";
	if (adm->oper)
		strcat (fls, "O");
	if (adm->tailing)
		strcat (fls, "T");
	if (adm->status)
		strcat (fls, "S");
	if (param)
		sock_write (*(sock_t *)param, "%s", fls);
	else
		fd_write (info.statsfile, "%s", fls);
}

int
is_pattern (const char *string)
{
	if (strchr (string, '*'))
		return 1;
	else if (strchr (string, '?'))
		return 1;
	else if (strchr (string, '.'))
		return 1;
	return 0;
}

int
is_number (const char *string)
{
	int i, length;
	if ((length = ice_strlen(string)) == 0) {
		return 0;
	}
	
	for(i=0; i < length; i++) {
		if (isalpha((int)string[i]) != 0) 
			return 0;
	}
	return 1;
}

const char *
con_host (connection_t *con)
{
	static char null[5] = "null";

	if (!con)
	{
		write_log (LOG_DEFAULT, "WARNING: con_host called with NULL connection");
		return null;
	}

	if (con->hostname)
		return con->hostname;
	else if (con->host)
		return con->host;
	else
		return null;
}

char *
my_strdup (const char *string)
{
	const char *ptr = string;
	if (!string)
	{
		xa_debug (1, "DEBUG: my_strdup called with NULL pointer!");
		return NULL;
	}
	while (ptr && *ptr && *ptr == ' ')
		ptr++;
	return nstrdup (ptr);
}

char alphabet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

char *util_base64_encode(char *message)
{
	char *encoded;
	unsigned long length, encoded_length;
	unsigned long left, bitqueue, i = 0, j = 0;

	length = ice_strlen(message);

	if (length == 0) return NULL;

	encoded_length = (4 * (length + ((3 - (length % 3)) % 3)) / 3);
	encoded = (char *)nmalloc(encoded_length + 1);

	while (i < length) {
		left = length - i;

		if (left > 2) {
			bitqueue = message[i++];
			bitqueue = (bitqueue << 8) + message[i++];
			bitqueue = (bitqueue << 8) + message[i++];
			
			encoded[j++] = alphabet[(bitqueue & 0xFC0000) >> 18];
			encoded[j++] = alphabet[(bitqueue & 0x3F000) >> 12];
			encoded[j++] = alphabet[(bitqueue & 0xFC0) >> 6];
			encoded[j++] = alphabet[bitqueue & 0x3F];
		} else if (left == 2) {
			bitqueue = message[i++];
			bitqueue = (bitqueue << 8) + message[i++];
			bitqueue <<= 8;

			encoded[j++] = alphabet[(bitqueue & 0xFC0000) >> 18];
			encoded[j++] = alphabet[(bitqueue & 0x3F000) >> 12];
			encoded[j++] = alphabet[(bitqueue & 0xFC0) >> 6];
			encoded[j++] = '=';			
		} else {
			bitqueue = message[i++];
			bitqueue <<= 16;

			encoded[j++] = alphabet[(bitqueue & 0xFC0000) >> 18];
			encoded[j++] = alphabet[(bitqueue & 0x3F000) >> 12];
			encoded[j++] = '=';
			encoded[j++] = '=';
		}
	}

	return encoded;
}

char unalphabet(char alpha)
{
	if (alpha >= 'A' && alpha <= 'Z')
		return (alpha - 'A');
	else if (alpha >= 'a' && alpha <= 'z')
		return (alpha - 'a' + 26);
	else if (alpha >= '0' && alpha <= '9')
		return (alpha - '0' + 52);
	else if (alpha == '+')
		return 62;
	else if (alpha == '/')
		return 63;
	else if (alpha == '=')
		return 64;
	else 
		return 65;
}

char *
util_base64_decode(char *message)
{
	char *decoded, temp;
	long length, decoded_length;
	long bitqueue, pad, i = 0, j = 0;

	length = ice_strlen(message);

	if (((length % 4) != 0) || (length == 0)) return NULL;

	decoded_length = length / 4 * 3;

	if (message[length - 1] == '=') {
		decoded_length--;
		if (message[length - 2] == '=')
			decoded_length--;
	}

	decoded = (char *)nmalloc(decoded_length + 1);
	memset (decoded, 0, decoded_length + 1);

	while (i < length) {
		pad = 0;

		temp = unalphabet(message[i++]);
		if (temp == 64) {
			free(decoded);
			return NULL;
		}
		bitqueue = temp;

		temp = unalphabet(message[i++]);
		if (temp == 64) {
			free(decoded);
			return NULL;
		}
		bitqueue <<= 6;
		bitqueue += temp;

		temp = unalphabet(message[i++]);
		if (temp == 64) {
			if (i != length - 1) {
				free(decoded);
				return NULL;
			}
			temp = 0; pad++;
		}
		bitqueue <<= 6;
		bitqueue += temp;

		temp = unalphabet(message[i++]);
		if (pad == 1 && temp != 64) {
				free(decoded);
				return NULL;
		}
		
		if (temp == 64) {
			if (i != length) {
				free(decoded);
				return NULL;
			}
			temp = 0; pad++;
		}
		bitqueue <<= 6;
		bitqueue += temp;

		decoded[j++] = ((bitqueue & 0xFF0000) >> 16);
		if (pad < 2) {
			decoded[j++] = ((bitqueue & 0xFF00) >> 8);
			if (pad < 1)
				decoded[j++] = (bitqueue & 0xFF);
		}
	}

	return decoded;
}

char *
safe_strcat (char *dest, const char *src, int maxsize)
{
	int size = 0;
	if (!dest || !src)
		return dest;
	if (ice_strlen (dest) + ice_strlen (src) > maxsize)
	{
		size = maxsize - (ice_strlen (dest) + ice_strlen (src));
		return strncat (dest, src, size);
	} else 
		return strcat (dest, src);
}

char *
mutex_to_string (mutex_t *mutex, char *out)
{
	if (mutex == &info.source_mutex)
		strcpy (out, "Source Tree Mutex");
	else if (mutex == &info.relay_mutex)
		strcpy (out, "Relay Tree Mutex");
	else if (mutex == &info.admin_mutex)
		strcpy (out, "Admin Tree Mutex");
	else if (mutex == &info.directory_mutex)
		strcpy (out, "Directory Tree Mutex");
	else if (mutex == &info.alias_mutex)
		strcpy (out, "Alias Tree Mutex");
	else if (mutex == &info.misc_mutex)
		strcpy (out, "Misc. Mutex");
	else if (mutex == &info.mount_mutex)
		strcpy (out, "Mount Point Mutex");
	else if (mutex == &info.hostname_mutex)
		strcpy (out, "Hostname Tree Mutex");
	else if (mutex == &info.acl_mutex)
		strcpy (out, "ACL Tree Mutex");
	else if (mutex == &info.double_mutex)
		strcpy (out, "Double Mutex Mutex");
	else if (mutex == &info.thread_mutex)
		strcpy (out, "Thread Tree Mutex");
#ifdef DEBUG_MEMORY
	else if (mutex == &info.memory_mutex)
		strcpy (out, "Memory Tree Mutex");
#endif
#ifdef DEBUG_SOCKETS
	else if (mutex == &sock_mutex) {
		strcpy (out, "Socket Tree Mutex");
	}
#endif
	else if (mutex == &info.resolvmutex)
		strcpy (out, "DNS Lookup Mutex");
	else if (mutex == &library_mutex)
		strcpy (out, "Library Mutex");
	else if (mutex == &authentication_mutex)
		strcpy (out, "Authentication Mutex");
	else 
		strcpy (out, "Unknown Mutex (probably source)");


	return out;
}

const char *
skip_before (const char *ptr, const char *search)
{
  return (const char *)strstr(ptr, search);
}

const char *
skip_after (const char *ptr, const char *search)
{
	char *hit = strstr (ptr, search);
	if (hit)
		return (const char *)hit + ice_strlen (search);
	else
		return NULL;
}

char *create_malloced_ascii_host (struct in_addr *in)
{
	char *buf = (char *)nmalloc(20);

	if (!in) {
		xa_debug(1, "ERROR: Dammit, don't send NULL's to create_malloced_ascii_host()");
		return NULL;
	}

	return makeasciihost(in, buf);
}

char *makeasciihost(const struct in_addr *in, char *buf)
{
	if (!buf) {
		write_log(LOG_DEFAULT, "ERROR: makeasciihost called with NULL arguments");
		return NULL;
	}
  
#ifdef HAVE_INET_NTOA

	strcpy(buf, inet_ntoa(*in));

#else

	unsigned char *s = (unsigned char *)in;
	int a, b, c, d;
	a = (int)*s++;
	b = (int)*s++;
	c = (int)*s++;
	d = (int)*s;

	sprintf(buf, "%d.%d.%d.%d", a, b, c, d);

#endif

	return buf;
}

char *
nice_time_minutes (unsigned long int minutes, char *buf)
{
	unsigned long int days, hours, remains;
	char buf2[BUFSIZE];
	
	if (!buf)
	{
		write_log (LOG_DEFAULT, "ERROR: nice_time_minutes called with NULL argument");
		return NULL;
	}

	buf[0] = '\0';

	days = minutes / 1440;
	remains = minutes % 1440;
	hours = remains / 60;
	remains = remains % 60;

	if (days > 0)
		sprintf(buf, "%lu days, %lu hours, %lu minutes", days, hours, remains);
	else if (hours > 0)
		sprintf(buf, "%lu hours, %lu minutes", hours, remains);
	else
	{
		sprintf(buf, "%lu minutes", remains);
		return buf;
	}

	if (remains > 0)
	{
		sprintf (buf2, " and %lu minutes", remains);
		strcat (buf, buf2);
	}

	return buf;
}

char *
nice_time (unsigned long int seconds, char *buf)
{
	unsigned long int days, hours, minutes, nseconds, remains;
	char buf2[BUFSIZE];
	
	if (!buf)
	{
		write_log (LOG_DEFAULT, "ERROR: nice_time called with NULL argument");
		return NULL;
	}

	buf[0] = '\0';

	days = seconds / 86400;
	remains = seconds % 86400;
	hours = remains / 3600;
	remains = remains % 3600;
	minutes = remains / 60;
	nseconds = remains % 60;
	if (days > 0)
		sprintf(buf, "%lu days, %lu hours, %lu minutes", days, hours, minutes);
	else if (hours > 0)
		sprintf(buf, "%lu hours, %lu minutes", hours, minutes);
	else if (minutes > 0)
		sprintf(buf, "%lu minutes", minutes);
	else
	{
		/* Only seconds */
		sprintf(buf, "%lu seconds", nseconds);
		return buf;
	}

	if (nseconds > 0)
	{
		sprintf (buf2, " and %lu seconds", nseconds);
		strcat (buf, buf2);
	}

	return buf;
}

char *
ice_sprintf (const char *template, const char *arg)
{
	char *ptr;

	if (!template)
	{
		if (!arg)
			return nstrdup ("(null)");
		return nstrdup (arg);
	}
		
	if (!arg)
		return nstrdup ("(null)");
	
	if (strchr (template, '%') == NULL)
		return nstrdup (template);

	ptr = (char *) nmalloc (ice_strlen (arg) + ice_strlen (template) + 2);

	sprintf (ptr, template, arg);
	return ptr;
}

size_t
ice_strlen (const char *string)
{
	if (!string)
	{
		xa_debug (1, "ERROR: ice_strlen() called with NULL pointer!");
		return 0;
	}
	return strlen (string);
}

int
ice_strcmp (const char *s1, const char *s2)
{
	if (!s1 || !s2)
	{
		xa_debug (1, "ERROR: ice_strcmp() called with NULL pointers!");
		return 0;
	}
	return strcmp (s1, s2);
}

int
ice_strncmp (const char *s1, const char *s2, size_t n)
{
	if (!s1 || !s2)
	{
		xa_debug (1, "ERROR: ice_strncmp() called with NULL pointers!");
		return 0;
	}
	return strncmp (s1, s2, n);
}

int
ice_strcasecmp (const char *s1, const char *s2)
{
	if (!s1 || !s2)
	{
		xa_debug (1, "ERROR: ice_strcasecmp() called with NULL pointers");
		return 0;
	}
#ifdef _WIN32
	return stricmp (s1, s2);
#else
	return strcasecmp (s1, s2);
#endif
}

const char *
nullcheck_string (const char *string)
{
	if (!string)
		return const_null;
	return string;
}

void 
catsprintf (char *line, const char *fmt, ...)
{
	char buff[BUFSIZE];
	va_list ap;
	
	va_start (ap, fmt);
	
#ifdef HAVE_VSNPRINTF
	vsnprintf(buff, BUFSIZE, fmt, ap);
#else
	vsprintf(buff, fmt, ap);
#endif
	
	va_end(ap);

	strcat(line, buff);
}



