#include "hypermail.h"
#include "setup.h"
#include "print.h"
#include "printfile.h"

/*
** printfile - print html header/footer file and fill in values 
**             substituting for magic cookies. Return allocated
**             memory with the header/footer filled in.
**
**      Substitution cookies supported
**
**              %% - '%' character
**              %~ - storage directory
**              %a - Other Archives URL
**              %b - About Archive URL
**              %c - Charset META TAG - Not valid on index pages
**              %e - email addr of message author - Not valid on index pages
**              %f - file name of the HTML document
**              %g - date and time archive generated
**              %h - HMURL
**              %i - Message-id - Not valid on index pages
**              %l - archive label
**              %m - Mailto address
**              %p - PROGNAME
**              %s - Subject of message or Index Title
**              %t - path to top directory ("" if no folders; usually "../",
**                                          sometimes "../../" with folders)
**              %v - VERSION
**              %u - Expanded version link (HMURL,PROGNAME,VERSION)
**              %S - Subject META TAG - Not valid on index pages
**              %A - Author META TAG - Not valid on index pages
**              %D - Date META TAG - Not valid on index pages
**              \n - newline character
**              \t - tab character
**
*/

int printfile(FILE *fp, char *format, char *label, char *subject,
	      char *dir, char *name, char *email, char *message_id,
	      char *charset, char *date, char *filename)
{
    register char *cp;
    register char *aptr;
    char c;
    char *ptr;

    aptr = format;

    while ((c = *aptr++)) {
	if (c == '\\') {
	    switch (*aptr++) {
	    case 'n':		/* Add the \n character */
		putc('\n', fp);
		continue;
	    case 't':		/* Add the \t character */
		putc('\t', fp);
		continue;
	    default:
		break;
	    }			/* end switch */
	}
	else if (c == '%') {
            char nextone = *aptr++;
	    switch (nextone) {
	    case '%':		/* Add the % character */
		putc('%', fp);
		continue;
	    case '~':		/* %~ - storage directory */
		for (cp = dir; *cp; cp++)
		    putc(*cp, fp);
		continue;
	    case 'A':		/* %e - email address of message author */
		if (email && name) {
		    fprintf(fp,
			    "<meta name=\"Author\" content=\"%s (%s)\">",
			    name, email);
		}
		continue;
	    case 'a':		/* %a - Other Archives URL */
		if (set_archives) {
		    for (cp = set_archives; *cp; cp++)
			putc(*cp, fp);
		}
		continue;
	    case 'B':
	      printf("Warning: the %%B option has been disabled. Use a\n"
		     "style sheet instead. See the INSTALL file for more info.\n");
	      continue;
	    case 'b':		/* %b - About this archive URL */
		if (set_about) {
		    for (cp = set_about; *cp; cp++)
			putc(*cp, fp);
		}
		continue;
	    case 'c':
		if (charset && *charset) {
		    /* only output this if we have a charset */
		    fprintf(fp, "<meta http-equiv=\"Content-Type\""
			    " content=\"text/html; charset=%s\">\n",
			    charset);
		}
		continue;

	    case 'D':		/* %D - date of message */
		if (date) {
		    fprintf(fp,
			    "<meta name=\"Date\" content=\"%s\">",
			    date);
		}
		continue;
	    case 'e':		/* %e - email address of message author */
		if (email) {
		    for (cp = email; *cp; cp++)
			putc(*cp, fp);
		}
		continue;
	    case 'f':		/* %f - file name */
		if (filename) {
		    for (cp = filename; *cp; cp++)
			putc(*cp, fp);
		}
		continue;
	    case 'g':		/* %g - date and time archive generated */
		for (cp = getlocaltime(); *cp; cp++)
		    putc(*cp, fp);
		continue;
	    case 'h':		/* %h - Hypermail Resource Center */
		for (cp = HMURL; *cp; cp++)
		    putc(*cp, fp);
		continue;
	    case 'i':		/* %m - Message-ID of message */
		if (message_id) {
		    for (cp = message_id; *cp; cp++)
			putc(*cp, fp);
		}
		continue;
	    case 'l':		/* %l - Archive label  */
		for (cp = label; *cp; cp++)
		    putc(*cp, fp);
		continue;
	    case 'm':		/* %m - mailto */
		if (set_mailto) {
		    for (cp = set_mailto; *cp; cp++)
			putc(*cp, fp);
		}
		continue;
	    case 'p':		/* %p - PROGNAME */
		for (cp = PROGNAME; *cp; cp++)
		    putc(*cp, fp);
		continue;
	    case 's':		/* %s - Subject of message or Index Title */
		for (ptr = cp = convchars(subject); *cp; cp++)
		    putc(*cp, fp);
		free(ptr);
		continue;
	    case 'S':		/* %s - Subject of message or Index Title */
		fprintf(fp, "<meta name=\"Subject\" content=\"%s\">",
			cp = convchars(subject));
		free(cp);
		continue;
	    case 't':
	      {
		struct emailinfo *ep;
		if(hashnumlookup(0, &ep))
		    fprintf(fp, ep->subdir ? ep->subdir->rel_path_to_top : "");
		continue;
	      }
	    case 'v':		/* %v - VERSION */
		for (cp = VERSION; *cp; cp++)
		    putc(*cp, fp);
		continue;
	    case 'u':		/* %u - Expanded Version link */
		fprintf(fp, "<a href=\"%s\">%s %s</a>",
			HMURL, PROGNAME, VERSION);
		continue;
	    default:
		putc('%', fp);
		putc(nextone, fp);
		continue;
	    }			/* end switch */
	}
	putc(c, fp);
    }				/* end while */

    fflush(fp);
    return (0);
}

/*
** Prints the standard page header 
*/

void print_main_header(FILE *fp, bool index_header, char *label, char *name,
		       char *email, char *subject, char *charset,
		       char *date, char *filename)
{
    char *title;
    char *rp;

    fprintf(fp,
	    "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"\n"
            "                      "
	    "\"http://www.w3.org/TR/html4/strict.dtd\">\n");
    fprintf(fp, "<html lang=\"%s\">\n", set_language);
    fprintf(fp, "<head>\n");

    if (charset && *charset) {
	/* charset info "as early as possible within the HEAD of the document"
	 */
	fprintf(fp, "<meta http-equiv=\"Content-Type\""
		" content=\"text/html; charset=%s\">\n", charset);
    }
    fprintf(fp, "<meta name=\"generator\" content=\"%s %s, see %s\">\n",
                PROGNAME, VERSION, HMURL);

    /* 
     * Strip off any trailing whitespace in TITLE so weblint is happy. 
     */

    trio_asprintf(&title, "%s: %s", label, rp = convchars(subject));
    free(rp);

    rp = title + (strlen(title) - 1);
    while (isspace(*rp))
	*rp-- = '\0';

    fprintf(fp, "<title>%s</title>\n", title);
    free(title);

    if (name && email)
	fprintf(fp, "<meta name=\"Author\" content=\"%s (%s)\">\n",name,email);
    fprintf(fp, "<meta name=\"Subject\" content=\"%s\">\n", rp =
	    convchars(subject));
	fprintf(fp, "<meta name=\"Date\" content=\"%s\">\n",date);
    free(rp);
    if (use_mailto)
	fprintf(fp, "<link rev=\"made\" href=\"mailto:%s\">\n", set_mailto);

    /* print the css url according to the type of header */
    if (index_header && set_icss_url && *set_icss_url) {
      fprintf(fp, "<link rel=\"stylesheet\" href=\"%s\" type=\"text/css\">\n",
              set_icss_url);

    } else if (!index_header && set_mcss_url && *set_mcss_url) {
      fprintf(fp, "<link rel=\"stylesheet\" href=\"%s\" type=\"text/css\">\n",
              set_mcss_url);

    } else {
      /*
       * if style sheets are not specified, emit a default one.
       */
      fprintf(fp, "<style type=\"text/css\">\n");
      fprintf(fp, "body {color: black; background: #ffffff}\n");
      fprintf(fp, "h1.center {text-align: center}\n");
      fprintf(fp, "div.center {text-align: center}\n");
      fprintf(fp, "</style>\n");
    }

    fprintf(fp, "</head>\n");
    fprintf(fp, "<body>\n");
}

/*
** Prints the header for an index page.
*/

void print_msg_header(FILE *fp, char *label, char *subject,
		      char *dir, char *name, char *email, char *msgid,
		      char *charset, time_t date, char *filename)
{
    char *ptr;
    if (mhtmlheaderfile)
	printfile(fp, mhtmlheaderfile, set_label, subject, set_dir, name, 
		  email, msgid, charset, secs_to_iso_meta(date), filename);
    else {
	print_main_header(fp, FALSE, set_label, name, email, subject,
			  charset, secs_to_iso_meta(date), filename);
#if 0 /* JK modified this */       
	fprintf(fp, "<h1 class=\"center\">%s</h1>\n",
		ptr = convchars(subject));
	free(ptr);
#endif
	fprintf(fp, "<h1>%s</h1>\n",
		ptr = convchars(subject));
	free(ptr);
#if 0 /* JK: and removed this as it looked a bit strange */
	if (!set_usetable)
	    fprintf(fp, "<hr>\n<p>\n");
#endif
    }
}

/*
** Prints the header for the by_date index page.
*/

void print_index_header(FILE *fp, char *label, char *dir, char *subject,
			char *filename)
{
    if (ihtmlheaderfile)
	printfile(fp, ihtmlheaderfile, label, subject, dir, NULL, NULL,
		  NULL, NULL, NULL, filename);
    else {
	print_main_header(fp, TRUE, label, NULL, NULL, subject, NULL, NULL, NULL);
	fprintf(fp, "<h1 class=\"center\">%s<br>%s</h1>\n", label, subject);
#if 0 /*@@ JK: removed it */	
	if (!set_usetable)
	    fprintf(fp, "<hr>\n");
#endif
    }
}

/*
** Prints the HTML page footer.
*/

void printfooter(FILE *fp, char *htmlfooter, char *label, char *dir,
		 char *subject, char *filename)
{
    printcomment(fp, "trailer", "footer");

    if (htmlfooter)
	printfile(fp, htmlfooter, label, subject,
		  dir, NULL, NULL, NULL, NULL, NULL, filename);
    else {
	if (set_showhr && !set_usetable)
	    fprintf(fp, "<hr>\n");
	fprintf(fp, "<p><small><em>\n");
	fprintf(fp, "%s ", lang[MSG_ARCHIVE_GENERATED_BY]);
	fprintf(fp, "<a href=\"%s\">%s %s</a> \n", HMURL, PROGNAME, VERSION);
	fprintf(fp, ": %s\n", getlocaltime());
	fprintf(fp, "</em></small></p>\n");
	fprintf(fp, "</body>\n</html>\n");
    }
}
