<?php
// $Horde: horde/lib/Text.php,v 1.6.2.5 2002/04/17 18:23:10 jan Exp $

/**
 * The Text:: class provides common methods for manipulating text.
 *
 * @author  Jon Parise <jon@horde.org>
 * @version $Revision: 1.6.2.5 $
 * @since   Horde 1.3
 * @package horde
 */
class Text {

    /**
     * Filter the given text based on the words found in $words.
     *
     * @param string $text         The text to filter.
     * @param string $words_file   Filename containing the words to replace.
     * @param string $replacement  The replacement string.
     *
     * @return string The filtered version of $text.
     */
    function filter($text, $words_file, $replacement)
    {
        if (@is_readable($words_file)) {
            /* Read the file and iterate through the lines. */
            $lines = file($words_file);
            foreach ($lines as $line) {
                /* Strip whitespace and comments. */
                $line = trim($line);
                $line = preg_replace('|#.*$|', '', $line);

                /* Filter the text. */
                if (!empty($line)) {
                    $text = preg_replace("/(\b(\w*)$line\b|\b$line(\w*)\b)/i",
                                         $replacement, $text);
                }
            }
        }

        return $text;
    }

    /**
     * Wraps the text of a message.
     *
     * @param string $text         String containing the text to wrap.
     * @param int    $length       Wrap $text at this number of characters.
     * @param string $break_char   Character to use when breaking lines.
     *
     * @return  string      String containing the wrapped text.
     */
    function wrap($text, $length = 80, $break_char = "\n")
    {
        $paragraphs = explode("\n", $text);
        for ($i = 0; $i < count($paragraphs); $i++) {
            $paragraphs[$i] = wordwrap($paragraphs[$i], $length, $break_char);
        }
        return implode($break_char, $paragraphs);
    }

    /**
     * Turns all URLs in the text into hyperlinks.
     *
     * @param string $text               The text to be transformed.
     * @param optional boolean $capital  Sometimes it's useful to generate <A and </A>
     *                                   so you can know which tags you just generated.
     *
     * @return string              The linked text.
     */
    function linkUrls($text, $capital = false, $class = '')
    {
        if ($capital) {
            $a = 'A';
            $text = str_replace('</A>', '</a>', $text); // make sure that the original message doesn't contain any capital </A> tags, so we can assume we generated them
            $text = str_replace('<A', '<a', $text);     // ditto for open <A> tags
        } else {
            $a = 'a';
        }
        if (!empty($class)) {
            $class = ' class="' . $class . '"';
        }
        return preg_replace('|(\w+)://([^\s"<]*[\w+#?/&=])|', '<' . $a . ' href="\1://\2" target="_blank"' . $class . '>\1://\2</' . $a . '>', $text);
    }

    /**
     * Re-convert links generated by Text::linkUrls() to working
     * hrefs, after htmlspecialchars() has been called on the
     * text. This is an awkward chain, but necessary to filter out
     * other HTML.
     *
     * @param          string $text    The text to convert.
     * @param optional string $target  The link target.
     *
     * @return  The converted text.
     */
    function enableCapitalLinks($text, $target = '_blank')
    {
        $text = str_replace('&lt;A href=&quot;', '<a class="fixed" href="', $text);
        $text = str_replace('&quot; target=&quot;_blank&quot;&gt;', '" target="' . $target . '">', $text);
        $text = str_replace('&quot;&gt;','">', $text);
        $text = str_replace('&lt;/A&gt;', '</a>', $text); // only reconvert capital /A tags - the ones we generated

        return $text;
    }

    /**
     * Replace occurences of %VAR% with VAR, if VAR exists in the
     * webserver's environment. Ignores all text after a # character
     * (shell-style comments).
     *
     * @param string $text	The text to expand.
     * @return string		The expanded text.
     */
    function expandEnvironment($text)
    {
        if (preg_match("|([^#]*)#.*|", $text, $regs)) {
            $text = $regs[1];

            if (strlen($text) > 0) {
                $text = $text . "\n";
            }
        }

        while (preg_match("|%([A-Za-z_]+)%|", $text, $regs)) {
            $text = preg_replace("|%([A-Za-z_]+)%|", getenv($regs[1]), $text);
        }
        return $text;
    }

    /**
     * Convert a line of text to display properly in HTML.
     *
     * @param string $text  The string of text to convert.
     *
     * @return string       The HTML-compliant converted text.
     */
    function htmlSpaces($text = '')
    {
        $text = htmlentities($text);
        $text = str_replace("\t", '&nbsp; &nbsp; &nbsp; &nbsp; ', $text);
        $text = str_replace('  ', '&nbsp; ', $text);
        $text = str_replace('  ', ' &nbsp;', $text);

        return $text;
    }

    /**
     * Same as htmlSpaces() but converts all spaces to &nbsp;
     * @see htmlSpaces()
     *
     * @param string $text  The string of text to convert.
     *
     * @return string       The HTML-compliant converted text.
     */
    function htmlAllSpaces($text = '')
    {
        $text = Text::htmlSpaces($text);
        $text = str_replace(' ', '&nbsp;', $text);

        return $text;
    }

    /**
     * Removes some common entities and high-ascii or otherwise
     * nonstandard characters common in text pasted from Microsoft
     * Word into a browser.
     *
     * @param string $text  The text to be cleaned.
     *
     * @return string       The cleaned text.
     */
    function cleanEntities($text)
    {
        // The '' entry may look wrong, depending on your editor,
        // but it's not - that's not really a single quote.
        $from = array('', '', '', '', '', '', '', '', '', '', chr(167), '&#61479;', '&#61572;', '&#61594;', '&#61640;', '&#61623;', '&#61607;');
        $to   = array('...',     "'", "'",    '"',    '"',    '*',    '-',    '-',    '*', '*',      '*',        '.',        '*',        '*',        '-',        '-',        '*');

        return str_replace($from, $to, $text);
    }

}
?>
