/*
 * Copyright 2003 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _TEMPLATE_
#define _TEMPLATE_

#include <assert.h>

struct personality;
struct subsystem;
struct ip_hdr;

struct subsystem_container {
	TAILQ_ENTRY(subsystem_container) next;

	struct subsystem *sub;
};


struct condition;
struct template {
	SPLAY_ENTRY(template) node;
	SPLAY_ENTRY(template) sub_node;		/* for subsystem sharing */
	TAILQ_ENTRY(template) next;		/* for subsystem sharing */

	char *name;

	struct porttree ports;

	struct action icmp;
	struct action tcp;
	struct action udp;

	struct personality *person;

	int id;
	uint32_t seq;
	int seqcalls;
	uint32_t timestamp;
	struct timeval tv;

	uint16_t drop_inrate;
	uint16_t drop_synrate;

	uid_t uid;
	gid_t gid;

	TAILQ_HEAD(subsysqueue, subsystem_container) subsystems;

	/* Condition on which this template is activated */
	TAILQ_HEAD(conditionqueue, condition) dynamic;
	
	/* Special handling for templates */
	int flags;
	struct interface *inter;

	/* Reference counter */
	uint16_t refcnt;
};

#define TEMPLATE_EXTERNAL	0x0001	/* Real machine on external network */
#define TEMPLATE_DYNAMIC	0x0002	/* Pointer to templates */

struct template *template_create(char *);
int template_add(struct template *, int, int, struct action *);
int template_subsystem(struct template *, char *, int, int);
struct template *template_clone(char *, struct template *, int);
struct template *template_find(char *);
struct template *template_find_best(char *, struct ip_hdr *, u_short);

int template_insert_dynamic(struct template *, struct template *,
    struct condition *);

void template_free_all(void);
void template_subsystem_free(struct subsystem *);

int templ_compare(struct template *, struct template *);

void template_deallocate(struct template *);

#define template_free(x)	do {					\
	if ((x) == NULL)						\
		break;							\
	/* Decrease ref counter */					\
	(x)->refcnt--;							\
	if ((x)->refcnt <= 0)						\
		template_deallocate(x);					\
} while (0)

static __inline struct template *
template_ref(struct template *tmpl)
{
	if (tmpl != NULL) {
		tmpl->refcnt++;
		assert(tmpl->refcnt);
	}
	return (tmpl);
}

#endif /* _TEMPLATE_ */
