/***************************************************************************
                          languagedefinition.h  -  description
                             -------------------
    begin                : Wed Nov 28 2001
    copyright            : (C) 2001-2007 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef LANGUAGEDEFINITION_H
#define LANGUAGEDEFINITION_H

#include <string>
#include <map>
#include <iostream>
#include <fstream>
#include <iterator>
#include <sstream>

#include "configurationreader.h"
#include "platform_fs.h"
#include "enums.h"
#include "re/Pattern.h"

namespace highlight {

class RegexElement;

/** maps keywords and the corresponding class IDs*/
typedef map <string, int> KeywordMap;


/**\brief Contains specific data of the programming language being processed.

   The load() method will only read a new language definition if the given
   file path is not equal to the path of the current language definition.

* @author Andre  Simon
*/

class LanguageDefinition  {

  public:

    LanguageDefinition();

    ~LanguageDefinition();

    /**\return Symbol string, containg all known symbols with the referencing state ids*/
    string &getSymbolString();

    /** \return Prefix of raw strings */
    unsigned char getRawStringPrefix();

   /** \return Continuation Character */
    unsigned char getContinuationChar();

    // /** \return List of characters allowed within identifiers */
    //string &getAllowedChars();

    /** \return true if syntax highlighting is enabled*/
    bool getSyntaxHighlight();

    /** \return True if language is case sensitive */
    bool isIgnoreCase();

    /** \param s String
         \return class id of keyword, 0 if s is not a keyword */
    int isKeyword(const string &s);

    /** Load new language definition
        \param langDefPath Path of language definition
        \param clear Test if former data should be deleted
        \return True if successfull  */
    bool load(const string& langDefPath, bool clear=true);

     /** \return  True if programming language is VHDL */
    bool isVHDL();

    /** \return True if multi line comments may be nested */
    bool allowNestedMLComments();

    /** \return True if highlighting is disabled */
    bool highlightingDisabled();

    /** \return True the next load() call will load a new language definition
        \param  langDefPath Path to language definition  */
    bool needsReload(const string &langDefPath);

    /** \return True if current language may be reformatted (c, c++, c#, java) */
    bool enableReformatting();

    /** \return True if escape sequences are allowed outsde of strings */
    bool allowExtEscSeq();

    /** \return Class ID of given keyword delimiter prefix
        \param prefix Keyword delimiter prefix   */
    unsigned int getDelimPrefixClassID(const string& prefix);

    /** \return keywords*/
    const KeywordMap& getKeywords() const;

    /** \return keyword classes*/
    const vector<string>& getKeywordClasses() const;

    /** \return regular expressions */
     vector<RegexElement*>& getRegexElements()  {return regex;};

    /** \return description of the programming language */
    const string & getDescription () {return langDesc;}

  private:
    // string containing symbols and their IDs of the programming language
    string symbolString;

    // path to laoded language definition
    string currentPath;

    // Language description
    string langDesc;

    KeywordMap keywords;

    vector <string> keywordClasses;

    vector <RegexElement*> regex;

    KeywordMap delimiterPrefixes;

    // keywords are not case sensitive if set
    bool ignoreCase,

    // highlighting is disabled
    disableHighlighting,

    // Escape sequences are allowed outrside of strings
    allowExtEscape,

    // switch to enable VHDL workarounds
    vhdlMode,

    // allow nested multi line comment blocks
    allowNestedComments,

    // single line comments have to start in coloumn 1 if set
    fullLineComment,

    // code formatting is enabled if set
    reformatCode;

    // character which is prefix of raw string (c#)
    unsigned char rawStringPrefix,

   //character which continues curreent style on next line
                  continuationChar;

    /** setzt Membervariablen auf Defaultwerte */
    void reset();

    // add a symbol sequencs to the symbolStream
    void addSimpleSymbol(stringstream& symbolStream, State state,
                         const string& paramValue );

    void addSymbol(stringstream& symbolStream,
                                       State stateBegin,
                                       State stateEnd,
                                       bool isDelimiter,
                                       const string& paramValue,
                                       unsigned int classID=0 );

    // add a delimiter symbol sequencs to the symbolStream
    void addDelimiterSymbol(stringstream& symbolStream,
                            State stateBegin, State stateEnd,
                            const string& paramValue,
                            unsigned int classID=0);

    bool getFlag( string& paramValue);

    unsigned char getSymbol(const string& paramValue);

    // generate a unique class ID of the class name
    unsigned int generateNewKWClass(const string& newClassName);

    // add keywords to the given class
    void addKeywords(const string &kwList,State stateBegin, State stateEnd, int classID);

    string extractRE(const string &paramValue);

  };


  class RegexElement {
    public:
    RegexElement():open(STANDARD), end(STANDARD), rePattern(NULL), kwClass(0)
        {
        }

        RegexElement(State oState, State eState,
                     Pattern *re, unsigned int cID=0):open(oState), end(eState), rePattern(re), kwClass(cID)
        {
       // cerr << "new re element "<<  rePattern->getPattern() <<" open: "<<open<<" end "<<end<<"\n";
        }

        ~RegexElement() { delete rePattern; }

        State open, end;
        Pattern *rePattern;
        unsigned int kwClass;
  };

}
#endif
