/***************************************************************************
                          languagedefinition.cpp  -  description
                             -------------------
    begin                : Wed Nov 28 2001
    copyright            : (C) 2001-2007 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "languagedefinition.h"

#define IDENTIFIER_RE "[a-zA-Z_]\\w*"

#define NUMBER_RE     "(?:0x|0X)[0-9a-fA-F]+|\\d*[\\.]?\\d+(?:[eE][\\-\\+]\\d+)?[lLuU]?"


using namespace std;

namespace highlight {

LanguageDefinition::LanguageDefinition():
    ignoreCase(false),
    disableHighlighting(false),
    allowExtEscape(false),
    vhdlMode(false),
    allowNestedComments(true),
    reformatCode(false)
{}

LanguageDefinition::~LanguageDefinition(){
  vector<RegexElement*>::iterator it;
  for (it=regex.begin(); it!=regex.end();it++){
    delete *it;
  }
}

int LanguageDefinition::isKeyword(const string &s)
{
  return (s.length() && keywords.count(s)) ? keywords[s] : 0;
}

void LanguageDefinition::addSymbol(stringstream& symbolStream,
                                   State stateBegin,
                                   State stateEnd,
                                   bool isDelimiter,
                                   const string& paramValue,
                                   unsigned int classID ) {
  string re = extractRE(paramValue);
  if (!re.empty() ){
    Pattern* p = Pattern::compile(re);
    if (p!=NULL)
      regex.push_back(new RegexElement(stateBegin,stateEnd, p));
    else
      cerr << "highlight: Regex failure ( "<< re <<" )\n";
    return;
  }
  if (isDelimiter) {
      addDelimiterSymbol(symbolStream, stateBegin, stateEnd, paramValue, classID);
  } else {
      addSimpleSymbol( symbolStream, stateBegin,paramValue );
  }
}

string LanguageDefinition::extractRE(const string &paramValue){
    if (paramValue.find("regex") != string::npos ){
        return StringTools::getParantheseVal(paramValue);
    }
    return string();
}

void LanguageDefinition::addSimpleSymbol(stringstream& symbolStream,
                                        State state,
                                        const string& paramValue ) {
  istringstream valueStream(paramValue);
  bool valExists=false;
  string value;
   while (valueStream >> value)
    {
      symbolStream << " " << value;
      valExists = true;
    }
  if (valExists)
    {
      symbolStream << " " << state;
    }
}

void LanguageDefinition::addDelimiterSymbol(stringstream& symbolStream,
                                        State stateBegin, State stateEnd,
                                        const string& paramValue,
                                        unsigned int classID) {
  istringstream valueStream(paramValue);
  string delimPrefix, delimSuffix;
  while (valueStream>>delimPrefix){
    valueStream >> delimSuffix;
    symbolStream << " "<<delimPrefix <<" " << stateBegin;
    symbolStream <<" "<< delimSuffix<<" "<< stateEnd;
    delimiterPrefixes.insert(make_pair(delimPrefix, classID));
  };
}

bool LanguageDefinition::getFlag( string& paramValue){
      return (StringTools::lowerCase(paramValue)=="true");
}

unsigned char LanguageDefinition::getSymbol(const string& paramValue){
      istringstream valueStream(paramValue);
      unsigned char symbol;
      valueStream >> symbol;
      return symbol;
}

void LanguageDefinition::addKeywords(const string &kwList,
                                     State stateBegin, State stateEnd,
                                     int classID){
  string re = extractRE(kwList);
  if (!re.empty()){
      Pattern* p = Pattern::compile(re);
      if (p!=NULL)
        regex.push_back(new RegexElement(stateBegin,stateEnd, p, classID));
      else
        cerr << "highlight: Regex compilation failure ("<< re <<")\n";
     return;
  }
  istringstream valueStream(kwList);
  string keyword;
  while (valueStream >> keyword){
    keywords.insert(make_pair(keyword, classID));
  }
}

unsigned int LanguageDefinition::generateNewKWClass(const string& newClassName){
  unsigned int newClassID=0;
  bool found=false;
  while (newClassID<keywordClasses.size() && !found){
   found= (newClassName==keywordClasses[newClassID++]);
  }
  if (!found){
   newClassID++;
   keywordClasses.push_back(newClassName);
  }
  return newClassID;
}

unsigned int LanguageDefinition::getDelimPrefixClassID(const string& prefix){
  if (delimiterPrefixes.count(prefix)){
    return delimiterPrefixes[prefix];
  }
  return 0;
}

bool LanguageDefinition::load(const string& langDefPath, bool clear)
{
  if (clear)  reset();

  ConfigurationReader langDef(langDefPath);
  if (langDef.found()) {
      currentPath=langDefPath;
      disableHighlighting=false;
      string token;
      stringstream symbolStrStream;

      addSymbol(symbolStrStream,
                STRING,
                STRING_END,
                false,
                langDef.getParameter("stringdelimiters") );

      addSymbol(symbolStrStream,
                STRING,
                STRING_END,
                true,
                langDef.getParameter("string_unequal") );

      addSymbol(symbolStrStream,
                DIRECTIVE,
                DIRECTIVE_END,
                false,
                langDef.getParameter("directive") );

      addSymbol(symbolStrStream,
                ESC_CHAR,
                ESC_CHAR_END,
                false,
                langDef.getParameter("escchar") );

      addSymbol(symbolStrStream,
                SL_COMMENT,
                SL_COMMENT_END,
                false,
                langDef.getParameter("sl_comment") );

      addSymbol(symbolStrStream,
                ML_COMMENT,
                ML_COMMENT_END,
                true,
                langDef.getParameter("ml_comment") );

      addSymbol(symbolStrStream,
                SYMBOL,
                SYMBOL_END,
                false,
                langDef.getParameter("symbols") );

      string paramName, className, classValue;
      vector<string> paramNames=langDef.getParameterNames();
      for (unsigned int i=0;i<paramNames.size();i++){
         paramName=paramNames[i];
         className=StringTools::getParantheseVal(paramName);
         classValue=langDef.getParameter(paramName);
         if (paramName.find("kw_list") != string::npos ){
           addKeywords(classValue, KEYWORD, KEYWORD_END, generateNewKWClass(className));
         }
         if (paramName.find("kw_re") != string::npos ){
            addKeywords(classValue,  KEYWORD, KEYWORD_END, generateNewKWClass(className));
         }
         if (paramName.find("tag_delim") != string::npos ) {
	    addSymbol(symbolStrStream, TAG_BEGIN, TAG_END, true,
                      classValue, generateNewKWClass(className));
        }
      }

      // use hardcoded regex if not defined in language definition
      string user_def_re = extractRE(langDef.getParameter("digit"));
      string re_digit = (user_def_re.empty()) ? NUMBER_RE : user_def_re;

      user_def_re = extractRE(langDef.getParameter("identifier"));
      string re_identifier= (user_def_re.empty()) ? IDENTIFIER_RE : user_def_re;

      // insert identifier and number regex after keyword regexes

      regex.push_back(new RegexElement(IDENTIFIER_BEGIN, IDENTIFIER_END,
                      Pattern::compile(re_identifier)));
      regex.push_back(new RegexElement(NUMBER, NUMBER_END,
                      Pattern::compile(re_digit)));

      symbolString = symbolStrStream.str();

      ignoreCase = getFlag(langDef.getParameter("ignorecase"));
      allowNestedComments = getFlag(langDef.getParameter("allownestedcomments"));
      disableHighlighting = getFlag(langDef.getParameter("disablehighlighting"));
      reformatCode = getFlag(langDef.getParameter("reformatting"));
      rawStringPrefix = getSymbol(langDef.getParameter("rawstringprefix"));
      continuationChar = getSymbol(langDef.getParameter("continuationsymbol"));
      allowExtEscape = getFlag(langDef.getParameter("allowextescape"));
      vhdlMode= getFlag(langDef.getParameter("vhdl_mode"));

      langDesc = langDef.getParameter("description");

      string fileToInclude=langDef.getParameter("include");
      if (!fileToInclude.empty()){
        string::size_type Pos = langDefPath.find_last_of(Platform::pathSeparator);
        string includeLangDefPath = langDefPath.substr(0, Pos+1) + fileToInclude;
        load(includeLangDefPath, false);
      }
      return true;
    } else {
      currentPath.clear();
      return false;
    }
}

void LanguageDefinition::reset()
{
  keywords.clear();
  keywordClasses.clear();
  delimiterPrefixes.clear();
  ignoreCase= false;
  allowNestedComments= reformatCode = false;
  rawStringPrefix = continuationChar = '\0';
  disableHighlighting=false;

  // TODO eigene methode
  vector<RegexElement*>::iterator it;
  for (it=regex.begin(); it!=regex.end();it++){
      delete *it;
  }
  regex.clear();
}

bool LanguageDefinition::allowNestedMLComments(){
  return allowNestedComments;
}

bool LanguageDefinition::highlightingDisabled(){
  return disableHighlighting;
}

bool LanguageDefinition::needsReload(const string &langDefPath){
  return currentPath!=langDefPath;
}

bool LanguageDefinition::enableReformatting(){
  return reformatCode;
}

const KeywordMap& LanguageDefinition::getKeywords() const{
  return keywords;
}

string &LanguageDefinition::getSymbolString() {
  return symbolString;
}

unsigned char LanguageDefinition::getRawStringPrefix(){
  return rawStringPrefix;
}

unsigned char LanguageDefinition::getContinuationChar(){
  return continuationChar;
}

bool LanguageDefinition::getSyntaxHighlight() {
  return !disableHighlighting;
}

bool LanguageDefinition::isIgnoreCase() {
  return ignoreCase;
}

bool LanguageDefinition::isVHDL() {
  return vhdlMode;
}

const vector<string>&LanguageDefinition::getKeywordClasses() const{
  return keywordClasses;
}

bool LanguageDefinition::allowExtEscSeq() {
    return allowExtEscape;
}

}
