/***************************************************************************
                          gui.cpp  -  description
                             -------------------
    begin                : 06.08.02
    copyright            : (C) 2002-2006 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "gui.h"
#include <wx/version.h>
#include <wx/stdpaths.h>
#include <wx/filefn.h>
#include <wx/ffile.h>
//#include <wx/splash.h>


#include "gui_failure_dlg.h"
#include "gui_showfile_dlg.h"
#include "highlight.xpm"

#define MAX_LINE__WIDTH       80

#ifdef __WXMSW__
	#define CHOICE_HEIGHT 22
	#define BUTTON_HEIGHT 25
#else
	#define CHOICE_HEIGHT 28
	#define BUTTON_HEIGHT 28
#endif

using namespace std;

// Main Application

bool HighlightApp::OnInit()
{
   const wxCmdLineEntryDesc cmdLineDesc[] ={
    { wxCMD_LINE_PARAM,  NULL, NULL, "input file", wxCMD_LINE_VAL_STRING,
        wxCMD_LINE_PARAM_MULTIPLE | wxCMD_LINE_PARAM_OPTIONAL },
    { wxCMD_LINE_OPTION, "u", "user-profile",  "user profile directory",
        wxCMD_LINE_VAL_STRING,
        wxCMD_LINE_PARAM_OPTIONAL },
    {wxCMD_LINE_NONE}
   };


   wxCmdLineParser parser(argc, argv);
   parser.SetDesc(cmdLineDesc);
   parser.Parse();

   HighlightFrame *win = new HighlightFrame(HIGHLIGHT_VERSION2);

   wxString userProfileDir;
   if (parser.Found("user-profile", &userProfileDir)){
        win->setUserProfilePath(userProfileDir.c_str());
   } else {
        win->setUserProfilePath(wxStandardPathsBase::Get().GetUserDataDir().c_str());
   }

   win->initInterface();

   //todo return fatalError
   if (win->getFatalError()) return false;

   if (parser.GetParamCount()){
     wxArrayString inputFileList;
     for (unsigned int i=0;i<parser.GetParamCount();i++){
       inputFileList.Add(parser.GetParam(i));
     }
     win->addInputFiles(inputFileList, "", true);
   }

   win->Centre();
   win->Show(true);
   SetTopWindow(win);
   win->showStartupTips();

   /* Do not return FALSE or the app will terminate immediately */
   return true;
}

void HighlightFrame::setUserProfilePath(const char* path){
      userPrefBaseDir=string(path);
}

// Main Window
HighlightFrame::HighlightFrame(const wxString &title)
    : wxFrame((wxFrame *) NULL, -1, title, wxDefaultPosition, wxDefaultSize,
               wxDEFAULT_FRAME_STYLE),
     styleOutFileName(NULL),
     language("English"),
     i18n(NULL)
{
	dataDir.searchDataDir("");
}

void HighlightFrame::initInterface(){

   preferencesPath=userPrefBaseDir+Platform::pathSeparator+"highlight.conf";
   defaultProjectPath=userPrefBaseDir+Platform::pathSeparator+"highlight.hpr";

   currentProjectPath=defaultProjectPath;

   ConfigurationReader prefConf(preferencesPath);
   string lastProject;
   if (prefConf.found()){
      language=prefConf.getParameter("language");
      showTips=prefConf.getParameter("showtips")=="1";
      StringTools::str2num<int>(lastTip, prefConf.getParameter("lasttip"), std::dec);
      lastProject = prefConf.getParameter("lastproject");
      if (!lastProject.empty() && lastProject != defaultProjectPath)
          currentProjectPath=lastProject;
   }

   fatalError=!loadLanguage(language);

   //SetIcon(wxIcon(highlight_xpm, wxBITMAP_TYPE_XPM));
   SetIcon(wxIcon(highlight_xpm));

   menuFile = new wxMenu(),
   menuHelp = new wxMenu();
   menuLang = new wxMenu();
   menuPrj = new wxMenu();

   MenuBar = new wxMenuBar();
   // create a menu bar
   menuFile->Append(ID_MENU_OPENFILES,wxString(i18n->getCParameter ("m001")));
   menuFile->AppendSeparator();
   menuFile->Append(PROGRAM_QUIT,wxString(i18n->getCParameter ("m002")));
   menuHelp->Append(ID_MENU_HELP, wxString(i18n->getCParameter("m003")));
   menuHelp->Append(ID_MENU_TIPS, wxString(i18n->getCParameter("m009")));
   menuHelp->AppendSeparator();
   menuHelp->Append(ID_MENU_LICENSE, wxString(i18n->getCParameter("m004")));
   menuHelp->Append(ID_MENU_CHANGELOG, wxString(i18n->getCParameter("m005")));
   menuHelp->AppendSeparator();
   menuHelp->Append(ID_MENU_ABOUT, wxString(i18n->getCParameter ("m006")));
   menuPrj->Append(ID_MENU_PRJ_LOAD, wxString(i18n->getCParameter ("m011")));
   menuPrj->Append(ID_MENU_PRJ_SAVE, wxString(i18n->getCParameter ("m012")));
   menuPrj->AppendSeparator();
   menuPrj->Append(ID_MENU_PRJ_LOAD_DEF, wxString(i18n->getCParameter ("m013")));


   wxFileName i18nFileName;
   wxArrayString i18nFiles;
   wxDir::GetAllFiles(wxString(dataDir.getI18nDir().c_str()),
                      &i18nFiles, wxString("*.txt"), (int)wxDIR_FILES);
   i18nFiles.Sort();
   if ( i18nFiles.IsEmpty()) {
       wxMessageBox(wxString("i18n folder not found. Check installation."),
                    wxString("Error."),
                    wxICON_EXCLAMATION);
       return;
   }
   // apart from English.txt, more translation files exist
   bool translationsAvailable=i18nFiles.GetCount()>1;
   if (translationsAvailable){
     for (unsigned int i=0; i<i18nFiles.GetCount(); i++){
       Connect(ID_MENU_LANGUAGE_DYN + i,  -1, wxEVT_COMMAND_MENU_SELECTED,
               (wxObjectEventFunction) (wxEventFunction) (wxCommandEventFunction)
               &HighlightFrame::OnCheckLanguage);
       i18nFileName.Assign(wxString(i18nFiles[i].c_str()));
       menuLang->AppendRadioItem(ID_MENU_LANGUAGE_DYN + i, i18nFileName.GetName());
     }
     menuLang->Check(menuLang->FindItem(wxString(language.c_str())), true);
  }

   // Add it to the menu bar
   MenuBar->Append(menuFile, wxString(i18n->getCParameter ("m007")));
   MenuBar->Append(menuPrj, wxString(i18n->getCParameter ("m010")));
   MenuBar->Append(menuLang, wxString(i18n->getCParameter ("m008")));
   MenuBar->Append(menuHelp, wxString(i18n->getCParameter ("m003")));

   SetMenuBar(MenuBar);
   MenuBar->EnableTop(1, translationsAvailable);

   panel =new wxPanel(this, ID_PANEL);
   butSource = new wxButton(panel, ID_BUTTON_OPENFILES,
                            wxString(i18n->getCParameter ("l001")),
                            wxDefaultPosition, wxSize(125, BUTTON_HEIGHT), 0 );

   listBox = new wxListBox(panel, ID_LISTBOX, wxDefaultPosition,
                           wxSize(200,300), 0, NULL,
                           wxLB_NEEDED_SB |wxLB_EXTENDED /*| wxLB_MULTIPLE*/ | wxLB_HSCROLL);

   butRemoveSelection=new wxButton(panel, ID_BUTTON_REMOVE,
                                    wxString(i18n->getCParameter ("l033")),
                                    wxDefaultPosition, wxSize(150, BUTTON_HEIGHT), 0 );

   butClearList = new wxButton(panel, ID_BUTTON_CLEAR,
                                    wxString(i18n->getCParameter ("l003")),
                                    wxDefaultPosition, wxSize(150, BUTTON_HEIGHT), 0 );

   // Spanish out directory label is very long, split in 2 lines
   wxString outDirLbl=wxString::Format(i18n->getCParameter ("l002"),"\n");
   lblOutDir = new wxStaticText(panel, -1, outDirLbl);

   outDir = new  wxTextCtrl(panel, ID_OUTDIRTEXT, "", wxDefaultPosition,
                           wxSize(175, CHOICE_HEIGHT));

   butOutDir = new wxButton( panel, ID_BUTTON_OUTDIR, "..." , wxDefaultPosition,
                             wxSize(20, 20), 0 );

   writeToSourceDir= new wxCheckBox(panel, ID_WRITE_TO_SRCDIR,
                                 wxString( i18n->getCParameter ("l032") ));

   lblOutputFormat = new wxStaticText(panel, -1, wxString(i18n->getCParameter("l004")));

   wxString formatNames[] = {"HTML","XHTML","LaTeX","TeX", "RTF", "XML"};
   outputFormatChoice=new wxChoice(panel, ID_FORMAT_CHOICE, wxDefaultPosition,
                            wxSize(80, CHOICE_HEIGHT), wxArrayString(NUMBER_FORMATS, formatNames));
   outputFormatChoice->SetSelection(0);

   linenumbers = new wxCheckBox(panel, ID_INCLUDELINENUMBERS,
                                wxString(i18n->getCParameter ("l013")));

   attachAnchors= new wxCheckBox(panel, ID_ATTACHANCHORS,
                                 wxString(i18n->getCParameter ("l006")));

   useFileAsAnchorName= new wxCheckBox(panel, ID_USEFILEASANCHORNAME,
                                 wxString(i18n->getCParameter ("l042")));


   orderedList= new wxCheckBox(panel, ID_ORDERED_LIST,
                                 wxString(i18n->getCParameter ("l037")));

   lnFillZeroes = new wxCheckBox(panel, ID_FILL_ZEROES,
                                 wxString(i18n->getCParameter ("l031")));

   wrapLines= new wxCheckBox(panel, ID_WRAP,
                                 wxString(i18n->getCParameter ("l027")));

   wrapLinesIntelligent= new wxCheckBox(panel, ID_WRAP_CAUTIOUS,
                                 wxString(i18n->getCParameter ("l028")));

   fragment = new wxCheckBox(panel, ID_FRAGMENT,
                             wxString(i18n->getCParameter ("l014")));

   includeStyle = new wxCheckBox(panel, ID_INCLUDECSS,
                               wxString(i18n->getCParameter ("l005")));

   generateHtmlIndex=new wxCheckBox(panel, ID_GENERATEINDEX,
                                    wxString(i18n->getCParameter ("l008")));

   replaceLatexQuotes=new wxCheckBox(panel, ID_REPLACE_QUOTES,
                                    wxString(i18n->getCParameter ("l029")));
   enableLaTeXBabelCompat=new wxCheckBox(panel, ID_BABEL_COMPAT,
                                    wxString(i18n->getCParameter ("l038")));

   lblColourTheme= new wxStaticText(panel, -1, wxString(i18n->getCParameter ("l010")));

   themeChoice=new wxChoice(panel, ID_THEMECHOICE, wxDefaultPosition,
                            wxSize(120, CHOICE_HEIGHT), 0, 0, wxCB_SORT);

   reformatCode= new wxCheckBox(panel, ID_REFORMATCODE,
                                wxString(i18n->getCParameter ("l007")),
                                wxDefaultPosition, wxSize(100, CHOICE_HEIGHT));

   reformatChoice=new wxChoice(panel, ID_REFORMATCHOICE, wxDefaultPosition,
                             wxSize(65, CHOICE_HEIGHT), 0, 0, wxCB_SORT);

   lblTabWidth=new wxStaticText(panel, ID_TAB_WIDTH, wxString(i18n->getCParameter ("l030")));
   tabWidth= new wxSpinCtrl(panel, ID_TAB_WIDTH,"4", wxDefaultPosition,
                            wxSize(40, CHOICE_HEIGHT), wxSP_ARROW_KEYS, 0, 8 , 4);

   lblLineLength=new wxStaticText(panel, ID_LINE_LEN, wxString(i18n->getCParameter ("l046")));
   lineLength= new wxSpinCtrl(panel, ID_LINE_LEN,"80", wxDefaultPosition,
                            wxSize(60, CHOICE_HEIGHT), wxSP_ARROW_KEYS, 60, 160, 80 );

   useEncoding= new wxCheckBox(panel, ID_USE_ENCODING,
                                wxString(i18n->getCParameter ("l036")),
                                wxDefaultPosition, wxSize(100, CHOICE_HEIGHT));

   wxString encodingNames[] = { "ISO-8859-1", "ISO-8859-2", "ISO-8859-3",
                                "ISO-8859-4", "ISO-8859-5", "ISO-8859-6",
                                "ISO-8859-7", "ISO-8859-8", "ISO-8859-9",
                                "ISO-8859-10", "ISO-8859-11", "ISO-8859-13",
                                "ISO-8859-14", "ISO-8859-15", "UTF-8"};

   encodingChoice=new wxComboBox(panel, ID_ENCODING_LIST, "", wxDefaultPosition,
                                 wxSize(95, CHOICE_HEIGHT),
                                 wxArrayString (NUMBER_ENCODINGS, encodingNames),
                                 wxCB_DROPDOWN );
   encodingChoice->SetSelection(0);

   lblPageSize= new wxStaticText(panel, -1, wxString(i18n->getCParameter ("l039")));
   wxString pageSizes[] = {"A3","A4","A5","B4", "B5", "B6", "Letter", "Legal"};
   pageSizeChoice=new wxChoice(panel, ID_FORMAT_CHOICE, wxDefaultPosition,
                            wxSize(70, CHOICE_HEIGHT), wxArrayString(8, pageSizes));
   pageSizeChoice->SetSelection(1);

   wxString str=wxString::Format("%s %s", i18n->getCParameter ("l011"),
                   ((styleOutFileName!=NULL)?(styleOutFileName->GetValue()).c_str():
                      CSS_DEFAULT_NAME));

   lblStyleIn= new wxStaticText(panel, -1,str);
   styleInFileName = new wxTextCtrl(panel, ID_CSS_INFILE, "",
                                  wxDefaultPosition,wxSize(120, CHOICE_HEIGHT));
   butChooseStyleInfile = new wxButton( panel, ID_BUTTON_OPEN_CSS_INFILE, "...",
                                     wxDefaultPosition, wxSize(20, 20), 0);

   lblStyleOut= new wxStaticText(panel, -1, wxString(i18n->getCParameter ("l009")));

   styleOutFileName = new wxTextCtrl(panel, ID_CSS_OUTFILE, CSS_DEFAULT_NAME,
                                   wxDefaultPosition,wxSize(150, CHOICE_HEIGHT));

   butStart = new wxButton(panel, ID_START, wxString(i18n->getCParameter("l015")),
                           wxDefaultPosition, wxSize(150, 30), 0 );
   butStart->SetFont(wxFont(11, wxDEFAULT , wxNORMAL,wxBOLD));


   progress= new wxGauge(panel, ID_PROGRESS, 100, wxDefaultPosition,
                         wxSize(150, 20),wxGA_SMOOTH );

  // Layout ***************************************************************

  wxBoxSizer *leftsizer = new wxBoxSizer( wxVERTICAL );
  leftsizer->Add( butSource, 0, wxTOP | wxALIGN_CENTER_HORIZONTAL, 5 );
  leftsizer->Add(
    listBox,
    1,            // make vertically stretchable
    wxGROW |    // make horizontally stretchable
    wxTOP|wxBOTTOM,        //   and make border all around
    10 );         // set border width to 10

  leftsizer->Add( butRemoveSelection, 0, wxTOP | wxALIGN_CENTER_HORIZONTAL, 5 );
  leftsizer->Add(butClearList, 0, wxTOP | wxALIGN_CENTER_HORIZONTAL, 5 );
  //leftsizer->Add(0,5);

   wxStaticBox *outDirBox= new wxStaticBox(panel, -1, "" );
   wxStaticBoxSizer *outDirBoxSizer = new wxStaticBoxSizer( outDirBox, wxVERTICAL );
   wxBoxSizer* outDirSelSizer = new wxBoxSizer(wxHORIZONTAL);
   outDirSelSizer->Add( outDir, 1, wxALIGN_CENTRE|wxALL , 5 );
   outDirSelSizer->Add( butOutDir, 0, wxALIGN_CENTER_VERTICAL|wxTOP, 0 );
   outDirBoxSizer->Add( lblOutDir, 0, wxALIGN_CENTER_HORIZONTAL | wxALL, 2 );
   outDirBoxSizer->Add( outDirSelSizer, 0, wxALIGN_CENTER_HORIZONTAL | wxBOTTOM, 5);
   outDirBoxSizer->Add( writeToSourceDir, 0, wxALIGN_CENTER_HORIZONTAL | wxBOTTOM, 5);
   leftsizer->Add(outDirBoxSizer,0, wxALIGN_CENTER_HORIZONTAL | wxTOP | wxBOTTOM, 5);

    wxBoxSizer *rightsizer = new wxBoxSizer( wxVERTICAL );

    wxStaticBox *formatChoiceBox= new wxStaticBox(panel, -1, "");
    wxStaticBoxSizer *formatChoiceBoxSizer = new wxStaticBoxSizer( formatChoiceBox, wxVERTICAL );

    wxBoxSizer *formatSizer = new wxBoxSizer(wxHORIZONTAL );
    formatSizer->Add( lblOutputFormat, 0,  wxTOP | wxALIGN_CENTER_VERTICAL , 3 );
    formatSizer->Add(5, 0, 1);
    formatSizer->Add( outputFormatChoice, 0,  wxTOP , 5 );
    formatChoiceBoxSizer->Add( formatSizer, 0,  wxTOP| wxBOTTOM |wxGROW );

    rightsizer->Add(formatChoiceBoxSizer,0, wxGROW|wxTOP, 0);


    wxStaticBox *optBox= new wxStaticBox(panel, -1, "");
    wxStaticBoxSizer *optBoxSizer = new wxStaticBoxSizer( optBox, wxVERTICAL );
    optBoxSizer->Add(0, 5, 1);
    optBoxSizer->Add(linenumbers, 0, wxALIGN_LEFT, 5 );
    optBoxSizer->Add( attachAnchors, 0,  wxTOP | wxALIGN_LEFT | wxLEFT, 5 );
    optBoxSizer->Add( useFileAsAnchorName, 0,  wxTOP | wxALIGN_LEFT | wxLEFT, 5 );
    optBoxSizer->Add( orderedList, 0,  wxTOP | wxALIGN_LEFT | wxLEFT, 5 );
    optBoxSizer->Add( lnFillZeroes, 0,  wxTOP | wxALIGN_LEFT | wxLEFT, 5 );

    optBoxSizer->Add( wrapLines, 0,  wxTOP | wxALIGN_LEFT, 10 );
    optBoxSizer->Add( wrapLinesIntelligent, 0,  wxTOP | wxALIGN_LEFT | wxLEFT, 5 );

    optBoxSizer->Add( fragment, 0,  wxTOP | wxALIGN_LEFT, 10 );
    optBoxSizer->Add( includeStyle, 0,  wxTOP | wxALIGN_LEFT, 5 );
    optBoxSizer->Add( generateHtmlIndex, 0,  wxTOP | wxBOTTOM | wxALIGN_LEFT, 5 );
    optBoxSizer->Add( replaceLatexQuotes, 0,  wxTOP | wxALIGN_LEFT, 5 );
    optBoxSizer->Add( enableLaTeXBabelCompat, 0,  wxTOP |wxBOTTOM| wxALIGN_LEFT, 5 );
    rightsizer->Add(optBoxSizer,0, wxGROW|wxTOP, 5);

    wxStaticBox *choicesBox= new wxStaticBox(panel, -1, "");
    wxStaticBoxSizer *choicesBoxSizer = new wxStaticBoxSizer( choicesBox, wxVERTICAL );

    wxBoxSizer *themeSizer = new wxBoxSizer(wxHORIZONTAL );
    themeSizer->Add( lblColourTheme, 0,  wxTOP | wxALIGN_CENTER_VERTICAL , 3 );
    themeSizer->Add(5, 0, 1);
    themeSizer->Add( themeChoice, 0,  wxTOP , 5 );
    choicesBoxSizer->Add( themeSizer, 0,  wxTOP| wxBOTTOM |wxGROW );
    choicesBoxSizer->Add( new wxStaticLine(panel, -1, wxDefaultPosition,
                          wxDefaultSize, wxLI_HORIZONTAL ), 0, wxGROW|wxTOP|wxLEFT|wxRIGHT, 5 );
    wxBoxSizer *reformatSizer = new wxBoxSizer(wxHORIZONTAL );
    reformatSizer->Add(reformatCode,0, wxTOP | wxALIGN_CENTER_VERTICAL, 5);
    reformatSizer->Add(5, 0, 1);
    reformatSizer->Add(reformatChoice,0, wxTOP, 5);
    choicesBoxSizer->Add( reformatSizer, 0,  wxTOP |wxGROW, 5 );
    choicesBoxSizer->Add( new wxStaticLine(panel, -1, wxDefaultPosition,
                          wxDefaultSize, wxLI_HORIZONTAL ), 0, wxGROW|wxTOP|wxLEFT|wxRIGHT, 5 );
    wxBoxSizer *tabSizer = new wxBoxSizer(wxHORIZONTAL );
    tabSizer->Add(lblTabWidth, 0, wxTOP|wxALIGN_CENTER_VERTICAL , 5);
    tabSizer->Add(5, 0, 1);
    tabSizer->Add(tabWidth, 0, wxTOP, 3);
    wxBoxSizer *lenSizer = new wxBoxSizer(wxHORIZONTAL );
    lenSizer->Add(lblLineLength, 0, wxTOP|wxALIGN_CENTER_VERTICAL , 5);
    lenSizer->Add(5, 0, 1);
    lenSizer->Add(lineLength, 0, wxTOP, 3);
    choicesBoxSizer->Add( tabSizer, 0, wxTOP| wxBOTTOM| wxGROW, 5 );
    choicesBoxSizer->Add( lenSizer, 0, wxTOP| wxBOTTOM| wxGROW, 5 );
    choicesBoxSizer->Add( new wxStaticLine(panel, -1, wxDefaultPosition,
                          wxDefaultSize, wxLI_HORIZONTAL ), 0, wxGROW|wxTOP|wxLEFT|wxRIGHT, 5 );
    wxBoxSizer *encodingSizer = new wxBoxSizer(wxHORIZONTAL );
    encodingSizer->Add(useEncoding,0, wxTOP | wxALIGN_CENTER_VERTICAL, 5);
    encodingSizer->Add(5, 0, 1);
    encodingSizer->Add(encodingChoice,0, wxTOP, 5);
    choicesBoxSizer->Add( encodingSizer, 0,  wxTOP |wxGROW, 5 );
    choicesBoxSizer->Add( new wxStaticLine(panel, -1, wxDefaultPosition,
                          wxDefaultSize, wxLI_HORIZONTAL ), 0, wxGROW|wxTOP|wxLEFT|wxRIGHT, 5 );

    wxBoxSizer *pageSizeSizer = new wxBoxSizer(wxHORIZONTAL );
    pageSizeSizer->Add( lblPageSize, 0,  wxTOP | wxALIGN_CENTER_VERTICAL , 3 );
    pageSizeSizer->Add(5, 0, 1);
    pageSizeSizer->Add( pageSizeChoice, 0,  wxTOP , 5 );
    choicesBoxSizer->Add( pageSizeSizer, 0,  wxTOP| wxBOTTOM |wxGROW );


    rightsizer->Add(choicesBoxSizer,0, wxGROW|wxTOP, 5);

    wxStaticBox *styleOptBox= new wxStaticBox(panel, -1, "");
    wxStaticBoxSizer *styleOptBoxBoxSizer = new wxStaticBoxSizer( styleOptBox, wxVERTICAL );
    styleOptBoxBoxSizer->Add( lblStyleOut, 0 );
    styleOptBoxBoxSizer->Add( styleOutFileName, 0,  wxTOP | wxGROW | wxALIGN_LEFT, 5 );
    styleOptBoxBoxSizer->Add( lblStyleIn, 0,  wxTOP | wxALIGN_LEFT, 10 );
    wxBoxSizer *styleInSizer = new wxBoxSizer(wxHORIZONTAL );
    styleInSizer->Add( styleInFileName, 1,  wxTOP | wxGROW | wxALIGN_LEFT, 5 );
    styleInSizer->Add( butChooseStyleInfile, 0,  wxBOTTOM|wxTOP| wxLEFT| wxALIGN_CENTER_VERTICAL, 5 );
    styleOptBoxBoxSizer->Add( styleInSizer, 0, wxBOTTOM|wxGROW, 5);
    #ifdef ADD_PREVIEW
    rightsizer->Add(styleOptBoxBoxSizer,0, wxGROW  | wxTOP | wxBOTTOM, 5);
    #else
    rightsizer->Add(styleOptBoxBoxSizer,0, wxGROW|wxTOP, 5);
    #endif

    rightsizer->Add(0, 20, 1);

    #ifndef ADD_PREVIEW
    rightsizer->Add( butStart, 0,  wxTOP | wxGROW, 10 );
    rightsizer->Add( progress, 0,  wxTOP | wxBOTTOM| wxGROW | wxALIGN_BOTTOM, 10 );
    #endif

    #ifdef ADD_PREVIEW
    wxBoxSizer *previewsizer = new wxBoxSizer( wxVERTICAL );
    wxString pvLblText ;
    pvLblText.Printf(i18n->getCParameter ("l043"), i18n->getCParameter ("l045"));
    lblPreview= new wxStaticText(panel, -1,  pvLblText);
    previewsizer->Add( lblPreview, 0,  wxTOP, 10 );
    preview = new wxHtmlWindow(panel, wxID_ANY,wxDefaultPosition,wxSize(350, 200),wxHW_NO_SELECTION);
    //preview-> SetFonts("Courier New", "Courier New", NULL);
    previewsizer->Add( preview, 1,  wxTOP | wxGROW , 10 );

    wxBoxSizer *convertsizer = new wxBoxSizer( wxHORIZONTAL );
    convertsizer->Add(butStart, 0,  wxTOP, 5 );
    convertsizer->Add(5, 0, 0);
    convertsizer->Add(progress, 1, wxGROW | wxTOP , 5 );
    previewsizer->Add(convertsizer, 0,  wxTOP |wxBOTTOM |wxGROW, 5 );
    #endif

    wxBoxSizer *panelsizer = new wxBoxSizer( wxHORIZONTAL );

    panelsizer->Add(leftsizer, 0, wxGROW|wxRIGHT|wxLEFT, 10);
    panelsizer->Add(rightsizer, 0,wxGROW|wxRIGHT|wxLEFT, 10);
    #ifdef ADD_PREVIEW
    panelsizer->Add(previewsizer, 1,wxGROW|wxRIGHT|wxLEFT, 10);
    #endif

   panelsizer->SetSizeHints( this );   // set size hints to honour minimum size
   panel->SetSizer( panelsizer );      // use the sizer for layout

   CreateStatusBar(2);
   if (lastProject != defaultProjectPath){
     wxString pName, pExt;
     wxFileName::SplitPath(lastProject.c_str(), NULL, &pName, &pExt);
     SetStatusText(wxString::Format("%s.%s", pName.c_str(), pExt.c_str()).Upper(),0);
   }else {
     SetStatusText(wxString(i18n->getCParameter("l041")).Upper(),0);
   }
   SetStatusText(wxString(i18n->getCParameter("l024")),1);


   fillChoiceCtrl(themeChoice, wxString(dataDir.getThemeDir().c_str()), "*.style");
   fillChoiceCtrl(reformatChoice, wxString(dataDir.getIndentSchemesDir().c_str()), "*.indent");

   assignToolTip(butSource,"t001");
   assignToolTip(butOutDir,"t002");
   assignToolTip(butChooseStyleInfile,"t003");
   assignToolTip(butClearList,"t004");
   assignToolTip(styleOutFileName,"t005");
   assignToolTip(styleInFileName,"t006");
   assignToolTip(outDir,"t007");
   assignToolTip(includeStyle,"t008");
   assignToolTip(fragment,"t009");
   assignToolTip(linenumbers,"t010");
   assignToolTip(attachAnchors,"t011");
   assignToolTip(useFileAsAnchorName,"t029");
   assignToolTip(reformatCode,"t012");
   assignToolTip(outputFormatChoice,"t013");
   assignToolTip(butStart,"t014");
   assignToolTip(generateHtmlIndex,"t015");
   assignToolTip(reformatChoice,"t016");
   assignToolTip(themeChoice,"t017");
   assignToolTip(wrapLines,"t018");
   assignToolTip(wrapLinesIntelligent,"t019");
   assignToolTip(replaceLatexQuotes,"t020");
   assignToolTip(enableLaTeXBabelCompat,"t027");
   assignToolTip(tabWidth,"t021");
   assignToolTip(lineLength,"t031");
   assignToolTip(lnFillZeroes,"t022");
   assignToolTip(writeToSourceDir,"t023");
   assignToolTip(butRemoveSelection,"t024");
   assignToolTip(useEncoding,"t025");
   assignToolTip(orderedList,"t026");
   assignToolTip(pageSizeChoice,"t028");
   #ifdef ADD_PREVIEW
   assignToolTip(preview,"t030");
   #endif

   wxFFile txtFile (string(dataDir.getExtDir()+"fileopenfilter.conf").c_str(), wxT("r"));
   txtFile.ReadAll(&fileFilter);
   fileFilter.Replace("\n",""); // wegen GTK

   if (fileFilter.IsEmpty()){
     fileFilter="All source code (*.*)|*.*|";
   }

   bool extensionsFound=loadFileTypeConfig(&extensions, &shebangs);
   if (!extensionsFound){
     wxMessageBox(wxString(i18n->getCParameter ("e008")),
                           wxString(i18n->getCParameter ("l025")),
                           wxICON_EXCLAMATION);
   }

   loadProject(currentProjectPath);

   // initialize drag and drop
   listBox->SetDropTarget (new DropFiles (this));

   #ifdef _WIN32
/*
     wxImage::AddHandler(new wxPNGHandler());
     wxBitmap bitmap;
  if (bitmap.LoadFile("splash.png", wxBITMAP_TYPE_PNG))
  {
      wxSplashScreen* splash = new wxSplashScreen(bitmap,
          wxSPLASH_CENTRE_ON_SCREEN|wxSPLASH_TIMEOUT,
          5000, NULL, -1, wxDefaultPosition, wxDefaultSize,
          wxSIMPLE_BORDER|wxSTAY_ON_TOP);
          wxYield();
  }
*/
  #endif

}

bool HighlightFrame::loadLanguage(const string &newLanguage){
    wxString i18nPath=wxString::Format("%s%s.txt", dataDir.getI18nDir().c_str(),
                                                   newLanguage.c_str() );
    delete i18n;
    i18n= new ConfigurationReader(i18nPath.c_str());
    if (!i18n->found()){
      wxString errMsg=wxString::Format("Could not load GUI language file (%s). Check installation.",
                    i18nPath.c_str());
      wxMessageBox(errMsg, "Highlight Fatal Error", wxICON_EXCLAMATION);
      return false;
    }
    language=newLanguage;
    return true;
}

bool HighlightFrame::loadFileTypeConfig(StringMap* extMap, StringMap* shebangMap) {
  if (!extMap || !shebangMap) return false;

  ConfigurationReader config(dataDir.getConfDir() + "filetypes.conf");
  if (config.found())
  {
        stringstream values;
        string paramName, paramVal;
        for (unsigned int i=0;i<config.getParameterNames().size();i++){
            paramName = config.getParameterNames()[i];

            if (paramName.find("ext") != string::npos){
                values.str(config.getParameter(paramName)) ;
		paramName = StringTools::getParantheseVal(paramName);
            	while (values >> paramVal) {
	   		extMap->insert(make_pair( paramVal,  paramName));
            	}
            	values.clear();
            }
	    else if (paramName.find("shebang") != string::npos){
                values.str(config.getParameter(paramName)) ;
		paramName = StringTools::getParantheseVal(paramName);
            	while (values >> paramVal) {
	   		shebangMap->insert(make_pair( paramVal,  paramName));
            	}
            	values.clear();
            }

        }
        return true;
    }
  return false;
}

void HighlightFrame::OnClickButtonSource(wxCommandEvent & WXUNUSED(event))
{
  wxArrayString chosenFiles;
  wxFileDialog openFiles(this,wxString(i18n->getCParameter("l001")),
                         openFileDir,  "",
                         fileFilter,

                         wxMULTIPLE | wxFILE_MUST_EXIST);
  openFiles.ShowModal();
  openFiles.GetFilenames(chosenFiles);
  openFileDir=openFiles.GetDirectory();
  if (openFileDir[openFileDir.length()-1] != Platform::pathSeparator ) {
    openFileDir += Platform::pathSeparator;
  }
  if (outDir->GetValue().IsEmpty()){
      outDir->SetValue(openFileDir);
  }
  addInputFiles(chosenFiles, openFileDir, false);
}

void HighlightFrame::OnTips(wxCommandEvent & WXUNUSED(event)){
  showStartupTips(true);
}

void HighlightFrame::addInputFiles(const wxArrayString& inputFiles,
                               const wxString &prefix, bool clear){
  wxString absInFilePath;
  if (clear) listBox->Clear();
  for (unsigned int i=0; i< inputFiles.GetCount(); i++){
     absInFilePath.Printf("%s%s", prefix.c_str(), inputFiles[i].GetData());
     if (listBox->FindString(absInFilePath) == -1) {
       listBox->Append(absInFilePath);
     }
  }
}

void HighlightFrame::OnClickButtonTargetDir(wxCommandEvent & WXUNUSED(event))
{
  wxDirDialog targetDir(this,wxString(i18n->getCParameter("l002")), "");
  targetDir.ShowModal();
  outDir->SetValue(targetDir.GetPath());
}

void HighlightFrame::OnClickButtonClear(wxCommandEvent & WXUNUSED(event)){
  listBox->Clear();
}

void HighlightFrame::OnClickButtonRemSelection(wxCommandEvent &event){
  wxArrayInt selectedItems;
  listBox->GetSelections (selectedItems);
  size_t cnt = selectedItems.GetCount();
  if (cnt){
    for (size_t i=0;i<cnt;i++){
      listBox->Delete(selectedItems[i]-i);
    }
  }
}

void HighlightFrame::OnHelp(wxCommandEvent & WXUNUSED(event)){
#ifdef WIN32
   showTextFile("README.txt");
#else
   showTextFile("README");
#endif
}
void HighlightFrame::OnLicense(wxCommandEvent & WXUNUSED(event)){
#ifdef WIN32
   showTextFile("COPYING.txt");
#else
   showTextFile("COPYING");
#endif
}
void HighlightFrame::OnChangelog(wxCommandEvent & WXUNUSED(event)){
#ifdef WIN32
   showTextFile("ChangeLog.txt");
#else
   showTextFile("ChangeLog");
#endif
}

void HighlightFrame::OnCheckLanguage(wxCommandEvent & event){
  bool langFound=loadLanguage(menuLang->GetLabel( event.GetId()).c_str());
  if (langFound){
    wxMessageBox(wxString(i18n->getCParameter("l026")),
                 wxString(i18n->getCParameter("l019")),
                 wxICON_INFORMATION);
  }
}

void HighlightFrame::OnStyleOutfileChanged(wxCommandEvent & WXUNUSED(event)){
    setStyleInLabel();
}

void HighlightFrame::OnClickButtonOpenStyleFile(wxCommandEvent & WXUNUSED(event))
{
  wxFileDialog openFiles(this,i18n->getCParameter ("l001"),"", "",
                         (outputFormatChoice->GetSelection()<2) ? "CSS input file (*.css)|*.css|":
                                                                  "LaTeX style (*.sty)|*.sty|",
                         wxFILE_MUST_EXIST );
  openFiles.ShowModal();
  if (!openFiles.GetFilename().IsEmpty()){
    wxString chosenFile=openFiles.GetDirectory();
    chosenFile.Append(Platform::pathSeparator);
    chosenFile.Append(openFiles.GetFilename());
    styleInFileName->SetValue(chosenFile);
}
}

void HighlightFrame::OnClickSaveProject(wxCommandEvent & WXUNUSED(event))
{
  wxFileDialog openFiles(this,i18n->getCParameter ("l040"),"", "",
                         "Highlight Project (*.hpr)|*.hpr|" , wxSAVE | wxOVERWRITE_PROMPT );
  openFiles.ShowModal();
  if (!openFiles.GetFilename().IsEmpty()){
    wxString chosenFile=openFiles.GetDirectory();
    chosenFile.Append(Platform::pathSeparator);
    chosenFile.Append(openFiles.GetFilename());
    bool res = writeProject(chosenFile.c_str());
    if (res){
        SetStatusText(openFiles.GetFilename().Upper(),0);
        currentProjectPath=chosenFile.c_str();
    }
  }
}

void HighlightFrame::OnClickOpenProject(wxCommandEvent & WXUNUSED(event))
{
  wxFileDialog openFiles(this,i18n->getCParameter ("l001"),"", "",
                         "Highlight Project (*.hpr)|*.hpr|",
                         wxFILE_MUST_EXIST );
  openFiles.ShowModal();
  if (!openFiles.GetFilename().IsEmpty()){
    wxString chosenFile=openFiles.GetDirectory();
    chosenFile.Append(Platform::pathSeparator);
    chosenFile.Append(openFiles.GetFilename());
    bool res = loadProject(chosenFile.c_str());
    if (res){
        SetStatusText(openFiles.GetFilename().Upper(),0);
        currentProjectPath=chosenFile.c_str();
    }
  }
}

void HighlightFrame::OnClickDefaultProject(wxCommandEvent & WXUNUSED(event))
{
     bool res = loadProject(defaultProjectPath);
    if (res){
        SetStatusText(wxString(i18n->getCParameter("l041")).Upper(),0);
        currentProjectPath=defaultProjectPath;
    }
}

void HighlightFrame::OnQuit2(wxCommandEvent & WXUNUSED(event)){
     Close(true); // ruft OnQuit auf
}

void HighlightFrame::OnClickElement(wxCommandEvent & event){
     plausiChecks();
     event.Skip();
}

#ifdef ADD_PREVIEW
void HighlightFrame::OnUpdatePreview(wxCommandEvent & event){
     updatePreview();
     event.Skip();
}
#endif


void HighlightFrame::OnQuit(wxCloseEvent & WXUNUSED(event))
{

  wxString userDataDir =  wxString(userPrefBaseDir.c_str());
  if (!wxDirExists(userDataDir)) wxMkdir(userDataDir);
  if (!writePreferences()) {
    wxMessageBox (wxString(i18n->getCParameter ("e006")),
                  wxString(i18n->getCParameter("l025")), wxICON_EXCLAMATION);
  }
  writeProject(currentProjectPath);
  if (i18n) delete i18n;

  this->Destroy();
}

void HighlightFrame::OnAbout(wxCommandEvent & WXUNUSED(event)){
   wxString translatorsList;
   wxFFile txtFile (string(dataDir.getI18nDir()+"Translators.list").c_str(), wxT("r"));
   txtFile.ReadAll(&translatorsList);

   if (translatorsList.IsEmpty()){
     translatorsList="Error: File Translators.list not found.\n";
   }
   wxString formatMsg;
   formatMsg.Printf(MSG_INFO_TXT, HIGHLIGHT_VERSION, wxMAJOR_VERSION,
                    wxMINOR_VERSION, wxRELEASE_NUMBER,
                    translatorsList.c_str(),HIGHLIGHT_URL);
   wxMessageBox(formatMsg,wxString(i18n->getCParameter("l019")),
                wxICON_INFORMATION);
}

void HighlightFrame::showTextFile(const wxString &file){
   wxString cmd;
   cmd.Printf("%s%s",dataDir.getDocDir().c_str(),file.c_str());
   ShowFileDlg dlg(this, -1,file, cmd );
   if (dlg.fileFound()){
      dlg.ShowModal();
   }
}

void HighlightFrame::assignToolTip(wxWindow* widget, const char* tipName){
   wxString tip =  i18n->getCParameter(tipName);
   tip.Replace("\\n", "\n");
   widget ->SetToolTip(tip);
}

void HighlightFrame::fillChoiceCtrl(wxChoice* choice, const wxString &dir,
                                    const wxString &ext){
   if (!choice) return;

   wxFileName fName;
   wxArrayString files;
   wxDir::GetAllFiles(dir, &files, ext);
   for (unsigned int i=0;i<files.GetCount(); i++){
     fName.Assign(files[i]);
     choice->Append(fName.GetName());
   }
   choice->SetSelection(0);
}

void HighlightFrame::showStartupTips(bool forceTips){
  if (showTips || forceTips ){
     wxString tipsFile;
     const char *i18nPath = dataDir.getI18nDir().c_str();
     tipsFile.Printf("%s%s.tips",i18nPath, language.c_str());
     if (!::wxFileExists(tipsFile)){
       tipsFile.Printf("%s%s.tips", i18nPath, "English");
     }
     wxTipProvider *tipProvider = wxCreateFileTipProvider(tipsFile, lastTip);
     showTips=wxShowTip(this, tipProvider,showTips);
     lastTip=tipProvider->GetCurrentTip();
     delete tipProvider;
  }
}

void HighlightFrame::plausiChecks(){

  bool webOutput=(outputFormatChoice->GetSelection()<2);
  bool extStyleEnabled=(outputFormatChoice->GetSelection()<4);
  bool write2outdir = !writeToSourceDir ->GetValue();
  includeStyle->Enable(extStyleEnabled || !write2outdir);
  styleOutFileName->Enable(extStyleEnabled && !includeStyle->GetValue()&&write2outdir);
  styleInFileName->Enable(extStyleEnabled);
  lblStyleOut->Enable(extStyleEnabled&&!includeStyle->GetValue()&&write2outdir);
  lblStyleIn->Enable(extStyleEnabled);
  butChooseStyleInfile->Enable(extStyleEnabled);
  attachAnchors->Enable(linenumbers->GetValue() && webOutput);
  useFileAsAnchorName->Enable(attachAnchors->IsEnabled() && attachAnchors->GetValue());
  orderedList->Enable(linenumbers->GetValue() && webOutput);
  lnFillZeroes->Enable(linenumbers->GetValue()&&!(orderedList->GetValue()));
  reformatChoice->Enable(reformatCode->GetValue());
  wrapLinesIntelligent->Enable(wrapLines->GetValue());
  replaceLatexQuotes->Enable(outputFormatChoice->GetSelection()==2
                             && fragment->GetValue());
  enableLaTeXBabelCompat->Enable(outputFormatChoice->GetSelection()==2 );
  lblTabWidth->Enable(!reformatCode->GetValue());
  tabWidth->Enable(!reformatCode->GetValue());
  lineLength->Enable(wrapLines->GetValue());
  generateHtmlIndex->Enable(webOutput && write2outdir);

  lblOutDir ->Enable(write2outdir);
  butOutDir->Enable(write2outdir);
  outDir->Enable(write2outdir);
  if (!write2outdir)  includeStyle->SetValue(extStyleEnabled);
/*
  if (tabWidth->GetValue()<0)
    tabWidth->SetValue(0);
  else if (tabWidth->GetValue()>8)
    tabWidth->SetValue(8);
*/
  if (extStyleEnabled){
    checkStyleReferences(styleOutFileName, webOutput);
    checkStyleReferences(styleInFileName, webOutput);
  }
  bool enableEncoding =    outputFormatChoice->GetSelection()<3
                        || outputFormatChoice->GetSelection()>4;
  useEncoding->Enable(enableEncoding);
  encodingChoice->Enable(enableEncoding && useEncoding->GetValue());

  pageSizeChoice->Enable(outputFormatChoice->GetSelection()==4);
  lblPageSize->Enable(outputFormatChoice->GetSelection()==4);

  setStyleInLabel();
}

void HighlightFrame::checkStyleReferences(wxTextCtrl *styleCtrl, bool webOutput){
    wxString stylePath = styleCtrl->GetValue();
    if (stylePath.Matches( (webOutput)? "*.sty" : "*.css")) {
      stylePath.Printf((webOutput)?"%s.css":"%s.sty",
                       stylePath.Mid(0, stylePath.Length()-4).c_str());
      styleCtrl->SetValue (stylePath);
    }
}

void HighlightFrame::setStyleInLabel(){
    wxString str=wxString::Format("%s %s",i18n->getCParameter ("l011"),
              ( (includeStyle->GetValue()) ? i18n->getCParameter ("l035"):
                  ((styleOutFileName!=NULL)?(styleOutFileName->GetValue()).c_str():
                                           CSS_DEFAULT_NAME )));
    lblStyleIn->SetLabel(str);
}

bool HighlightFrame::writePreferences(){

    ofstream pref (preferencesPath.c_str());
    if (!pref.fail()){
      pref << "# WinHighlight Preferences - Please do not edit.";
      pref << "\n$language="<<language;
      pref << "\n$showtips="<<showTips;
      pref << "\n$lasttip="<<lastTip;
      pref << "\n$lastproject="<<currentProjectPath;

      pref.close();
      return true;
    }
    return false;
}

bool HighlightFrame::writeProject(const string& projectFile){
    ofstream prjFile (projectFile.c_str());
    if (!prjFile.fail()){
      prjFile << "# WinHighlight Project File - Please do not edit.";
      prjFile << "\n$outdir="<<outDir->GetValue().c_str();
      prjFile << "\n$outputformat="<< outputFormatChoice->GetSelection() ;
      prjFile << "\n$outcssfile="<<(styleOutFileName->GetValue()).GetData();
      prjFile << "\n$incssfile="<<(styleInFileName->GetValue()).GetData();
      prjFile << "\n$themechoice="<<themeChoice->GetSelection();
      prjFile << "\n$openfiledir="<<openFileDir.c_str();
      prjFile << "\n$includecss="<<includeStyle->GetValue();
      prjFile << "\n$fragment="<<fragment->GetValue();
      prjFile << "\n$linenumbers="<<linenumbers->GetValue();
      prjFile << "\n$attachanchors="<<attachAnchors->GetValue();
      prjFile << "\n$fileisanchorname="<<useFileAsAnchorName->GetValue();
      prjFile << "\n$reformatcode="<<reformatCode->GetValue();
      prjFile << "\n$formatchoice="<<reformatChoice->GetSelection();
      prjFile << "\n$encodingval="<<encodingChoice->GetValue();
      prjFile << "\n$useencoding="<<useEncoding->GetValue();
      prjFile << "\n$generatehtmlindex="<<generateHtmlIndex->GetValue();
      prjFile << "\n$wrap="<<wrapLines->GetValue();
      prjFile << "\n$wrapsimple="<<wrapLinesIntelligent->GetValue();
      prjFile << "\n$replacequotes="<<replaceLatexQuotes->GetValue();
      prjFile << "\n$babel="<<enableLaTeXBabelCompat->GetValue();
      prjFile << "\n$tabwidth="<<tabWidth->GetValue();
      prjFile << "\n$linelength="<<lineLength->GetValue();
      prjFile << "\n$fillzeroes="<<lnFillZeroes->GetValue();
      prjFile << "\n$write2src="<<writeToSourceDir->GetValue();
      prjFile << "\n$orderedlist="<<orderedList->GetValue();
      prjFile << "\n$pagesize="<<pageSizeChoice->GetSelection();

      for (int fileCount=0; fileCount < listBox->GetCount(); fileCount++)
      {
         prjFile << "\n$inputfile_"<<fileCount<<"="<<listBox->GetString(fileCount).c_str();
      }

      prjFile.close();
      return true;
    }
    return false;
}


bool HighlightFrame::loadProject(const string& projectFile){
    string trueStr("1");
    ConfigurationReader projectConf(projectFile);
    if (projectConf.found()){
      outDir->SetValue(projectConf.getParameter("outdir").c_str());
      int intVal=0;
      StringTools::str2num<int>(intVal, projectConf.getParameter("outputformat"), std::dec);
      outputFormatChoice->SetSelection(intVal);
      StringTools::str2num<int>(intVal, projectConf.getParameter("themechoice"), std::dec);
      themeChoice->SetSelection(intVal);
      openFileDir=projectConf.getParameter("openfiledir").c_str();
      includeStyle->SetValue(projectConf.getParameter("includecss")==trueStr);
      fragment->SetValue(projectConf.getParameter("fragment")==trueStr);
      linenumbers->SetValue(projectConf.getParameter("linenumbers")==trueStr);
      attachAnchors->SetValue(projectConf.getParameter("attachanchors")==trueStr);
      useFileAsAnchorName->SetValue(projectConf.getParameter("fileisanchorname")==trueStr);
      reformatCode->SetValue(projectConf.getParameter("reformatcode")==trueStr);
      generateHtmlIndex->SetValue(projectConf.getParameter("generatehtmlindex")==trueStr);
      wrapLines->SetValue(projectConf.getParameter("wrap")==trueStr);
      wrapLinesIntelligent->SetValue(projectConf.getParameter("wrapsimple")==trueStr);
      replaceLatexQuotes->SetValue(projectConf.getParameter("replacequotes")==trueStr);
      enableLaTeXBabelCompat->SetValue(projectConf.getParameter("babel")==trueStr);
      StringTools::str2num<int>(intVal, projectConf.getParameter("formatchoice"), std::dec);
      reformatChoice->SetSelection(intVal);
      StringTools::str2num<int>(intVal, projectConf.getParameter("pagesize"), std::dec);
      pageSizeChoice->SetSelection(intVal);

      tabWidth->SetValue(wxString(projectConf.getCParameter("tabwidth")));
      lineLength->SetValue(wxString(projectConf.getCParameter("linelength")));
      lnFillZeroes->SetValue(projectConf.getParameter("fillzeroes")==trueStr);
      styleOutFileName->SetValue(projectConf.getParameter("outcssfile").c_str());
      styleInFileName->SetValue(projectConf.getParameter("incssfile").c_str());
      writeToSourceDir->SetValue(projectConf.getParameter("write2src")==trueStr);

      encodingChoice->SetValue(wxString(projectConf.getCParameter("encodingval")));
      useEncoding->SetValue(projectConf.getParameter("useencoding")==trueStr);
      orderedList->SetValue(projectConf.getParameter("orderedlist")==trueStr);

      int fileCount=0;
      wxString infile;
      bool done=false;
      listBox->Clear();
      while (!done){
          infile = wxString(projectConf.getCParameter(wxString::Format("inputfile_%d", fileCount).c_str()));
          fileCount++;
          if (!infile.empty()){
             if (listBox->FindString(infile) == -1) {
                listBox->Append(infile);
             }
          } else {
             done = true;
          }
      }
      setStyleInLabel();
      plausiChecks();
      return true;
    }
    return false;
}

void HighlightFrame::OnClickButtonStart(wxCommandEvent & WXUNUSED(event)){

    if (!listBox->GetCount() )
        return;

    if (!writeToSourceDir->GetValue() && !wxDir::Exists(outDir->GetValue()))
    {
      wxMessageBox(wxString(i18n->getCParameter("l020")),
                   wxString(i18n->getCParameter("l025")), wxICON_EXCLAMATION);
      outDir->SetFocus();
      return;
    }
    if (outputFormatChoice->GetSelection()<2
        && (includeStyle->GetValue()==false)
        &&(styleOutFileName->GetValue().IsEmpty()))
    {
      wxMessageBox(wxString(i18n->getCParameter("l021")),
                   wxString(i18n->getCParameter("l025")), wxICON_EXCLAMATION);
      styleOutFileName->SetFocus();
      return;
    }

    wxStopWatch stopWatch;

    string outPath((outDir->GetValue()).c_str());
    if (outPath[outPath.length()-1] != Platform::pathSeparator ) {
      outPath +=Platform::pathSeparator;
    }
    wxString stylePath=wxString::Format("%s%s.style", dataDir.getThemeDir().c_str(),
                                   themeChoice->GetStringSelection().c_str());

    highlight::CodeGenerator *generator =
      highlight::CodeGenerator::getInstance(getOutputType());

    generator->initTheme(stylePath.c_str());
    generator->setSpecialOptions(attachAnchors->GetValue(),
                                 orderedList->GetValue(),
                                 replaceLatexQuotes->GetValue(),
                                 enableLaTeXBabelCompat->GetValue(),
                                 pageSizeChoice->GetStringSelection().Lower().c_str(),
                                 "l");
    generator->setStyleInputPath((styleInFileName->GetValue()).GetData());
    generator->setStyleOutputPath((styleOutFileName->GetValue()).GetData());
    generator->setIncludeStyle(includeStyle->GetValue());
    generator->setPrintLineNumbers(linenumbers->GetValue());
    generator->setPrintZeroes(lnFillZeroes->GetValue());
    generator->setFragmentCode(fragment->GetValue());

    // TODO neue Optionen als Eingabefelder
    generator->setPreformatting(getWrappingStyle(),
                               (generator->getPrintLineNumbers()) ?
                                lineLength->GetValue() - 5 : lineLength->GetValue(),
                                tabWidth->GetValue());

    if (useEncoding->GetValue()) {
        generator->setEncoding((encodingChoice->GetValue()).GetData());
    } else {
        generator->setEncoding("none");
    }
    // TODO
    //generator->setBaseFont(options.getBaseFont()) ;
    //generator->setBaseFontSize(options.getBaseFontSize()) ;
    generator->setLineNumberWidth(5);  /*options.getNumberWidth()*/

    if (reformatCode->GetValue()){
       wxString indentSchemePath=wxString::Format("%s%s.indent",
                                  dataDir.getIndentSchemesDir().c_str(),
                                  reformatChoice->GetStringSelection().c_str());
      generator->initIndentationScheme(indentSchemePath.c_str());
    }

    string outfileName;
    string langDefPath;
    string suffix;
    string outfileSuffix = getOutFileSuffix();
    int fileCount;
    highlight::ParseError error;
    highlight::LoadResult loadRes;
    wxArrayString reformatFailures, outputFailures, inputFailures;
    wxString currentInputFile;

    SetStatusText(wxString(i18n->getCParameter ("l017")),1);
    wxBeginBusyCursor();
    butStart->Disable();

    for (fileCount=0; fileCount < listBox->GetCount(); fileCount++){

       currentInputFile= listBox->GetString(fileCount);
       suffix = getFileType(getFileSuffix(currentInputFile.c_str()), currentInputFile.c_str());
       langDefPath = dataDir.getLangDefDir() + suffix + ".lang";
       loadRes=generator->initLanguage(langDefPath);
       if (loadRes==highlight::LOAD_FAILED) {
          wxString formatMsg;
          formatMsg.Printf(wxString(i18n->getCParameter ("e002")), suffix.c_str());
          wxMessageBox(formatMsg, wxString(i18n->getCParameter("l025")),
                       wxICON_EXCLAMATION);
       } else {
          if (reformatCode->GetValue()&& !generator->formattingIsPossible()){
             reformatFailures.Add(currentInputFile);
          }
          if (writeToSourceDir->GetValue()) { //write output to source directories?
            outfileName = string(currentInputFile.c_str());
          } else {
            outfileName = outPath;
            outfileName += wxFileName(currentInputFile).GetFullName().c_str();
          }
          outfileName += outfileSuffix;

          string::size_type pos=currentInputFile.Find(Platform::pathSeparator, true);
          wxString inFileName = currentInputFile.Mid(pos+1);

          if (useFileAsAnchorName->GetValue()){
            generator->setHTMLAnchorPrefix(inFileName.c_str());
          }
          generator->setTitle(inFileName.c_str());

          error = generator->generateFile(currentInputFile.c_str(),
                                         outfileName.c_str() );
          if (error != highlight::PARSE_OK){
            stopWatch.Pause();
            if (error == highlight::BAD_INPUT) {
              inputFailures.Add(currentInputFile);
            }
            else {
              outputFailures.Add(wxString(outfileName.c_str()));
            }
            stopWatch.Resume();
          }
          progress->SetValue(100 * fileCount / listBox->GetCount());
	  wxYield();
       }
   }
   // print external style definition file
   if (includeStyle->IsEnabled() && !includeStyle->GetValue()){
      string css_name = (styleOutFileName->GetValue()).GetData();
      string csspath = outPath;
      unsigned int pos = css_name.find_last_of(Platform::pathSeparator);
      if (pos == string::npos) {
         pos = css_name.find_last_of('/');
      }
      if ((pos != string::npos) && (pos < css_name.length())) {
         csspath += css_name.substr(pos+1, css_name.length());
      } else {
         csspath += css_name;
      }
      bool styleFileOK=generator -> printExternalStyle(csspath);

      if (!styleFileOK) {
         outputFailures.Add(wxString(csspath.c_str()));
      }
    }

    // print index file
    if (generateHtmlIndex->GetValue() && !writeToSourceDir->GetValue()) {
      vector <string>  fileList;
      for (int i=0; i < listBox->GetCount(); i++){
          fileList.push_back(string(listBox->GetString(i)));
       }
       bool indexFileOK=generator->printIndexFile(fileList,  outPath);
       if (!indexFileOK) {
          wxMessageBox(wxString(i18n->getCParameter("e007")),
                       wxString(i18n->getCParameter("l025")),
                       wxICON_EXCLAMATION);
       }
    }

   highlight::CodeGenerator::deleteInstance();

   progress->SetValue(0);
   wxEndBusyCursor();
   butStart->Enable();

   wxString formatMsg;
   formatMsg.Printf(i18n->getCParameter ("l023"), fileCount, stopWatch.Time());
   SetStatusText(formatMsg, 1);

   if (!reformatFailures.IsEmpty()||
       !inputFailures.IsEmpty() ||
       !outputFailures.IsEmpty()) {
       FailureDlg dlg(this, -1,wxString(i18n->getCParameter ("l034")),
                      wxString(i18n->getCParameter ("e010")), reformatFailures,
                      wxString(i18n->getCParameter ("e003")), inputFailures,
                      wxString(i18n->getCParameter ("e004")), outputFailures );
       dlg.ShowModal() ;
   }
}

highlight::WrapMode HighlightFrame::getWrappingStyle(){
  if (!wrapLinesIntelligent->GetValue() && wrapLinesIntelligent->IsEnabled())
     return highlight::WRAP_SIMPLE;
  return (wrapLines->GetValue())?highlight::WRAP_DEFAULT:highlight::WRAP_DISABLED;
}

string HighlightFrame::analyzeFile(const string& file){
    ifstream inFile(file.c_str());
    string firstLine;
    getline (inFile, firstLine);
    StringMap::iterator it;
     for (it=shebangs.begin(); it!=shebangs.end();it++){
       if (Pattern::matches(it->first, firstLine)) return it->second;
     }
    return "";
}

string HighlightFrame::getFileType(const string& suffix, const string &inputFile)
{
    string fileType = (extensions.count(suffix)) ? extensions[suffix] : suffix ;
    if (!fileType.empty()) return fileType;
    return analyzeFile(inputFile);
}

string HighlightFrame::getFileSuffix(const string& fileName)
{
  unsigned int ptPos=fileName.rfind(".");
  return (ptPos == string::npos) ? "" : fileName.substr(ptPos+1,
                                        fileName.length());
}

string HighlightFrame::getOutFileSuffix(){
  switch (outputFormatChoice->GetSelection()) {
    case 0: return ".html";
    case 1: return ".xhtml";
    case 2:
    case 3: return ".tex";
    case 4: return ".rtf";
    case 5: return ".xml";
  }
  return ".html";
}

highlight::OutputType HighlightFrame::getOutputType(){
    switch (outputFormatChoice->GetSelection()){
     case 0: return highlight::HTML;
     case 1: return highlight::XHTML;
     case 2: return highlight::LATEX;
     case 3: return highlight::TEX;
     case 4: return highlight::RTF;
     case 5: return highlight::XML;
    }
    return highlight::HTML;
}

#ifdef ADD_PREVIEW
void HighlightFrame::updatePreview(){
    wxArrayInt selections;
    listBox->GetSelections(selections);
    if (!selections.GetCount()) return;

    wxBeginBusyCursor();
    SetStatusText(wxString(i18n->getCParameter ("l044")),1);
highlight::PreviewGenerator pwgenerator;
    pwgenerator.setMaxInputLineCnt(500);

    wxString stylePath=wxString::Format("%s%s.style", dataDir.getThemeDir().c_str(),
                                        themeChoice->GetStringSelection().c_str());
    pwgenerator.initTheme(stylePath.c_str());

    pwgenerator.setPreformatting(getWrappingStyle(),
                               (pwgenerator.getPrintLineNumbers()) ?
                                lineLength->GetValue() - 5 : lineLength->GetValue(),
                                tabWidth->GetValue());

    if (reformatCode->GetValue()){
       wxString indentSchemePath=wxString::Format("%s%s.indent",
                                  dataDir.getIndentSchemesDir().c_str(),
                                  reformatChoice->GetStringSelection().c_str());
       pwgenerator.initIndentationScheme(indentSchemePath.c_str());
    }

    pwgenerator.setPrintLineNumbers(linenumbers->GetValue());
    pwgenerator.setPrintZeroes(lnFillZeroes->GetValue());

    wxString previewData;
    wxString currentInputFile= listBox->GetString(selections.Item(0));
    string suffix = getFileType(getFileSuffix(currentInputFile.c_str()), currentInputFile.c_str());
    string langDefPath = dataDir.getLangDefDir() + suffix + ".lang";
    if (pwgenerator.initLanguage(langDefPath)!=highlight::LOAD_FAILED) {
          previewData = wxString(pwgenerator.generateStringFromFile(currentInputFile.c_str()).c_str());
    }
    if (!previewData.IsEmpty()) {
        wxString pwLabel;
        pwLabel.Printf(i18n->getCParameter ("l043"), currentInputFile.c_str());
        lblPreview->SetLabel(pwLabel);
        preview->SetPage(previewData);
    }
    wxEndBusyCursor();
    SetStatusText(wxString(i18n->getCParameter("l024")),1);
}
#endif


bool DropFiles::OnDropFiles (wxCoord x, wxCoord y, const wxArrayString& filenames) {
  for (size_t n = 0; n < filenames.Count(); n++) {
    if (wxFileName(filenames[n]).IsDir()) {
        wxMessageBox (_("Directories are not allowed!"),
                      _("Error"), wxOK | wxICON_EXCLAMATION);
        return false;
    }
  }
  m_frame->addInputFiles(filenames, "", false);
  return true;
}
