#include "config.h"

#ifdef HAVE_XSLT

#include <stdio.h>
#include <string.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>
#include "libstr.h"
#include "send.h"
#include "log.h"

#define VALUE_SIZE 16
#define XSLT_INDEX "/index.xslt\0"
#define XSLT_INDEX_LEN 12
#define MAX_PARAMETERS 200 /* must be even */
#define MAX_NAME_LEN 100

extern char *hs_conlen;

/* XSLT transform parameter functions
 */
static void add_parameter(const char **params, char *key, char *value, int *i) {
	size_t key_len;
	char *str;

	if ((key == NULL) || (value == NULL)) {
		return;
	}

	if (*i + 2 > MAX_PARAMETERS) {
		return;
	}

	key_len = strlen(key);
	if ((str = (char*)malloc(key_len + strlen(value) + 4)) == NULL) {
		return;
	}

	if (forbidden_chars_present(value)) {	
		return;
	} else if ((value = strdup(value)) == NULL) {
		return;
	}
	url_decode(value);

	if (strchr(value, '\'') == NULL) {
		sprintf(str, "%s%c'%s'", key, 0, value);
	} else if (strchr(value, '"') == NULL) {
		sprintf(str, "%s%c\"%s\"", key, 0, value);
	} else {
		free(value);
		return;
	}
	free(value);

	params[(*i)++] = str;
	params[(*i)++] = str + key_len + 1;
}

static void add_headerfield(t_session *session, const char **params, char *header, char *key, int *i) {
	char *value;

	if ((value = get_headerfield(header, session->headerfields)) != NULL) {
		add_parameter(params, key, value, i);
	}
}

static void add_parameter_line(const char **params, char *line, char *prefix, int *i) {
	char *item, *key, *value, name[MAX_NAME_LEN + 1];
	size_t prefix_len;

	prefix_len = strlen(prefix);
	while (line != NULL) {
		split_string(line, &item, &line, '&');
		if (split_string(item, &key, &value, '=') == -1) {
			continue;
		} else if (prefix_len + strlen(key) > MAX_NAME_LEN) {
			continue;
		}

		memcpy(name, prefix, prefix_len);
		strcpy(name + prefix_len, key);

		add_parameter(params, name, value, i);
	}
}

static const char **get_transform_parameters(t_session *session) {
	const char **params;
	int i = 0;
	char ip[MAX_IP_STR_LEN];
	char value[20], *cookie;

	if ((params = (const char**)malloc(sizeof(void*) * (MAX_PARAMETERS + 1))) != NULL) {
		add_parameter(params, "REQUEST_METHOD", session->method, &i);
		add_parameter(params, "REQUEST_URI", session->request_uri, &i);
		add_parameter(params, "SERVER_PROTOCOL", session->http_version, &i);
		if (inet_ntop(session->ip_address.family, &(session->ip_address.value), ip, MAX_IP_STR_LEN) != NULL) {
			add_parameter(params, "REMOTE_ADDR", ip, &i);
		}

		value[19] = '\0';
		snprintf(value, 9, "%d", session->binding->port);
		add_parameter(params, "SERVER_PORT", value, &i);
		add_parameter(params, "SERVER_NAME", *(session->host->hostname.item), &i);
		add_parameter(params, "SERVER_SOFTWARE", "Hiawatha v"VERSION, &i);
#ifdef HAVE_SSL
		if (session->binding->use_ssl) {
			add_parameter(params, "HTTP_SCHEME", "https", &i);
		} else
#endif
			add_parameter(params, "HTTP_SCHEME", "http", &i);

		if (session->remote_user != NULL) {
			if (session->http_auth == basic) {
				add_parameter(params, "AUTH_TYPE", "Basic", &i);
			} else if (session->http_auth == digest) {
				add_parameter(params, "AUTH_TYPE", "Digest", &i);
			}
			add_parameter(params, "REMOTE_USER", session->remote_user, &i);
		}

		add_headerfield(session, params, "Accept:", "HTTP_ACCEPT", &i);
		add_headerfield(session, params, "Accept-Charset:", "HTTP_ACCEPT_CHARSET", &i);
		add_headerfield(session, params, "Accept-Language:", "HTTP_ACCEPT_LANGUAGE", &i);
		add_headerfield(session, params, "Client-IP:", "HTTP_CLIENT_IP", &i);
		add_headerfield(session, params, "From:", "HTTP_FROM", &i);
		add_headerfield(session, params, "Host:", "HTTP_HOST", &i);
		add_headerfield(session, params, "Range:", "HTTP_RANGE", &i);
		add_headerfield(session, params, "Referer:", "HTTP_REFERER", &i);
		add_headerfield(session, params, "User-Agent:", "HTTP_USER_AGENT", &i);
		add_headerfield(session, params, "Via:", "HTTP_VIA", &i);
		add_headerfield(session, params, "X-Forwarded-For:", "HTTP_X_FORWARDED_FOR", &i);

		value[19] = '\0';
		snprintf(value, 19, "%d", session->return_code);
		add_parameter(params, "HTTP_RETURN_CODE", value, &i);
		if (session->error_code != -1) {
			snprintf(value, 9, "%d", session->error_code);
			add_parameter(params, "HTTP_GENERATED_ERROR", value, &i);
		}

		if (session->vars != NULL) {
			add_parameter(params, "QUERY_STRING", session->vars, &i);
		}
		if (session->body != NULL) {
			value[19] = '\0';
			snprintf(value, 19, "%ld", session->content_length);
			add_parameter(params, "CONTENT_LENGTH", value, &i);
			add_headerfield(session, params, "Content-Type:", "CONTENT_TYPE", &i);
		}

		add_parameter_line(params, session->vars, "GET_", &i);
		add_parameter_line(params, session->body, "POST_", &i);
		if (session->cookie != NULL) {
			add_parameter_line(params, session->cookie, "COOKIE_", &i);
		} else if ((cookie = get_headerfield("Cookie:", session->headerfields)) != NULL) {
			add_parameter_line(params, cookie, "COOKIE_", &i);
		}

		params[i] = NULL;
	}

	return params;
}

static void dispose_transform_parameters(const char **params) {
	char **param;

	if (params == NULL) {
		return;
	}

	param = (char**)params;
	while (*param != NULL) {
		free(*param);
		param += 2;
	}

	free(params);
}

/* XSLT transform functions
 */
void init_xslt_module() {
	xmlInitMemory();
	xmlInitParser();
}

bool can_transform_with_xslt(t_session *session) {
	char *ext, *xslt, *slash;
	size_t len;
	FILE *fp;

	/* Check virtual host settings
	 */
	if (session->host->use_xslt == false) {
		return false;
	}

	/* Check extension
	 */
	if ((ext = file_extension(session->file_on_disk)) == NULL) {
		return false;
	} else if (strcmp(ext, "xml") != 0) {
		return false;
	}

	/* Check for XSLT existence: <file>.xslt
	 */
	if ((len = strlen(session->file_on_disk)) < 4) {
		return false;
	} else if ((xslt = (char*)malloc(len + 2)) == NULL) {
		return false;
	}
	memcpy(xslt, session->file_on_disk, len - 3);
	memcpy(xslt + len - 3, "xslt\0", 5);
	if ((fp = fopen(xslt, "r")) != NULL) {
		fclose(fp);
		session->xslt_file = xslt;
		return true;
	} else {
		free(xslt);
	}

	/* Check for XSLT existence: index.xslt in directory
	 */
	if ((slash = strrchr(session->file_on_disk, '/')) == NULL) {
		return false;
	}
	len = slash - session->file_on_disk;
	if ((xslt = (char*)malloc(len + XSLT_INDEX_LEN)) == NULL) {
		return false;
	}
	memcpy(xslt, session->file_on_disk, len);
	memcpy(xslt + len, XSLT_INDEX, XSLT_INDEX_LEN);
	if ((fp = fopen(xslt, "r")) != NULL) {
		fclose(fp);
		session->xslt_file = xslt;
		return true;
	} else {
		free(xslt);
	}

	/* Check for XSLT existence: /index.xslt
	 */
	if ((xslt = (char*)malloc(session->host->website_root_len + XSLT_INDEX_LEN)) == NULL) {
		return false;
	}
	memcpy(xslt, session->host->website_root, session->host->website_root_len);
	memcpy(xslt + session->host->website_root_len, XSLT_INDEX, XSLT_INDEX_LEN);
	if ((fp = fopen(xslt, "r")) != NULL) {
		fclose(fp);
		session->xslt_file = xslt;
		return true;
	} else {
		free(xslt);
	}

	return false;
}

int transform_xml(t_session *session) {
	xmlDocPtr data_xml, style_xml, result_xml;
	xsltStylesheetPtr xslt;
	xmlOutputBufferPtr output;
	char value[VALUE_SIZE + 1];
	const char **params;
	int result = 200;

	/* Read XML data
	 */
	data_xml = xmlReadFile(session->file_on_disk, NULL, 0);
	if (data_xml == NULL) {
		log_file_error(session, session->file_on_disk, "invalid XML");
		return 500;
	}

	/* Read XSLT sheet
	 */
	style_xml = xmlReadFile(session->xslt_file, NULL, 0);
	if (style_xml == NULL) {
		log_file_error(session, session->xslt_file, "invalid XML");
		xmlFreeDoc(data_xml);
		return 500;
	}
	xslt = xsltParseStylesheetDoc(style_xml);
	if (xslt == NULL) {
		log_file_error(session, session->xslt_file, "invalid XSLT");
		xmlFreeDoc(style_xml);
		xmlFreeDoc(data_xml);
		return 500;
	}

	/* Mimetype
	 */
	session->mimetype = get_mimetype(session->file_on_disk, session->config->mimetype);
	if (xslt->method != NULL) {
		if (strcmp((char*)xslt->method, "html") == 0) {
			session->mimetype = get_mimetype(".html", session->config->mimetype);
		}
	}

	/* Transform XML to HTML
	 */
	params = get_transform_parameters(session);
	result_xml = xsltApplyStylesheet(xslt, data_xml, params);
	dispose_transform_parameters(params);

	/* Handle transformation result
	 */
	if (result_xml == NULL) {
		log_file_error(session, session->file_on_disk, "transformation error");
		xsltFreeStylesheet(xslt);
		xmlFreeDoc(data_xml);
		return 500;
	}
	if ((output = xmlAllocOutputBuffer(NULL)) == NULL) {
		xsltFreeStylesheet(xslt);
		xmlFreeDoc(data_xml);
		return 500;
	}
    if (xsltSaveResultTo(output, result_xml, xslt) == -1) {
		log_file_error(session, session->file_on_disk, "transformation error");
		xmlFreeDoc(result_xml);
		xsltFreeStylesheet(xslt);
		xmlFreeDoc(data_xml);
		return 500;
	}

	/* Print HTML
	 */
	value[VALUE_SIZE] = '\0';
	if (send_header(session, 200) == -1) {
		result = 500;
	} else if (send_buffer(session, hs_conlen, 16) == -1) {
		result = -1;
	} else if (snprintf(value, VALUE_SIZE, "%d\r\n\r\n", output->buffer->use) == -1) {
		result = -1;
	} else if (send_buffer(session, value, strlen(value)) == -1) {
		result = -1;
	} else if (send_buffer(session, (char*)output->buffer->content, output->buffer->use) == -1) {
		result = -1;
	}

	/* Free buffers
	 */
	xmlOutputBufferClose(output);
	xmlFreeDoc(result_xml);
	xsltFreeStylesheet(xslt);
	xmlFreeDoc(data_xml);

	return result;
}

#endif
