#ifndef _SERVERCONFIG_H
#define _SERVERCONFIG_H

#include "config.h"
#include <stdbool.h>
#include <pthread.h>
#include "global.h"
#include "mimetype.h"
#include "libfs.h"
#include "liblist.h"
#include "userconfig.h"
#ifdef HAVE_SSL
#include "libssl.h"
#endif

typedef enum { root, part } t_pathmatch;
typedef enum { basic, digest } t_auth_method;
typedef enum { hiawatha, common, extended } t_log_format;

typedef struct type_cgi_handler {
	char          *handler;
	t_charlist    extension;
	struct type_cgi_handler *next;
} t_cgi_handler;

typedef struct type_connect_to {
	int           port;
	char          *host;
	bool          available;

	struct type_connect_to *next;
} t_connect_to;

typedef struct type_cgi_session {
	t_ipv4 client_ip;
	t_connect_to  *connect_to;
	int timer;

	struct type_cgi_session *next;
} t_cgi_session;

typedef struct type_fcgi_server {
	char          *fcgi_id;
	t_connect_to  *connect_to;
	t_charlist    extension;
	int           session_timeout;
	char          *chroot;
	size_t        chroot_len;

	t_cgi_session *cgi_session_list[256];
	pthread_mutex_t cgi_session_mutex[256];

	struct type_fcgi_server *next;
} t_fcgi_server;

typedef struct type_throttle {
	char          *filetype;
	unsigned long upload_speed;

	struct type_throttle *next;
} t_throttle;

typedef struct type_binding {
	int           port;
	t_ipv4        interface;
	char          *binding_id;
#ifdef HAVE_SSL
	bool          use_ssl;
	char          *server_key;
	SSL_CTX       *ssl_context;
	char          *ca_cert;
	int           verify_depth;
#endif

	bool          enable_trace;
	bool          enable_alter;
	int           max_keepalive;
	long          max_request_size;
	int           time_for_1st_request;
	int           time_for_request;

	int           socket;

	struct type_binding *next;
} t_binding;

typedef struct type_directory {
	char          *path;
	t_pathmatch   path_match;
	char          *wrap_cgi;
	t_groups      groups;
	char          *start_file;
	bool          execute_cgi;
	bool          execute_cgiset;
	bool          show_index;
	bool          show_index_set;
	bool          follow_symlinks;
	bool          follow_symlinks_set;
	bool          use_gz_file;
	bool          use_gz_file_set;
	t_auth_method auth_method;
	char          *passwordfile;
	char          *groupfile;
	t_charlist    required_group;
	t_accesslist  *access_list;
	t_accesslist  *alter_list;
	t_charlist    alter_group;
	mode_t        alter_fmode;
	t_charlist    image_referer;
	char          *imgref_replacement;
	t_keyvalue    *envir_str;

	/* Uploadspeed control
	 */
	int           max_clients;
	int           nr_of_clients;
	long          upload_speed;
	long          session_speed;
	pthread_mutex_t client_mutex;

	struct type_directory *next;
} t_directory;

typedef struct type_host {
	char          *website_root;
	char          *start_file;
	char          *error_handler;
	int           return_code;
	char          *access_logfile;
	FILE          *access_fileptr;
	FILE          **access_fp;
	char          *error_logfile;
	t_charlist    hostname;
	bool          user_websites;
	bool          execute_cgi;
	bool          show_index;
	bool          use_gz_file;
	bool          require_resolve_ip;
	char          *login_message;
	char          *passwordfile;
	t_auth_method auth_method;
	char          *groupfile;
	char          *index_style;
	t_charlist    required_binding;
	t_denybotlist *deny_bot;
	t_charlist    required_group;
	t_charlist    alter_group;
	char          *wrap_cgi;
	t_groups      groups;
	t_charlist    volatile_object;
	t_accesslist  *access_list;
	t_accesslist  *alter_list;
	mode_t        alter_fmode;
	t_charlist    image_referer;
	char          *imgref_replacement;
	t_keyvalue    *envir_str;
	t_keyvalue    *alias;
#ifdef HAVE_SSL
    bool          require_ssl;
#endif
	bool          prevent_cmdi;
	bool          prevent_sqli;
	bool          prevent_xss;
	bool          follow_symlinks;
	t_charlist    fast_cgi;

	struct type_host *next;

	bool          secure_url;
} t_host;

typedef struct type_config {
	char          *throttle_config;
	char          *mimetype_config;

	char          *server_root;
	uid_t         server_uid;
	gid_t         server_gid;
	t_groups      groups;
	char          *server_string;
	t_binding     *binding;
	t_log_format  log_format;
#ifdef HAVE_SSL
	char          *server_key;
	char          *ca_cert;
	int           verify_depth;
	char          *dh_file;
	char          *allowed_ciphers;
#endif
	t_charlist    cgi_extension;
	int           total_connections;
	int           connections_per_ip;
	int           time_for_cgi;
	bool          kill_timedout_cgi;
	char          *system_logfile;
	char          *garbage_logfile;
	char          *pidfile;
	t_accesslist  *logfile_mask;
	char          *user_directory;
	bool          user_directory_set;
	
	t_mimetype    *mimetype;
	t_host        *first_host;
	t_directory   *directory;
	t_throttle    *throttle;
	t_cgi_handler  *cgi_handler;
	char          *cgi_wrapper;
	bool          wrap_user_cgi;

	int           ban_on_garbage;
	int           ban_on_max_per_ip;
	int           ban_on_flooding;
	int           ban_on_max_request_size;
	int           ban_on_cmdi;
	int           ban_on_sqli;
	int           ban_on_timeout;
	bool          kick_on_ban;
	bool          reban_during_ban;
	int           flooding_count;
	int           flooding_time;
	int           reconnect_delay;
	t_accesslist  *banlist_mask;
	t_fcgi_server *fcgi_server;

#ifdef HAVE_CACHE
	off_t         cache_size;
	off_t         cache_max_filesize;
	off_t         cache_min_filesize;
#endif

#ifdef HAVE_COMMAND
	t_binding     *command_port;
#endif
} t_config;

t_config *default_config(void);
int check_configuration(t_config *config);
int read_main_configfile(char *configfile, t_config *config, bool config_check);
int read_user_configfile(char *configfile, t_host *host, t_tempdata **tempdata);
t_host *get_hostrecord(t_host *host, char *hostname, t_binding *binding);
int read_throttleconfig(char *configfile, t_throttle **throttle);
unsigned short get_throttlespeed(char *type, t_throttle *throttle);

#ifdef HAVE_SSL
#ifndef HAVE_DEV_URANDOM
void fill_random_buffer(t_config *config, char *buffer, int size);
#endif
#endif

#endif
