/* Hiawatha | command.c
 *
 * CommandChannel routines
 */
#include "config.h"

#ifdef HAVE_COMMAND

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <time.h>
#include "libstr.h"
#include "libmd5.h"
#include "command.h"
#include "client.h"
#ifdef HAVE_CACHE
#include "cache.h"
#endif

#define MAX_IDLE_TIME  30
#define MAX_CMD_SIZE   64
#define TIMESTAMP_SIZE 50

#define KILOBYTE    1024
#define MEGABYTE 1048576

static t_admin *adminlist, *loop, *next;
static char *prompt = "\033[01;34mhcc>\033[00m ";

static pthread_mutex_t command_mutex;

static char start_time[TIMESTAMP_SIZE];
extern char *version_string;

static volatile unsigned long counters[COUNTER_MAX];
static volatile unsigned long long transfer[TRANSFER_MAX];

void increment_counter(int counter) {
	counters[counter]++;
}

void increment_transfer(int counter, long bytes) {
	transfer[counter] += bytes;
}

static void clear_counters(void) {
	int i;

	for (i = 0; i < COUNTER_MAX; i++) {
		counters[i] = 0;
	}
	for (i = 0; i < TRANSFER_MAX; i++) {
		transfer[i] = 0;
	}
}

/* Initialize the CommandChannel
 */
void init_command_module(void) {
	time_t t;
	struct tm *s;

    adminlist = NULL;
	loop = NULL;
	next = NULL;
	pthread_mutex_init(&command_mutex, NULL);

	time(&t);
	s = localtime(&t);
	start_time[TIMESTAMP_SIZE - 1] = '\0';
	strftime(start_time, TIMESTAMP_SIZE - 1, "%a %d %b %Y %T %z", s);

	clear_counters();
}

/* An administrator has connected to the CommandChannel.
 */
int add_admin(int sock) {
	t_admin *new;

	if ((new = (t_admin*)malloc(sizeof(t_admin))) != NULL) {
		if ((new->fp = fdopen(sock, "r+")) != NULL) {
			fprintf(new->fp, "\n\033[02;31mHiawatha v"VERSION" CommandChannel\033[00m\n");
			fprintf(new->fp, "Password: \0337\033[00;30;40m"); // Save cursor position and change color to black
			fflush(new->fp);

			new->next = adminlist;
			new->socket = sock;
			new->authenticated = false;
			new->timer = MAX_IDLE_TIME;
			adminlist = new;


			return 0;
		} else {
			return -1;
		}
	} else {
		return -1;
	}
}

/* An administrator has left the CommandChannel
 */
void remove_admin(int sock) {
	t_admin *deadone = NULL, *check;

	if (adminlist != NULL) {
		if (adminlist->socket == sock) {
			deadone = adminlist;
			adminlist = adminlist->next;
		} else {
			check = adminlist;
			while (check->next != NULL) {
				if (check->next->socket == sock) {
					deadone = check->next;
					check->next = deadone->next;
					break;
				}
				check = check->next;
			}
		}
	}

	if (deadone != NULL) {
		fclose(deadone->fp);
		free(deadone);
	}
}

/* Disconnect al the administrators.
 */
void disconnect_admins(void) {
	t_admin *admin;

	while (adminlist != NULL) {
		admin = adminlist;
		adminlist = adminlist->next;

		close(admin->socket);
		free(admin);
	}
}

/* Return the first admin record.
 */
t_admin *first_admin(void) {
	if ((loop = adminlist) != NULL) {
		next = loop->next;
		return loop;
	} else {
		next = NULL;
	}

	return NULL;
}

/* Return the next admin record.
 */
t_admin *next_admin(void) {
	if ((loop = next) != NULL) {
		next = loop->next;
		return loop;
	}

	return NULL;
}

/* Check administratos (only auto-logout timers for now).
 */
void check_adminlist(void) {
	t_admin *admin, *prev_admin = NULL, *next_admin;

	admin = adminlist;
	while (admin != NULL) {
		next_admin = admin->next;
		if (admin->timer == 0) {
			fprintf(admin->fp, "\033[00m(auto-logout)\n");
			fflush(admin->fp);
			close(admin->socket);
			if (prev_admin == NULL) {
				adminlist = next_admin;
			} else {
				prev_admin->next = next_admin;
			}
			free(admin);
			admin = prev_admin;
		} else {
			admin->timer--;
		}
		prev_admin = admin;
		admin = next_admin;
	}
}

/* Show help info.
 */
static void show_help(FILE *fp) {
	fprintf(fp,	"  ban <ip>[,<time>]: ban an IP (for <time> seconds)\n"
				"  clear screen     : clear the screen\n"
#ifdef HAVE_CACHE
				"        cache      : remove all unlocked files from the cache\n"
#endif
				"        counters   : set all counters to zero\n"
				"  kick <id>        : kick client by its id (show clients)\n"
				"       <ip>        : kick client by its IP\n"
				"       all         : disconnect all clients\n"
				"  show bans        : show the ban list\n"
#ifdef HAVE_CACHE
				"       cache       : show the file in the cache\n"
#endif
				"       clients     : show the connected clients\n"
				"       status      : show general information\n"
				"  quit/exit        : quit CommandChannel\n"
				"  unban <ip>       : unban an IP address\n");
}

static void show_status(FILE *fp) {
	fprintf(fp, "  %s\n", version_string);
	fprintf(fp, "  Start time        : %s\n\n", start_time);

#ifdef HAVE_CACHE
	fprintf(fp, "  Size of cache     : %9.1f kB\n", ((float)size_of_cache()) / KILOBYTE);
#endif
	fprintf(fp, "  Number of clients : %7d\n", number_of_clients());
	fprintf(fp, "  Number of bans    : %7d\n\n", number_of_bans());

	fprintf(fp, "  Clients served    : %7lu\n", counters[COUNTER_CLIENT]);
	fprintf(fp, "  Files requested   : %7lu\n", counters[COUNTER_FILE]);
	fprintf(fp, "  CGIs requested    : %7lu\n", counters[COUNTER_CGI]);
	fprintf(fp, "  Indexes requested : %7lu\n", counters[COUNTER_INDEX]);
	fprintf(fp, "  Data received     : %9.1f MB\n", ((float)transfer[TRANSFER_RECEIVED]) / MEGABYTE);
	fprintf(fp, "  Data send         : %9.1f MB\n\n", ((float)transfer[TRANSFER_SEND]) / MEGABYTE);

	fprintf(fp, "  Clients banned    : %7lu\n", counters[COUNTER_BAN]);
	fprintf(fp, "  Connections denied: %7lu\n", counters[COUNTER_DENY]);
}

static int run_command(char *line, FILE *fp, t_config *config) {
	char *cmd, *param, *param2;
	unsigned long ip;
	int retval = 0, time, id, count;
	
	split_string(line, &cmd, &param, ' ');

	if (strcmp(cmd, "ban") == 0) {
		if (param == NULL) {
			fprintf(fp, "  ban what?");
		} else {
			if (split_string(param, &param, &param2, ',') == 0) {
				time = str2int(param2);
			} else {
				time = TIMER_OFF;
			}
			if (parse_ip(param, &ip) == -1) {
				fprintf(fp, "  invalid IP!\n");
			} else switch (count = ban_ip(ip, time, config->kick_on_ban)) {
				case -1:
					fprintf(fp, "  error while banning!\n");
					break;
				case 0:
					fprintf(fp, "  IP rebanned\n");
					break;
				default:
					fprintf(fp, "  %d IPs banned", count);
					if (config->kick_on_ban) {
						fprintf(fp, " and kicked\n");
					} else {
						fprintf(fp, "\n");
					}
			}
		}
	} else if (strcmp(cmd, "clear") == 0) {
		if (param == NULL) {
			fprintf(fp, "  clear what?\n");
#ifdef HAVE_CACHE
		} else if (strcmp(param, "cache") == 0) {
			fprintf(fp, "  %d files removed from the cache.\n", clear_cache());
#endif	
		} else if (strcmp(param, "screen") == 0) {
			fprintf(fp, "\033[2J\033[H");
		} else if (strcmp(param, "counters") == 0) {
			clear_counters();
		} else {
			fprintf(fp, "  clear it yourself!\n");
		}
	} else if (strcmp(cmd, "help") == 0) {
		show_help(fp);
	} else if (strcmp(cmd, "kick") == 0) {
		if (param == NULL) {
			fprintf(fp, "  kick what?\n");
		} else if (strcmp(param, "all") == 0) {
			fprintf(fp, "   %d clients have been kicked\n", disconnect_clients(config));
		} else if ((id = str2int(param)) != -1) {
			if (kick_client(id) == 1) {
				fprintf(fp, "  client has been kicked\n");
			} else {
				fprintf(fp, "  client not found!\n");
			}
		} else if (parse_ip(param, &ip) != -1) {
			fprintf(fp, "  %d clients have been kicked\n", kick_ip(ip));
		} else if (strcmp(param, "yourself") == 0) {
			fprintf(fp, "  I can't. I'm a computer.\n");
		} else if (strcmp(param, "me") == 0) {
			fprintf(fp, "  you need help...\n");
		} else {
			fprintf(fp, "  %s kicked back. Ouch!\n", param);
		}
	} else if (strcmp(cmd, "show") == 0) {
		if (param == NULL) {
			fprintf(fp, "  show what?\n");
		} else if (strcmp(param, "bans") == 0) {
			print_ban_list(fp);
#ifdef HAVE_CACHE
		} else if (strcmp(param, "cache") == 0) {
			print_cache_list(fp);
#endif
		} else if (strcmp(param, "clients") == 0) {
			print_client_list(fp);
		} else if (strcmp(param, "status") == 0) {
			show_status(fp);
		} else {
			fprintf(fp, "  can't show that!\n");
		}
	} else if (strcmp(cmd, "unban") == 0) {
		if (param == NULL) {
			fprintf(fp, "  unban who?\n");
		} else if (strcmp(param, "all") == 0) {
			fprintf(fp, "  %d IPs have been unbanned\n", unban_ip(0));
		} else if (parse_ip(param, &ip) == -1) {
			fprintf(fp, "  invalid IP!\n");
		} else if (unban_ip(ip) == 1) {
			fprintf(fp, "  IP has been unbanned\n");
		} else {
			fprintf(fp, "  IP not found!\n");
		}
	} else if ((strcmp(cmd, "quit") == 0) || (strcmp(cmd, "exit") == 0)) {
		retval = cc_DISCONNECT;
	} else if (strcmp(cmd, "") != 0) {
		fprintf(fp, "  unknown command!\n");
	}

	return retval;
}

/* Handle a administrator command.
 */
int handle_admin(t_admin *admin, t_config *config) {
	int retval = cc_OKE;
	char line[MAX_CMD_SIZE + 1], *pwd, encrypted[33]; //, salt[3];

	if (fgets(line, MAX_CMD_SIZE, admin->fp) != NULL) {
		line[MAX_CMD_SIZE] = '\0';
		admin->timer = MAX_IDLE_TIME;
		if (strlen(line) >= MAX_CMD_SIZE - 1) {
			do {
				fgets(line, MAX_CMD_SIZE, admin->fp);
			} while (strlen(line) >= MAX_CMD_SIZE - 1);
			if (admin->authenticated == false) {
				fprintf(admin->fp, "\033[00m");
				retval = cc_DISCONNECT;
			}
			fprintf(admin->fp, "  don't do that!\n");
		} else if (admin->authenticated) {
			retval = run_command(remove_spaces(line), admin->fp, config);
		} else {
			fprintf(admin->fp, "\0338\033[K\0338\n"); // Restore cursor position and color and erase to end of line
			pwd = remove_spaces(line);
			md5_hash(pwd, strlen(pwd), encrypted);
			if ((admin->authenticated = (strcmp(encrypted, config->command_port->binding_id) == 0)) == false) {
				retval = cc_DISCONNECT;
				fprintf(admin->fp, "Password incorrect\n\n");
			} else {
				fprintf(admin->fp, "Welcome. Use 'help' for help. Auto-logout after %d seconds idle-time.\n\n", MAX_IDLE_TIME);
				fflush(admin->fp);
			}
		}
	} else {
		fprintf(admin->fp, "  read error!\n");
		retval = cc_DISCONNECT;
	}

	if (retval == 0) {
		fprintf(admin->fp, "%s", prompt);
		fflush(admin->fp);
	}

	return retval;
}

#endif
