#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include "libstr.h"

typedef struct type_line {
	char *key, *value, *file;
	int linenr;
	struct type_line *next;
} t_line;

t_line *last_result(t_line *config) {
	if (config != NULL) {
		while (config->next != NULL) {
			config = config->next;
		}
	}
	
	return config;
}

t_line *add_result(t_line *config, char *key, char *value, char *file, int linenr) {
	t_line *new;

	if (config == NULL) {
		if ((new = config = (t_line*)malloc(sizeof(t_line))) == NULL) {
			return config;
		}
	} else {
		new = last_result(config);
		if ((new->next = (t_line*)malloc(sizeof(t_line))) == NULL) {
			return config;
		}
		new = new->next;
	}
	new->next = NULL;

	new->key = key;
	new->value = value;
	new->file = file;
	new->linenr = linenr;

	return config;
}

t_line *search_key(t_line *config, char *key) {
	t_line *result = NULL;

	while (config != NULL) {
		if (strcmp(config->key, key) == 0) {
			if (config->value == NULL) {
				printf("'%s' on line %d in "CONFIG_DIR"/%s requires a parameter.\n", config->key, config->linenr, config->file);
				exit(-1);
			} else {
				result = add_result(result, config->key, config->value, config->file, config->linenr);
			}
		}
		config = config->next;
	}

	return result;
}

bool in_result(t_line *result, char *value) {
	while (result != NULL) {
		if (strcmp(result->value, value) == 0) {
			return true;
		}
		result = result->next;
	}

	return false;
}

void dispose_result(t_line *config) {
	t_line *prev;

	while (config != NULL) {
		prev = config;
		config = config->next;
		free(prev);
	}
}

t_line *read_file(char *config_file, bool handle_include) {
	FILE *fp;
	char *line, *data, *value;
	t_line *config = NULL, *aliases = NULL, *item, *alias;
	bool is_alias;
	int linenr = 1;

	if ((fp = fopen(config_file, "r")) != NULL) {
		if ((line = (char*)malloc(1025)) != NULL) {
			while (fgets(line, 1024, fp) != NULL) {
				data = strip_line(line);
				if ((data[0] == '#') || (data[0] == '\0')) {
					continue;
				}

				if (handle_include && (strncasecmp(data, "include ", 8) == 0)) {
					if ((item = last_result(config)) == NULL) {
						if ((config = read_file(strdup(data + 8), false)) == NULL) {
							return NULL;
						}
					} else {
						if ((item->next = read_file(strdup(data + 8), false)) == NULL) {
							return NULL;
						}
					}
					continue;
				} else if (strncmp(data, "set ", 4) == 0) {
					is_alias = true;
					data += 4;
				} else {
					is_alias = false;
				}

				data = strdup(data);
				split_string(data, &data, &value, '=');
				if (is_alias) {
					aliases = add_result(aliases, data, value, config_file, linenr);
				} else {
					config = add_result(config, strlower(data), value, config_file, linenr);
				}

				linenr++;
			}
			free(line);
		}
		fclose(fp);

		item = config;
		while (item != NULL) {
			alias = aliases;
			while (alias != NULL) {
				if (item->value != NULL) {
					if ((value = str_replace(item->value, alias->key, alias->value)) != NULL) {
						item->value = value;
					}
				}
				alias = alias->next;
			}
			item = item->next;
		}
		dispose_result(aliases);
	} else {
		printf("Warning: can't open %s\n", config_file);
	}

	return config;
}

bool is_ip_address(char *str) {
	char *digit;
	int digits = 0;

	while (str != NULL) {
		split_string(str, &digit, &str, '.');
		if (str2int(digit) == -1) {
			return false;
		}
		digits++;
	}

	return (digits == 4);
}

int check_main_config() {
	int errors = 0;
	t_line *config, *haystack, *needles, *needle;
	char *item, *rest, *info;
	bool inside_section;

	if ((config = read_file("httpd.conf", true)) == NULL) {
		printf("Can't find httpd.conf.\n");
		return 1;
	}

	/* FastCGI Id check
	 */
	haystack = search_key(config, "fcgiserverid");
	needles = needle = search_key(config, "fastcgi");
	while (needle != NULL) {
		if (in_result(haystack, needle->value) == false) {
			printf("FastCGI ServerId '%s' on line %d in "CONFIG_DIR"/%s not found in FastCGIserver section.\n", needle->value, needle->linenr, needle->file);
			errors++;
		}
		needle = needle->next;
	}
	dispose_result(haystack);
	dispose_result(needles);

	/* Binding Id check
	 */
	haystack = search_key(config, "bindingid");
	needles = needle = search_key(config, "requiredbinding");
	while (needle != NULL) {
		if (in_result(haystack, needle->value) == false) {
			printf("RequiredBinding '%s' on line %d in "CONFIG_DIR"/%s not found in Binding section.\n", needle->value, needle->linenr, needle->file);
			errors++;
		}
		needle = needle->next;
	}
	dispose_result(haystack);
	dispose_result(needles);

	/* Extension check
	 */
	haystack = NULL;
	needles = needle = search_key(config, "cgiextension");
	while (needle != NULL) {
		rest = strdup(needle->value);
		while (rest != NULL) {
			split_string(rest, &item, &rest, ',');
			if (in_result(haystack, item) == false) {
				haystack = add_result(haystack, needle->key, item, needle->file, needle->linenr);
			} else {
				printf("Duplicate extension found (%s) in CGIextension.\n", item);
				errors++;
			}
		}
		needle = needle->next;
	}
	dispose_result(needles);

	needles = needle = search_key(config, "cgihandler");
	while (needle != NULL) {
		rest = strdup(needle->value);
		split_string(rest, &info, &rest, ':');
		while (rest != NULL) {
			split_string(rest, &item, &rest, ',');
			if (in_result(haystack, item) == false) {
				haystack = add_result(haystack, needle->key, item, needle->file, needle->linenr);
			} else {
				printf("Duplicate extension found (%s) in CGIhandler %s.\n", item, info);
				errors++;
			}
		}
		needle = needle->next;
	}
	dispose_result(needles);

	dispose_result(haystack);

	/* Default-website hostname check
	 */
	inside_section = false;
	haystack = config;
	while (haystack != NULL) {
		if (strncmp(haystack->key, "virtualhost", 11) == 0) {
			inside_section = true;
		} else if (strcmp(haystack->key, "}") == 0) {
			inside_section = false;
		} else if (inside_section == false) {
			if (strcmp(haystack->key, "hostname") == 0) {
				if (is_ip_address(haystack->value) == false) {
					printf("Warning: it is wise to use your IP address as the hostname of the default website (line %d in "CONFIG_DIR"/%s) and give it a blank webpage. By doing so, automated webscanners won't find your possible vulnerable website.\n", haystack->linenr, haystack->file);
				}
				break;
			}
		}

		haystack = haystack->next;
	}

	/* ... check
	 */

	return errors;
}

int main(int argc, char *argv[]) {
	int errors_found = 0;
	char *config_dir = CONFIG_DIR;
	bool quiet = false;

	if (argc > 1) {
		if (strcmp(argv[1], "-q") == 0) {
			quiet = true;
		}
	}

	chdir(config_dir);
	errors_found += check_main_config();

	if ((quiet == false) && (errors_found == 0))  {
		printf("No non-fatal errors found in the Hiawatha configuration.\n");
	}

	return errors_found;
}
