#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <dirent.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "client.h"
#include "log.h"

#define ip_index(ip) (ip & 255)
#define TIMER_OFF -1

typedef struct type_client {
	t_session *session;
	int delayed_remove;

	struct type_client *next;
} t_client;

typedef struct type_banned {
	long ip;
	int  timer;
	int  bantime;
	unsigned long connect_attempts;
	
	struct type_banned *next;
} t_banned;

static t_client *client_list[256];
static pthread_mutex_t client_mutex[256];
static t_banned *banlist;
static pthread_mutex_t ban_mutex;

/* Initialize this module.
 */
void init_client_module(void) {
	int i;

	for (i = 0; i < 256; i++) {
		client_list[i] = NULL;
		pthread_mutex_init(&client_mutex[i], NULL);
	}
	banlist = NULL;
	pthread_mutex_init(&ban_mutex, NULL);
}

/* Add the session record of a client to the client_list.
 */
int add_client(t_session *session) {
	t_client *new;
	int i;

	if ((new = (t_client*)malloc(sizeof(t_client))) != NULL) {
		new->session = session;
		new->delayed_remove = TIMER_OFF;

		i = ip_index(session->ip_address);

		pthread_mutex_lock(&client_mutex[i]);

		new->next = client_list[i];
		client_list[i] = new;

		pthread_mutex_unlock(&client_mutex[i]);

		return 0;
	} else {
		return -1;
	}
}

/* Remember the clientrecord for flooding prevention
 */
void mark_client_for_removal(t_session *session, int delay) {
	t_client *list;
	int i;

	i = ip_index(session->ip_address);

	pthread_mutex_lock(&client_mutex[i]);

	list = client_list[i];
	while (list != NULL) {
		if (list->session == session) {
			close_socket(list->session);
			list->delayed_remove = delay - 1;
			break;
		}
		list = list->next;
	}

	pthread_mutex_unlock(&client_mutex[i]);
}

/* Check the delayed_remove timers and remove client
 * when timer has reached 0
 */
void check_delayed_remove_timers(void) {
	t_client *client, *prev = NULL, *next;
	int i;

	for (i = 0; i < 256; i++) {
		pthread_mutex_lock(&client_mutex[i]);

		client = client_list[i];
		while (client != NULL) {
			next = client->next;
			switch (client->delayed_remove) {
				case TIMER_OFF:
					break;
				case 0:
					free(client->session);
					free(client);
					client = prev;
					if (prev == NULL) {
						client_list[i] = next;
					} else {
						prev->next = next;
					}
					break;
				default:
					client->delayed_remove--;
			}
			prev = client;
			client = next;
		}

		pthread_mutex_unlock(&client_mutex[i]);
	}
}

/* Remove a client from the client_list.
 */
void remove_client(t_session *session, bool free_session) {
	t_client *to_be_removed = NULL, *list;
	int i;

	i = ip_index(session->ip_address);

	pthread_mutex_lock(&client_mutex[i]);

	if (client_list != NULL) {
		if (client_list[i]->session == session) {
			to_be_removed = client_list[i];
			client_list[i] = client_list[i]->next;
		} else {
			list = client_list[i];
			while (list->next != NULL) {
				if (list->next->session == session) {
					to_be_removed = list->next;
					list->next = to_be_removed->next;
					break;
				}
				list = list->next;
			}
		}
	}

	pthread_mutex_unlock(&client_mutex[i]);

	if (to_be_removed != NULL) {
		if (free_session) {
			close_socket(to_be_removed->session);
			free(to_be_removed->session);
		}
		free(to_be_removed);
	}
}

/* Check whether to allow or deny a new connection.
 */
int connection_allowed(long ip, int maxperip, int maxtotal) {
	bool banned = false;
	t_banned *ban;
	int perip = 0, total = 0, i;
	t_client *client;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->connect_attempts++;
			banned = true;
			break;
		}
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);

	if (banned) {
		return ca_BANNED;
	}

	for (i = 0; i < 256; i++) {
		pthread_mutex_lock(&client_mutex[i]);

		client = client_list[i];
		while (client != NULL) {
			if (client->session->ip_address == ip) {
				perip++;
			}
			if (client->delayed_remove == TIMER_OFF) {
				total++;
			}
			client = client->next;
		}

		pthread_mutex_unlock(&client_mutex[i]);
	}

	if (perip < maxperip) {
		if (total < maxtotal) {
			return total;
		} else {
			return ca_TOOMUCH_TOTAL;
		}
	} else {
		return ca_TOOMUCH_PERIP;
	}
}

/* Disconnect all connected clients.
 */
void disconnect_clients(t_config *config) {
	t_client *client;
	t_directory *dir;
	int max_wait = 30, i;

	for (i = 0; i < 256; i++) {
		pthread_mutex_lock(&client_mutex[i]);

		client = client_list[i];
		while (client != NULL) {
			client->session->force_quit = true;
			client = client->next;
		}

		pthread_mutex_unlock(&client_mutex[i]);
	}

	for (i = 0; i < 256; i++) {
		while ((client_list[i] != NULL) && (max_wait-- > 0)) {
			usleep(100000);
		}
	}

	dir = config->directory;
	while (dir != NULL) {
		dir->nr_of_clients = 0;
		dir = dir->next;
	}
}

/* Kick an IP address.
 */
void kick_ip(long ip) {
	t_client *client;

	pthread_mutex_lock(&client_mutex[ip_index(ip)]);

	client = client_list[ip_index(ip)];
	while (client != NULL) {
		if (client->session->ip_address == ip) {
			client->session->force_quit = true;
		}
		client = client->next;
	}

	pthread_mutex_unlock(&client_mutex[ip_index(ip)]);
}

/* Check if the client is flooding the server with requests
 */
bool client_is_flooding(t_session *session) {
	time_t time_passed;

	time_passed = time(NULL) - session->flooding_timer + 1;

	return ((session->kept_alive * session->config->flooding_time) > (session->config->flooding_count * time_passed));
}

#ifdef HAVE_COMMAND
/* Extra client functions
 */
void print_client_list(FILE *fp) {
	t_client *client;
	char *host;
	unsigned char *ip_address;
	int i;

	for (i = 0; i < 256; i++) {
		pthread_mutex_lock(&client_mutex[i]);

		client = client_list[i];
		while (client != NULL) {
			fprintf(fp, "session id  : %d\n", client->session->client_id);
			ip_address = (unsigned char*)&(client->session->ip_address);
			fprintf(fp, "IP-address  : %hhu.%hhu.%hhu.%hhu\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
			fprintf(fp, "socket      : %d\n", client->session->client_socket);
			fprintf(fp, "Kept alive  : %d\n", client->session->kept_alive);
			if (client->session->remote_user != NULL) {
			fprintf(fp, "Remote user : %s\n", client->session->remote_user);
			}
			if (client->session->file_on_disk != NULL) {
				fprintf(fp, "File on disk: %s\n", client->session->file_on_disk);
				if ((host = get_headerfield("Host:", client->session->headerfields)) != NULL) {
					fprintf(fp, "Host        : %s\n", host);
				}
			}
			client = client->next;
			if (client != NULL) {
				fprintf(fp, "---\n");
			}
		}

		pthread_mutex_unlock(&client_mutex[i]);
	}
}
#endif

/* Disconnect a client.
 */
void kick_client(int id) {
	t_client *client;
	int i;

	for (i = 0; i < 256; i++) {
		pthread_mutex_lock(&client_mutex[i]);

		client = client_list[i];
		while (client != NULL) {
			if (client->session->client_id == id) {
				client->session->force_quit = true;
				break;
			}
			client = client->next;
		}

		pthread_mutex_unlock(&client_mutex[i]);
	}
}

/* IP ban functions
 */
int ban_ip(long ip, int timer, bool kick_on_ban) {
	int retval = 0;
	t_banned *ban;
	bool new_ip = true;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->timer = timer;
			ban->bantime = timer;
			new_ip = false;
			break;
		}
		ban = ban->next;
	}

	if (new_ip) {
		if ((ban = (t_banned*)malloc(sizeof(t_banned))) != NULL) {
			ban->ip = ip;
			ban->timer = timer;
			ban->bantime = timer;
			ban->connect_attempts = 0;
			ban->next = banlist;

			banlist = ban;
		} else {
			retval = -1;
		}
	}

	pthread_mutex_unlock(&ban_mutex);

	if (kick_on_ban) {
		kick_ip(ip);
	}

	return retval;
}

/* Reset the timer of a banned IP address.
 */
void reban_ip(long ip) {
	t_banned *ban;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->timer = ban->bantime;
			break;
		}
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);
}

/* Check the timers of the banlist.
 */
void check_banlist(t_config *config) {
	t_banned *ban, *prev = NULL, *next;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		next = ban->next;
		switch (ban->timer) {
			case TIMER_OFF:
				break;
			case 0:
				if (prev == NULL) {
					banlist = next;
				} else {
					prev->next = next;
				}
				log_unban(config->system_logfile, ban->ip, ban->connect_attempts);
				free(ban);
				ban = prev;
				break;
			default:
				ban->timer--;
		}
		prev = ban;
		ban = next;
	}
	
	pthread_mutex_unlock(&ban_mutex);
}

/* Unban an IP address.
 */
void unban_ip(long ip) {
	t_banned *ban, *prev = NULL;

	pthread_mutex_lock(&ban_mutex);
	
	ban = banlist;
	while (ban != NULL) {
		if ((ban->ip == ip) || (ip == 0)) {
			if (prev == NULL) {
				banlist = ban->next;
			} else {
				prev->next = ban->next;
			}
			free(ban);
			break;
		}
		prev = ban;
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);
}

#ifdef HAVE_COMMAND
/* Print the list of banned IP addresses.
 */
void print_banlist(FILE *fp) {
	t_banned *ban;
	unsigned char *ip_address;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		ip_address = (unsigned char*)&(ban->ip);
		fprintf(fp, "IP-address  : %hhu.%hhu.%hhu.%hhu\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
		fprintf(fp, "seconds left: %d\n", ban->timer);
		ban = ban->next;
		if (ban != NULL) {
			fprintf(fp, "---\n");
		}
	}

	pthread_mutex_unlock(&ban_mutex);
}
#endif
