/* Hiawatha | cgi.c
 *
 * All the routines for handling CGI requests.
 */
#include "config.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <netdb.h>
#include <errno.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#include "libstr.h"
#include "send.h"
#include "cgi.h"
#include "envir.h"
#include "log.h"

#define MAX_DELAY_TIMER 60
#define ip_index(ip) (ip & 255)

static int delay_timer = 0;

/*
 * Load balancer
 * ==============
 */

void init_load_balancer(t_fcgi_server *fcgi_server) {
	int i;

	while (fcgi_server != NULL) {
		for (i = 0; i < 256; i++) {
			fcgi_server->cgi_session_list[i] = NULL;
			pthread_mutex_init(&fcgi_server->cgi_session_mutex[i], NULL);
		}
		fcgi_server = fcgi_server->next;
	}
}

t_connect_to *select_connect_to(t_fcgi_server *fcgi_server, unsigned long client_ip) {
	t_connect_to  *connect_to = NULL;
	t_cgi_session *cgi_session;
	bool search_new_fcgi_server = true;
	int i;

	/* Only one connect_to?
	 */
	if (fcgi_server->connect_to->next == fcgi_server->connect_to) {
		return fcgi_server->connect_to;
	}

	i = ip_index(client_ip);
	pthread_mutex_lock(&fcgi_server->cgi_session_mutex[i]);

	/* Search in cgi_session_list
	 */
	cgi_session = fcgi_server->cgi_session_list[i];
	while (cgi_session != NULL) {
		if (cgi_session->client_ip == client_ip) {
			if (cgi_session->timer - delay_timer > 0) {
				if (cgi_session->connect_to->available) {
					cgi_session->timer = fcgi_server->session_timeout + delay_timer;
					connect_to = cgi_session->connect_to;
				} else {
					cgi_session->client_ip = 0;
				}
				search_new_fcgi_server = false;
			}
			break;
		}
		cgi_session = cgi_session->next;
	}

	if (search_new_fcgi_server) {
		connect_to = fcgi_server->connect_to;
		while (connect_to->available == false) {
			if ((connect_to = connect_to->next) == fcgi_server->connect_to) {
				break;
			}
		}
		fcgi_server->connect_to = connect_to->next;

		/* Add to cgi_session_list
		 */
		if (fcgi_server->session_timeout > 0) {
			if ((cgi_session = (t_cgi_session*)malloc(sizeof(t_cgi_session))) != NULL) {
				cgi_session->client_ip = client_ip;
				cgi_session->connect_to = connect_to;
				cgi_session->timer = fcgi_server->session_timeout + delay_timer;

				cgi_session->next = fcgi_server->cgi_session_list[i];
				fcgi_server->cgi_session_list[i] = cgi_session;
			}
		}
	}
	
	pthread_mutex_unlock(&fcgi_server->cgi_session_mutex[i]);

	return connect_to;
}

void check_load_balancer(t_config *config) {
	t_fcgi_server *fcgi_server;
	t_cgi_session *cgi_session, *last, *next = NULL;
	t_connect_to  *connect_to = NULL;
	int i, sock;

	if (++delay_timer == MAX_DELAY_TIMER) {
		fcgi_server = config->fcgi_server;
		while (fcgi_server != NULL) {
			/* Check session timeouts
			 */
			for (i = 0; i < 256; i++) {
				pthread_mutex_lock(&fcgi_server->cgi_session_mutex[i]);

				last = NULL;
				cgi_session = fcgi_server->cgi_session_list[i];
				while (cgi_session != NULL) {
					next = cgi_session->next;

					if (((cgi_session->timer -= MAX_DELAY_TIMER) <= 0) || (cgi_session->connect_to->available == false)) {
						if (last == NULL) {
							fcgi_server->cgi_session_list[i] = next;
						} else {
							last->next = next;
						}
						free(cgi_session);
					}

					last = cgi_session;
					cgi_session = next;
				}

				pthread_mutex_unlock(&fcgi_server->cgi_session_mutex[i]);
			}

			/* Check if offline FastCGI servers are available again
			 */
			connect_to = fcgi_server->connect_to;
			do { 
				if (connect_to->available == false) {
					if ((sock = connect_to_fcgi_server(connect_to)) != -1) {
						close(sock);
						connect_to->available = true;
					} else {
						log_string(config->system_logfile, "FastCGI server is still unavailable");
					}
				}
				connect_to = connect_to->next;
			} while (connect_to != fcgi_server->connect_to);

			fcgi_server = fcgi_server->next;
		}

		delay_timer = 0;
	}
}

/*
 * Search FastCGI server
 * ======================
 */

t_fcgi_server *fcgi_server_match(t_fcgi_server *fcgi_server, t_charlist *fastcgi, char *cgi_script) {
	char *extension;

	extension = file_extension(cgi_script);
	while (fcgi_server != NULL) {
		if (in_charlist(fcgi_server->fcgi_id, fastcgi)) {
			if (in_charlist(extension, &(fcgi_server->extension))) {
				break;
			}
		}
		fcgi_server = fcgi_server->next;
	}

	return fcgi_server;
}


/*
 * Normal CGI processes
 * =====================
 */

int fork_cgi_process(t_session *session, t_cgi_info *cgi_info) {
	int cgi_pid, post_pipe[2], html_pipe[2], error_pipe[2], i;
	char *slash, *run[7], cgi_time[16];

	do {
		if (pipe(post_pipe) != -1) {
			if (pipe(html_pipe) != -1) {
				if (pipe(error_pipe) != -1) {
					break;
				}
				close(html_pipe[0]);
				close(html_pipe[1]);
			}
			close(post_pipe[0]);
			close(post_pipe[1]);
		}
		return -1;
	} while (false);

	switch (cgi_pid = fork()) {
		case -1:
			break;
		case 0:
			/* Child, executes CGI program.
			 */
			dup2(post_pipe[0], STDIN_FILENO);
			dup2(html_pipe[1], STDOUT_FILENO);
			dup2(error_pipe[1], STDERR_FILENO);

			close(post_pipe[0]);
			close(post_pipe[1]);
			close(html_pipe[0]);
			close(html_pipe[1]);
			close(error_pipe[0]);
			close(error_pipe[1]);

			fcntl(0, F_SETFD, 0);
			fcntl(1, F_SETFD, 0);
			fcntl(2, F_SETFD, 0);

			set_environment(session, NULL);
/*
#ifdef HAVE_SSL
			if (session->binding->use_ssl) {
				ssl_set_environment(session->ssl_data);
			}
#endif
*/
			if ((slash = strrchr(session->file_on_disk, '/')) != NULL) {
				*slash = '\0';
				chdir(session->file_on_disk);
				*slash = '/';
			}

			i = 0;
			if (cgi_info->wrap_cgi) {
				run[i++] = session->config->cgi_wrapper;
				if (session->host->cgi_wrap_id != NULL) {
					setenv("CGIWRAP_ID", session->host->cgi_wrap_id, 1);
				} else {
					setenv("CGIWRAP_ID", session->local_user, 1);
				}
				if (session->host->follow_symlinks) {
					setenv("CGIWRAP_FOLLOWSYMLINKS", "true", 1);
				} else {
					setenv("CGIWRAP_FOLLOWSYMLINKS", "false", 1);
				}
				cgi_time[15] = '\0';
				snprintf(cgi_time, 15, "%d", session->config->time_for_cgi);
				setenv("CGIWRAP_TIMEFORCGI", cgi_time, 1);
			} else if (setsid() == -1) {
				exit(-1);
			}
			if (session->cgi_handler != NULL) {
				run[i++] = session->cgi_handler;
				run[i++] = session->cgi_handler;
			} else {
				if (cgi_info->wrap_cgi) {
					run[i++] = "-";
				}
				run[i++] = session->file_on_disk;
			}
			run[i++] = session->file_on_disk;
			run[i] = NULL;

			execvp(run[0], run + 1);
			exit(-1);
		default:
			/* Parent, reads CGI output
			 */
			close(post_pipe[0]);
			close(error_pipe[1]);
			close(html_pipe[1]);

			cgi_info->to_cgi = post_pipe[1];
			cgi_info->from_cgi = html_pipe[0];
			cgi_info->cgi_error = error_pipe[0];

			if (cgi_info->from_cgi > cgi_info->cgi_error) {
				cgi_info->highest_fd = cgi_info->from_cgi + 1;
			} else {
				cgi_info->highest_fd = cgi_info->cgi_error + 1;
			}

			/* Send POST data to CGI program.
			 */
			if ((session->body != NULL) && (session->content_length > 0)) {
				write(cgi_info->to_cgi, session->body, session->content_length);
			}
	}

	return cgi_pid;
}

int read_from_cgi_process(t_session *session, t_cgi_info *cgi_info) {
	bool read_again;
	fd_set read_fds;
	int bytes_read;

	cgi_info->input_result = cgi_info->error_result = cd_NOTHING;

	do {
		read_again = false;

		FD_ZERO(&read_fds);
		FD_SET(cgi_info->from_cgi, &read_fds);
		FD_SET(cgi_info->cgi_error, &read_fds);

		switch (select(cgi_info->highest_fd, &read_fds, NULL, NULL, &(cgi_info->select_timeout))) {
			case -1:
				return cr_ERROR;
			case 0:
				if (session->force_quit) {
					return cr_FORCE_QUIT;
				} else if (--(cgi_info->timer) <= 0) {
					return cr_TIMEOUT;
				} else {
					cgi_info->select_timeout.tv_sec = 1;
					cgi_info->select_timeout.tv_usec = 0;
					read_again = true;
				}
		}
	} while (read_again);

	do {
		read_again = false;

		if (FD_ISSET(cgi_info->from_cgi, &read_fds)) {
			bytes_read = read(cgi_info->from_cgi, cgi_info->input_buffer + cgi_info->input_len, cgi_info->input_buffer_size - cgi_info->input_len);
			if (bytes_read == -1) {
				if (errno != EINTR) {
					cgi_info->input_result = cd_READ_ERROR;
				} else {
					read_again = true;
				}
			} else if (bytes_read == 0) {
				cgi_info->input_result = cd_END_OF_DATA;
			} else if (bytes_read > 0) {
				cgi_info->input_result = cd_READ_DATA;
				cgi_info->input_len += bytes_read;
			}
		}
	} while (read_again);

	do {
		read_again = false;

		if (FD_ISSET(cgi_info->cgi_error, &read_fds)) {
			bytes_read = read(cgi_info->cgi_error, cgi_info->error_buffer + cgi_info->error_len, cgi_info->error_buffer_size - cgi_info->error_len);
			if (bytes_read == -1) {
				if (errno != EINTR) {
					cgi_info->error_result = cd_READ_ERROR;
				} else {
					read_again = true;
				}
			} else if (bytes_read == 0) {
				cgi_info->error_result = cd_END_OF_DATA;
			} else if (bytes_read > 0) {
				cgi_info->error_result = cd_READ_DATA;
				cgi_info->error_len += bytes_read;
			}
		}
	} while (read_again);

	return cr_OKE;
}


/*
 *  Fast CGI processes
 *  ===================
 */

int connect_to_fcgi_server(t_connect_to *connect_to) {
	int sock;
	struct hostent *hostinfo;
	struct sockaddr_in saddr;

	if ((hostinfo = gethostbyname(connect_to->host)) != NULL) {
		if ((sock = socket(AF_INET, SOCK_STREAM, 0)) > 0) {
			bzero(&saddr, sizeof(struct sockaddr_in));
			saddr.sin_family = AF_INET;
			saddr.sin_port = htons(connect_to->port);
			memcpy(&saddr.sin_addr.s_addr, hostinfo->h_addr, 4);
			if (connect(sock, (struct sockaddr*)&saddr, sizeof(struct sockaddr_in)) != 0) {
				close(sock);
				sock = -1;
			}
		} else {
			sock = -1;
		}
	} else {
		sock = -1;
	}

	return sock;
}

int	send_fcgi_data(t_session *session, int sock) {
	t_fcgi_buffer fcgi_buffer;

	fcgi_buffer.sock = sock;
	fcgi_buffer.size = 0;

	fcgi_buffer.mode = 4;
	if (send_directly(sock, "\x01\x01\x00\x01" "\x00\x08\x00\x00" "\x00\x01\x00\x00" "\x00\x00\x00\x00", 16) == -1) {
		return -1;
	}
	set_environment(session, &fcgi_buffer);
	if (send_fcgi_buffer(&fcgi_buffer, NULL, 0) == -1) {
		return -1;
	}

	fcgi_buffer.mode = 5;
	if ((session->body != NULL) && (session->content_length > 0)) {
		/* Send POST data to CGI program.
		 */
		if (send_fcgi_buffer(&fcgi_buffer, session->body, session->content_length) == -1) {
			return -1;
		}
	}
	if (send_fcgi_buffer(&fcgi_buffer, NULL, 0) == -1) {
		return -1;
	}

	return 0;
}

static int read_fcgi_socket(t_session *session, t_cgi_info *cgi_info, char *buffer, int size) {
	int bytes_read = -1;
	fd_set read_fds;
	struct timeval select_timeout;
	bool read_again;

	select_timeout.tv_sec = 1;
	select_timeout.tv_usec = 0;

	do {
		FD_ZERO(&read_fds);
		FD_SET(cgi_info->from_cgi, &read_fds);

		switch (select(cgi_info->from_cgi + 1, &read_fds, NULL, NULL, &select_timeout)) {
			case -1:
				return cr_ERROR;
			case 0:
				if (session->force_quit) {
					return cr_FORCE_QUIT;
				} else if (--(cgi_info->timer) <= 0) {
					return cr_TIMEOUT;
				} else {
					select_timeout.tv_sec = 1;
					select_timeout.tv_usec = 0;
				}
				break;
			default:
				do {
					read_again = false;
					if ((bytes_read = read(cgi_info->from_cgi, buffer, size)) == -1) {
						if (errno != EINTR) {
							return cr_ERROR;
						} else {
							read_again = true;
						}
					}
				} while (read_again);
		}
	} while (bytes_read < 0);

	return bytes_read;
}

int read_from_fcgi_server(t_session *session, t_cgi_info *cgi_info) {
	char *buffer;
	bool read_again;
	unsigned char header[8];
	int bytes_read, size, padding, offset = 0, bytes_left;

	cgi_info->input_result = cgi_info->error_result = cd_NOTHING;

	/* Read header
	 */
	size = 8;
	do {
		read_again = false;

		switch (bytes_read = read_fcgi_socket(session, cgi_info, (char*)header + 8 - size, size)) {
			case cr_TIMEOUT:
			case cr_FORCE_QUIT:
				return bytes_read;
			case cr_ERROR:
				if (errno != EINTR) {
					cgi_info->input_result = cd_READ_ERROR;
					return cr_OKE;
				} else {
					read_again = true;
				}
				break;
			case 0:
				cgi_info->input_result = cd_END_OF_DATA;
				return cr_OKE;
			default:
				if ((size -= bytes_read) == 0) {
					if (header[1] == FCGI_END_REQUEST) {
						cgi_info->input_result = cd_END_OF_DATA;
						return cr_OKE;
					}
				} else {
					read_again = true;
				}
		}
	} while (read_again);

	/* Determine the size and type of the data
	 */
	padding = (int)header[6];
	size = 256 * (int)header[4] + (int)header[5] + padding;
	switch (header[1]) {
		case FCGI_STDOUT:
			buffer = cgi_info->input_buffer;
			break;
		case FCGI_STDERR:
			buffer = cgi_info->error_buffer;
			break;
		case FCGI_END_REQUEST:
			cgi_info->input_result = cd_END_OF_DATA;
			return cr_OKE;
		default:
			lseek(cgi_info->from_cgi, size, SEEK_CUR);
			return cr_OKE;
	}

	/* Read data
	 */
	bytes_left = size;
	do {
		read_again = false;

		switch (bytes_read = read_fcgi_socket(session, cgi_info, buffer + offset, bytes_left)) {
			case cr_TIMEOUT:
			case cr_FORCE_QUIT:
				return bytes_read;
			case cr_ERROR:
				if (errno != EINTR) {
					cgi_info->input_result = cd_READ_ERROR;
					return cr_OKE;
				} else {
					read_again = true;
				}
				break;
			case 0:
				if (size == 0) {
					cgi_info->input_result = cd_END_OF_DATA;
				}
				return cr_OKE;
			default:
				offset += bytes_read;
				if ((bytes_left -= bytes_read) > 0) {
					read_again = true;
				}
		}
	} while (read_again);

	switch (header[1]) {
		case FCGI_STDOUT:
			cgi_info->input_result = cd_READ_DATA;
			cgi_info->input_len = size - padding;
			break;
		case FCGI_STDERR:
			cgi_info->error_result = cd_READ_DATA;
			cgi_info->error_len = size - padding;
			break;
		default:
			break;
	}

	return cr_OKE;
}
