/**************************************
  SixXS Heartbeat Client
  by Jeroen Massar <jeroen@sixxs.net>
***************************************

 $Author: jeroen $
 $Id: heartbeat-client.c,v 1.12 2003/10/26 15:22:43 jeroen Exp $
 $Date: 2003/10/26 15:22:43 $

**************************************/
#define _XOPEN_SOURCE
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <syslog.h>

#include "../common/config.h"
#include "../common/common.h"
#include "hb.h"

// Config
int		g_os			= -1;
char		*g_ipv4_interface	= NULL;
char		*g_ipv4_pop		= NULL;
char		*g_ipv4_local		= NULL;
char		*g_ipv4_local_resolve	= NULL;
char		*g_ipv6_interface	= NULL;
char		*g_ipv6_pop		= NULL;
char		*g_ipv6_local		= NULL;
int		g_ipv6_prefixlen	= 0;
char		*g_password		= NULL;
int		g_sendinterval		= 60;
int		g_timeout		= 300;
int		g_statictunnel		= 0;
int		g_behindnat		= 0;
int		g_daemonize		= 1;

#define CONF_STRING	1
#define CONF_INT	2

enum {
	OS_LINUX = 42,
	OS_FREEBSD,
	OS_OPENBSD,
	OS_NETBSD };

struct os
{
	char	*label;
	int	val;
} os[] = {
	{"linux",	OS_LINUX },
	{"freebsd",	OS_FREEBSD },
	{"openbsd",	OS_OPENBSD },
	{"netbsd",	OS_NETBSD },
	{NULL,		0 },
};

struct conf
{
	char	*label;
	int	type;
	void	*var;
} conf[] = {
	{"ipv4_interface",	CONF_STRING,	&g_ipv4_interface},
	{"ipv4_pop",		CONF_STRING,	&g_ipv4_pop},
	{"ipv4_local_resolve",	CONF_STRING,	&g_ipv4_local_resolve},
	{"ipv6_interface",	CONF_STRING,	&g_ipv6_interface},
	{"ipv6_pop",		CONF_STRING,	&g_ipv6_pop},
	{"ipv6_local",		CONF_STRING,	&g_ipv6_local},
	{"ipv6_prefixlen",	CONF_INT,	&g_ipv6_prefixlen},
	{"hb_password",		CONF_STRING,	&g_password},
	{"hb_sendinterval",	CONF_INT,	&g_sendinterval},
	{"hb_timeout",		CONF_INT,	&g_timeout},
	{"hb_behindnat",	CONF_INT,	&g_behindnat},
	{"hb_statictunnel",	CONF_INT,	&g_statictunnel},
	{"daemonize",		CONF_INT,	&g_daemonize},
	{NULL,			CONF_INT,	NULL},
};

/**************************************
  Functions
**************************************/
// Change the local endpoint of this tunnel
void tunnel_change()
{
	char buf[1000];
	char *cmd;

	// Don't touch the tunnel
	// In this mode the client can be run as non-root btw
	if (g_statictunnel != 0) return;

	hblog(LOG_INFO, "IP address change to %s detected, reconfiguring\n", g_ipv4_local);

	buf[0] = '\0';

	switch (g_os)
	{
		case OS_LINUX:
				cmd = "ip tunnel change %s local %s";
				break;
		case OS_FREEBSD:
				cmd = "gifconfig %s %s %s";
				break;
		case OS_OPENBSD:
				cmd = "ifconfig %s giftunnel %s %s";
				break;
		case OS_NETBSD:
				cmd = "ifconfig %s tunnel %s %s";
				break;
		default:
				/* Should not happen */
				hblog(LOG_ERR, "Unknown OS while changing endpoint.\n");
				hblog(LOG_ERR, "Exiting...\n");
				exit(-1);
	}
	snprintf(buf, sizeof(buf), cmd, g_ipv6_interface, g_ipv4_local, g_ipv4_pop);
	system(buf);
}

// configure this client
int loadconfig(char *filename)
{
	FILE		*f;
	char		buf[1000], *end = NULL, *val = NULL, *p = NULL;
	unsigned int	line = 0, i, len, found;
	struct conf	*c;
	struct os	*o;

	f = fopen(filename, "r");
	if (!f)
	{
		fprintf(stderr, "Could not open config file \"%s\"\n", filename);
		return -1;
	}

	while (fgets(buf, sizeof(buf), f))
	{
		line++;

		/* Chop off \n and \r and white space */
		p = &buf[strlen(buf)-1];
		while (	p >= buf && (
			*p == '\n' ||
			*p == '\r' ||
			*p == '\t' ||
			*p == ' ')) *p-- = '\0';

		/* Ignore comments and emtpy lines */
		if (	buf[0] == '#' ||
			buf[0] == ';' ||
			buf[0] == ' ' ||
			(buf[0] == '/' && buf[1] == '/') ||
			strlen(buf) < 1) continue;

		found = 0;

		// Get the end of the first argument
		p = buf;
		end = &buf[strlen(buf)-1];
		// Skip until whitespace
		while (	p < end &&
			*p != ' ' &&
			*p != '\t') p++;
		// Terminate this argument
		*p = '\0';
		*p++;

		// Skip whitespace
		while ( p < end &&
			*p == ' ' &&
			*p == '\t') p++;

		// Start of the value
		val = p;

		// If starting with quotes, skip until next quotes
		if (*p == '"' || *p == '\'')
		{
			p++;
			// Find next quote
			while (p <= end &&
				*p != *val &&
				*p != '\0') p++;
			// Terminate
			*p = '\0';
			// Skip the first quote
			val++;
		}
		// Otherwise it is already terminated above

		if (strncasecmp(buf, "os", 2) == 0)
		{
			o = os;
			while (o->label != NULL)
			{
				if (strcasecmp(val, o->label) == 0)
				{
					g_os = o->val;
					break;
				}
				o++;
			}
			if (g_os == -1) fprintf(stderr, "Unknown OS \"%s\"\n", val);
			continue;
		}

		c = conf;
		while (c->label != NULL)
		{
			len = strlen(c->label);
			if (strncasecmp(buf, c->label, len) == 0)
			{
				if (c->type == CONF_STRING)
				{
					char **str = (char **)c->var;
					*str = strdup(val);
				}
				else if (c->type == CONF_INT)
				{
					sscanf(val, "%d", (int *)c->var);
				}
				found = 1;
				break;
			}
			c++;
		}
		if (found == 0) fprintf(stderr, "Unknown configuration statement: %s \"%s\"\n", buf, val);
	}
	fclose(f);

	// Is OS not set?
	if (g_os == -1)
	{
		fprintf(stderr, "Required parameter \"os\" not set\n");
		return -1;
	}

	return 1;
}

void beat()
{
	// Send heartbeat
	int address_changed = 0;
	int sockfd = heartbeat_socket(&address_changed, g_statictunnel, g_ipv4_interface, &g_ipv4_local, g_ipv4_pop, g_ipv4_local_resolve);
	if (sockfd >= 0)
	{
		if (address_changed == 1) tunnel_change();
		sendhb(sockfd, g_ipv4_local, g_ipv6_local, g_password, g_behindnat);
		close(sockfd);
	}
}

void sighup(int i)
{
	beat();
	signal(SIGHUP, &sighup);
}

int main(int argc, char *argv[])
{
	int i;

	if (argc != 2)
	{
		fprintf(stderr,
			"heartbeat-client <configfile>\n");
		return -1;
	}

	if (huprunning() == 1)
	{
		fprintf(stdout, "Already running instance HUP'ed, exiting\n");
		return 0;
	}

	if (!loadconfig(argv[1]))
	{
		fprintf(stderr, "Configuration incomplete\n");
		return -1;
	}

	if (g_daemonize)
	{
		// Daemonize
		i = fork();
		if (i < 0)
		{
			fprintf(stderr, "Couldn't fork\n");
			return -1;
		}
		// Exit the mother fork
		if (i != 0) return 0;

		// Child fork
		setsid();
		// Cleanup stdin/out/err
		freopen("/dev/null","r",stdin);
		freopen("/dev/null","w",stdout);
		freopen("/dev/null","w",stderr);

		savepid();
	}

	// Handle a SIGHUP to force a heartbeat
	signal(SIGHUP, &sighup);

	for (;;)
	{
		beat();

		// Sleep
		sleep(g_sendinterval);
	}
}
