/*
 * hangul utilities
 */
#ifdef HAVE_CONFIG_H
#include "xtermcfg.h"
#endif

#if OPT_HANGUL
#include <stdio.h>
#include <stdlib.h> /* malloc */

#include "ptyx.h"
#include "data.h"
#include "hangul.h"
#include "xterm.h"
#include "hfont.h"


int han_in_seq = FALSE;
int han_no_status_line = FALSE;
int han_font_width_check = TRUE;

int
han_split_font_name(char *fontname, char *efontname, char *hfontname)
{
    char *s;
    if (!fontname) return 0;
    s = strchr(fontname, ',');
    if (!s) return 0;
    if (s - fontname > 79) return 0;
    strncpy(efontname, fontname, s - fontname);
    efontname[s - fontname] = '\0';
    if (strlen(s+1) > 79) return 0;
    strcpy(hfontname, s + 1);
    return 1;
}

char *
han_composite_font_name(char *efontname, char *hfontname)
{
    char *fn;
    int len;
    if (efontname == NULL || hfontname == NULL) return NULL;
    len = strlen(efontname);
    fn = (char *)malloc(len + strlen(hfontname) + 2);
    if (!fn) return NULL;
    strcpy(fn, efontname);
    fn[len] = ',';
    strcpy(fn + len + 1, hfontname);
    return fn;
}

int
han_set_font_encoding(char *hfontname, XFontStruct *hfs)
{
    char *charset_registry = NULL;
    char *hfont_fullname = NULL;
    TScreen *screen = &term->screen;

    if (screen->han_font_type != HANGUL_FONT_TYPE_GUESS) {
      switch(screen->han_font_type)
	{
	  case HANGUL_FONT_TYPE_KS:
	    charset_registry = "ksc5601.1987-0";
	    break;
	  case HANGUL_FONT_TYPE_KS1:
	    charset_registry = "ksc5601.1987-1";
	    break;
	  case HANGUL_FONT_TYPE_JOHAB:
	    charset_registry = "johab-0";
	    break;
	  case HANGUL_FONT_TYPE_JOHAB844:
	    charset_registry = "johab844";
	    break;
	}
    } else {
	XFontProp *pp;
	int i;
	static char charset_buf[64];
	char *charset_encoding = NULL;
	Atom atom_FONT = XInternAtom(screen->display,"FONT",False);
	Atom atom_CHARSET = XInternAtom(screen->display,"CHARSET_REGISTRY",False);
	Atom atom_CHARSET_ENCODING = XInternAtom(screen->display,"CHARSET_ENCODING",False);

	for (i=0, pp = hfs->properties; i < hfs->n_properties; i++, pp++)
	    if (pp->name == atom_CHARSET) {
		charset_registry = XGetAtomName(screen->display, pp->card32);
	    } else if (pp->name == atom_CHARSET_ENCODING) {
		charset_encoding = XGetAtomName(screen->display, pp->card32);
	    } else if (pp->name == atom_FONT) {
		hfont_fullname = XGetAtomName(screen->display, pp->card32);
	    }
	if(charset_registry && charset_encoding) {
	    sprintf(charset_buf, "%s-%s", charset_registry, charset_encoding);
	    charset_registry = charset_buf;
	}
    }

    if(charset_registry == NULL) {
      charset_registry = hfont_fullname;
    }
    han_set_code_to_font(charset_registry);
}

void
han_correct_font_width(XFontStruct *hfs)
{
    XChar2b tmpstr[4];
    int n = kscode_to_font[term->screen.han_font_type]("", tmpstr, 2);
    hfs->max_bounds.width = XTextWidth16(hfs, tmpstr, n);
}

int
han_check_font_ratio(XFontStruct *nfs, char *name, XFontStruct *hfs,
		     char *hname)
{
    if (hfs->max_bounds.width == nfs->max_bounds.width * 2
	|| (hfs->max_bounds.width == 1 && nfs->max_bounds.width == 1))
	return 0;

#if 0
    fprintf(stderr,"Warning: font width ratio between hangul");
    fprintf(stderr,"and english font is not good\n");
    fprintf(stderr,"hangul  font: %s\n",hname);
    fprintf(stderr,"english font: %s\n",name);
    fprintf(stderr,"(hangul font width) != (english font width) * 2\n");
    fprintf(stderr,"(%d) != (%d) * 2\n",hfs->max_bounds.width,nfs->max_bounds.width);
#endif

    return -1;
}

void HandleExecEvent(Widget w, XEvent *event, String *params, Cardinal *nparams)
{
    register TScreen *screen = &term->screen;
    int fd[2];
    int row, x = 0, y = 0, i, j, n, repeat, nstr, col;
    char buf[256];
    char *buf_sav[256];
    unsigned long mask;
    XGCValues xgcv;
    GC gc = NULL;

    if (*nparams != 1 && *nparams != 2) return;

    if (event->type == ButtonPress || event->type == ButtonRelease ) {
	/* we have  position */
	xgcv.function = GXxor;
	xgcv.foreground = screen->foreground ^ term->core.background_pixel;
	mask = GCForeground | GCFunction;
	gc = XtGetGC((Widget)term, mask, &xgcv);
	x = event->xbutton.x;
	y = event->xbutton.y;
	row = (y - screen->border) / FontHeight(screen);
	y = row * FontHeight(screen) + screen->border;
	col = (x - OriginX(screen)) / FontWidth(screen);
	x = col*FontWidth(screen) + screen->border;
	sprintf(buf,"%03d",col);
	strncpy(buf+3, SCRN_BUF_CHARS(screen,row+screen->topline),
		screen->max_col);
	XFillRectangle(screen->display, TextWindow(screen),gc,x,
		       y, FontWidth(screen),FontHeight(screen));
	XFlush(screen->display);
    } else {
	/* by keyboard event ? */
	buf[0] = 0;
    }

    pipe(fd);
    if(fork() == 0) {
	/* child */
	/* first turn off privilege from setuid bit */
	setuid(getuid());
	setgid(getgid());
	close(1);
	dup(fd[1]);
	close(fd[0]);
	if( buf[0] != 0) execlp(*params,*params,buf,0);
	else execlp(*params,*params,0);
	/* not reached */
	exit(1);
    }
    /* parent */
    nstr = 0;
    close(fd[1]);
    while( (n = read(fd[0],buf,256)) > 0) {
	buf_sav[nstr] = XtMalloc(n+1);
	strncpy(buf_sav[nstr],buf,n);
	buf_sav[nstr][n] = 0; /* null padding */
	nstr++;
    }
    close(fd[0]);
    if(*nparams == 2)
	repeat = atoi(params[1]);
    else
	repeat = 1;

    for(j=1;j<=repeat;j++) {
	for(i=0;i<nstr;i++)
	    StringInput (screen, buf_sav[i], strlen(buf_sav[i]));
    }
    if (event->type == ButtonPress || event->type == ButtonRelease ) {
	XFillRectangle(screen->display, TextWindow(screen),gc,x,
		       y , FontWidth(screen),FontHeight(screen));
	XtReleaseGC((Widget)term, gc);
    }
    for(i=0;i<nstr;i++) XtFree(buf_sav[i]);
}

void
set_default_color(TScreen *screen)
{
    XSetForeground(screen->display, NormalGC(screen), screen->foreground);
    XSetBackground(screen->display, NormalGC(screen),
		   term->core.background_pixel);
    XSetForeground(screen->display, NormalHGC(screen), screen->foreground);
    XSetBackground(screen->display, NormalHGC(screen),
		   term->core.background_pixel);
}

GC
updatedXtermHGC(TScreen *screen, int flags, int fg_bg, Bool hilite)
{
    Pixel fg_pix = getXtermForeground(flags,extract_fg(fg_bg,flags));
    Pixel bg_pix = getXtermBackground(flags,extract_bg(fg_bg,flags));
    GC hgc;

    if ( (!hilite && (flags & INVERSE) != 0)
	 ||  (hilite && (flags & INVERSE) == 0) ) {
	hgc = ReverseHGC(screen);

	XSetForeground(screen->display, hgc, bg_pix);
	XSetBackground(screen->display, hgc, fg_pix);
    } else {
	hgc = NormalHGC(screen);

	XSetForeground(screen->display, hgc, fg_pix);
	XSetBackground(screen->display, hgc, bg_pix);
    }
    return hgc;
}

int
han_conv_ksc_to_ct(Char *str, int len, Char *des)
{
    int hangul = 0;
    int ctlen = 0;
    if (len <= 0) len = strlen(str);
    if (des) {
	while (len > 0) {
	    if (*str & 0x80) {
		if (!hangul) {
		    *des++ = 27;
		    *des++ = '$';
		    *des++ = '(';
		    *des++ = 'C';
		    hangul = 1;
		}
		*des++ = *str++ & 0x7f;
		*des++ = *str++ & 0x7f;
		len -= 2;
	    } else {
		if (hangul) {
		    *des++ = 27;
		    *des++ = '(';
		    *des++ = 'B';
		    hangul = 0;
		}
		*des++ = *str++;
		len--;
	    }
	}
    } else {
	while (len > 0) {
	    if (*str & 0x80) {
		if (!hangul) {
		    ctlen += 4;
		    hangul = 1;
		}
		ctlen += 2;
		str += 2;
		len -= 2;
	    } else {
		if (hangul) {
		    ctlen += 3;
		    hangul = 0;
		}
		ctlen++;
		str++;
		len--;
	    }
	}
    }
    return ctlen;
}

#endif /* OPT_HANGUL */
