/* GtkBalls
 * Copyright (C) 1998 Eugene V. Morozov
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>

#include "gtkballs.h"
#include "interface.h"
#include "scoreboard.h"
#include "preferences.h"

extern struct score_board scoreboard[11];

static GtkWidget *dialog = NULL;
static int saved_score = 0;
static char *user_name = NULL;
static gchar last_player_name[15];

struct point { int x; int y; };

void my_usleep (int x)
{
  struct timeval st_delay;
  
  st_delay.tv_usec = x;
  st_delay.tv_sec = 0;
  select (32, NULL, NULL, NULL, &st_delay);
}

void 
init_names_and_scores (GtkWidget ** names, GtkWidget ** scores)
{
  int i;
  char str[128];

  if (!read_score (scoreboard))
    {
      g_print ("Unable to read score.\n");
    }

  for (i = 0; i < 10; i++)
    {
      sprintf (str, "%-16s", scoreboard[i].name);
      names[i] = gtk_label_new (str);
      if (scoreboard[i].score == 0)
	str[0] = '\0';
      else
	sprintf (str, "%5d", scoreboard[i].score);
      scores[i] = gtk_label_new (str);
    }
}

void 
show_hall_of_fame (GtkWidget * widget, gpointer data)
{
  GtkWidget *hall_of_fame;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *vbox, *button_box;
  GtkWidget *names[10], *scores[10];
  GtkWidget *close_button;
  int i;

  hall_of_fame = gtk_widget_new (gtk_window_get_type (),
				 "GtkObject::user_data", NULL,
				 "GtkWindow::title", "Hall of Fame",
				 "GtkWindow::allow_grow", FALSE,
				 "GtkWindow::allow_shrink", FALSE,
				 "GtkContainer::border_width", 5,
				 NULL);

  gtk_widget_set_usize (hall_of_fame, 230, 350);

  gtk_signal_connect (GTK_OBJECT (hall_of_fame), "delete_event",
		      GTK_SIGNAL_FUNC (delete_event), NULL);
  gtk_signal_connect (GTK_OBJECT (hall_of_fame), "destroy",
		      GTK_SIGNAL_FUNC (destroy), hall_of_fame);
  
  frame = gtk_frame_new (" Hall of Fame ");

  vbox = gtk_vbox_new (FALSE, 0);
  button_box = gtk_hbox_new (TRUE, 0);

  gtk_container_border_width (GTK_CONTAINER (vbox), 10);

  gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  table = gtk_table_new (9, 1, FALSE);

  init_names_and_scores (names, scores);

  for (i = 0; i < 10; i++)
    {
      gtk_table_attach_defaults (GTK_TABLE (table), names[i], 0, 1, i, i + 1);
      gtk_table_attach_defaults (GTK_TABLE (table), scores[i], 1, 2, i, i + 1);
      gtk_widget_show (names[i]);
      gtk_widget_show (scores[i]);
    }

  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_widget_show (table);

  close_button = gtk_button_new_with_label ("   Close   ");
  gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
		      GTK_SIGNAL_FUNC (destroy), hall_of_fame);
  gtk_box_pack_start (GTK_BOX (button_box), close_button, FALSE, FALSE, 0);
  gtk_widget_show (close_button);

  gtk_box_pack_start (GTK_BOX (vbox), button_box, FALSE, FALSE, 4);
  gtk_widget_show (button_box);

  gtk_container_add (GTK_CONTAINER (hall_of_fame), vbox);

  gtk_widget_show (vbox);

  gtk_widget_show (close_button);
  gtk_widget_show (hall_of_fame);
  gtk_grab_add (hall_of_fame);
}

/* 
   shows simple message box
 */

void 
simple_message_box (char *message, char *label)
{
  GtkWidget *dialog;
  GtkWidget *prompt_label;
  GtkWidget *button;

  dialog = gtk_dialog_new ();
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);

  prompt_label = gtk_label_new (message);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), prompt_label, TRUE,
		      TRUE, 5);
  gtk_widget_show (prompt_label);

  button = gtk_button_new_with_label (label);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (destroy), dialog);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
		      TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
}

/* 
   returns board coordinate of the window position
 */
int 
calculate_board_coord (int window_coord)
{
  return (window_coord - 1) / 32;
}

void 
update_rectangle (GtkWidget * widget, int x, int y, int w, int h)
{
  GdkRectangle r;

  r.x = x;
  r.y = y;
  r.width = w;
  r.height = h;
  gtk_widget_draw (widget, &r);
}

void 
draw_ball (GtkWidget * widget, GdkPixmap * ball, GdkPixmap * pixmap, int x, int y, int yoffset)
{
  gdk_draw_rectangle (pixmap,
		      widget->style->white_gc,
		      TRUE,
		      x * 32, y * 32,
		      30, 30);

  gdk_draw_pixmap (pixmap,
		   widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		   ball,
		   0, 0,
		   x * 32 + 6, y * 32 + 1 + yoffset,
		   21, 21);

  update_rectangle (widget, x * 32, y * 32, 30, 30);
}

void 
draw_paw (GtkWidget * widget, GdkPixmap * paw, GdkPixmap * pixmap, int x, int y)
{
  gdk_draw_rectangle (pixmap,
		      widget->style->white_gc,
		      TRUE,
		      x * 32, y * 32,
		      30, 30);

  gdk_draw_pixmap (pixmap,
		   widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		   paw,
		   0, 0,
		   x * 32 + 1, y * 32 + 1,
		   28, 28);

  update_rectangle (widget, x * 32, y * 32, 30, 30);
}

void 
set_hi_score_label (void)
{
  char str[128];

  sprintf (str, "Hi-score: %i", hi_score);
  gtk_label_set (GTK_LABEL (hi_score_label), str);
}

void 
set_user_score_label (void)
{
  char str[128];

  if (score > hi_score)
    {
      hi_score = score;
      set_hi_score_label ();
    }

  sprintf (str, "Your score: %i", score);
  gtk_label_set (GTK_LABEL (user_score_label), str);
}

void
add_point_if_first (struct point *balls_to_delete, int *number_of_balls_to_delete,
		    int x, int y)
{
  int i;

  for (i = 0; i < *number_of_balls_to_delete; i++)
    if ((balls_to_delete[i].x == x) && (balls_to_delete[i].y == y))
      return;
  balls_to_delete[*number_of_balls_to_delete].x = x;
  balls_to_delete[*number_of_balls_to_delete].y = y;
  (*number_of_balls_to_delete)++;
}

void 
clear_line (int x, int y, int length, int direction, 
	    struct point *balls_to_delete, 
	    int *number_of_balls_to_delete)
{
  int i, x1, y1;

  switch (direction)
    {
    case 0:
      for (i = y - length; i < y; i++)
	add_point_if_first (balls_to_delete, number_of_balls_to_delete, x, i);
      break;
    case 1:
      for (i = x - length; i < x; i++)
	add_point_if_first (balls_to_delete, number_of_balls_to_delete, i, y);
      break;
    case 2:
      /* save length */
      i = length;
      for (x1 = x - 1, y1 = y - 1; length > 0; x1--, y1--, length--)
	add_point_if_first (balls_to_delete, number_of_balls_to_delete, x1, y1);
      /* restore length */
      length = i;
      break;
    case 3:
      i = length;
      for (x1 = x - 1, y1 = y + 1; length > 0; x1--, y1++, length--)
	add_point_if_first (balls_to_delete, number_of_balls_to_delete, x1, y1);
      length = i;
      break;
    case 4:
      i = length;
      for (x1 = x - 1, y1 = y - 1; length > 0; x1--, y1--, length--)
	add_point_if_first (balls_to_delete, number_of_balls_to_delete, x1, y1);
      length = i;
      break;
    case 5:
      i = length;
      for (x1 = x - 1, y1 = y + 1; length > 0; x1--, y1++, length--)
	add_point_if_first (balls_to_delete, number_of_balls_to_delete, x1, y1);
      length = i;
      break;
    }

/*   score = score + 10 + 2 * (length - 5) * (length - 5); */
/*   set_user_score_label (); */
}

int 
destroy_lines (GtkWidget * widget, GdkPixmap * pixmap)
{
  int x, y, length;
  int x1, y1;
  int retval = FALSE;
  int number_of_balls_to_delete = 0;
  struct point *balls_to_delete;
  int i, color;

  balls_to_delete = malloc (NUMBER_OF_CELLS * NUMBER_OF_CELLS * sizeof (struct point));

  /* test vertical rows */
  for (x = 0; x < NUMBER_OF_CELLS; x++)
    {
      length = 1;
      for (y = 1; y < NUMBER_OF_CELLS - 1; y++)
	if ((board[x][y] == board[x][y - 1]) && (board[x][y] != 0))
	  {
	    while ((board[x][y] == board[x][y - 1]) && (y < NUMBER_OF_CELLS))
	      {
		length++;
		y++;
	      }
	    if (length > 4)
	      {
		clear_line (x, y, length, 0, balls_to_delete, &number_of_balls_to_delete);
		retval = TRUE;
	      }
	    length = 1;
	  }
    }

  /* test horizontal rows */
  for (y = 0; y < NUMBER_OF_CELLS; y++)
    {
      length = 1;
      for (x = 1; x < NUMBER_OF_CELLS - 1; x++)
	if ((board[x][y] == board[x - 1][y]) && (board[x][y] != 0))
	  {
	    while ((board[x][y] == board[x - 1][y]) && (x < NUMBER_OF_CELLS))
	      {
		length++;
		x++;
	      }
	    if (length > 4)
	      {
		clear_line (x, y, length, 1, balls_to_delete, &number_of_balls_to_delete);
		retval = TRUE;
	      }
	    length = 1;
	  }
    }

  /* test diagonal rows from the left side to the right */
  for (y = 0; y < NUMBER_OF_CELLS; y++)
    {
      length = 1;
      /* from the top to the bottom */
      for (x1 = 1, y1 = y + 1; (y1 < NUMBER_OF_CELLS - 1) && (x1 < NUMBER_OF_CELLS - 1); x1++, y1++)
	if ((board[x1][y1] == board[x1 - 1][y1 - 1]) && (board[x1][y1] != 0))
	  {
	    while ((board[x1][y1] == board[x1 - 1][y1 - 1]) && (x1 < NUMBER_OF_CELLS)
		   && (y1 < NUMBER_OF_CELLS))
	      {
		length++;
		x1++;
		y1++;
	      }
	    if (length > 4)
	      {
		clear_line (x1, y1, length, 2, balls_to_delete, &number_of_balls_to_delete);
		retval = TRUE;
	      }
	    length = 1;
	  }

      length = 1;
      /* from bottom to top */
      for (x1 = 1, y1 = y - 1; (y1 >= 0) && (x1 < NUMBER_OF_CELLS); x1++, y1--)
	if ((board[x1][y1] == board[x1 - 1][y1 + 1]) && (board[x1][y1] != 0))
	  {
	    while ((board[x1][y1] == board[x1 - 1][y1 + 1]) && (x1 < NUMBER_OF_CELLS) && (y1 >= 0))
	      {
		length++;
		x1++;
		y1--;
	      }
	    if (length > 4)
	      {
		clear_line (x1, y1, length, 3, balls_to_delete, &number_of_balls_to_delete);
		retval = TRUE;
	      }
	    length = 1;
	  }
    }

  /* test diagonal rows from the top to the bottom */
  for (x = 0; x < NUMBER_OF_CELLS; x++)
    {
      length = 1;
      /* from the top to the bottom */
      for (x1 = x + 1, y1 = 1; (y1 < NUMBER_OF_CELLS - 1) && (x1 < NUMBER_OF_CELLS - 1); x1++, y1++)
	if ((board[x1][y1] == board[x1 - 1][y1 - 1]) && (board[x1][y1] != 0))
	  {
	    while ((board[x1][y1] == board[x1 - 1][y1 - 1]) && (x1 < NUMBER_OF_CELLS)
		   && (y1 < NUMBER_OF_CELLS))
	      {
		length++;
		x1++;
		y1++;
	      }
	    if (length > 4)
	      {
		clear_line (x1, y1, length, 4, balls_to_delete, &number_of_balls_to_delete);
		retval = TRUE;
	      }
	    length = 1;
	  }

      length = 1;
      /* from bottom to top */
      for (x1 = x + 1, y1 = NUMBER_OF_CELLS - 2; (y1 >= 0) && (x1 < NUMBER_OF_CELLS); x1++, y1--)
	if ((board[x1][y1] == board[x1 - 1][y1 + 1]) && (board[x1][y1] != 0))
	  {
	    while ((board[x1][y1] == board[x1 - 1][y1 + 1]) && (x1 < NUMBER_OF_CELLS)
		   && (y1 >= 0))
	      {
		length++;
		x1++;
		y1--;
	      }
	    if (length > 4)
	      {
		clear_line (x1, y1, length, 5, balls_to_delete, &number_of_balls_to_delete);
		retval = TRUE;
	      }
	    length = 1;
	  }
    }

  if (retval)
    {
      for (color = BLUE_BALL; color <= CYAN_BALL; color++)
	{
	  for (i = 0; i < number_of_balls_to_delete; i++)
	    {
	      board[balls_to_delete[i].x][balls_to_delete[i].y] = color;
	      draw_board (widget, pixmap);
	      gtk_widget_draw (widget, NULL);
	    }
	}
      
      for (i = 0; i < number_of_balls_to_delete; i++)
	board[balls_to_delete[i].x][balls_to_delete[i].y] = 0;
      
      draw_board (widget, pixmap); 
      gtk_widget_draw (widget, NULL);

      score = score + 10 + 2 * (number_of_balls_to_delete - 5) * (number_of_balls_to_delete - 5);
      if (!show_next_colors) 
	score++;
      set_user_score_label ();
    }


  free (balls_to_delete);
  return retval;
}

void 
read_entry (GtkWidget * widget, gpointer data)
{
  struct score_board current_entry;

  user_name = gtk_entry_get_text (GTK_ENTRY (data));
  if (strlen (user_name) == 0)
    strcpy (current_entry.name, "Anonymous");
  else
    strcpy (current_entry.name, user_name);
  strcpy (last_player_name, user_name);
  current_entry.score = saved_score;
  if (!read_score (scoreboard))
    {
      g_print ("Unable to read score.\n");
    }
  insert_entry_in_score_board (scoreboard, current_entry);
  if (!write_score (scoreboard))
    {
      g_print ("Unable to save score.\n");
    }
  gtk_grab_remove (dialog);
  gtk_widget_destroy (dialog);
  if (!write_score (scoreboard))
    g_print ("Unable to save score.\n");
}

char *
input_name_dialog (void)
{
  GtkWidget *prompt_label;
  GtkWidget *name;
  GtkWidget *button;

  /* we have to save score, because they will be set to 0 in new_game() */
  saved_score = score;

  dialog = gtk_dialog_new ();

  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);

  prompt_label = gtk_label_new ("Enter your name");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), prompt_label, TRUE,
		      TRUE, 0);
  gtk_widget_show (prompt_label);

  name = gtk_entry_new_with_max_length (14);
  /* restore the last player's name */
  if (user_name != NULL) {
    gtk_entry_set_text (GTK_ENTRY (name), last_player_name);
    gtk_entry_select_region (GTK_ENTRY (name), 0, strlen (gtk_entry_get_text (GTK_ENTRY (name))));
  }
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), name, TRUE,
		      TRUE, 5);
  gtk_widget_show (name);

  button = gtk_button_new_with_label ("OK");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (read_entry), name);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
		      TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (dialog);
  gtk_grab_add (dialog);

  return NULL;
}
