/* GtkBalls
 * Copyright (C) 1998 Eugene V. Morozov
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <locale.h>
#include <sys/param.h>
#include <time.h>
#include <gtk/gtk.h>
#include <X11/Xlocale.h>

#include "gtkballs.h"
#include "interface.h"
#include "preferences.h"
#include "scoreboard.h"
#include "path.h"
#include "license.h"

/* Backing pixmap for drawing area */
static GdkPixmap *pixmap = NULL;
static GdkPixmap *small_balls[NUMBER_OF_BALLS + 1];
static GdkPixmap *balls[NUMBER_OF_BALLS + 1];
static GdkPixmap *paws[8];
static GdkPixmap *icon;
static GtkWidget *drawing_area = NULL;
GtkWidget *hi_score_label = NULL, *user_score_label = NULL;
/* Three balls that shows what colors will appear on next turn
 * I use this variables in preferences.c */
GtkWidget *ball1, *ball2, *ball3;
/* This flag shows should be next colors to be shown
 * This variable is also used in preferences.c */
gint show_next_colors = TRUE;
/* This flag indicates wether to show the path of the ball
 * This variable is also used in preferences.c */
gint show_path = TRUE;
/* This flagsindicates wether to show the footprints of the ball
 * This variable is also used in preferences.c */
gint show_footprints = TRUE;
int board[NUMBER_OF_CELLS][NUMBER_OF_CELLS];
static int animation_in_progress = FALSE;
int jumping_ball = 0;		/* number of the jumping ball */
int jumping_ball_color = 0;
static int paw_x, paw_y;
/* The next two variables is also used in interface.c */
int score = 0, hi_score = 0;
struct score_board scoreboard[10];
static guint timer_tag;
static int next_colors[5] = {0, 0, 0, 0, 0};

int move_ball (GtkWidget * widget, int source_node, int target_ball);
void new_game ();

static GtkItemFactoryEntry menu_items[] = 
{
  {"/Game/New", "<control>N", new_game, 0, "<Item>"},
  {"/Game/", NULL, NULL, 0, "<Separator>"},
  {"/Game/Hall of fame", NULL, show_hall_of_fame, 0, "<Item>"},
  {"/Game/", NULL, NULL, 0, "<Separator>"},
  {"/Game/Quit", "<control>Q", destroy, 0, "<Item>"},
  {"/Settings/Preferences", "<control>P", preferences_dialog, 0, "<Item>"},
  {"/Help", NULL, NULL, 0, "<LastBranch>"},
  {"/Help/About", NULL, about, 0, "<Item>"}
};

void get_main_menu (GtkWidget *window, GtkWidget **menubar)
{
  int nmenu_items = sizeof (menu_items)/sizeof (menu_items[0]);
  GtkItemFactory *factory;

  factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<Main>", NULL);

  gtk_item_factory_create_items (factory, nmenu_items, menu_items, NULL);
  gtk_window_add_accel_group (GTK_WINDOW (window), factory->accel_group);

  if (menubar)
    *menubar = factory->widget;
}

int
count_free_cells (void)
{
  int i, j, counter = 0;

  for (i = 0; i < NUMBER_OF_CELLS; i++)
    for (j = 0; j < NUMBER_OF_CELLS; j++)
      if (board[i][j] == 0)
	counter++;

  return counter;
}

int
new_turn (int number)
{
  int i, j, k = 0, free_cells_number;

  if ((free_cells_number = count_free_cells ()) == 0)
    return FALSE;

  k = 0;
  do
    {
      i = (int) (((float) NUMBER_OF_CELLS) * rand () / (RAND_MAX + 1.0));
      j = (int) (((float) NUMBER_OF_CELLS) * rand () / (RAND_MAX + 1.0));
      if (board[i][j] == 0)
	{
	  board[i][j] = next_colors[k];
	  k++;
	}
      else
	continue;
      /* I hope that k!=0 */
      next_colors[k-1] = 1 + (int) (((float) NUMBER_OF_BALLS) * rand () / (RAND_MAX + 1.0));
    }
  while (k < ((number < free_cells_number) ? number : free_cells_number));

  return TRUE;
}

/* 
   start a new game:
   clear the game field, create five new balls
   and set score to 0.
 */
void
new_game ()
{
  int i, j;

  if (animation_in_progress == TRUE)
    {
      gtk_timeout_remove (timer_tag);
      animation_in_progress = FALSE;
    }

  for (i = 0; i < NUMBER_OF_CELLS; i++)
    for (j = 0; j < NUMBER_OF_CELLS; j++)
      board[i][j] = 0;

  new_turn (5);
  score = 0;
  set_user_score_label ();
  if (drawing_area != NULL)
    {
      gtk_pixmap_set (GTK_PIXMAP (ball1), 
		      small_balls[next_colors[0]], 
		      GTK_PIXMAP (ball1)->mask);
      gtk_pixmap_set (GTK_PIXMAP (ball2), 
		      small_balls[next_colors[1]], 
		      GTK_PIXMAP (ball2)->mask);
      gtk_pixmap_set (GTK_PIXMAP (ball3), 
		      small_balls[next_colors[2]], 
		      GTK_PIXMAP (ball3)->mask);
      gtk_widget_draw (drawing_area, NULL);
    }
}

void
about (GtkWidget * widget, gpointer data)
{
  GtkWidget *window;
  GtkWidget *vbox, *buttons_box;
  GdkPixmap *splash; /* GtkBalls logo and copying conditions */
  GtkWidget *pixmap;
  GdkBitmap *mask;
  GtkStyle *style;
  GtkWidget *separator;
  GtkWidget *ok_button, *license_button;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed), &window);
  gtk_window_set_title (GTK_WINDOW (window), "About");
  gtk_container_border_width (GTK_CONTAINER (window), 2);
  
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  style = gtk_widget_get_style (window);
  splash = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
				       &style->bg[GTK_STATE_NORMAL],
				       PREFIX "/share/gtkballs/gtkballs_about.xpm");
  pixmap = gtk_pixmap_new (splash, mask);
  gtk_box_pack_start (GTK_BOX (vbox), pixmap, TRUE, TRUE, 1);
  gtk_widget_show (pixmap);

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, FALSE, 5);
  gtk_widget_show (separator);

  buttons_box = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), buttons_box, TRUE, TRUE, 2);
  gtk_widget_show (buttons_box);

  ok_button = gtk_button_new_with_label (" OK ");
  gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (buttons_box), ok_button, TRUE, FALSE, 0);
  gtk_widget_show (ok_button);

  license_button = gtk_button_new_with_label (" Show license ");
  gtk_signal_connect (GTK_OBJECT (license_button), "clicked",
		      GTK_SIGNAL_FUNC (gnu_license_dialog), NULL);
  gtk_box_pack_start (GTK_BOX (buttons_box), license_button, TRUE, FALSE, 0);
  gtk_widget_show (license_button);

  gtk_widget_show (window);
}

gint
delete_event (GtkWidget *widget, gpointer data)
{
  return (FALSE);
}

gint
animate_ball (gpointer data)
{
  GtkWidget *widget;
  int x, y;
  static int phase = 0;		/* 1 through NUMBER_OF_PHASES */

  widget = GTK_WIDGET (data);

  phase = ++phase % NUMBER_OF_PHASES;
  x = find_x_of_the_node (jumping_ball, NUMBER_OF_CELLS);
  y = find_y_of_the_node (jumping_ball, NUMBER_OF_CELLS);
  if (widget != NULL)
    draw_ball (widget, balls[jumping_ball_color], pixmap, x, y, 
		   (phase > NUMBER_OF_PHASES/2) ? phase + 1 : 3 - phase);
  
  return TRUE;
}

void
draw_board (GtkWidget * widget, GdkPixmap * pixmap)
{
  int i, j;

  for (i = 0; i < NUMBER_OF_CELLS; i++)
    for (j = 0; j < NUMBER_OF_CELLS; j++)
      if (animation_in_progress == TRUE)
	{
	  if (((j * NUMBER_OF_CELLS + i) != jumping_ball) && (paw_x != i) && (paw_y != j))
	    {
	      gdk_draw_rectangle (pixmap,
				  widget->style->white_gc,
				  TRUE,
				  i * 32 + 1, j * 32 + 1,
				  29, 29);
	      gdk_draw_pixmap (pixmap,
			       widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			       balls[board[i][j]],
			       0, 0,
			       i * 32 + 6, j * 32 + 6,
			       21, 21);
	    }
	}
      else
	{
	  gdk_draw_rectangle (pixmap,
			      widget->style->white_gc,
			      TRUE,
			      i * 32 + 1, j * 32 + 1,
			      29, 29);
	  gdk_draw_pixmap (pixmap,
			   widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			   balls[board[i][j]],
			   0, 0,
			   i * 32 + 6, j * 32 + 6,
			   21, 21);
	}
}

void
destroy (GtkWidget * widget, gpointer data)
{
  if (data != NULL)
    {
      gtk_widget_destroy (GTK_WIDGET (data));
      return;
    }

  gtk_main_quit ();
}

/* Create a new backing pixmap of the appropriate size */
static gint
configure_event (GtkWidget * widget, GdkEventConfigure * event)
{
  pixmap = gdk_pixmap_new (widget->window,
			   widget->allocation.width,
			   widget->allocation.height,
			   -1);

  gdk_draw_rectangle (pixmap,
		      widget->style->bg_gc[GTK_STATE_NORMAL],
		      TRUE,
		      0, 0,
		      widget->allocation.width,
		      widget->allocation.height);
  return TRUE;
}

/* Refill the screen from the backing pixmap */
static gint
expose_event (GtkWidget * widget, GdkEventExpose * event)
{
  int x, y;

  for (y = 0; y < widget->allocation.width;
       y += widget->allocation.height / NUMBER_OF_CELLS)
    gdk_draw_line (pixmap,
		   widget->style->black_gc,
		   0, y,
		   widget->allocation.width - 1, y);

  for (x = 0; x < widget->allocation.width;
       x += widget->allocation.width / NUMBER_OF_CELLS)
    gdk_draw_line (pixmap,
		   widget->style->black_gc,
		   x, 0,
		   x, widget->allocation.height - 1);

  draw_board (widget, pixmap);

  gdk_draw_pixmap (widget->window,
		   widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		   pixmap,
		   event->area.x, event->area.y,
		   event->area.x, event->area.y,
		   event->area.width, event->area.height);

  return FALSE;
}

static gint
button_press_event (GtkWidget * widget, GdkEventButton * event)
{
  int x, y;
  
  if (event->button == 1 && pixmap != NULL)
    {
      x = calculate_board_coord (event->x);
      y = calculate_board_coord (event->y);
      if (board[x][y] != 0)
	{
	  if ((animation_in_progress == TRUE)
	      && (x == find_x_of_the_node (jumping_ball, NUMBER_OF_CELLS))
	      && (y == find_y_of_the_node (jumping_ball, NUMBER_OF_CELLS)))
	    return FALSE;
	  if (animation_in_progress == TRUE)
	    gtk_timeout_remove (timer_tag);

	  /* following line makes sure that ball isn't in the highest position */
	  draw_ball (widget, balls[jumping_ball_color], pixmap,
			 find_x_of_the_node (jumping_ball, NUMBER_OF_CELLS),
			 find_y_of_the_node (jumping_ball, NUMBER_OF_CELLS),
			 5);

	  jumping_ball = y * NUMBER_OF_CELLS + x;
	  jumping_ball_color = board[x][y];
	  animation_in_progress = TRUE;
	  timer_tag = gtk_timeout_add (110, animate_ball, widget);
	}
      else if (animation_in_progress == TRUE)
	{
	  gtk_timeout_remove (timer_tag);
	  animation_in_progress = FALSE;

	  if (move_ball (widget, jumping_ball, y * NUMBER_OF_CELLS + x) == 0)
	    {
	      timer_tag = gtk_timeout_add (110, animate_ball, widget);
	      animation_in_progress = TRUE;
	      return FALSE;
	    }
	  if (destroy_lines (widget, pixmap) == FALSE)
	    {
	      new_turn (3);
	      gtk_pixmap_set (GTK_PIXMAP (ball1), 
			      small_balls[next_colors[0]], 
			      GTK_PIXMAP (ball1)->mask);
	      gtk_pixmap_set (GTK_PIXMAP (ball2), 
			      small_balls[next_colors[1]], 
			      GTK_PIXMAP (ball2)->mask);
	      gtk_pixmap_set (GTK_PIXMAP (ball3), 
			      small_balls[next_colors[2]], 
			      GTK_PIXMAP (ball3)->mask);
	    }
	  /* if fifth ball will appear */
	  destroy_lines (widget, pixmap);
	  if (count_free_cells () == 0)
	    {
	      if (score > scoreboard[9].score)
		input_name_dialog ();
	      new_game (widget);
	    }
	  draw_board (widget, pixmap);
	  gtk_widget_draw (widget, NULL);
	}
    }

  return FALSE;
}

int
move_ball (GtkWidget * widget, int source_ball, int target_ball)
{
  int nodes[(NUMBER_OF_CELLS) * (NUMBER_OF_CELLS)];
  int path[(NUMBER_OF_CELLS) * (NUMBER_OF_CELLS)];
  int i, j, k, phase;
  int x, y, color;
  int direction, offset;

  i = find_x_of_the_node (target_ball, NUMBER_OF_CELLS);
  j = find_y_of_the_node (target_ball, NUMBER_OF_CELLS);
  if (board[i][j] != 0)
    return 2;

  for (i = 0; i < NUMBER_OF_CELLS; i++)
    for (j = 0; j < NUMBER_OF_CELLS; j++)
      if (board[i][j] != 0)
	nodes[j * NUMBER_OF_CELLS + i] = -1;
      else
	nodes[j * NUMBER_OF_CELLS + i] = 0;

  nodes[source_ball] = nodes[target_ball] = 0;
  if (!find_path (nodes, NUMBER_OF_CELLS, source_ball,
		  target_ball, path))
    return 0;

  x = find_x_of_the_node (source_ball, NUMBER_OF_CELLS);
  y = find_y_of_the_node (source_ball, NUMBER_OF_CELLS);
  color = board[x][y];
  board[x][y] = 0;
  phase = 0;

  if (show_path)
    {
      animation_in_progress = TRUE;
      offset = 0;
      draw_ball (widget, balls[0], pixmap, x, y, 6);
      
      for (k = path[0] - 1; k; k--)
	{
	  i = find_x_of_the_node (path[k], NUMBER_OF_CELLS);
	  j = find_y_of_the_node (path[k], NUMBER_OF_CELLS);
	  jumping_ball = j * NUMBER_OF_CELLS + i;
	  if (k == path[0] - 1)
	    {
	      /* x and y are the coordinates of starting position */
	      paw_x = x;
	      paw_y = y;
	    }
	  else
	    {
	      paw_x = find_x_of_the_node (path[k+1], NUMBER_OF_CELLS);
	      paw_y = find_y_of_the_node (path[k+1], NUMBER_OF_CELLS);
	    }
	  if (paw_y < j)
	    direction = 2; /* up */
	  else if (paw_y > j)
	    direction = 0; /* down */
	  else if (paw_x < i)
	    direction = 1; /* left */
	  else if (paw_x > i)
	    direction = 3; /* right */
	  if (show_footprints)
	    draw_paw (widget, paws[direction+offset], pixmap, paw_x, paw_y);
	  offset = (offset == 0) ? 4 : 0;
	  for (phase = 0; phase < NUMBER_OF_PHASES; phase++)
	    {
	      draw_ball (widget, balls[jumping_ball_color], pixmap, i, j, 
			 (phase > NUMBER_OF_PHASES/2) ? phase + 1 : 3 - phase);
	    }
	  gdk_draw_rectangle (pixmap,
			      widget->style->white_gc,
			      TRUE,
			      i * 32 + 1, j * 32 + 1,
			      29, 29);
	  update_rectangle (widget, i * 32 + 1, j * 32 + 1, 29, 29);
	}
      i = find_x_of_the_node (target_ball, NUMBER_OF_CELLS);
      j = find_y_of_the_node (target_ball, NUMBER_OF_CELLS);
      jumping_ball = j * NUMBER_OF_CELLS + i;
      if (k == path[0] - 1)
	{
	  /* x and y are the coordinates of starting position */
	  paw_x = x;
	  paw_y = y;
	}
      else
	{
	  paw_x = find_x_of_the_node (path[k+1], NUMBER_OF_CELLS);
	  paw_y = find_y_of_the_node (path[k+1], NUMBER_OF_CELLS);
	}
      if (paw_y < j)
	direction = 2; /* up */
      else if (paw_y > j)
	direction = 0; /* down */
      else if (paw_x < i)
	direction = 1; /* left */
      else if (paw_x > i)
	direction = 3; /* right */
      if (show_footprints)
	draw_paw (widget, paws[direction+offset], pixmap, paw_x, paw_y);
      offset = (offset == 0) ? 4 : 0;
    }
  
  animation_in_progress = FALSE;

  i = find_x_of_the_node (target_ball, NUMBER_OF_CELLS);
  j = find_y_of_the_node (target_ball, NUMBER_OF_CELLS);
  board[i][j] = color;

  draw_ball (widget, balls[color], pixmap, i, j, 6);

  return 1;
}

int
main (int argc, char **argv)
{
  GtkWidget *main_window;
  GtkWidget *menubar;
  GtkWidget *vbox, *hbox, *drawing_area_box, *small_balls_box;
  GdkBitmap *mask;
  GtkStyle *style;
  int i, j;

  setlocale (LC_ALL, "");

  if (!read_score (scoreboard))
    {
      strcpy (scoreboard[0].name, "John");
      scoreboard[0].score = 100;
      for (i = 1; i < 10; i++)
	{
	  scoreboard[i].name[0] = '\0';
	  scoreboard[i].score = 0;
	}
    }

  /* load user's preferences */
  load_preferences ();

  /* initialize random seed */
  srand ((unsigned int) time (NULL));

  for (i = 0; i < NUMBER_OF_CELLS; i++)
    for (j = 0; j < NUMBER_OF_CELLS; j++)
      board[i][j] = 0;
  
  for (i = 0; i < 5; i++)
    next_colors[i] = 1 + (int) (((float) NUMBER_OF_BALLS) * rand () / (RAND_MAX + 1.0));
  
  new_turn (5);
  
  gtk_init (&argc, &argv);

  main_window = gtk_widget_new (gtk_window_get_type (),
				"GtkObject::user_data", NULL,
				"GtkWindow::type", GTK_WINDOW_TOPLEVEL,
				"GtkWindow::title", "GtkBalls",
				"GtkWindow::allow_grow", FALSE,
				"GtkWindow::allow_shrink", FALSE,
				NULL);

  gtk_widget_set_usize (main_window, 330, 360);

  gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
		      GTK_SIGNAL_FUNC (delete_event), NULL);
  gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
		      GTK_SIGNAL_FUNC (destroy), NULL);
  gtk_widget_show (main_window);

  vbox = gtk_vbox_new (FALSE, 5);
  hbox = gtk_hbox_new (TRUE, 0);
  drawing_area_box = gtk_hbox_new (FALSE, 0);
  small_balls_box = gtk_hbox_new (TRUE, 0);

  get_main_menu (main_window, &menubar);
  gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);

  hi_score = scoreboard[0].score;
  score = 0;

  hi_score_label = gtk_label_new ("Hi-score: 0");
  set_hi_score_label ();
  gtk_box_pack_start (GTK_BOX (hbox), hi_score_label, FALSE, FALSE, 0);
  gtk_widget_show (hi_score_label);

  style = gtk_widget_get_style (main_window);
  small_balls[BLUE_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
						       &style->bg[GTK_STATE_NORMAL],
						       PREFIX "/share/gtkballs/small_blue_ball.xpm");
  small_balls[BROWN_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
							&style->bg[GTK_STATE_NORMAL],
							PREFIX "/share/gtkballs/small_brown_ball.xpm");
  small_balls[GREEN_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
							&style->bg[GTK_STATE_NORMAL],
							PREFIX "/share/gtkballs/small_green_ball.xpm");
  small_balls[RED_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
						      &style->bg[GTK_STATE_NORMAL],
						      PREFIX "/share/gtkballs/small_red_ball.xpm");
  small_balls[YELLOW_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
							 &style->bg[GTK_STATE_NORMAL],
							 PREFIX "/share/gtkballs/small_yellow_ball.xpm");
  small_balls[PINK_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
						       &style->bg[GTK_STATE_NORMAL],
						       PREFIX "/share/gtkballs/small_pink_ball.xpm");
  small_balls[CYAN_BALL] = gdk_pixmap_create_from_xpm (main_window->window, &mask,
						       &style->bg[GTK_STATE_NORMAL],
						       PREFIX "/share/gtkballs/small_cyan_ball.xpm");
  
  ball1 = gtk_pixmap_new (small_balls[next_colors[0]], mask);
  gtk_box_pack_start (GTK_BOX (small_balls_box), ball1, FALSE, FALSE, 0);

  ball2 = gtk_pixmap_new (small_balls[next_colors[1]], mask);
  gtk_box_pack_start (GTK_BOX (small_balls_box), ball2, FALSE, FALSE, 0);

  ball3 = gtk_pixmap_new (small_balls[next_colors[2]], mask);
  gtk_box_pack_start (GTK_BOX (small_balls_box), ball3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), small_balls_box, FALSE, FALSE, 0);
  if (show_next_colors)
    {
      gtk_widget_show (ball1);
      gtk_widget_show (ball2);
      gtk_widget_show (ball3);
    }
  gtk_widget_show (small_balls_box);
  
  user_score_label = gtk_label_new ("Your score: 0");
  set_user_score_label ();
  gtk_box_pack_start (GTK_BOX (hbox), user_score_label, FALSE, FALSE, 0);
  gtk_widget_show (user_score_label);

  gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (hbox), FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  drawing_area = gtk_drawing_area_new ();

  gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area), 289, 289);

  gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "configure_event",
		      (GtkSignalFunc) configure_event, NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
		      (GtkSignalFunc) button_press_event, NULL);
  gtk_widget_set_events (drawing_area,
			 GDK_EXPOSURE_MASK |
			 GDK_BUTTON_PRESS_MASK);

  gtk_box_pack_start (GTK_BOX (drawing_area_box), drawing_area, TRUE, FALSE, 10);
  gtk_widget_show (drawing_area);

  gtk_box_pack_start (GTK_BOX (vbox), drawing_area_box, FALSE, FALSE, 0);
  gtk_widget_show (GTK_WIDGET (drawing_area_box));

  gtk_container_add (GTK_CONTAINER (main_window), vbox);
  gtk_widget_show (vbox);
  
  /* load balls */
  balls[0] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					 &style->bg[GTK_STATE_NORMAL],
					 PREFIX "/share/gtkballs/empty_cell.xpm");
  balls[BLUE_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						 &style->bg[GTK_STATE_NORMAL],
						 PREFIX "/share/gtkballs/blue_ball.xpm");
  balls[BROWN_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						  &style->bg[GTK_STATE_NORMAL],
						  PREFIX "/share/gtkballs/brown_ball.xpm");
  balls[GREEN_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						  &style->bg[GTK_STATE_NORMAL],
						  PREFIX "/share/gtkballs/green_ball.xpm");
  balls[RED_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						&style->bg[GTK_STATE_NORMAL],
						PREFIX "/share/gtkballs/red_ball.xpm");
  balls[YELLOW_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						   &style->bg[GTK_STATE_NORMAL],
						   PREFIX "/share/gtkballs/yellow_ball.xpm");
  balls[PINK_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						 &style->bg[GTK_STATE_NORMAL],
						 PREFIX "/share/gtkballs/pink_ball.xpm");
  balls[CYAN_BALL] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
						 &style->bg[GTK_STATE_NORMAL],
						 PREFIX "/share/gtkballs/cyan_ball.xpm");

  paws[0] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_up_1.xpm");
  paws[1] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_right_1.xpm");
  paws[2] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_down_1.xpm");
  paws[3] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_left_1.xpm");
  paws[4] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_up_2.xpm");
  paws[5] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_right_2.xpm");
  paws[6] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_down_2.xpm");
  paws[7] = gdk_pixmap_create_from_xpm (drawing_area->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					PREFIX "/share/gtkballs/paw_left_2.xpm");

  icon = gdk_pixmap_create_from_xpm (main_window->window, &mask,
				     &style->bg[GTK_STATE_NORMAL],
				     PREFIX "/share/gtkballs/gtkballs.xpm");
  gdk_window_set_icon (main_window->window,
		       NULL,
		       icon,
		       NULL);
  gtk_main ();

  return 0;
}
