/* GtkamlCompiler.vala
 * 
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 1996-2002, 2004, 2005, 2006 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "GtkamlCompiler.h"
#include <stdlib.h>
#include <string.h>
#include <vala/valacodecontext.h>
#include <vala/valareport.h>
#include <stdio.h>
#include <vala/valasourcefile.h>
#include <glib/gstdio.h>
#include <vala/valasourcereference.h>
#include <gobject/valaccodegenerator.h>
#include <vala/valausingdirective.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valasymbol.h>
#include <vala/valagenieparser.h>
#include <vala/valasymbolresolver.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valacodewriter.h>
#include <vala/valaflowanalyzer.h>
#include <vala/valanullchecker.h>
#include <vala/valacodegenerator.h>
#include <gobject/valagirwriter.h>
#include <gobject/valaccodecompiler.h>
#include <config.h>
#include "GtkamlParser.h"
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_COMPILER (vala_compiler_get_type ())
#define VALA_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMPILER, ValaCompiler))
#define VALA_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMPILER, ValaCompilerClass))
#define VALA_IS_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMPILER))
#define VALA_IS_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMPILER))
#define VALA_COMPILER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMPILER, ValaCompilerClass))

typedef struct _ValaCompiler ValaCompiler;
typedef struct _ValaCompilerClass ValaCompilerClass;
typedef struct _ValaCompilerPrivate ValaCompilerPrivate;
typedef struct _ValaParamSpecCompiler ValaParamSpecCompiler;

struct _ValaCompiler {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCompilerPrivate * priv;
};

struct _ValaCompilerClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCompiler *self);
};

struct _ValaParamSpecCompiler {
	GParamSpec parent_instance;
};



static char* string_substring (const char* self, glong offset, glong len);
struct _ValaCompilerPrivate {
	ValaCodeContext* context;
};

#define VALA_COMPILER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_COMPILER, ValaCompilerPrivate))
enum  {
	VALA_COMPILER_DUMMY_PROPERTY
};
static char* vala_compiler_basedir = NULL;
static char* vala_compiler_directory = NULL;
static gboolean vala_compiler_version = FALSE;
static char** vala_compiler_sources = NULL;
static char** vala_compiler_vapi_directories = NULL;
static char* vala_compiler_library = NULL;
static char** vala_compiler_packages = NULL;
static char* vala_compiler_target_glib = NULL;
static char** vala_compiler_implicits_directories = NULL;
static gint vala_compiler_implicits_directories_length1 = 0;
static gint vala_compiler_implicits_directories_size = 0;
static gboolean vala_compiler_ccode_only = FALSE;
static char* vala_compiler_header_filename = NULL;
static gboolean vala_compiler_compile_only = FALSE;
static char* vala_compiler_output = NULL;
static gboolean vala_compiler_debug = FALSE;
static gboolean vala_compiler_thread = FALSE;
static gboolean vala_compiler_disable_assert = FALSE;
static gboolean vala_compiler_enable_checking = FALSE;
static gboolean vala_compiler_deprecated = FALSE;
static gboolean vala_compiler_experimental = FALSE;
static gboolean vala_compiler_non_null_experimental = FALSE;
static gboolean vala_compiler_disable_dbus_transformation = FALSE;
static char* vala_compiler_cc_command = NULL;
static char** vala_compiler_cc_options = NULL;
static char* vala_compiler_dump_tree = NULL;
static gboolean vala_compiler_save_temps = FALSE;
static gboolean vala_compiler_quiet_mode = FALSE;
static gint vala_compiler_quit (ValaCompiler* self);
static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg);
static gint vala_compiler_run (ValaCompiler* self);
static gboolean vala_compiler_ends_with_dir_separator (const char* s);
static char* vala_compiler_realpath (const char* name);
static gint vala_compiler_main (char** args, int args_length1);
static ValaCompiler* vala_compiler_construct (GType object_type);
static ValaCompiler* vala_compiler_new (void);
static GParamSpec* vala_param_spec_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
static gpointer vala_value_get_compiler (const GValue* value);
static void vala_value_set_compiler (GValue* value, gpointer v_object);
static gpointer vala_compiler_parent_class = NULL;
static void vala_compiler_finalize (ValaCompiler* obj);
static GType vala_compiler_get_type (void);
static gpointer vala_compiler_ref (gpointer instance);
static void vala_compiler_unref (gpointer instance);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);

static const GOptionEntry VALA_COMPILER_options[] = {{"vapidir", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", (gchar) 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_library, "Library name", "NAME"}, {"basedir", 'b', 0, G_OPTION_ARG_FILENAME, &vala_compiler_basedir, "Base source directory", "DIRECTORY"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_compiler_directory, "Output directory", "DIRECTORY"}, {"version", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_version, "Display version number", NULL}, {"ccode", 'C', 0, G_OPTION_ARG_NONE, &vala_compiler_ccode_only, "Output C code", NULL}, {"header", 'H', 0, G_OPTION_ARG_FILENAME, &vala_compiler_header_filename, "Output C header file", "FILE"}, {"compile", 'c', 0, G_OPTION_ARG_NONE, &vala_compiler_compile_only, "Compile but do not link", NULL}, {"output", 'o', 0, G_OPTION_ARG_FILENAME, &vala_compiler_output, "Place output in file FILE", "FILE"}, {"debug", 'g', 0, G_OPTION_ARG_NONE, &vala_compiler_debug, "Produce debug information", NULL}, {"thread", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_thread, "Enable multithreading support", NULL}, {"disable-assert", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_assert, "Disable assertions", NULL}, {"enable-checking", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_enable_checking, "Enable additional run-time checks", NULL}, {"enable-deprecated", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_deprecated, "Enable deprecated features", NULL}, {"enable-experimental", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_experimental, "Enable experimental features", NULL}, {"enable-non-null-experimental", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_non_null_experimental, "Enable experimental enhancements for non-null types", NULL}, {"disable-dbus-transformation", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_dbus_transformation, "Disable transformation of D-Bus member names", NULL}, {"cc", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_cc_command, "Use COMMAND as C compiler command", "COMMAND"}, {"Xcc", 'X', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_cc_options, "Pass OPTION to the C compiler", "OPTION..."}, {"dump-tree", (gchar) 0, 0, G_OPTION_ARG_FILENAME, &vala_compiler_dump_tree, "Write code tree to FILE", "FILE"}, {"save-temps", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_save_temps, "Keep temporary files", NULL}, {"implicitsdir", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_implicits_directories, "Look for implicit add and creation methods and their parameters in DIRECTORY", "DIRECTORY..."}, {"quiet", 'q', 0, G_OPTION_ARG_NONE, &vala_compiler_quiet_mode, "Do not print messages to the console", NULL}, {"target-glib", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_target_glib, "Target version of glib for code generation", "MAJOR.MINOR"}, {"", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_sources, NULL, "FILE..."}, {NULL}};


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static gint vala_compiler_quit (ValaCompiler* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, 0);
	_tmp0 = FALSE;
	if (vala_report_get_errors () == 0) {
		_tmp0 = vala_report_get_warnings () == 0;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		return 0;
	}
	if (vala_report_get_errors () == 0) {
		if (!vala_compiler_quiet_mode) {
			fprintf (stdout, "Compilation succeeded - %d warning(s)\n", vala_report_get_warnings ());
		}
		return 0;
	} else {
		if (!vala_compiler_quiet_mode) {
			fprintf (stdout, "Compilation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (), vala_report_get_warnings ());
		}
		return 1;
	}
}


static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg) {
	GError * inner_error;
	char* package_path;
	ValaSourceFile* _tmp2;
	char* _tmp4;
	char* _tmp3;
	char* _tmp5;
	char* deps_filename;
	gboolean _tmp13;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (context != NULL, FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	inner_error = NULL;
	if (vala_code_context_has_package (context, pkg)) {
		/* ignore multiple occurences of the same package*/
		return TRUE;
	}
	package_path = vala_code_context_get_package_path (context, pkg, vala_compiler_vapi_directories, _vala_array_length (vala_compiler_vapi_directories));
	if (package_path == NULL) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, package_path = (g_free (package_path), NULL), _tmp1);
	}
	vala_code_context_add_package (context, pkg);
	_tmp2 = NULL;
	vala_code_context_add_source_file (context, _tmp2 = vala_source_file_new (context, package_path, TRUE, NULL));
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_file_unref (_tmp2), NULL));
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp5 = NULL;
	deps_filename = (_tmp5 = g_build_filename (_tmp3 = g_path_get_dirname (package_path), _tmp4 = g_strdup_printf ("%s.deps", pkg), NULL), _tmp4 = (g_free (_tmp4), NULL), _tmp3 = (g_free (_tmp3), NULL), _tmp5);
	if (g_file_test (deps_filename, G_FILE_TEST_EXISTS)) {
		{
			char* deps_content;
			gulong deps_len;
			char* _tmp8;
			gboolean _tmp7;
			char* _tmp6;
			deps_content = NULL;
			deps_len = 0UL;
			_tmp8 = NULL;
			_tmp6 = NULL;
			_tmp7 = g_file_get_contents (deps_filename, &_tmp6, &deps_len, &inner_error);
			deps_content = (_tmp8 = _tmp6, deps_content = (g_free (deps_content), NULL), _tmp8);
			_tmp7;
			if (inner_error != NULL) {
				deps_content = (g_free (deps_content), NULL);
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				goto __finally0;
			}
			{
				char** _tmp9;
				char** dep_collection;
				int dep_collection_length1;
				int dep_it;
				_tmp9 = NULL;
				dep_collection = _tmp9 = g_strsplit (deps_content, "\n", 0);
				dep_collection_length1 = _vala_array_length (_tmp9);
				for (dep_it = 0; dep_it < _vala_array_length (_tmp9); dep_it = dep_it + 1) {
					const char* _tmp11;
					char* dep;
					_tmp11 = NULL;
					dep = (_tmp11 = dep_collection[dep_it], (_tmp11 == NULL) ? NULL : g_strdup (_tmp11));
					{
						if (_vala_strcmp0 (dep, "") != 0) {
							if (!vala_compiler_add_package (self, context, dep)) {
								char* _tmp10;
								_tmp10 = NULL;
								vala_report_error (NULL, _tmp10 = g_strdup_printf ("%s, dependency of %s, not found in specified Vala API directories", dep, pkg));
								_tmp10 = (g_free (_tmp10), NULL);
							}
						}
						dep = (g_free (dep), NULL);
					}
				}
				dep_collection = (_vala_array_free (dep_collection, dep_collection_length1, (GDestroyNotify) g_free), NULL);
			}
			deps_content = (g_free (deps_content), NULL);
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp12;
				_tmp12 = NULL;
				vala_report_error (NULL, _tmp12 = g_strdup_printf ("Unable to read dependency file: %s", e->message));
				_tmp12 = (g_free (_tmp12), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally0:
		if (inner_error != NULL) {
			package_path = (g_free (package_path), NULL);
			deps_filename = (g_free (deps_filename), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return FALSE;
		}
	}
	return (_tmp13 = TRUE, package_path = (g_free (package_path), NULL), deps_filename = (g_free (deps_filename), NULL), _tmp13);
}


static gint vala_compiler_run (ValaCompiler* self) {
	ValaCodeContext* _tmp0;
	gboolean _tmp1;
	gboolean _tmp2;
	gint glib_major;
	gint glib_minor;
	gboolean _tmp7;
	ValaCCodeGenerator* _tmp8;
	char** _tmp21;
	GtkamlParser* parser;
	ValaGenieParser* genie_parser;
	ValaSymbolResolver* resolver;
	ValaSemanticAnalyzer* analyzer;
	ValaFlowAnalyzer* flow_analyzer;
	gint _tmp39;
	g_return_val_if_fail (self != NULL, 0);
	_tmp0 = NULL;
	self->priv->context = (_tmp0 = vala_code_context_new (), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp0);
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	if (!vala_compiler_ccode_only) {
		_tmp2 = !vala_compiler_compile_only;
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		_tmp1 = vala_compiler_output == NULL;
	} else {
		_tmp1 = FALSE;
	}
	/* default to build executable*/
	if (_tmp1) {
		/* strip extension if there is one
		 else we use the default output file of the C compiler*/
		if (g_utf8_strrchr (vala_compiler_sources[0], (glong) (-1), (gunichar) '.') != NULL) {
			glong dot;
			char* _tmp4;
			char* _tmp3;
			dot = g_utf8_pointer_to_offset (vala_compiler_sources[0], g_utf8_strrchr (vala_compiler_sources[0], (glong) (-1), (gunichar) '.'));
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_compiler_output = (_tmp4 = g_path_get_basename (_tmp3 = string_substring (vala_compiler_sources[0], (glong) 0, dot)), vala_compiler_output = (g_free (vala_compiler_output), NULL), _tmp4);
			_tmp3 = (g_free (_tmp3), NULL);
		}
	}
	vala_code_context_set_library (self->priv->context, vala_compiler_library);
	vala_code_context_set_assert (self->priv->context, !vala_compiler_disable_assert);
	vala_code_context_set_checking (self->priv->context, vala_compiler_enable_checking);
	vala_code_context_set_deprecated (self->priv->context, vala_compiler_deprecated);
	vala_code_context_set_experimental (self->priv->context, vala_compiler_experimental);
	vala_code_context_set_non_null_experimental (self->priv->context, vala_compiler_non_null_experimental);
	vala_code_context_set_dbus_transformation (self->priv->context, !vala_compiler_disable_dbus_transformation);
	vala_report_set_verbose_errors (!vala_compiler_quiet_mode);
	vala_code_context_set_ccode_only (self->priv->context, vala_compiler_ccode_only);
	vala_code_context_set_compile_only (self->priv->context, vala_compiler_compile_only);
	vala_code_context_set_header_filename (self->priv->context, vala_compiler_header_filename);
	vala_code_context_set_output (self->priv->context, vala_compiler_output);
	if (vala_compiler_basedir != NULL) {
		char* _tmp5;
		_tmp5 = NULL;
		vala_code_context_set_basedir (self->priv->context, _tmp5 = vala_compiler_realpath (vala_compiler_basedir));
		_tmp5 = (g_free (_tmp5), NULL);
	}
	if (vala_compiler_directory != NULL) {
		char* _tmp6;
		_tmp6 = NULL;
		vala_code_context_set_directory (self->priv->context, _tmp6 = vala_compiler_realpath (vala_compiler_directory));
		_tmp6 = (g_free (_tmp6), NULL);
	} else {
		vala_code_context_set_directory (self->priv->context, vala_code_context_get_basedir (self->priv->context));
	}
	vala_code_context_set_debug (self->priv->context, vala_compiler_debug);
	vala_code_context_set_thread (self->priv->context, vala_compiler_thread);
	vala_code_context_set_save_temps (self->priv->context, vala_compiler_save_temps);
	glib_major = 2;
	glib_minor = 12;
	_tmp7 = FALSE;
	if (vala_compiler_target_glib != NULL) {
		_tmp7 = sscanf (vala_compiler_target_glib, "%d.%d", &glib_major, &glib_minor, NULL) != 2;
	} else {
		_tmp7 = FALSE;
	}
	if (_tmp7) {
		vala_report_error (NULL, "Invalid format for --target-glib");
	}
	vala_code_context_set_target_glib_major (self->priv->context, glib_major);
	vala_code_context_set_target_glib_minor (self->priv->context, glib_minor);
	if (vala_code_context_get_target_glib_major (self->priv->context) != 2) {
		vala_report_error (NULL, "This version of valac only supports GLib 2");
	}
	_tmp8 = NULL;
	vala_code_context_set_codegen (self->priv->context, (ValaCodeGenerator*) (_tmp8 = vala_ccode_generator_new ()));
	(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_visitor_unref (_tmp8), NULL));
	/* default packages */
	if (!vala_compiler_add_package (self, self->priv->context, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (!vala_compiler_add_package (self, self->priv->context, "gobject-2.0")) {
		vala_report_error (NULL, "gobject-2.0 not found in specified Vala API directories");
	}
	if (vala_compiler_packages != NULL) {
		char** _tmp11;
		{
			char** package_collection;
			int package_collection_length1;
			int package_it;
			package_collection = vala_compiler_packages;
			package_collection_length1 = _vala_array_length (vala_compiler_packages);
			for (package_it = 0; package_it < _vala_array_length (vala_compiler_packages); package_it = package_it + 1) {
				const char* _tmp10;
				char* package;
				_tmp10 = NULL;
				package = (_tmp10 = package_collection[package_it], (_tmp10 == NULL) ? NULL : g_strdup (_tmp10));
				{
					if (!vala_compiler_add_package (self, self->priv->context, package)) {
						char* _tmp9;
						_tmp9 = NULL;
						vala_report_error (NULL, _tmp9 = g_strdup_printf ("%s not found in specified Vala API directories", package));
						_tmp9 = (g_free (_tmp9), NULL);
					}
					package = (g_free (package), NULL);
				}
			}
		}
		_tmp11 = NULL;
		vala_compiler_packages = (_tmp11 = NULL, vala_compiler_packages = (_vala_array_free (vala_compiler_packages, _vala_array_length (vala_compiler_packages), (GDestroyNotify) g_free), NULL), _tmp11);
	}
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	{
		char** source_collection;
		int source_collection_length1;
		int source_it;
		source_collection = vala_compiler_sources;
		source_collection_length1 = _vala_array_length (vala_compiler_sources);
		for (source_it = 0; source_it < _vala_array_length (vala_compiler_sources); source_it = source_it + 1) {
			const char* _tmp20;
			char* source;
			_tmp20 = NULL;
			source = (_tmp20 = source_collection[source_it], (_tmp20 == NULL) ? NULL : g_strdup (_tmp20));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					char* rpath;
					gboolean _tmp13;
					rpath = vala_compiler_realpath (source);
					_tmp13 = FALSE;
					if (g_str_has_suffix (source, ".vala")) {
						_tmp13 = TRUE;
					} else {
						_tmp13 = g_str_has_suffix (source, ".gs");
					}
					if (_tmp13) {
						ValaSourceFile* source_file;
						ValaUsingDirective* _tmp15;
						ValaUnresolvedSymbol* _tmp14;
						source_file = vala_source_file_new (self->priv->context, rpath, FALSE, NULL);
						/* import the GLib namespace by default (namespace of backend-specific standard library)*/
						_tmp15 = NULL;
						_tmp14 = NULL;
						vala_source_file_add_using_directive (source_file, _tmp15 = vala_using_directive_new ((ValaSymbol*) (_tmp14 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), NULL));
						(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_code_node_unref (_tmp15), NULL));
						(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_code_node_unref (_tmp14), NULL));
						vala_code_context_add_source_file (self->priv->context, source_file);
						(source_file == NULL) ? NULL : (source_file = (vala_source_file_unref (source_file), NULL));
					} else {
						if (g_str_has_suffix (source, ".gtkaml")) {
							ValaSourceFile* _tmp16;
							_tmp16 = NULL;
							vala_code_context_add_source_file (self->priv->context, _tmp16 = vala_source_file_new (self->priv->context, rpath, FALSE, NULL));
							(_tmp16 == NULL) ? NULL : (_tmp16 = (vala_source_file_unref (_tmp16), NULL));
						} else {
							if (g_str_has_suffix (source, ".vapi")) {
								ValaSourceFile* _tmp17;
								_tmp17 = NULL;
								vala_code_context_add_source_file (self->priv->context, _tmp17 = vala_source_file_new (self->priv->context, rpath, TRUE, NULL));
								(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_source_file_unref (_tmp17), NULL));
							} else {
								if (g_str_has_suffix (source, ".c")) {
									vala_code_context_add_c_source_file (self->priv->context, rpath);
								} else {
									char* _tmp18;
									_tmp18 = NULL;
									vala_report_error (NULL, _tmp18 = g_strdup_printf ("%s is not a supported source file type. Only .vala, .vapi, .gs, and .c files are supported.", source));
									_tmp18 = (g_free (_tmp18), NULL);
								}
							}
						}
					}
					rpath = (g_free (rpath), NULL);
				} else {
					char* _tmp19;
					_tmp19 = NULL;
					vala_report_error (NULL, _tmp19 = g_strdup_printf ("%s not found", source));
					_tmp19 = (g_free (_tmp19), NULL);
				}
				source = (g_free (source), NULL);
			}
		}
	}
	_tmp21 = NULL;
	vala_compiler_sources = (_tmp21 = NULL, vala_compiler_sources = (_vala_array_free (vala_compiler_sources, _vala_array_length (vala_compiler_sources), (GDestroyNotify) g_free), NULL), _tmp21);
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	parser = gtkaml_parser_new ();
	gtkaml_parser_parse (parser, self->priv->context, vala_compiler_implicits_directories, vala_compiler_implicits_directories_length1);
	genie_parser = vala_genie_parser_new ();
	vala_genie_parser_parse (genie_parser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp23;
		return (_tmp23 = vala_compiler_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), _tmp23);
	}
	resolver = vala_symbol_resolver_new ();
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp24;
		return (_tmp24 = vala_compiler_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), _tmp24);
	}
	analyzer = vala_semantic_analyzer_new ();
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_compiler_dump_tree != NULL) {
		ValaCodeWriter* code_writer;
		code_writer = vala_code_writer_new (TRUE);
		vala_code_writer_write_file (code_writer, self->priv->context, vala_compiler_dump_tree);
		(code_writer == NULL) ? NULL : (code_writer = (vala_code_visitor_unref (code_writer), NULL));
	}
	if (vala_report_get_errors () > 0) {
		gint _tmp25;
		return (_tmp25 = vala_compiler_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), _tmp25);
	}
	flow_analyzer = vala_flow_analyzer_new ();
	vala_flow_analyzer_analyze (flow_analyzer, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp26;
		return (_tmp26 = vala_compiler_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), (flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL)), _tmp26);
	}
	if (vala_code_context_get_non_null_experimental (self->priv->context)) {
		ValaNullChecker* null_checker;
		null_checker = vala_null_checker_new ();
		vala_null_checker_check (null_checker, self->priv->context);
		if (vala_report_get_errors () > 0) {
			gint _tmp27;
			return (_tmp27 = vala_compiler_quit (self), (null_checker == NULL) ? NULL : (null_checker = (vala_code_visitor_unref (null_checker), NULL)), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), (flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL)), _tmp27);
		}
		(null_checker == NULL) ? NULL : (null_checker = (vala_code_visitor_unref (null_checker), NULL));
	}
	vala_code_generator_emit (vala_code_context_get_codegen (self->priv->context), self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp28;
		return (_tmp28 = vala_compiler_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), (flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL)), _tmp28);
	}
	if (vala_compiler_library != NULL) {
		ValaCodeWriter* interface_writer;
		char* vapi_filename;
		gboolean _tmp29;
		ValaGIRWriter* gir_writer;
		char* gir_filename;
		gboolean _tmp31;
		char* _tmp33;
		interface_writer = vala_code_writer_new (FALSE);
		vapi_filename = g_strdup_printf ("%s.vapi", vala_compiler_library);
		_tmp29 = FALSE;
		if (vala_compiler_directory != NULL) {
			_tmp29 = !g_path_is_absolute (vapi_filename);
		} else {
			_tmp29 = FALSE;
		}
		/* put .vapi file in current directory unless -d has been explicitly specified*/
		if (_tmp29) {
			char* _tmp30;
			_tmp30 = NULL;
			vapi_filename = (_tmp30 = g_strdup_printf ("%s%c%s", vala_code_context_get_directory (self->priv->context), (gint) G_DIR_SEPARATOR, vapi_filename), vapi_filename = (g_free (vapi_filename), NULL), _tmp30);
		}
		vala_code_writer_write_file (interface_writer, self->priv->context, vapi_filename);
		gir_writer = vala_gir_writer_new ();
		gir_filename = g_strdup_printf ("%s.gir", vala_compiler_library);
		_tmp31 = FALSE;
		if (vala_compiler_directory != NULL) {
			_tmp31 = !g_path_is_absolute (gir_filename);
		} else {
			_tmp31 = FALSE;
		}
		/* put .gir file in current directory unless -d has been explicitly specified*/
		if (_tmp31) {
			char* _tmp32;
			_tmp32 = NULL;
			gir_filename = (_tmp32 = g_strdup_printf ("%s%c%s", vala_code_context_get_directory (self->priv->context), (gint) G_DIR_SEPARATOR, gir_filename), gir_filename = (g_free (gir_filename), NULL), _tmp32);
		}
		vala_gir_writer_write_file (gir_writer, self->priv->context, gir_filename);
		_tmp33 = NULL;
		vala_compiler_library = (_tmp33 = NULL, vala_compiler_library = (g_free (vala_compiler_library), NULL), _tmp33);
		(interface_writer == NULL) ? NULL : (interface_writer = (vala_code_visitor_unref (interface_writer), NULL));
		vapi_filename = (g_free (vapi_filename), NULL);
		(gir_writer == NULL) ? NULL : (gir_writer = (vala_code_visitor_unref (gir_writer), NULL));
		gir_filename = (g_free (gir_filename), NULL);
	}
	if (!vala_compiler_ccode_only) {
		ValaCCodeCompiler* ccompiler;
		gboolean _tmp34;
		ccompiler = vala_ccode_compiler_new ();
		_tmp34 = FALSE;
		if (vala_compiler_cc_command == NULL) {
			_tmp34 = g_getenv ("CC") != NULL;
		} else {
			_tmp34 = FALSE;
		}
		if (_tmp34) {
			char* _tmp36;
			const char* _tmp35;
			_tmp36 = NULL;
			_tmp35 = NULL;
			vala_compiler_cc_command = (_tmp36 = (_tmp35 = g_getenv ("CC"), (_tmp35 == NULL) ? NULL : g_strdup (_tmp35)), vala_compiler_cc_command = (g_free (vala_compiler_cc_command), NULL), _tmp36);
		}
		if (vala_compiler_cc_options == NULL) {
			gint _tmp38_length1;
			char** _tmp38;
			char** _tmp37;
			_tmp38 = NULL;
			_tmp37 = NULL;
			vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, (_tmp38 = (_tmp37 = g_new0 (char*, 0 + 1), _tmp37), _tmp38_length1 = 0, _tmp38), 0);
			_tmp38 = (_vala_array_free (_tmp38, _tmp38_length1, (GDestroyNotify) g_free), NULL);
		} else {
			vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, vala_compiler_cc_options, _vala_array_length (vala_compiler_cc_options));
		}
		(ccompiler == NULL) ? NULL : (ccompiler = (vala_ccode_compiler_unref (ccompiler), NULL));
	}
	return (_tmp39 = vala_compiler_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), (flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL)), _tmp39);
}


static gboolean vala_compiler_ends_with_dir_separator (const char* s) {
	g_return_val_if_fail (s != NULL, FALSE);
	return G_IS_DIR_SEPARATOR (g_utf8_get_char (g_utf8_offset_to_pointer (s, g_utf8_strlen (s, -1) - 1)));
}


/* ported from glibc */
static char* vala_compiler_realpath (const char* name) {
	char* rpath;
	const char* start;
	const char* end;
	glong root_len;
	gboolean _tmp9;
	g_return_val_if_fail (name != NULL, NULL);
	rpath = NULL;
	/* start of path component*/
	start = NULL;
	/* end of path component*/
	end = NULL;
	if (!g_path_is_absolute (name)) {
		char* _tmp0;
		/* relative path*/
		_tmp0 = NULL;
		rpath = (_tmp0 = g_get_current_dir (), rpath = (g_free (rpath), NULL), _tmp0);
		start = end = name;
	} else {
		char* _tmp1;
		/* set start after root*/
		start = end = g_path_skip_root (name);
		/* extract root*/
		_tmp1 = NULL;
		rpath = (_tmp1 = string_substring (name, (glong) 0, g_utf8_pointer_to_offset (name, start)), rpath = (g_free (rpath), NULL), _tmp1);
	}
	root_len = g_utf8_pointer_to_offset (rpath, g_path_skip_root (rpath));
	for (; g_utf8_get_char (start) != 0; start = end) {
		glong len;
		while (G_IS_DIR_SEPARATOR (g_utf8_get_char (start))) {
			start = g_utf8_next_char (start);
		}
		/* find end of path component*/
		len = (glong) 0;
		for (end = start; TRUE; end = g_utf8_next_char (end)) {
			gboolean _tmp2;
			_tmp2 = FALSE;
			if (g_utf8_get_char (end) != 0) {
				_tmp2 = !G_IS_DIR_SEPARATOR (g_utf8_get_char (end));
			} else {
				_tmp2 = FALSE;
			}
			if (!_tmp2) {
				break;
			}
			len++;
		}
		if (len == 0) {
			break;
		} else {
			gboolean _tmp3;
			_tmp3 = FALSE;
			if (len == 1) {
				_tmp3 = g_utf8_get_char (start) == '.';
			} else {
				_tmp3 = FALSE;
			}
			if (_tmp3) {
			} else {
				gboolean _tmp4;
				_tmp4 = FALSE;
				if (len == 2) {
					_tmp4 = g_str_has_prefix (start, "..");
				} else {
					_tmp4 = FALSE;
				}
				if (_tmp4) {
					/* back up to previous component, ignore if at root already*/
					if (g_utf8_strlen (rpath, -1) > root_len) {
						do {
							char* _tmp5;
							_tmp5 = NULL;
							rpath = (_tmp5 = string_substring (rpath, (glong) 0, g_utf8_strlen (rpath, -1) - 1), rpath = (g_free (rpath), NULL), _tmp5);
						} while (!vala_compiler_ends_with_dir_separator (rpath));
					}
				} else {
					char* _tmp8;
					char* _tmp7;
					if (!vala_compiler_ends_with_dir_separator (rpath)) {
						char* _tmp6;
						_tmp6 = NULL;
						rpath = (_tmp6 = g_strconcat (rpath, (G_DIR_SEPARATOR_S), NULL), rpath = (g_free (rpath), NULL), _tmp6);
					}
					_tmp8 = NULL;
					_tmp7 = NULL;
					rpath = (_tmp8 = g_strconcat (rpath, _tmp7 = (string_substring (start, (glong) 0, len)), NULL), rpath = (g_free (rpath), NULL), _tmp8);
					_tmp7 = (g_free (_tmp7), NULL);
				}
			}
		}
	}
	_tmp9 = FALSE;
	if (g_utf8_strlen (rpath, -1) > root_len) {
		_tmp9 = vala_compiler_ends_with_dir_separator (rpath);
	} else {
		_tmp9 = FALSE;
	}
	if (_tmp9) {
		char* _tmp10;
		_tmp10 = NULL;
		rpath = (_tmp10 = string_substring (rpath, (glong) 0, g_utf8_strlen (rpath, -1) - 1), rpath = (g_free (rpath), NULL), _tmp10);
	}
	if (G_DIR_SEPARATOR != '/') {
		char** _tmp12;
		gint components_size;
		gint components_length1;
		char** _tmp11;
		char** components;
		char* _tmp13;
		/* don't use backslashes internally,
		 to avoid problems in #include directives*/
		_tmp12 = NULL;
		_tmp11 = NULL;
		components = (_tmp12 = _tmp11 = g_strsplit (rpath, "\\", 0), components_length1 = _vala_array_length (_tmp11), components_size = components_length1, _tmp12);
		_tmp13 = NULL;
		rpath = (_tmp13 = g_strjoinv ("/", components), rpath = (g_free (rpath), NULL), _tmp13);
		components = (_vala_array_free (components, components_length1, (GDestroyNotify) g_free), NULL);
	}
	return rpath;
}


static gint vala_compiler_main (char** args, int args_length1) {
	GError * inner_error;
	ValaCompiler* compiler;
	gint _tmp3;
	inner_error = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala Gtkaml Compiler");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_COMPILER_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &inner_error);
		if (inner_error != NULL) {
			(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
			if (inner_error->domain == G_OPTION_ERROR) {
				goto __catch1_g_option_error;
			}
			goto __finally1;
		}
		(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
	}
	goto __finally1;
	__catch1_g_option_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			gint _tmp0;
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			return (_tmp0 = 1, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp0);
		}
	}
	__finally1:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return 0;
	}
	if (vala_compiler_version) {
		fprintf (stdout, "Gtkaml %s (based on Vala 0.5.6)\n", PACKAGE_VERSION);
		return 0;
	}
	if (vala_compiler_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		return 1;
	}
	compiler = vala_compiler_new ();
	return (_tmp3 = vala_compiler_run (compiler), (compiler == NULL) ? NULL : (compiler = (vala_compiler_unref (compiler), NULL)), _tmp3);
}


int main (int argc, char ** argv) {
	g_type_init ();
	return vala_compiler_main (argv, argc);
}


static ValaCompiler* vala_compiler_construct (GType object_type) {
	ValaCompiler* self;
	self = (ValaCompiler*) g_type_create_instance (object_type);
	return self;
}


static ValaCompiler* vala_compiler_new (void) {
	return vala_compiler_construct (VALA_TYPE_COMPILER);
}


static void vala_value_compiler_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_compiler_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_compiler_unref (value->data[0].v_pointer);
	}
}


static void vala_value_compiler_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_compiler_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_compiler_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_compiler_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCompiler* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_compiler_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_compiler_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCompiler** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_compiler_ref (value->data[0].v_pointer);
	}
	return NULL;
}


static GParamSpec* vala_param_spec_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCompiler* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_COMPILER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


static gpointer vala_value_get_compiler (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_COMPILER), NULL);
	return value->data[0].v_pointer;
}


static void vala_value_set_compiler (GValue* value, gpointer v_object) {
	ValaCompiler* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_COMPILER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_COMPILER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_compiler_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_compiler_unref (old);
	}
}


static void vala_compiler_class_init (ValaCompilerClass * klass) {
	vala_compiler_parent_class = g_type_class_peek_parent (klass);
	VALA_COMPILER_CLASS (klass)->finalize = vala_compiler_finalize;
	g_type_class_add_private (klass, sizeof (ValaCompilerPrivate));
}


static void vala_compiler_instance_init (ValaCompiler * self) {
	self->priv = VALA_COMPILER_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_compiler_finalize (ValaCompiler* obj) {
	ValaCompiler * self;
	self = VALA_COMPILER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
}


static GType vala_compiler_get_type (void) {
	static GType vala_compiler_type_id = 0;
	if (vala_compiler_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_compiler_init, vala_value_compiler_free_value, vala_value_compiler_copy_value, vala_value_compiler_peek_pointer, "p", vala_value_compiler_collect_value, "p", vala_value_compiler_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCompiler), 0, (GInstanceInitFunc) vala_compiler_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_compiler_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCompiler", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_compiler_type_id;
}


static gpointer vala_compiler_ref (gpointer instance) {
	ValaCompiler* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


static void vala_compiler_unref (gpointer instance) {
	ValaCompiler* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_COMPILER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




