/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

/* menu_functions.c : define the functions associated to menu items. */

#include <config.h>
#include <glib.h>
#include <gnome.h>
#include <glade/glade.h>

#include "thumb-cache.h"
#include "catalog-list.h"
#include "clist-utils.h"
#include "comments.h"
#include "dir-list.h"
#include "dlg-bookmarks.h"
#include "dlg-png-exporter.h"
#include "dlg-file-utils.h"
#include "dlg-preferences.h"
#include "file-list.h"
#include "file-utils.h"
#include "fullscreen.h"
#include "main.h"
#include "menu-callbacks.h"
#include "pixbuf-utils.h"
#include "image-viewer.h"
#include "image-list.h"
#include "image-list-utils.h"


typedef enum {
	WALLPAPER_ALIGN_TILED     = 0,
	WALLPAPER_ALIGN_CENTERED  = 1,
	WALLPAPER_ALIGN_STRETCHED = 2,
	WALLPAPER_ALIGN_SCALED    = 3
} WallpaperAlign;


void
new_window_cb (GtkWidget *widget, 
	       void *data)
{
	GThumbWindow *window;

	window = window_new (FALSE);
	gtk_widget_show_all (window->app);
}


void
close_cb (GtkWidget *widget, 
	  void *data)
{
	g_return_if_fail (data != NULL);
		
	window_close ((GThumbWindow*)data);

	if (window_list == NULL)
		gtk_main_quit ();
}


void
quit_cb (GtkWidget *widget, 
	 void *data)
{
	GList *scan;

	for (scan = window_list; scan; scan = window_list)
		close_cb (widget, scan->data);
}


void
delete_history_cb (GtkWidget *widget, 
		   void *data)
{
	Bookmarks * history;

	history = bookmarks_new (RC_HISTORY_FILE);
	bookmarks_write_to_disk (history);
	bookmarks_free (history);

	all_windows_update_history_list ();
}


void
export_as_image_cb (GtkWidget *widget, 
		    void *data)
{
	GThumbWindow *window = data;
	dlg_exporter (window);
}


/* edit menu callbacks. */


void
rotate_cb (GtkWidget *widget, 
	   void *data)
{
	GThumbWindow *window = data;
	image_viewer_alter (IMAGE_VIEWER (window->viewer), ALTER_ROTATE_90);
}


void
rotate_180_cb (GtkWidget *widget, 
	       void *data)
{
	GThumbWindow *window = data;
	image_viewer_alter (IMAGE_VIEWER (window->viewer), ALTER_ROTATE_180);
}


void
flip_cb (GtkWidget *widget, 
	 void *data)
{
	GThumbWindow *window = data;	
	image_viewer_alter (IMAGE_VIEWER (window->viewer), ALTER_FLIP);
}


void
mirror_cb (GtkWidget *widget, 
	   void *data)
{
	GThumbWindow *window = data;
	image_viewer_alter (IMAGE_VIEWER (window->viewer), ALTER_MIRROR);
}


void
rename_cb (GtkWidget *widget,
	   void *data)
{
	GThumbWindow *window;
	gint pos;
	ImageList *ilist;

	window = data;
	ilist = IMAGE_LIST (window->file_list->ilist);
	pos = GPOINTER_TO_INT (ilist->selection->data);
	image_list_start_editing (ilist, pos);
}


void
delete_cb (GtkWidget *widget,
	   void *data)
{
	GThumbWindow *window = data;
	GList *list;
	ImageList *ilist;

	window = data;
	ilist = IMAGE_LIST (window->file_list->ilist);

	list = ilist_utils_get_file_list_selection (ilist);
	dlg_file_delete (window, list);

	/* the list is dealloated when the dialog is closed. */
}


void
copy_cb (GtkWidget *widget,
	 void *data)
{
	GThumbWindow *window = data;
	GList *list;
	ImageList *ilist;

	window = data;
	ilist = IMAGE_LIST (window->file_list->ilist);

	list = ilist_utils_get_file_list_selection (ilist);
	dlg_file_copy (window, list);

	/* the list is deallocated when the dialog is closed. */
}


void
move_cb (GtkWidget *widget,
	 void *data)
{
	GThumbWindow *window = data;
	GList *list;
	ImageList *ilist;

	window = data;
	ilist = IMAGE_LIST (window->file_list->ilist);

	list = ilist_utils_get_file_list_selection (ilist);
	dlg_file_move (window, list);

	/* the list is deallocated when the dialog is closed. */
}


void
select_all_cb (GtkWidget *widget, 
	       void *data)
{
	GThumbWindow *window = data;

	image_list_select_all (IMAGE_LIST (window->file_list->ilist));
	window_update_menu_sensitivity (window);
}


void
delete_comment_cb (GtkWidget *widget, 
		   void *data)
{
	GThumbWindow *window = data;
	ImageList *ilist;
	GList *list, *scan;

	ilist = IMAGE_LIST (window->file_list->ilist);
	list = ilist_utils_get_file_list_selection (ilist);
	for (scan = list; scan; scan = scan->next) {
		comment_delete (scan->data);
		all_windows_notify_update_comment (scan->data);
	}
	path_list_free (list);
}



/* view menu callbacks. */


void
fullscreen_cb (GtkWidget *widget, 
	       void *data)
{
	GThumbWindow *window = data;

	if (!window->fullscreen)
		fullscreen_start (fullscreen, window);
	else
		fullscreen_stop (fullscreen);
}


void
browser_cb (GtkWidget *widget, 
	    void *data)
{
	GThumbWindow *window = data;

	if (window->sidebar_visible)
		window_hide_sidebar (window);
	else
		window_show_sidebar (window);
}


void
dir_browser_cb (GtkWidget *widget,
		void *data)
{
	GThumbWindow *window = data;

	if ((window->sidebar_visible) && (window->sidebar_content == DIR_LIST))
		window_hide_sidebar (window);
	else
		window_set_sidebar_content (window, DIR_LIST);
}


void
catalog_browser_cb (GtkWidget *widget,
		    void *data)
{
	GThumbWindow *window = data;

	if ((window->sidebar_visible) && (window->sidebar_content == CATALOG_LIST))
		window_hide_sidebar (window);
	else
		window_set_sidebar_content (window, CATALOG_LIST);
}


void
toggle_animation_cb (GtkWidget *widget, 
		    void *data)
{
	GThumbWindow *window = data;
	ImageViewer *viewer = IMAGE_VIEWER (window->viewer);

	if (! viewer->is_animation)
		return;

	if (! viewer->running_animation)
		image_viewer_start_animation (viewer);
	else
		image_viewer_stop_animation (viewer);
}


void
step_animation_cb (GtkWidget *widget, 
		   void *data)
{
	GThumbWindow *window = data;
	ImageViewer *viewer = IMAGE_VIEWER (window->viewer);

	if (! viewer->is_animation)
		return;

	image_viewer_step_animation (viewer);
}


void
refresh_cb (GtkWidget *widget, 
	    void *data)
{
	GThumbWindow *window = data;
	window_refresh (window);
}


void
sort_by_name_cb (GtkWidget *widget, 
		 void *data)
{
	GThumbWindow *window = data;
	GtkCheckMenuItem *mitem = GTK_CHECK_MENU_ITEM (widget);
	
	if (! mitem->active)
		return;
	file_list_set_sort_method (window->file_list, SORT_BY_NAME);
}


void
sort_by_size_cb (GtkWidget *widget, 
		 void *data)
{
	GThumbWindow *window = data;
	GtkCheckMenuItem *mitem = GTK_CHECK_MENU_ITEM (widget);
	
	if (! mitem->active)
		return;
	file_list_set_sort_method (window->file_list, SORT_BY_SIZE);
}


void
sort_by_time_cb (GtkWidget *widget, 
		 void *data)
{
	GThumbWindow *window = data;
	GtkCheckMenuItem *mitem = GTK_CHECK_MENU_ITEM (widget);
	
	if (! mitem->active)
		return;
	file_list_set_sort_method (window->file_list, SORT_BY_TIME);
}


void
sort_by_path_cb (GtkWidget *widget, 
		 void *data)
{
	GThumbWindow *window = data;
	GtkCheckMenuItem *mitem = GTK_CHECK_MENU_ITEM (widget);
	
	if (! mitem->active)
		return;
	file_list_set_sort_method (window->file_list, SORT_BY_PATH);
}


void
reversed_order_cb (GtkWidget *widget, 
		   void *data)
{
	GThumbWindow *window = data;
	GtkSortType new_type;
	
	if (window->file_list->sort_type == GTK_SORT_ASCENDING)
		new_type = GTK_SORT_DESCENDING;
	else
		new_type = GTK_SORT_ASCENDING;

	file_list_set_sort_type (window->file_list, new_type);
}


void
slideshow_cb (GtkWidget *widget, 
	      void *data)
{
	GThumbWindow *window = data;
	
	if (!window->slideshow)
		window_start_slideshow (window);
	else
		window_stop_slideshow (window);
}


void
thumbnails_cb (GtkWidget *widget, 
	       void *data)
{
	GThumbWindow *window = data;

	file_list_enable_thumbs (window->file_list, ! (window->file_list->enable_thumbs));
}


void
trans_white_cb (GtkWidget *widget, 
		void *data)
{
	GThumbWindow *window = data;

	if (! GTK_CHECK_MENU_ITEM (widget)->active) 
		return;

	image_viewer_set_transp_type (IMAGE_VIEWER (window->viewer), 
				      TRANSP_TYPE_WHITE);
	image_viewer_update_view (IMAGE_VIEWER (window->viewer));
}


void
trans_gray_cb (GtkWidget *widget, 
	       void *data)
{
	GThumbWindow *window = data;

	if (! GTK_CHECK_MENU_ITEM (widget)->active) 
		return;

	image_viewer_set_transp_type (IMAGE_VIEWER (window->viewer), 
				      TRANSP_TYPE_GRAY);
	image_viewer_update_view (IMAGE_VIEWER (window->viewer));
}


void
trans_black_cb (GtkWidget *widget, 
		void *data)
{
	GThumbWindow *window = data;

	if (! GTK_CHECK_MENU_ITEM (widget)->active) 
		return;

	image_viewer_set_transp_type (IMAGE_VIEWER (window->viewer), 
				      TRANSP_TYPE_BLACK);
	image_viewer_update_view (IMAGE_VIEWER (window->viewer));
}


void
trans_checked_cb (GtkWidget *widget, 
		  void *data)
{
	GThumbWindow *window = data;

	if (! GTK_CHECK_MENU_ITEM (widget)->active) 
		return;

	image_viewer_set_transp_type (IMAGE_VIEWER (window->viewer),
				      TRANSP_TYPE_CHECKED);
	image_viewer_update_view (IMAGE_VIEWER (window->viewer));
}


/* Tools menu. */


void
remove_old_thumbnails_cb (GtkWidget *widget, 
			  void *data)
{
	cache_remove_old_previews_async (NULL, TRUE, FALSE);
}


void
remove_old_comments_cb (GtkWidget *widget, 
			void *data)
{
	comments_remove_old_comments_async (NULL, TRUE, FALSE);
}


static void
set_as_wallpaper (void *data,
		  WallpaperAlign align)
{
	GThumbWindow *window = data;
	gchar *command;
	gint i, n;
	gchar *new_wallpaper, *path, *e_path;
	gchar *wp_name, *wp_config_name;
	gboolean found;

	if ((window->image_path == NULL) || !path_is_file (window->image_path))
		return;

	path = g_strdup (window->image_path);

	/* try to find the image in the list of wallpapers. */
	found = FALSE;
	n = gnome_config_get_int ("/Background/Default/wallpapers=0");

	for (i = 1; (i <= n) && !found; i++) {
		wp_config_name = g_strdup_printf ("/Background/Default/wallpaper%d", i);
		if (wp_config_name == NULL)
			continue;

		wp_name = gnome_config_get_string (wp_config_name);
		if (wp_name == NULL)
			continue;

		found = strcmp (wp_name, path) == 0;

		g_free (wp_name);
		g_free (wp_config_name);
	}

	if (found) 
		gnome_config_set_string (
			"/Background/Default/wallpaper", path);
	else {
		/* add the image to wallpapers list. */
		
		n++;
		new_wallpaper = g_strdup_printf (
			"/Background/Default/wallpaper%d", n);
		gnome_config_set_string (
			"/Background/Default/wallpaper", path);
		gnome_config_set_int    ("/Background/Default/wallpapers", n);
		gnome_config_set_string (new_wallpaper, path);
		gnome_config_set_string (
			"/Background/Default/wallpapers_dir", path);
	
		g_free (new_wallpaper);
	}

	/* Alignment type. */
	gnome_config_set_int ("/Background/Default/wallpaperAlign", align);

	gnome_config_sync ();

	/* execute asynchronously */
	e_path = shell_escape (path);
	command = g_strconcat ("background-properties-capplet ",
			       "--init-session-settings --ignore "
			       "--background-image=",
			       e_path,
			       NULL);
	gnome_execute_shell (g_get_home_dir (), command);
	g_free (command); 
	g_free (e_path);
	g_free (path);
}


void
set_as_wallpaper_centered_cb (GtkWidget *widget, 
			      void *data)
{
	set_as_wallpaper (data, WALLPAPER_ALIGN_CENTERED);
}


void
set_as_wallpaper_tiled_cb (GtkWidget *widget, 
			   void *data)
{
	set_as_wallpaper (data, WALLPAPER_ALIGN_TILED);
}


void
set_as_wallpaper_scaled_cb (GtkWidget *widget, 
			    void *data)
{
	set_as_wallpaper (data, WALLPAPER_ALIGN_SCALED);
}


void
set_as_wallpaper_stretched_cb (GtkWidget *widget, 
			       void *data)
{
	set_as_wallpaper (data, WALLPAPER_ALIGN_STRETCHED);
}


void
restore_wallpaper_cb  (GtkWidget *widget, 
		       void *data)
{
	gchar *command;

	gnome_config_set_string ("/Background/Default/wallpaper", 
				 preferences.wallpaper);
	gnome_config_set_string ("/Background/Default/wallpapers_dir", 
				 preferences.wallpapers_dir);
	gnome_config_set_int ("/Background/Default/wallpaperAlign", 
				 preferences.wallpaperAlign);	
	gnome_config_sync ();

	/* execute asynchronously */
	command = g_strconcat ("background-properties-capplet ",
			       "--init-session-settings --ignore "
			       "--background-image=",
			       preferences.wallpaper,
			       NULL);
	gnome_execute_shell (g_get_home_dir (), command);
	g_free (command); 

}


void
preferences_cb (GtkWidget *widget, 
		void *data)
{
	GThumbWindow *window = data;

	dlg_preferences (window);
}


/* bookmarks menu. */

void
add_bookmark_cb (GtkWidget *widget, 
		 void *data)
{
	GThumbWindow *window = data;
	gchar *path = NULL;

	if (window->sidebar_content == CATALOG_LIST) {
	    	gint row;
		gchar *prefix, *catalog_path;
		
		if (GTK_CLIST (window->catalog_list->clist)->selection == NULL)
			return;

		row = GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data);
		if (catalog_list_is_search (window->catalog_list, row))
			prefix = g_strdup (SEARCH_PREFIX);
		else
			prefix = g_strdup (CATALOG_PREFIX);
		catalog_path = catalog_list_path_from_row (window->catalog_list, row);

		path = g_strconcat (prefix,
				    catalog_path,
				    NULL);
		g_free (catalog_path);
		g_free (prefix);
	} else {
		if (window->dir_list->path == NULL)
			return;

		path = g_strconcat (FILE_PREFIX,
				    window->dir_list->path,
				    NULL);
	}

	bookmarks_add (preferences.bookmarks, path);
	bookmarks_write_to_disk (preferences.bookmarks);
	all_windows_update_bookmark_list ();

	g_free (path);
}


void
edit_bookmarks_cb (GtkWidget *widget, 
		   void *data)
{
	GThumbWindow *window = data;

	dlg_edit_bookmarks (window);
}


/* help menu callbacks. */


void
about_cb (GtkWidget *widget, 
	  void *data)
{
	GtkWidget   *about;
	const gchar *authors[] = {
		"Paolo Bacchilega",
		NULL
	};

	about = gnome_about_new ("gThumb", VERSION,
				 "Copyright (C) 2001 The Free Software Foundation, Inc.",
				 authors,
				 _("An image viewer and browser for GNOME."),
				 NULL);

	gtk_widget_show_all (about);
}


/* toolbar specific callbacks. */


void
go_up_cb (GtkWidget *widget,
	  void *data)
{
	GThumbWindow *window = data;	
	gchar *up_dir;

	if (window->dir_list->path == NULL)
		return;

	up_dir = remove_level_from_path (window->dir_list->path);
	window_go_to_directory (window, up_dir);
	g_free (up_dir);
}


void
go_home_cb (GtkWidget *widget,
	    void *data)
{
	GThumbWindow *window = data;	
	window_go_to_directory (window, g_get_home_dir ());
}


void
zoom_in_cb (GtkWidget *widget,
	    void *data)
{
	GThumbWindow *window = data;

	image_viewer_zoom_in (IMAGE_VIEWER (window->viewer));
}


void   
zoom_out_cb (GtkWidget *widget,
	     void *data)
{
	GThumbWindow *window = data;

	image_viewer_zoom_out (IMAGE_VIEWER (window->viewer));
}


void
zoom_to_fit_cb (GtkWidget *widget,
		void *data)
{
	GThumbWindow *window = data;

	image_viewer_zoom_to_fit (IMAGE_VIEWER (window->viewer));
}


void   
no_zoom_cb (GtkWidget *widget,
	    void *data)
{
	GThumbWindow *window = data;

	image_viewer_set_zoom (IMAGE_VIEWER (window->viewer), 1.0);
}


void
high_quality_zoom_cb (GtkWidget *widget,
		      void *data)
{
	GThumbWindow *window = data;

	if (! GTK_CHECK_MENU_ITEM (widget)->active) 
		return;

	preferences.zoom_quality = ZOOM_QUALITY_HIGH;
	image_viewer_set_zoom_quality (IMAGE_VIEWER (window->viewer),
				       ZOOM_QUALITY_HIGH);
	image_viewer_update_view (IMAGE_VIEWER (window->viewer));
}


void
low_quality_zoom_cb (GtkWidget *widget,
		     void *data)
{
	GThumbWindow *window = data;

	if (! GTK_CHECK_MENU_ITEM (widget)->active) 
		return;

	preferences.zoom_quality = ZOOM_QUALITY_LOW;
	image_viewer_set_zoom_quality (IMAGE_VIEWER (window->viewer),
				       ZOOM_QUALITY_LOW);
	image_viewer_update_view (IMAGE_VIEWER (window->viewer));
}
