/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-pixmap.h>
#include <glade/glade.h>
#include <gtk/gtk.h>
#include "thumb-cache.h"
#include "comments.h"
#include "dlg-file-utils.h"
#include "file-utils.h"
#include "image-viewer.h"
#include "main.h"
#include "window.h"

#include "icons/toolbar_zoom_in.xpm"
#include "icons/toolbar_zoom_out.xpm"

enum {
	OVERWRITE_YES,
	OVERWRITE_NO,
	OVERWRITE_ALL,
	OVERWRITE_NONE
};


static int dlg_overwrite (GThumbWindow *window,
			  gint default_overwrite_mode,
			  gchar *old_filename,
			  gchar *new_filename,
			  gboolean show_overwrite_all_none); 


static void
dlg_show_error (gchar *title, GList *error_list)
{
	GtkWidget *gdialog;
	char *msg;
	const int max_files = 10;
	int n = 0;
	GList *scan;
	
	msg = g_strdup_printf ("%s\n\n", title);
	for (scan = error_list; scan && (n < max_files); scan = scan->next) {
		gchar *tmp = msg;

		msg = g_strconcat (tmp, scan->data, "\n", NULL);
		g_free (tmp);
		n++;
	}
	
	if (scan != NULL) {
		gchar *tmp = msg;
		msg = g_strconcat (tmp, "...\n", NULL);
		g_free (tmp);
	}

	gdialog = gnome_error_dialog (msg);
	gnome_dialog_run_and_close (GNOME_DIALOG (gdialog));
}


/* -- delete -- */


static void
reply_cb (gint reply, gpointer data)
{
	GList *list = (GList*) data;
	GList *scan;
	GList *error_list = NULL;
	gboolean error = FALSE;

	if (reply != 0)
		return;

	scan = list;
	while (scan) {
		if (unlink (scan->data) == 0) {
			cache_delete (scan->data);
			comment_delete (scan->data);
			scan = scan->next;
		} else {
			list = g_list_remove_link (list, scan);
			error_list = g_list_prepend (error_list, scan->data);
			g_list_free (scan);
			scan = list;
			error = TRUE;
		}
	}

	all_windows_notify_files_deleted (list);

	if (error) 
		dlg_show_error (_("Could not delete the following files:"),
				error_list);
	path_list_free (list);
	path_list_free (error_list);
}


void 
dlg_file_delete (GThumbWindow *window, 
		 GList *list)
{
	GtkWidget *dialog;
	gint result;

	if (! preferences.confirm_file_del) {
		reply_cb (0, list);
		return;
	}

	dialog = gnome_question_dialog_parented (_("Selected files will be deleted, are you sure ?"),
						 reply_cb,
						 list,
						 GTK_WINDOW (window->app));
	gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);

	result = gnome_dialog_run  (GNOME_DIALOG (dialog));
}


/* -- move -- */


static void
destroy_cb (GtkWidget *w,
	    GtkWidget *file_sel)
{
	GList *file_list;
	
	file_list = gtk_object_get_data (GTK_OBJECT (file_sel), "list");
	path_list_free (file_list);
}


static void 
file_move_ok_cb (GtkWidget *w,
		 GtkWidget *file_sel)
{
	gchar *path;
	GList *file_list, *scan;
	GList *error_list = NULL;
	GThumbWindow *window;
	gint len, overwrite_result;
	gboolean file_exists, show_ow_all_none;
	gboolean error = FALSE;
	gchar *new_name;

	window = gtk_object_get_data (GTK_OBJECT (file_sel), "gthumb_window");
	file_list = gtk_object_get_data (GTK_OBJECT (file_sel), "list");
	path = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_sel));

	if (path == NULL)
		return;

	/* ignore ending slash. */
	len = strlen (path);
	if (path[len - 1] == '/')
		path[len - 1] = 0;

	if (! path_is_dir (path)) 
		return;

	show_ow_all_none = g_list_length (file_list) > 1;
	overwrite_result = OVERWRITE_NO;
	for (scan = file_list; scan;) {
		new_name = g_strconcat (path,
					"/",
					file_name_from_path (scan->data),
					NULL);

		/* handle overwrite. */

		file_exists = path_is_file (new_name);

		if ((overwrite_result != OVERWRITE_ALL)
		    && (overwrite_result != OVERWRITE_NONE)
		    && file_exists)
			overwrite_result = dlg_overwrite (window, 
							  overwrite_result,
							  scan->data,
							  new_name,
							  show_ow_all_none);

		if (file_exists 
		    && ((overwrite_result == OVERWRITE_NONE)
			|| (overwrite_result == OVERWRITE_NO))) {
			/* file will not be deleted, delete the file from 
			 * the list. */
			GList *next = scan->next;
			
			file_list = g_list_remove_link (file_list, scan);
			g_free (scan->data);
			g_list_free (scan);
			scan = next;

			g_free (new_name);
			continue;
		} 
		
		if (file_move ((gchar*) scan->data, new_name)) {
			cache_move ((gchar*) scan->data, new_name);
			comment_move ((gchar*) scan->data, new_name);
			scan = scan->next;
		} else {
			file_list = g_list_remove_link (file_list, scan);
			error_list = g_list_prepend (error_list, scan->data);
			g_list_free (scan);
			scan = file_list;
			error = TRUE;
		}
		g_free (new_name);
	}

	all_windows_notify_files_deleted (file_list);
	all_windows_notify_update_directory (path);

	if (error) 
		dlg_show_error (_("Could not move the following files:"),
				error_list);
	path_list_free (error_list);

	/* file_list can change so re-set the data. */
	gtk_object_set_data (GTK_OBJECT (file_sel), "list", file_list);
	gtk_widget_destroy (file_sel);
}


void 
dlg_file_move (GThumbWindow *window, 
	       GList *list)
{
	gchar *path;
	GtkWidget *file_sel;

	file_sel = gtk_file_selection_new (_("Choose the destination directory"));
	
	if (window->dir_list->path != NULL)
		path = g_strconcat (window->dir_list->path,
				    "/",
				    NULL);
	else
		path = g_strconcat (g_get_home_dir (),
				    "/",
				    NULL);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_sel), path);
	g_free (path);

	gtk_object_set_data (GTK_OBJECT (file_sel), "list", list);
	gtk_object_set_data (GTK_OBJECT (file_sel), "gthumb_window", window);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->ok_button),
			    "clicked", 
			    (GtkSignalFunc) file_move_ok_cb, 
			    file_sel);

	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->cancel_button),
			    "clicked", 
			    (GtkSignalFunc) gtk_widget_destroy,
			    GTK_OBJECT (file_sel));

	gtk_signal_connect (GTK_OBJECT (file_sel),
			    "destroy", 
			    (GtkSignalFunc) destroy_cb,
			    file_sel);
    
	gtk_window_set_transient_for (GTK_WINDOW (file_sel), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (file_sel),TRUE);

	gtk_widget_hide (GTK_WIDGET (GTK_FILE_SELECTION (file_sel)->file_list)->parent);
	gtk_widget_show (file_sel);
}


/* -- copy -- */


static void 
file_copy_ok_cb (GtkWidget *w,
		 GtkWidget *file_sel)
{
	gchar *path;
	GList *file_list, *scan;
	GList *error_list = NULL;
	gint len, overwrite_result;
	gchar *new_name;
	gboolean file_exists, show_ow_all_none;
	gboolean error = FALSE;
	GThumbWindow *window;

	window = gtk_object_get_data (GTK_OBJECT (file_sel), "gthumb_window");
	file_list = gtk_object_get_data (GTK_OBJECT (file_sel), "list");
	path = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_sel));

	/* ignore ending slash. */
	len = strlen (path);
	if (path[len - 1] == '/')
		path[len - 1] = 0;

	show_ow_all_none = g_list_length (file_list) > 1;
	overwrite_result = OVERWRITE_NO;
	for (scan = file_list; scan; ) {
		new_name = g_strconcat (path,
					"/",
					file_name_from_path (scan->data),
					NULL);

		/* handle overwrite. */
		file_exists = path_is_file (new_name);

		if ((overwrite_result != OVERWRITE_ALL)
		    && (overwrite_result != OVERWRITE_NONE)
		    && file_exists)
			overwrite_result = dlg_overwrite (window, 
							  overwrite_result,
							  scan->data,
							  new_name,
							  show_ow_all_none);

		if (file_exists
		    && ((overwrite_result == OVERWRITE_NONE)
			|| (overwrite_result == OVERWRITE_NO))) {
			scan = scan->next;
			g_free (new_name);
			continue;
		}

		if (file_copy ((gchar*) scan->data, new_name)) {
			cache_copy ((gchar*) scan->data, new_name);
			comment_copy ((gchar*) scan->data, new_name);
			scan = scan->next;
		} else {
			file_list = g_list_remove_link (file_list, scan);
			error_list = g_list_prepend (error_list, scan->data);
			g_list_free (scan);
			scan = file_list;
			error = TRUE;
		}

		g_free (new_name);
	}

	if (file_list != NULL)
		all_windows_notify_update_directory (path);

	if (error) 
		dlg_show_error (_("Could not copy the following files:"),
				error_list);
	path_list_free (error_list);

	/* file_list can change so re-set the data. */
	gtk_object_set_data (GTK_OBJECT (file_sel), "list", file_list);
	gtk_widget_destroy (file_sel);
}


void 
dlg_file_copy (GThumbWindow *window,
	       GList  *list)
{
	GtkWidget *file_sel;
	gchar *path;

	file_sel = gtk_file_selection_new (_("Choose the destination directory"));

	if (window->dir_list->path != NULL)
		path = g_strconcat (window->dir_list->path,
				    "/",
				    NULL);
	else
		path = g_strconcat (g_get_home_dir (),
				    "/",
				    NULL);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_sel), path);
	g_free (path);

	gtk_object_set_data (GTK_OBJECT (file_sel), "list", list);
	gtk_object_set_data (GTK_OBJECT (file_sel), "gthumb_window", window);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->ok_button),
			    "clicked", 
			    (GtkSignalFunc) file_copy_ok_cb, 
			    file_sel);

	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->cancel_button),
			    "clicked", 
			    (GtkSignalFunc) gtk_widget_destroy,
			    GTK_OBJECT (file_sel));


	gtk_signal_connect (GTK_OBJECT (file_sel),
			    "destroy", 
			    (GtkSignalFunc) destroy_cb,
			    file_sel);
    
	gtk_window_set_transient_for (GTK_WINDOW (file_sel), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (file_sel),TRUE);

	gtk_widget_hide (GTK_WIDGET (GTK_FILE_SELECTION (file_sel)->file_list)->parent);
	gtk_widget_show (file_sel);
}


/* -- overwrite dialog -- */

#define MAX_LABEL_LENGTH 50


static gchar *
get_filename_label (gchar *filename)
{
	gchar *filename_lbl, *tmp_lbl;
	gint l;

	l = strlen (filename);
	if (l > MAX_LABEL_LENGTH) {
		tmp_lbl = g_strndup (filename + l - MAX_LABEL_LENGTH,
				     MAX_LABEL_LENGTH);
		filename_lbl = g_strconcat ("...",
					    tmp_lbl,
					    NULL);
		g_free (tmp_lbl);
	} else
		filename_lbl = g_strndup (filename, MAX_LABEL_LENGTH);

	return filename_lbl;
}


#define PREVIEW_SIZE 180

static int
dlg_overwrite (GThumbWindow *window,
	       gint default_overwrite_mode,
	       gchar *old_filename, 
	       gchar *new_filename,
	       gboolean show_overwrite_all_none)
{
	GladeXML *gui;
	GtkWidget *dialog;
	GtkWidget *overwrite_filename_label;
	GtkWidget *old_image_frame;
	GtkWidget *old_img_zoom_in_button;
	GtkWidget *old_img_zoom_out_button;
	GtkWidget *new_image_frame;
	GtkWidget *new_img_zoom_in_button;
	GtkWidget *new_img_zoom_out_button;
	GtkWidget *overwrite_yes_radiobutton;
	GtkWidget *overwrite_no_radiobutton;
	GtkWidget *overwrite_all_radiobutton;
	GtkWidget *overwrite_none_radiobutton;
	gchar *filename_lbl;
	gint result;
	GtkWidget *old_image_viewer, *new_image_viewer;
	GtkWidget *overwrite_radiobutton;

	gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_FILE, NULL);

	if (!gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return OVERWRITE_NO;
        }

	/* Get the widgets. */

	dialog = glade_xml_get_widget (gui, "overwrite_dialog");
	overwrite_filename_label = glade_xml_get_widget (gui, "overwrite_filename_label");
	old_image_frame = glade_xml_get_widget (gui, "old_image_frame");
	old_img_zoom_in_button = glade_xml_get_widget (gui, "old_img_zoom_in_button");
	old_img_zoom_out_button = glade_xml_get_widget (gui, "old_img_zoom_out_button");
	new_image_frame = glade_xml_get_widget (gui, "new_image_frame");
	new_img_zoom_in_button = glade_xml_get_widget (gui, "new_img_zoom_in_button");
	new_img_zoom_out_button = glade_xml_get_widget (gui, "new_img_zoom_out_button");
	overwrite_yes_radiobutton = glade_xml_get_widget (gui, "overwrite_yes_radiobutton");
	overwrite_no_radiobutton = glade_xml_get_widget (gui, "overwrite_no_radiobutton");
	overwrite_all_radiobutton = glade_xml_get_widget (gui, "overwrite_all_radiobutton");
	overwrite_none_radiobutton = glade_xml_get_widget (gui, "overwrite_none_radiobutton");

	/* Set widgets data. */

	gtk_container_add (GTK_CONTAINER (old_img_zoom_in_button),
			   gnome_pixmap_new_from_xpm_d (toolbar_zoom_in_xpm));
	gtk_widget_show_all (old_img_zoom_in_button);
	gtk_container_add (GTK_CONTAINER (old_img_zoom_out_button),
			   gnome_pixmap_new_from_xpm_d (toolbar_zoom_out_xpm));
	gtk_widget_show_all (old_img_zoom_out_button);

	gtk_container_add (GTK_CONTAINER (new_img_zoom_in_button),
			   gnome_pixmap_new_from_xpm_d (toolbar_zoom_in_xpm));
	gtk_widget_show_all (new_img_zoom_in_button);
	gtk_container_add (GTK_CONTAINER (new_img_zoom_out_button),
			   gnome_pixmap_new_from_xpm_d (toolbar_zoom_out_xpm));
	gtk_widget_show_all (new_img_zoom_out_button);

	/* * set filename labels. */

	filename_lbl = get_filename_label (new_filename);
	gtk_label_set (GTK_LABEL (overwrite_filename_label), filename_lbl);
	if (filename_lbl)
		g_free (filename_lbl);

	/* * set the default overwrite mode. */

	overwrite_radiobutton = overwrite_no_radiobutton;
	switch (default_overwrite_mode) {
	case OVERWRITE_YES:
		overwrite_radiobutton = overwrite_yes_radiobutton;
		break;

	case OVERWRITE_NO:
		overwrite_radiobutton = overwrite_no_radiobutton;
		break;

	case OVERWRITE_ALL:
		overwrite_radiobutton = overwrite_all_radiobutton;
		break;

	case OVERWRITE_NONE:
		overwrite_radiobutton = overwrite_none_radiobutton;
		break;
	}
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (overwrite_radiobutton), TRUE);

	if (!show_overwrite_all_none) {
		gtk_widget_hide (overwrite_all_radiobutton);
		gtk_widget_hide (overwrite_none_radiobutton);
	}

	/* * load images. */

	old_image_viewer = image_viewer_new ();
	gtk_container_add (GTK_CONTAINER (old_image_frame), old_image_viewer);
	image_viewer_size (IMAGE_VIEWER (old_image_viewer), 
			   PREVIEW_SIZE, PREVIEW_SIZE);
	image_viewer_set_zoom_quality (IMAGE_VIEWER (old_image_viewer),
				       ZOOM_QUALITY_LOW);
	gtk_widget_show (old_image_viewer);
	image_viewer_load_image (IMAGE_VIEWER (old_image_viewer), 
				 old_filename);

	new_image_viewer = image_viewer_new ();
	gtk_container_add (GTK_CONTAINER (new_image_frame), new_image_viewer);
	image_viewer_size (IMAGE_VIEWER (new_image_viewer), 
			   PREVIEW_SIZE, PREVIEW_SIZE);
	image_viewer_set_zoom_quality (IMAGE_VIEWER (new_image_viewer),
				       ZOOM_QUALITY_LOW);
	gtk_widget_show (new_image_viewer);
	image_viewer_load_image (IMAGE_VIEWER (new_image_viewer), 
				 new_filename);

	/* signals. */

	gtk_signal_connect_object (GTK_OBJECT (old_img_zoom_in_button),
				   "clicked",
				   (GtkSignalFunc) image_viewer_zoom_in,
				   GTK_OBJECT (old_image_viewer));
	gtk_signal_connect_object (GTK_OBJECT (old_img_zoom_out_button),
				   "clicked",
				   (GtkSignalFunc) image_viewer_zoom_out,
				   GTK_OBJECT (old_image_viewer));

	gtk_signal_connect_object (GTK_OBJECT (new_img_zoom_in_button),
				   "clicked",
				   (GtkSignalFunc) image_viewer_zoom_in,
				   GTK_OBJECT (new_image_viewer));
	gtk_signal_connect_object (GTK_OBJECT (new_img_zoom_out_button),
				   "clicked",
				   (GtkSignalFunc) image_viewer_zoom_out,
				   GTK_OBJECT (new_image_viewer));

	/* run dialog. */

	gtk_window_set_transient_for (GTK_WINDOW (dialog), 
				      GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	result = gnome_dialog_run (GNOME_DIALOG (dialog));

	if (result == -1)
		result = OVERWRITE_NO;
	else if (GTK_TOGGLE_BUTTON (overwrite_yes_radiobutton)->active)
		result = OVERWRITE_YES;
	else if (GTK_TOGGLE_BUTTON (overwrite_no_radiobutton)->active)
		result = OVERWRITE_NO;
	else if (GTK_TOGGLE_BUTTON (overwrite_all_radiobutton)->active)
		result = OVERWRITE_ALL;
	else if (GTK_TOGGLE_BUTTON (overwrite_none_radiobutton)->active)
		result = OVERWRITE_NONE;
	else
		result = OVERWRITE_NO;

	gnome_dialog_close (GNOME_DIALOG (dialog));
	gtk_object_unref (GTK_OBJECT (gui));

	return result;
}
