/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-dateedit.h>
#include <glade/glade.h>
#include "typedefs.h"
#include "main.h"
#include "window.h"
#include "image-list.h"
#include "image-list-utils.h"
#include "comments.h"


#define COMMENT_GLADE_FILE "gthumb_comments.glade"


typedef struct {
	GThumbWindow *  window;
	GladeXML *      gui;
	GList *         file_list;

	GtkWidget *     dialog;
	GtkWidget *     place_entry;
	GtkWidget *     date_checkbutton;
	GtkWidget *     date_dateedit;
	GtkWidget *     note_text;

	GtkWidget *     keyword_entry;
	GtkWidget *     add_key_button;
	GtkWidget *     remove_key_button;
	GtkWidget *     keywords_list;
} DialogData;


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget *widget, 
	    DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	g_list_foreach (data->file_list, (GFunc) g_free, NULL);
	g_list_free (data->file_list);
	g_free (data);
}


/* called when the "add keyword" button is pressed. */
static void
add_keyword_cb (GtkWidget *widget, 
		DialogData *data)
{
	GtkWidget *list_item;
	gchar *keyword;

	keyword = gtk_entry_get_text (GTK_ENTRY (data->keyword_entry));
	if (keyword == NULL)
		return;

	list_item = gtk_list_item_new_with_label (keyword);
	gtk_container_add (GTK_CONTAINER (data->keywords_list),
			   list_item);
	gtk_widget_show (list_item);
}


/* called when the "remove keyword" button is pressed. */
static void
remove_keyword_cb (GtkWidget *widget, 
		   DialogData *data)
{
	GList *selection;

	if (GTK_LIST (data->keywords_list)->selection == NULL)
		return;

	selection = g_list_copy (GTK_LIST (data->keywords_list)->selection);
	gtk_list_remove_items (GTK_LIST (data->keywords_list), selection); 
	g_list_free (selection);
}


static gint
keyword_key_press_cb (GtkWidget *widget, 
		      GdkEvent *event,
		      DialogData *data)
{
	if (event->key.keyval == GDK_Return) {
		add_keyword_cb (widget, data);
		gtk_entry_select_region (GTK_ENTRY (widget), 0, -1);
	}

	return FALSE;
}


static void
date_check_toggled_cb (GtkWidget *widget,
		       DialogData *data)
{
	gtk_widget_set_sensitive (data->date_dateedit,
				  GTK_TOGGLE_BUTTON (widget)->active);
}


/* called when the "ok" button is pressed. */
static void
ok_clicked_cb (GtkWidget *widget, 
	       DialogData *data)
{
	GList *scan;
	CommentData *cdata;
	gchar *text;
	struct tm tm = {0};
	int i;

	cdata = comment_data_new ();

	/* place */
	text = gtk_entry_get_text (GTK_ENTRY (data->place_entry));
	if (text != NULL) 
		cdata->place = g_strdup (text);
	
	/* date */
	if (GTK_TOGGLE_BUTTON (data->date_checkbutton)->active) {
		time_t t;
		struct tm *tm_date;

		t = gnome_date_edit_get_date (GNOME_DATE_EDIT (data->date_dateedit));
		tm_date = localtime (&t);
		tm.tm_year = tm_date->tm_year;
		tm.tm_mon = tm_date->tm_mon;
		tm.tm_mday = tm_date->tm_mday;
	}
	cdata->time = mktime (&tm);
	if (cdata->time < 0)
		cdata->time = 0;

	/* comment */
	cdata->comment = gtk_editable_get_chars (GTK_EDITABLE (data->note_text), 0, -1);

	/* keywords. */
	i = 0;
	scan = GTK_LIST (data->keywords_list)->children;
	cdata->keywords_n = g_list_length (scan);
	if (cdata->keywords_n > 0)
		cdata->keywords = g_new0 (char *, cdata->keywords_n + 1);

	for (; scan; scan = scan->next) {
		GtkLabel *label = GTK_LABEL (GTK_BIN (scan->data)->child);
		cdata->keywords[i++] = g_strdup (label->label);
	}

	/* save and close. */
	for (scan = data->file_list; scan; scan = scan->next) {
		comments_save_comment (scan->data, cdata);
		all_windows_notify_update_comment (scan->data);
	}
	comment_data_free (cdata);
	gtk_widget_destroy (data->dialog);
}


int
strcmp_null_tollerant (gchar *s1, gchar *s2)
{
	if ((s1 == NULL) && (s2 == NULL))
		return 0;
	else if ((s1 != NULL) && (s2 == NULL))
		return 1;
	else if ((s1 == NULL) && (s2 != NULL))
		return -1;
	else 
		return strcmp (s1, s2);
}


void
dlg_edit_comment (GtkWidget *widget, gpointer wdata)
{
	GThumbWindow *window = wdata;
	DialogData *data;
	ImageList *ilist;
	GtkWidget *btn_ok;
	GtkWidget *btn_cancel;
	CommentData *cdata;
	GList *scan;

	data = g_new (DialogData, 1);

	data->window = window;

	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" COMMENT_GLADE_FILE , NULL);
        if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	ilist = IMAGE_LIST (window->file_list->ilist);
	data->file_list = ilist_utils_get_file_list_selection (ilist);
	if (data->file_list == NULL) {
		g_free (data);
		return;
	}

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "comments_dialog");

	data->place_entry = glade_xml_get_widget (data->gui, "place_entry");
	data->date_checkbutton = glade_xml_get_widget (data->gui, "date_checkbutton");
	data->date_dateedit = glade_xml_get_widget (data->gui, "date_dateedit");
	data->note_text = glade_xml_get_widget (data->gui, "note_text");

	data->keyword_entry = glade_xml_get_widget (data->gui, "keyword_entry");
	data->add_key_button = glade_xml_get_widget (data->gui, "add_key_button");
	data->remove_key_button = glade_xml_get_widget (data->gui, "remove_key_button");
	data->keywords_list = glade_xml_get_widget (data->gui, "keywords_list");

        btn_ok = glade_xml_get_widget (data->gui, "ok_button");
        btn_cancel = glade_xml_get_widget (data->gui, "cancel_button");

	/* Set widgets data. */

	cdata = comments_load_comment (data->file_list->data);

	if (cdata != NULL) {
		/* NULL-ify a field if it is not the same in all comments. */
		for (scan = data->file_list->next; scan; scan = scan->next) {
			CommentData *scan_cdata;
			
			scan_cdata = comments_load_comment (scan->data);
			if (scan_cdata == NULL)
				break;

			if (strcmp_null_tollerant (cdata->place, 
						   scan_cdata->place) != 0) 
				if (cdata->place != NULL) {
					g_free (cdata->place);
					cdata->place = NULL;
				}
			
			if (cdata->time != scan_cdata->time)
				cdata->time = 0;
			
			if (strcmp_null_tollerant (cdata->comment, 
						   scan_cdata->comment) != 0) 
				if (cdata->comment != NULL) {
					g_free (cdata->comment);
					cdata->comment = NULL;
				}

			if (cdata->keywords_n == scan_cdata->keywords_n) {
				int i;
				for (i = 0; i < cdata->keywords_n; i++)
					if (strcmp (cdata->keywords[i], scan_cdata->keywords[i]) != 0) {
						comment_data_free_keywords (cdata);
						break;
					}
			} else 
				comment_data_free_keywords (cdata);

			comment_data_free (scan_cdata);
		}
	}

	gtk_widget_set_sensitive (data->date_dateedit, FALSE);

	if (cdata != NULL) {
		int i;	
		GtkWidget *list_item;

		if (cdata->comment != NULL)
			gtk_text_insert (GTK_TEXT (data->note_text), 
					 NULL, NULL, NULL,
					 cdata->comment, 
					 strlen (cdata->comment));

		if (cdata->place != NULL)
			gtk_entry_set_text (GTK_ENTRY (data->place_entry), 
					    cdata->place);

		if (cdata->time != 0) {
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->date_checkbutton), TRUE);
			gnome_date_edit_set_time (GNOME_DATE_EDIT (data->date_dateedit), cdata->time);

			gtk_widget_set_sensitive (data->date_dateedit, TRUE);
		}

		/* keywords */

		for (i = 0; i < cdata->keywords_n; i++) {
			list_item = gtk_list_item_new_with_label (cdata->keywords[i]);
			gtk_widget_show (list_item);
			gtk_container_add (GTK_CONTAINER (data->keywords_list),
					   list_item);
		}

		comment_data_free (cdata);
	}

	/* Set the signals handlers. */
	
	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) destroy_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (btn_ok), "clicked",
			    GTK_SIGNAL_FUNC (ok_clicked_cb),
			    data);
	gtk_signal_connect_object (GTK_OBJECT (btn_cancel), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));
	gtk_signal_connect (GTK_OBJECT (data->date_checkbutton),
			    "toggled",
			    GTK_SIGNAL_FUNC (date_check_toggled_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->add_key_button), "clicked",
			    GTK_SIGNAL_FUNC (add_keyword_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->remove_key_button), "clicked",
			    GTK_SIGNAL_FUNC (remove_keyword_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->keyword_entry), 
			    "key_press_event",
			    GTK_SIGNAL_FUNC (keyword_key_press_cb),
			    data);

	/* run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show (data->dialog);
}



