/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "catalog.h"
#include "typedefs.h"
#include "file-utils.h"


#define SEARCH_HEADER "# Search\n"
#define MAX_LINE_LENGTH 4096


char *
get_catalog_full_path (char *relative_path)
{
	char *path;

	path = g_strconcat (g_get_home_dir (),
			    "/",
			    RC_CATALOG_DIR,
			    (relative_path != NULL) ? "/" : NULL,
			    relative_path,
			    NULL);
	return path;
}


gboolean
delete_catalog_dir (char *full_path, 
		    gboolean recursive)
{
	gint result;

	result = rmdir (full_path);

	if (result != 0) {
		GList *files, *dirs;
		GList *scan;

		if (errno != ENOTEMPTY) {
			g_warning ("Cannot remove directory: %s\n", full_path);
			return FALSE;
		}

		if ((errno == ENOTEMPTY) && !recursive) {
			return FALSE;
		}

		path_list_new (full_path, &files, &dirs);

		/* delete files. */
		for (scan = dirs; scan; scan = scan->next) {
			result = unlink (scan->data);

			if (result == 0) 
				continue;

			g_warning ("Cannot remove file %s", (gchar*) scan->data);
			path_list_free (dirs);
			path_list_free (files);
			
			return FALSE;
		}

		/* delete sub-directories. */
		for (scan = dirs; scan; scan = scan->next) {		
			result = delete_catalog_dir (scan->data, TRUE);

			if (result == 0) 
				continue;

			path_list_free (dirs);
			path_list_free (files);
				
			return FALSE;
		}

		/* the directory is empty now, delete it. */
		result = rmdir (full_path);

		if (result != 0)
			g_warning ("Cannot remove directory: %s\n", full_path);

		path_list_free (dirs);
		path_list_free (files);

		return (result == 0);
	}

	return TRUE;
}


gboolean
delete_catalog (char *full_path)
{
	gint result;

	result = unlink (full_path);

	if (result != 0) {
		g_warning ("Cannot remove file: %s\n", full_path);
		return FALSE;
	}

	return TRUE;
}


gboolean
file_is_search_result (char *fullpath)
{
	gchar  line [MAX_LINE_LENGTH];
	FILE  *f;
	gboolean is_search;

	f = fopen (fullpath, "r");
	if (!f)	{
		g_print ("ERROR: Failed opening catalog file: %s\n", fullpath);
		return FALSE;
	}

	fgets (line, sizeof (line), f);
	fclose (f);

	is_search = (strcmp (line, SEARCH_HEADER) == 0);

	return is_search;
}


Catalog*  
catalog_new ()
{
	Catalog *catalog;

	catalog = g_new (Catalog, 1);
	catalog->path = NULL;
	catalog->list = NULL;
	catalog->search_data = NULL;

	return catalog;
}


void
catalog_free (Catalog *catalog)
{
	if (catalog->path)
		g_free (catalog->path);
	if (catalog->list) 
		path_list_free (catalog->list);
	if (catalog->search_data)
		search_data_free (catalog->search_data);
	g_free (catalog);
}


void
catalog_set_path (Catalog *catalog,
		  char *full_path)
{
	g_return_if_fail (catalog != NULL);

	if (catalog->path != NULL)
		g_free (catalog->path);

	catalog->path = NULL;
	if (full_path)
		catalog->path = g_strdup (full_path);
}


void
catalog_set_search_data (Catalog *catalog,
			 SearchData *search_data)
{
	g_return_if_fail (catalog != NULL);

	if (catalog->search_data != NULL)
		search_data_free (catalog->search_data);

	if (search_data != NULL) {
		catalog->search_data = search_data_new ();
		search_data_copy (catalog->search_data, search_data);
	}
}


gboolean
catalog_is_search_result (Catalog *catalog)
{
	g_return_val_if_fail (catalog != NULL, FALSE);
	return (catalog->search_data != NULL);
}


static void
copy_unquoted (char *unquoted, char *line)
{
	gint l = strlen (line);

	strncpy (unquoted, line + 1, l - 3);
	unquoted[l - 3] = 0;
}


void
catalog_load_from_disk (Catalog *catalog,
			char *fullpath)
{
	gchar  line [MAX_LINE_LENGTH];
	FILE  *f;

	f = fopen (fullpath, "r");
	if (!f)	{
		g_print ("ERROR: Failed opening catalog file: %s\n", fullpath);
		return;
	}

	if (catalog->path != NULL)
		g_free (catalog->path);
	if (catalog->list != NULL)
		path_list_free (catalog->list);
	if (catalog->search_data != NULL)
		search_data_free (catalog->search_data);

	catalog->path = g_strdup (fullpath);
	catalog->list = NULL;
	catalog->search_data = NULL;

	while (fgets (line, sizeof (line), f)) {
		gchar *file_name;

		/* The file name is quoted. */
		if (*line != '"') {
			gchar unquoted [MAX_LINE_LENGTH];
			time_t date;
			gint date_scope;

			/* search data starts with SEARCH_HEADER */

			if (strcmp (line, SEARCH_HEADER) != 0) continue;

			/* load search data. */

			catalog->search_data = search_data_new ();

			/* * start from */

			fgets (line, sizeof (line), f);
			copy_unquoted (unquoted, line);
			search_data_set_start_from (catalog->search_data,
						    unquoted);

			/* * recursive */

			fgets (line, sizeof (line), f);
			copy_unquoted (unquoted, line);
			search_data_set_recursive (catalog->search_data, strcmp (unquoted, "TRUE") == 0);

			/* * file pattern */

			fgets (line, sizeof (line), f);
			copy_unquoted (unquoted, line);
			search_data_set_file_pattern (catalog->search_data,
						      unquoted);

			/* * comment pattern */

			fgets (line, sizeof (line), f);
			copy_unquoted (unquoted, line);
			search_data_set_comment_pattern (catalog->search_data,
							 unquoted);

			/* * place pattern */

			fgets (line, sizeof (line), f);
			copy_unquoted (unquoted, line);
			search_data_set_place_pattern (catalog->search_data,
						       unquoted);

			/* * keywords pattern */

			fgets (line, sizeof (line), f);
			copy_unquoted (unquoted, line);
			search_data_set_keywords_pattern (catalog->search_data,
							  unquoted);

			/* * date */

			fscanf (f, "%ld\n", &date);
			search_data_set_date (catalog->search_data, date);

			/* * date scope */

			fscanf (f, "%d\n", &date_scope);
			search_data_set_date_scope (catalog->search_data, 
						    date_scope);
		}

		file_name = g_strndup (line + 1, strlen (line) - 3);
		catalog->list = g_list_prepend (catalog->list, file_name);
	}
	fclose (f);
}


void
catalog_write_to_disk (Catalog *catalog)
{
	FILE *f;
	gchar *path;
	GList *scan;

	g_return_if_fail (catalog != NULL);
	g_return_if_fail (catalog->path != NULL);

	path = catalog->path;
	f = fopen (path, "w");
	if (!f)	{
		g_warning ("Cannot open file %s", path);
		return;
	}

	if (catalog->search_data != NULL) {
		SearchData *search_data = catalog->search_data;

		/* write search data. */

		if (! fprintf (f, SEARCH_HEADER)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "\"%s\"\n", search_data->start_from)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "\"%s\"\n", (search_data->recursive ? "TRUE" : "FALSE"))) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "\"%s\"\n", search_data->file_pattern)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "\"%s\"\n", search_data->comment_pattern)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "\"%s\"\n", search_data->place_pattern)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "\"%s\"\n", search_data->keywords_pattern)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "%ld\n", search_data->date)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}

		if (! fprintf (f, "%d\n", catalog->search_data->date_scope)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}
	}

	/* write the file list. */

	for (scan = catalog->list; scan; scan = scan->next) 
		if (! fprintf (f, "\"%s\"\n", (gchar*) scan->data)) {
			g_warning ("Cannot save to file %s", path);
			fclose (f);
			return;
		}
	
	fclose (f);
}


void
catalog_add_item (Catalog *catalog,
		  const char *file_path)
{
	g_return_if_fail (catalog != NULL);
	g_return_if_fail (file_path != NULL);	

	if (g_list_find (catalog->list, file_path) == NULL)
		catalog->list = g_list_prepend (catalog->list, 
						g_strdup (file_path));
}


void
catalog_remove_item (Catalog *catalog,
		     const char *file_path)
{
	GList *scan;

	g_return_if_fail (catalog != NULL);
	g_return_if_fail (file_path != NULL);	

	for (scan = catalog->list; scan; scan = scan->next)
		if (strcmp ((gchar*) scan->data, file_path) == 0)
			break;

	if (scan == NULL)
		return;

	catalog->list = g_list_remove_link (catalog->list, scan);

	g_free (scan->data);
	g_list_free (scan);
}


void
catalog_remove_all_items (Catalog *catalog)
{
	g_return_if_fail (catalog != NULL);

	g_list_foreach (catalog->list, (GFunc) g_free, NULL);
	g_list_free (catalog->list);
	catalog->list = NULL;
}

