/*
 * GQview
 * (C) 2003 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "preferences.h"

#include "cache_maint.h"
#include "filelist.h"
#include "img-view.h"
#include "layout_config.h"
#include "layout_util.h"
#include "ui_clist_edit.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"

#include "icons/config.xpm"

typedef struct _ThumbSize ThumbSize;
struct _ThumbSize
{
	gint w;
	gint h;
};

static ThumbSize thumb_size_list[] =
{
	{ 24, 24 },
	{ 32, 32 },
	{ 48, 48 }, 
	{ 64, 64 }, 
	{ 85, 64 }, 
	{ 85, 85 },
	{ 100, 100 }, 
	{ 125, 125 }, 
	{ 150, 150 }, 
	{ 175, 175 }, 
	{ 200, 200 }, 
	{ 256, 256 }, 
	{ 0, 0 }
};

/* config memory values */
static gint startup_path_enable_c;
static gint confirm_delete_c;
static gint enable_delete_key_c;
static gint restore_tool_c;
static gint save_window_positions_c;
static gint zoom_mode_c;
static gint two_pass_zoom_c;
static gint fit_window_c;
static gint limit_window_size_c;
static gint zoom_to_fit_expands_c;
static gint max_window_size_c;
static gint progressive_key_scrolling_c;
static gint thumb_max_width_c;
static gint thumb_max_height_c;
static gint enable_thumb_caching_c;
static gint enable_thumb_dirs_c;
static gint use_xvpics_thumbnails_c;
static gint show_dot_files_c;
static gint file_filter_disable_c;
static gint slideshow_delay_c;
static gint slideshow_random_c;
static gint slideshow_repeat_c;

static gint mousewheel_scrolls_c;

static gint enable_in_place_rename_c;

static gint collection_rectangular_selection_c;
static gint collection_drop_insert_c;

static gint tile_cache_max_c;

static gint thumbnail_quality_c;
static gint zoom_quality_c;
static gint dither_quality_c;

static gint zoom_increment_c;

static gint enable_read_ahead_c;

#if 0
static gint place_dialogs_under_mouse_c;
#endif

static gint black_window_background_c;

static gint dupe_custom_threshold_c;

static GtkWidget *configwindow = NULL;
static GtkWidget *startup_path_entry;
static GtkWidget *filter_clist = NULL;
static GtkWidget *editor_name_entry[8];
static GtkWidget *editor_command_entry[8];

static GtkWidget *layout_widget;

static gint tree_descend_subdirs_c;


/*
 *-----------------------------------------------------------------------------
 * option widget callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void startup_path_set_current(GtkWidget *widget, gpointer data)
{
	gtk_entry_set_text(GTK_ENTRY(startup_path_entry), layout_get_path(NULL));
}

static void slideshow_delay_cb(GtkObject *adj, gpointer data)
{
	slideshow_delay_c = (gint)GTK_ADJUSTMENT(adj)->value;
}

static void dupe_custom_cb(GtkObject *adj, gpointer data)
{
	dupe_custom_threshold_c = (gint)GTK_ADJUSTMENT(adj)->value;
}

static void zoom_mode_original_cb(GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		zoom_mode_c = ZOOM_RESET_ORIGINAL;
}

static void zoom_mode_fit_cb(GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		zoom_mode_c = ZOOM_RESET_FIT_WINDOW;
}

static void zoom_mode_none_cb(GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		zoom_mode_c = ZOOM_RESET_NONE;
}

static void max_window_size_cb(GtkObject *adj, gpointer data)
{
	max_window_size_c = (gint)GTK_ADJUSTMENT(adj)->value;
}

static void tile_cache_max_cb(GtkObject *adj, gpointer data)
{
	tile_cache_max_c = (gint)GTK_ADJUSTMENT(adj)->value;
}

static void zoom_increment_cb(GtkObject *adj, gpointer data)
{
	zoom_increment_c = (gint)(GTK_ADJUSTMENT(adj)->value * 10.0 + 0.01);
}

static void cache_clear_cancel_cb(GenericDialog *gd, gpointer data)
{
	/* no op, only so cancel button appears */
}

static void cache_clear_ok_cb(GenericDialog *gd, gpointer data)
{
	cache_maintain_home(TRUE);
}

static void cache_clear_cb(GtkWidget *widget, gpointer data)
{
	GenericDialog *gd;

	gd = generic_dialog_new(_("Clear cache"), _("This will remove all thumbnails that have\nbeen saved to disk, continue?"),
				"GQview", "clear_cache", TRUE, cache_clear_cancel_cb, NULL);
	generic_dialog_add(gd, _("Ok"), cache_clear_ok_cb, TRUE);

	gtk_widget_show(gd->dialog);
}

/*
 *-----------------------------------------------------------------------------
 * sync progam to config window routine (private)
 *-----------------------------------------------------------------------------
 */ 

static void config_window_apply(void)
{
	const gchar *buf;
	gchar *l_conf;
	gint new_style;
	gint i;
	gint refresh = FALSE;

	for(i=0; i<8; i++)
		{
		g_free(editor_name[i]);
		editor_name[i] = NULL;
		buf = gtk_entry_get_text(GTK_ENTRY(editor_name_entry[i]));
		if (buf && strlen(buf) > 0) editor_name[i] = g_strdup(buf);

		g_free(editor_command[i]);
		editor_command[i] = NULL;
		buf = gtk_entry_get_text(GTK_ENTRY(editor_command_entry[i]));
		if (buf && strlen(buf) > 0) editor_command[i] = g_strdup(buf);
		}
	layout_edit_update_all();

	g_free(startup_path);
	startup_path = NULL;
	buf = gtk_entry_get_text(GTK_ENTRY(startup_path_entry));
	if (buf && strlen(buf) > 0) startup_path = remove_trailing_slash(buf);

	if (show_dot_files != show_dot_files_c) refresh = TRUE;
	if (file_filter_disable != file_filter_disable_c) refresh = TRUE;

	startup_path_enable = startup_path_enable_c;
	confirm_delete = confirm_delete_c;
	enable_delete_key = enable_delete_key_c;
	restore_tool = restore_tool_c;
	save_window_positions = save_window_positions_c;
	zoom_mode = zoom_mode_c;
	two_pass_zoom = two_pass_zoom_c;
	fit_window = fit_window_c;
	limit_window_size = limit_window_size_c;
	zoom_to_fit_expands = zoom_to_fit_expands_c;
	max_window_size = max_window_size_c;
	progressive_key_scrolling = progressive_key_scrolling_c;
	thumb_max_width = thumb_max_width_c;
	thumb_max_height = thumb_max_height_c;
	enable_thumb_caching = enable_thumb_caching_c;
	enable_thumb_dirs = enable_thumb_dirs_c;
	use_xvpics_thumbnails = use_xvpics_thumbnails_c;
	show_dot_files = show_dot_files_c;
	file_filter_disable = file_filter_disable_c;

	slideshow_random = slideshow_random_c;
	slideshow_repeat = slideshow_repeat_c;
	slideshow_delay = slideshow_delay_c;

	mousewheel_scrolls = mousewheel_scrolls_c;

	enable_in_place_rename = enable_in_place_rename_c;

	collection_rectangular_selection = collection_rectangular_selection_c;
	collection_drop_insert = collection_drop_insert_c;

	tile_cache_max = tile_cache_max_c;

	thumbnail_quality = thumbnail_quality_c;
	zoom_quality = zoom_quality_c;
	dither_quality = dither_quality_c;

	zoom_increment = zoom_increment_c;

	enable_read_ahead = enable_read_ahead_c;

#if 0
	place_dialogs_under_mouse = place_dialogs_under_mouse_c;
#endif

	if (black_window_background != black_window_background_c)
		{
		black_window_background = black_window_background_c;

		layout_colors_update();
		view_window_colors_update();
		}

	dupe_custom_threshold = dupe_custom_threshold_c;

	tree_descend_subdirs = tree_descend_subdirs_c;

	l_conf = layout_config_get(layout_widget, &new_style);

	if (new_style != layout_style ||
	    (l_conf == NULL) != (layout_order == NULL) ||
	    !layout_order ||
	    strcmp(buf, layout_order) != 0)
		{
		if (refresh) filter_rebuild();
		refresh = FALSE;

		g_free(layout_order);
		layout_order = l_conf;
		l_conf = NULL;

		layout_style = new_style;

		layout_styles_update();
		}

	g_free(l_conf);

	if (refresh)
		{
		filter_rebuild();
		layout_refresh(NULL);
		}
}

/*
 *-----------------------------------------------------------------------------
 * config window main button callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void config_window_close_cb(GtkWidget *widget, gpointer data)
{
	gtk_widget_destroy(configwindow);
	configwindow = NULL;
	filter_clist = NULL;
}

static gint config_window_delete(GtkWidget *w, GdkEventAny *event, gpointer data)
{
	config_window_close_cb(NULL, NULL);
	return TRUE;
}

static void config_window_ok_cb(GtkWidget *widget, gpointer data)
{
	config_window_apply();
	config_window_close_cb(NULL, NULL);
}

static void config_window_apply_cb(GtkWidget *widget, gpointer data)
{
	config_window_apply();
}

/*
 *-----------------------------------------------------------------------------
 * config window setup (private)
 *-----------------------------------------------------------------------------
 */ 

static void check_button_cb(GtkWidget *widget, gpointer data)
{
	gint *value_ptr = data;
	*value_ptr = GTK_TOGGLE_BUTTON (widget)->active;
}

static void add_check_button(gint option, gint *option_c, gchar *text, GtkWidget *box)
{
	GtkWidget *button;
	*option_c = option;
	button = gtk_check_button_new_with_label (text);
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(button), option);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(check_button_cb), option_c);
	gtk_widget_show(button);
}

static void quality_menu_cb(GtkWidget *widget, gpointer data)
{
	gint *option_c = data;

	*option_c = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), "value"));
}

static void add_menu_item(GtkWidget *menu, gchar *text, gint *option_c, gint type)
{
	GtkWidget *item;

	item = gtk_menu_item_new_with_label(text);
	gtk_object_set_data(GTK_OBJECT(item), "value", GINT_TO_POINTER(type));
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			   GTK_SIGNAL_FUNC(quality_menu_cb), option_c);
	gtk_menu_append(GTK_MENU (menu), item);
	gtk_widget_show(item);
}

static void add_quality_menu(gint option, gint *option_c, gchar *text, GtkWidget *box, gint allow_tiles)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *omenu;
	GtkWidget *menu;

	*option_c = option;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	add_menu_item(menu, _("Nearest (worst, but fastest)"), option_c, (gint)GDK_INTERP_NEAREST);
	if (allow_tiles) add_menu_item(menu, _("Tiles"), option_c, (gint)GDK_INTERP_TILES);
	add_menu_item(menu, _("Bilinear"), option_c, (gint)GDK_INTERP_BILINEAR);
	add_menu_item(menu, _("Hyper (best, but slowest)"), option_c, (gint)GDK_INTERP_HYPER);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu),
			(!allow_tiles && *option_c > (gint)GDK_INTERP_TILES) ? *option_c - 1 : *option_c);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);
}

static void add_dither_menu(gint option, gint *option_c, gchar *text, GtkWidget *box)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *omenu;
	GtkWidget *menu;

	*option_c = option;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	add_menu_item(menu, _("None"), option_c, (gint)GDK_RGB_DITHER_NONE);
	add_menu_item(menu, _("Normal"), option_c, (gint)GDK_RGB_DITHER_NORMAL);
	add_menu_item(menu, _("Best"), option_c, (gint)GDK_RGB_DITHER_MAX);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), *option_c);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);
}

static void thumb_size_menu_cb(GtkWidget *widget, gpointer data)
{
	gint n;

	n = GPOINTER_TO_INT(data);

	if (n >= 0 && n < sizeof(thumb_size_list) / sizeof(ThumbSize))
		{
		thumb_max_width_c = thumb_size_list[n].w;
		thumb_max_height_c = thumb_size_list[n].h;
		}
}

static void add_thumb_size_menu(gchar *text, GtkWidget *box)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *omenu;
	GtkWidget *menu;
	gint row;
	gint i;
	gint c;

	thumb_max_width_c = thumb_max_width;
	thumb_max_height_c = thumb_max_height;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	row = 0;
	i = 0;
	c = TRUE;
	while (c)
		{
		gint w, h;

		w = thumb_size_list[i].w;
		h = thumb_size_list[i].h;

		if ( w > 0 && h > 0)
			{
			GtkWidget *item;
			gchar *buf;

			buf = g_strdup_printf("%d x %d", w, h);
			item = gtk_menu_item_new_with_label(buf);
			g_free(buf);

			gtk_signal_connect(GTK_OBJECT(item), "activate",
					   GTK_SIGNAL_FUNC(thumb_size_menu_cb), GINT_TO_POINTER(i));
			gtk_menu_append(GTK_MENU (menu), item);
			gtk_widget_show(item);
			if (thumb_max_width >= w && thumb_max_width >= h) row = i;
			}
		else
			{
			c = FALSE;
			}
		i++;
		}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), row);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);
}

static void filter_clist_populate(void)
{
	GList *work;
	GtkCList *clist;

	if (!filter_clist) return;
	clist = GTK_CLIST(filter_clist);

	gtk_clist_clear(clist);

	work = filter_get_list();
	while (work)
		{
		FilterEntry *fe;
		gint row;
		gchar *buf[4];

		fe = work->data;
		work = work->next;

		buf[0] = (fe->enabled) ? "X" : "";
		buf[1] = fe->extensions;
		buf[2] = fe->description;
		buf[3] = NULL;

		row = gtk_clist_append(clist, buf);
		gtk_clist_set_row_data(clist, row, fe);
		}
}

static gint filter_clist_ext_cb(ClistEditData *ced,
				const gchar *old_text, const gchar *new_text, gpointer data)
{
	FilterEntry *fe = data;

	if (!fe || !new_text || strlen(new_text) < 1) return FALSE;

	g_free(fe->extensions);
	fe->extensions = g_strdup(new_text);

	filter_rebuild();

	return TRUE;
}

static gint filter_clist_desc_cb(ClistEditData *ced,
				 const gchar *old_text, const gchar *new_text, gpointer data)
{
	FilterEntry *fe = data;

	if (!fe || !new_text || strlen(new_text) < 1) return FALSE;

	g_free(fe->description);
	fe->description = g_strdup(new_text);

	return TRUE;
}

static gint filter_clist_click_cb(GtkWidget *clist, GdkEventButton *event, gpointer data)
{
	gint row = -1;
	gint column = -1;
	FilterEntry *fe;

	gtk_clist_get_selection_info(GTK_CLIST(clist), event->x, event->y, &row, &column);
	if (row == -1 || column == -1) return TRUE;

	fe = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	gtk_clist_select_row(GTK_CLIST(clist), row, 0);
	if (!fe || event->button != 1) return TRUE;

	if (column == 0)
		{
		fe->enabled = !fe->enabled;
		gtk_clist_set_text(GTK_CLIST(clist), row, column, (fe->enabled) ? "X" : "");
		filter_rebuild();
		}
	else if (column == 1)
		{
		clist_edit_by_row(GTK_CLIST(clist), row, column,
				  filter_clist_ext_cb, fe);
		}
	else if (column == 2)
		{
		clist_edit_by_row(GTK_CLIST(clist), row, column,
				filter_clist_desc_cb, fe);
		}
	
	return TRUE;
}

static void filter_add_cb(GtkWidget *widget, gpointer data)
{
	gint row;

	filter_add("description", ".new", TRUE);
	filter_clist_populate();

	row = GTK_CLIST(filter_clist)->rows - 1;
	gtk_clist_moveto(GTK_CLIST(filter_clist), row, 0, 0.0, 1.0);
	gtk_clist_select_row(GTK_CLIST(filter_clist), row, 0);
}

static void filter_remove_cb(GtkWidget *widget, gpointer data)
{
	gint row;
	FilterEntry *fe;

	if (!filter_clist || !GTK_CLIST(filter_clist)->selection) return;
	row = GPOINTER_TO_INT(GTK_CLIST(filter_clist)->selection->data);
	fe = gtk_clist_get_row_data(GTK_CLIST(filter_clist), row);
	if (!fe) return;

	filter_remove_entry(fe);
	filter_rebuild();
	filter_clist_populate();
}

static void filer_default_cancel_cb(GenericDialog *gd, gpointer data)
{
	/* no op */
}

static void filer_default_ok_cb(GenericDialog *gd, gpointer data)
{
	filter_reset();
	filter_add_defaults();
	filter_rebuild();
	filter_clist_populate();
}

static void filter_default_cb(GtkWidget *widget, gpointer data)
{
	GenericDialog *gd;

	gd = generic_dialog_new(_("Reset filters"), _("This will reset the file filters to the defaults.\nContinue?"),
				"GQview", "reset_filter", TRUE, filer_default_cancel_cb, NULL);
	generic_dialog_add(gd, _("Ok"), filer_default_ok_cb, TRUE);
	gtk_widget_show(gd->dialog);
}

static void config_window_create(void)
{
	GtkWidget *win_vbox;
	GtkWidget *hbox;
	GtkWidget *notebook;
	GtkWidget *frame;
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *vbox1;
	GtkWidget *button;
	GtkWidget *tabcomp;
	GtkWidget *radiobuttongroup;
	GtkWidget *table;
	GtkObject *adj;
	GtkWidget *spin;
	GtkWidget *scrolled;

	gchar *filter_titles[] = { "", _("filter"), _("description"), NULL};
	gint i;

	configwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT (configwindow), "delete_event",
			   GTK_SIGNAL_FUNC(config_window_delete), NULL);
	gtk_window_set_policy(GTK_WINDOW(configwindow), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(configwindow), _("GQview configuration"));
	gtk_window_set_wmclass(GTK_WINDOW(configwindow), "config", "GQview");
	gtk_container_border_width(GTK_CONTAINER (configwindow), 5);

	window_set_icon(configwindow, (const gchar **)config_xpm, NULL);

	win_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(configwindow), win_vbox);
	gtk_widget_show(win_vbox);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 0);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbox), 95, 40);
	gtk_box_pack_end(GTK_BOX(win_vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Ok"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(config_window_ok_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Apply"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(config_window_apply_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Cancel"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(config_window_close_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start (GTK_BOX(win_vbox), notebook, TRUE, TRUE, 0);

	/* general options tab */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_widget_show(frame);
	label = gtk_label_new(_("General"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), frame, label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	frame = gtk_frame_new(_("Initial directory"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_check_button(startup_path_enable, &startup_path_enable_c,
			 _("On startup, change to this directory:"), vbox1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	gtk_widget_realize(configwindow);

	tabcomp = tab_completion_new(&startup_path_entry, startup_path, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tabcomp, TRUE, TRUE, 0);
	gtk_widget_show(tabcomp);

	button = gtk_button_new_with_label (_("Use current"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(startup_path_set_current), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	frame = gtk_frame_new(_("Thumbnails"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_thumb_size_menu(_("Size:"), vbox1);

	add_quality_menu(thumbnail_quality, &thumbnail_quality_c, _("Quality:"), vbox1, TRUE);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);
	add_check_button(enable_thumb_caching, &enable_thumb_caching_c,
			 _("Cache thumbnails"), hbox);
	button = gtk_button_new_with_label(_("Clear cache"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(cache_clear_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	add_check_button(enable_thumb_dirs, &enable_thumb_dirs_c,
			 _("Cache thumbnails into .thumbnails"), vbox1);

	add_check_button(use_xvpics_thumbnails, &use_xvpics_thumbnails_c,
			 _("Use xvpics thumbnails when found (read only)"), vbox1);

	frame = gtk_frame_new(_("Slide show"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Delay before image change (seconds):"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	slideshow_delay_c = slideshow_delay;
	adj = gtk_adjustment_new((float)slideshow_delay_c, 1.0, 1200.0, 1, 1, 1);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
	gtk_box_pack_start( GTK_BOX(hbox), spin, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			   GTK_SIGNAL_FUNC(slideshow_delay_cb), NULL);
	gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
	gtk_widget_show(spin);
	
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	add_check_button(slideshow_random, &slideshow_random_c,
			 _("Random"), hbox);
	add_check_button(slideshow_repeat, &slideshow_repeat_c,
			 _("Repeat"), hbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
	gtk_widget_show(vbox1);

	add_check_button(enable_in_place_rename, &enable_in_place_rename_c,
			 _("In place renaming"), vbox1);
	add_check_button(enable_read_ahead, &enable_read_ahead_c,
			 _("Preload next image"), vbox1);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_end(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
	gtk_widget_show(vbox1);

	add_check_button(confirm_delete, &confirm_delete_c,
			 _("Confirm file delete"), vbox1);
	add_check_button(enable_delete_key, &enable_delete_key_c,
			 _("Enable Delete key"), vbox1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Custom similarity threshold:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	dupe_custom_threshold_c = dupe_custom_threshold;
	adj = gtk_adjustment_new((float)dupe_custom_threshold_c, 0.0, 100.0, 1, 1, 1);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			   GTK_SIGNAL_FUNC(dupe_custom_cb), NULL);
	gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
	gtk_widget_show(spin);

	/* image tab */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_widget_show(frame);
	label = gtk_label_new(_("Image"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);
	
	frame = gtk_frame_new(_("Quality"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_dither_menu(dither_quality, &dither_quality_c, _("Dithering method:"), vbox1);
	add_quality_menu(zoom_quality, &zoom_quality_c, _("Zoom (scaling):"), vbox1, !(GDK_PIXBUF_TILES_BROKEN));
	add_check_button(two_pass_zoom, &two_pass_zoom_c, _("Two pass zooming"), vbox1);

	frame = gtk_frame_new(_("When new image is selected:"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	zoom_mode_c = zoom_mode;
	radiobuttongroup = gtk_radio_button_new_with_label (NULL, _("Zoom to original size"));
	if (zoom_mode == ZOOM_RESET_ORIGINAL) gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(radiobuttongroup), 1);
	gtk_signal_connect(GTK_OBJECT(radiobuttongroup), "clicked",
			   GTK_SIGNAL_FUNC(zoom_mode_original_cb), NULL);
	gtk_box_pack_start(GTK_BOX(vbox1), radiobuttongroup, FALSE, FALSE, 0);
	gtk_widget_show(radiobuttongroup);

	button = gtk_radio_button_new_with_label (gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttongroup)),_("Fit image to window"));
	if (zoom_mode == ZOOM_RESET_FIT_WINDOW) gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(button), 1);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(zoom_mode_fit_cb), NULL);
	gtk_box_pack_start(GTK_BOX(vbox1), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_radio_button_new_with_label (gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttongroup)),_("Leave Zoom at previous setting"));
	if (zoom_mode == ZOOM_RESET_NONE) gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(button), 1);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(zoom_mode_none_cb), NULL);
	gtk_box_pack_start(GTK_BOX(vbox1), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	add_check_button(zoom_to_fit_expands, &zoom_to_fit_expands_c,
			 _("Expand images for zoom to fit."), vbox);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Offscreen cache size (Mb per image):"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	tile_cache_max_c = tile_cache_max;
	adj = gtk_adjustment_new((float)tile_cache_max_c, 0.0, 128.0, 1.0, 1.0, 1.0);
	spin = gtk_spin_button_new( GTK_ADJUSTMENT(adj), 1, 0 );
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			   GTK_SIGNAL_FUNC(tile_cache_max_cb), NULL);
	gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
	gtk_widget_show(spin);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Zoom increment:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	zoom_increment_c = zoom_increment;
	adj = gtk_adjustment_new(zoom_increment_c ? (float)zoom_increment_c / 10.0 : 1.0, 0.1, 4.0, 0.1, 0.1, 0.1);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, 1);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			   GTK_SIGNAL_FUNC(zoom_increment_cb), NULL);
	gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
	gtk_widget_show(spin);

	add_check_button(progressive_key_scrolling, &progressive_key_scrolling_c,
			 _("Progressive keyboard scrolling"), vbox);
	add_check_button(mousewheel_scrolls, &mousewheel_scrolls_c,
			 _("Mouse wheel scrolls image"), vbox);

	add_check_button(black_window_background, &black_window_background_c,
			 _("Black background"), vbox);

	/* window tab */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_widget_show(frame);
	label = gtk_label_new(_("Windows"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	add_check_button(save_window_positions, &save_window_positions_c,
			 _("Remember window positions"), vbox);
	add_check_button(restore_tool, &restore_tool_c,
			 _("Remember tool state (float/hidden)"), vbox);

	add_check_button(fit_window, &fit_window_c,
			 _("Fit window to image when tools are hidden/floating"), vbox);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	add_check_button(limit_window_size, &limit_window_size_c,
			 _("Limit size when auto-sizing window (%):"), hbox);

	max_window_size_c = max_window_size;
	adj = gtk_adjustment_new((float)max_window_size_c, 10.0, 150.0, 1.0, 1.0, 1.0);
        spin = gtk_spin_button_new( GTK_ADJUSTMENT(adj), 1, 0);
        gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 5);
        gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			   GTK_SIGNAL_FUNC(max_window_size_cb), NULL);
        gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
        gtk_widget_show(spin);

#if 0
	add_check_button(place_dialogs_under_mouse, &place_dialogs_under_mouse_c,
			 _("Place dialogs under mouse"), vbox);
#endif

	add_check_button(tree_descend_subdirs, &tree_descend_subdirs_c,
			 _("Descend folders in tree view"), vbox);

	frame = gtk_frame_new(_("Layout"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	layout_widget = layout_config_new();
	layout_config_set(layout_widget, layout_style, layout_order);
	gtk_container_add(GTK_CONTAINER(frame), layout_widget);
	gtk_widget_show(layout_widget);


	/* filtering tab */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_widget_show(frame);
	label = gtk_label_new(_("Filtering"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	add_check_button(show_dot_files, &show_dot_files_c,
			 _("Show entries that begin with a dot"), vbox);
	add_check_button(file_filter_disable, &file_filter_disable_c,
			 _("Disable File Filtering"), vbox);

	frame = gtk_frame_new(_("Include files of type:"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vbox1), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	filter_clist = gtk_clist_new_with_titles(3, filter_titles);
	gtk_clist_set_selection_mode(GTK_CLIST(filter_clist), GTK_SELECTION_SINGLE);
	gtk_clist_column_titles_passive(GTK_CLIST(filter_clist));
	gtk_clist_set_column_auto_resize(GTK_CLIST(filter_clist), 0, TRUE);
	gtk_clist_set_column_width(GTK_CLIST(filter_clist), 1, 150);
	filter_clist_populate();
	gtk_signal_connect(GTK_OBJECT(filter_clist), "button_press_event",
			   GTK_SIGNAL_FUNC(filter_clist_click_cb), NULL);
	gtk_container_add(GTK_CONTAINER(scrolled), filter_clist);
	gtk_widget_show(filter_clist);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX(vbox1), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Defaults"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(filter_default_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Remove"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(filter_remove_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Add"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(filter_add_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	/* collections tab */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_widget_show(frame);
	label = gtk_label_new(_("Collections"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);

	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	add_check_button(collection_rectangular_selection, &collection_rectangular_selection_c,
			 _("Rectangular selection"), vbox);
	add_check_button(collection_drop_insert, &collection_drop_insert_c,
			 _("Insert file drops at pointer location"), vbox);

	/* editor entry tab */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_widget_show(frame);
	label = gtk_label_new(_("Editors"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);

	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	table=gtk_table_new(3,9,FALSE);
	gtk_container_add (GTK_CONTAINER(vbox),table);
	gtk_widget_show(table);

	label = gtk_label_new(_("#"));
	gtk_table_attach_defaults(GTK_TABLE (table),label, 0, 1, 0, 1);
	gtk_widget_show(label);
	label = gtk_label_new(_("Menu name"));
	gtk_table_attach_defaults(GTK_TABLE (table),label, 1, 2, 0, 1);
	gtk_widget_show(label);
	label = gtk_label_new(_("Command Line"));
	gtk_table_attach_defaults(GTK_TABLE (table),label, 2, 3, 0, 1);
	gtk_widget_show(label);

	for (i=0; i<8; i++)
		{
		gchar *buf;

		buf = g_strdup_printf("%d", i+1);
		label = gtk_label_new(buf);
		g_free(buf);

		gtk_table_attach_defaults(GTK_TABLE (table),label, 0, 1, i+1, i+2);
		gtk_widget_show(label);

		editor_name_entry[i] = gtk_entry_new_with_max_length(32);
		gtk_widget_set_usize(editor_name_entry[i],80,-1);
		if (editor_name[i]) gtk_entry_set_text(GTK_ENTRY(editor_name_entry[i]),editor_name[i]);
		gtk_table_attach_defaults(GTK_TABLE (table),editor_name_entry[i],1,2,i+1,i+2);
		gtk_widget_show(editor_name_entry[i]);

		editor_command_entry[i] = gtk_entry_new_with_max_length(255);
		gtk_widget_set_usize(editor_command_entry[i],160,-1);
		tab_completion_add_to_entry(editor_command_entry[i], NULL, NULL);
		if (editor_command[i]) gtk_entry_set_text(GTK_ENTRY(editor_command_entry[i]), editor_command[i]);
		gtk_table_attach_defaults(GTK_TABLE (table),editor_command_entry[i],2,3,i+1,i+2);
		gtk_widget_show(editor_command_entry[i]);
		}

	gtk_widget_show(notebook);

	gtk_widget_show(configwindow);
}

/*
 *-----------------------------------------------------------------------------
 * config window show (public)
 *-----------------------------------------------------------------------------
 */ 

void show_config_window(void)
{
	if (configwindow) return;
	config_window_create();
}

/*
 *-----------------
 * about window
 *-----------------
 */

static GtkWidget *about = NULL;

static gint about_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	gtk_widget_destroy(about);
	about = NULL;

	return TRUE;
}

void show_about_window(void)
{
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *button;

	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	gchar *buf;

	if (about)
		{
		gdk_window_raise(about->window);
		return;
		}

	about = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_wmclass(GTK_WINDOW(about), "about", "GQview");
	gtk_signal_connect(GTK_OBJECT(about), "delete_event",
			   GTK_SIGNAL_FUNC(about_delete_cb), NULL);

	gtk_window_set_title(GTK_WINDOW(about), "About - GQview");
	gtk_container_border_width(GTK_CONTAINER(about), 5);

	frame = gtk_frame_new(NULL);
	gtk_container_add(GTK_CONTAINER(about), frame);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	pixbuf = gdk_pixbuf_new_from_xpm_data (logo);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pixbuf);
	
	button = gtk_pixmap_new(pixmap, mask);
	gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
	gtk_widget_show(button);

	gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	buf = g_strdup_printf(_("GQview %s\n\nCopyright (c) 2002 by John Ellis\nhttp://gqview.sourceforge.net\ngqview@users.sourceforge.net\n\nReleased under the GNU General Public License"), VERSION);
	label = gtk_label_new(buf);
	g_free(buf);

	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	window_set_icon(about, NULL, NULL);
	gtk_widget_show(about);
}

