/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include "ui2_typedefs.h"

#include "ui2_text.h"
#include "ui2_text_edit.h"

#include "ui2_display.h"
#include "ui2_editor.h"
#include "ui2_main.h"
#include "ui2_skin.h"
#include "ui2_widget.h"
#include "ui_fileops.h"
#include "ui_pixbuf_ops.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"


static void text_edit_page_add(GtkWidget *widget, gpointer data);


/*
 *-------
 * fonts
 *-------
 */

typedef struct _FontDetail FontDetail;
struct _FontDetail
{
	GtkWidget *extended_button;
};

typedef struct _FontListData FontListData;
struct _FontListData
{
	gchar *key;
	gchar *text_id;
	gchar *data;

	gchar *image;
	gint extended;
};

static GdkPixbuf *font_get_pixbuf(gpointer widget)
{
	FontData *font = widget;

	return font->overlay;
}

static void font_edit_write(FILE *f, WidgetData *wd, SkinData *skin, const gchar *dir)
{
	FontData *font = wd->widget;
	gchar *image;

	image = ui_edit_copy_unique_file(ui_widget_get_data(wd, "image"),
					 dir, font->overlay, "font_", wd->key);

	if (image) ui_edit_widget_set_path_key(wd, "image", dir, image);

	ui_edit_write_section(f, "font", wd->key);

	ui_edit_write_key_char(f, "image", image);
	ui_edit_write_key_bool(f, "extended", font->extended);

	g_free(image);
}

static FontListData *font_edit_list_find(GList *list, const gchar *image)
{
	GList *work;
	work = list;
	while(work)
		{
		FontListData *fd = work->data;
		if (strcmp(image, fd->image) == 0) return fd;
		work = work->next;
		}
	return NULL;
}

static gpointer font_edit_read(UIData *ui, WidgetData *wd, GList *list)
{
	FontListData *fd;
	FontData *font;
	const gchar *image;

	image = ui_widget_get_data(wd, "image");

	if (!image || font_edit_list_find(list, image)) return NULL;

	font = wd->widget;

	fd = g_new0(FontListData, 1);
	fd->image = g_strdup(image);
	fd->extended = font->extended;

	fd->key = g_strdup(wd->key);
	fd->data = g_strdup(ui_widget_get_data(wd, "data"));
	fd->text_id = g_strdup(wd->text_id);

	return fd;
}

static void font_edit_free(gpointer data)
{
	FontListData *fd = data;

	g_free(fd->key);
	g_free(fd->text_id);
	g_free(fd->data);
	g_free(fd->image);
	g_free(fd);
}

static gpointer font_edit_props(UIData *ui, WidgetData *wd, GtkWidget *vbox, gpointer detail_data)
{
	FontDetail *fd = detail_data;
	FontData *font = wd->widget;

	if (!fd)
		{
		fd = g_new0(FontDetail, 1);

		fd->extended_button = ui_edit_toggle_new(vbox, _("Extended"));
		gtk_widget_set_sensitive(fd->extended_button, FALSE);
		}

	ui_edit_toggle_set(fd->extended_button, font->extended);

	return fd;
}

static gint font_edit_removable(gpointer widget)
{
	FontData *font = widget;

	return (font->ref < 2);
}

void font_type_init_edit(WidgetObjectData *od)
{
	od->func_get_pixbuf = font_get_pixbuf;

	od->func_edit_write = font_edit_write;

	od->func_edit_read = font_edit_read;
	od->func_edit_free = font_edit_free;

	od->func_edit_props = font_edit_props;

	od->func_edit_page_add = text_edit_page_add;

	od->func_edit_removable = font_edit_removable;
}


/*
 *-------
 * text
 *-------
 */


typedef struct _TextDetail TextDetail;
struct _TextDetail
{
	TextData *text;
	GtkWidget *spin_length;
	GtkWidget *extended_button;
	GtkWidget *sizeable_button;
	GtkWidget *font_key_entry;

	WidgetData *wd;
	UIData *ui;
};

typedef struct _TextPage TextPage;
struct _TextPage
{
	GtkWidget *key_entry;
	GtkWidget *text_id_entry;
	GtkWidget *data_entry;

	GtkWidget *image_entry;
	GtkWidget *font_id_entry;

	GtkWidget *extended_button;
	GtkWidget *sizeable_button;
	GtkWidget *length_spin;

	GtkWidget *clist;
	GtkWidget *pixmap;

	GtkWidget *font_key_entry;

	EditData *ed;
};


static GdkPixbuf *text_get_pixbuf(gpointer data)
{
	TextData *text = data;
	return text->font->overlay;
}

static void text_edit_write(FILE *f, WidgetData *wd, SkinData *skin, const gchar *dir)
{
	TextData *text = wd->widget;
	gchar *image = NULL;
	const gchar *file;
	const gchar *font_id;

	file = ui_widget_get_data(wd, "image");
	font_id = ui_widget_get_data(wd, "font_id");
	if (file || (!file && !font_id))
		{
		image = ui_edit_copy_unique_file(file, dir, text->font->overlay, "text_", wd->key);
		}

	if (image) ui_edit_widget_set_path_key(wd, "image", dir, image);

	ui_edit_write_section(f, "text", wd->key);

	if (image)
		{
		ui_edit_write_key_char(f, "image", image);
		ui_edit_write_key_bool(f, "extended", text->font->extended);
		}
	else
		{
		ui_edit_write_key_char(f, "font", font_id);
		}
	ui_edit_write_key_int(f, "x", text->x);
	ui_edit_write_key_int(f, "y", text->y);
	ui_edit_write_key_int(f, "length", text->length);

	ui_edit_write_key_bool(f, "sizeable", text->sizeable);

	g_free(image);
}

static gpointer text_edit_read(UIData *ui, WidgetData *wd, GList *list)
{
	TextData *text = wd->widget;
	FontListData *fd;
	const gchar *image;

	image = ui_widget_get_data(wd, "image");

	if (!image || font_edit_list_find(list, image)) return NULL;

	fd = g_new0(FontListData, 1);
	fd->image = g_strdup(image);
	fd->extended = text->font->extended;

	fd->key = g_strdup(wd->key);
	fd->data = g_strdup(ui_widget_get_data(wd, "data"));
	fd->text_id = g_strdup(wd->text_id);

	return fd;
}

static void text_edit_props_sizeable_cb(GtkWidget *widget, gpointer data)
{
	TextDetail *td = data;

	td->text->sizeable = ui_edit_toggle_get(td->sizeable_button);
}

static void text_edit_props_length_cb(GtkAdjustment *adjustment, gpointer data)
{
	TextDetail *td = data;
	gint new_length;
	gint old_w, w;

	new_length = ui_edit_spin_get(td->spin_length);

	if (new_length * td->text->font->char_width > td->ui->skin->width)
		{
		new_length = td->ui->skin->width / td->text->font->char_width;
		ui_edit_spin_set_blocking(td->spin_length, new_length, td);
		}

	if (new_length == td->text->length) return;

	old_w = td->text->width;
	w = td->text->font->char_width * new_length;

	if (new_length > td->text->length &&
	    td->text->x + w > td->ui->skin->width)
		{
		ui_widget_set_coord(td->ui, td->wd, td->ui->skin->width - w, td->text->y, TRUE);
		}

	if (old_w > w) ui_edit_widget_draw_highlight(td->ui, td->wd, FALSE);

	td->text->length = new_length;
	td->text->width = w;
	td->text->size_width = w;
	gdk_pixbuf_unref(td->text->pixbuf);
	td->text->pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, w, td->text->height);

	ui_edit_widget_resync(td->ui, td->wd, TRUE, old_w > w ? old_w - w : 0, 0);
}

static gpointer text_edit_props(UIData *ui, WidgetData *wd, GtkWidget *vbox, gpointer detail)
{
	TextData *text = wd->widget;
	TextDetail *td = detail;

	if (!td)
		{
		GtkObject *adj;
		GtkWidget *hbox;

		td = g_new0(TextDetail, 1);

		td->extended_button = ui_edit_toggle_new(vbox, _("Extended"));
		gtk_widget_set_sensitive(td->extended_button, FALSE);

		td->sizeable_button = ui_edit_toggle_new(vbox, _("Dynamic size"));
		gtk_signal_connect(GTK_OBJECT(td->sizeable_button), "clicked",
				   text_edit_props_sizeable_cb, td);

		td->spin_length = ui_edit_spin_new(vbox, _("length:"), 1, 200, &adj);
		gtk_signal_connect(adj, "value_changed", text_edit_props_length_cb, td);

		hbox = ui_edit_frame_new(vbox, FALSE, NULL);

		td->font_key_entry = ui_edit_entry_new(hbox, _("Font key:"));
		ui_edit_frame_sensitive(td->font_key_entry, FALSE, TRUE);
		}

	td->text = text;
	td->wd = wd;
	td->ui = ui;

	ui_edit_toggle_set(td->extended_button, text->font->extended);
	ui_edit_toggle_set_blocking(td->sizeable_button, text->sizeable, td);

	ui_edit_spin_set_blocking(td->spin_length, text->length, td);
	ui_edit_entry_set(td->font_key_entry, ui_widget_get_data(wd, "font_id"));

	return td;
}

static void text_edit_page_font_add_cb(GtkWidget *widget, gpointer data)
{
	TextPage *pd = data;
	FontData *font;
	const gchar *key;
	const gchar *text_id;
	const gchar *image;
	gint extended;

	key = ui_edit_entry_get(pd->font_key_entry);
	text_id = ui_edit_entry_get(pd->text_id_entry);
	image = ui_edit_entry_get(pd->image_entry);
	extended = ui_edit_toggle_get(pd->extended_button);

	if (!key || !image || !isfile(image))
		{
		warning_dialog(_("Font error"), _("Font must contain a key and valid image."));
		return;
		}

	font = font_new_from_file(image, extended);
	
	if (font)
		{
		WidgetData *wd;
		wd = font_register(pd->ed->ui->skin, font, key, text_id);
		ui_edit_widget_add_finish(pd->ed, wd, image, ui_edit_entry_get(pd->data_entry));
		}

	font_unref(font);

	tab_completion_append_to_history(pd->image_entry, image);
}

static void text_edit_page_add_cb(GtkWidget *widget, gpointer data)
{
	TextPage *pd = data;
	TextData *text;
	const gchar *key;
	const gchar *text_id;
	const gchar *font_id;
	const gchar *image;

	key = ui_edit_entry_get(pd->key_entry);
	text_id = ui_edit_entry_get(pd->text_id_entry);
	font_id = ui_edit_entry_get(pd->font_id_entry);
	image = ui_edit_entry_get(pd->image_entry);

	if (!key ||
	    ( (!font_id || !ui_widget_exists(pd->ed->ui, font_id, font_type_id())) && (font_id || !image || !isfile(image)) ) )
		{
		warning_dialog(_("Text error"), _("Text must contain a key, and valid image or font key."));
		return;
		}

	if (font_id)
		{
		FontData *font;
		WidgetData *wd;

		wd = skin_widget_get_by_key(pd->ed->ui->skin, font_id, font_type_id());
		font = wd->widget;
		text = text_new(font, ui_edit_spin_get(pd->length_spin),
				ui_edit_toggle_get(pd->sizeable_button), 0, 0);
		image = NULL;
		}
	else
		{
		text = text_new_from_file(image, ui_edit_spin_get(pd->length_spin),
					  ui_edit_toggle_get(pd->extended_button),
					  ui_edit_toggle_get(pd->sizeable_button), 0, 0);
		}
	if (text)
		{
		WidgetData *wd;
		wd = text_register(pd->ed->ui->skin, text, key, text_id);
		ui_widget_set_data(wd, "font_id", font_id);
		ui_edit_widget_add_finish(pd->ed, wd, image, ui_edit_entry_get(pd->data_entry));
		}

	if (!font_id) tab_completion_append_to_history(pd->image_entry, image);
}

static void text_edit_page_sync(TextPage *pd, FontListData *fd)
{
	if (!fd) return;

	ui_edit_entry_set(pd->image_entry, fd->image);
	ui_edit_toggle_set(pd->extended_button, fd->extended);
}

static void text_edit_page_clist_cb(GtkWidget *clist, gint row, gint col, GdkEvent *event, gpointer data)
{
	TextPage *pd = data;
	FontListData *fd;

	fd = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	text_edit_page_sync(pd, fd);
}

static void text_edit_page_font_cb(GtkWidget *entry, gpointer data)
{
	TextPage *pd = data;
	gint state;

	state = (strlen(gtk_entry_get_text(GTK_ENTRY(pd->font_id_entry))) == 0);

	ui_edit_frame_sensitive(pd->image_entry, state, TRUE);
}

static void text_edit_page_destroy_cb(GtkWidget *widget, gpointer data)
{
	TextPage *pd = data;

	g_free(pd);
}

static GtkWidget *text_edit_page_new(EditData *ed)
{
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *frame;
	GtkWidget *button;
	GtkWidget *sep;
	TextPage *pd;
	gchar *titles[] = { _("Image"), _("Key"), _("Extended"), NULL };

	pd = g_new0(TextPage, 1);
	pd->ed = ed;

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_object_set_data(GTK_OBJECT(hbox), "page", pd);
	gtk_signal_connect(GTK_OBJECT(hbox), "destroy",
			   GTK_SIGNAL_FUNC(text_edit_page_destroy_cb), pd);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
	gtk_widget_show(vbox);

	pd->key_entry = ui_edit_key_entry_new(vbox, ed->ui, text_type_id());
	pd->data_entry = ui_edit_entry_new(vbox, _("Data:"));
	pd->text_id_entry = ui_edit_entry_new(vbox, _("Text id:"));

	pd->extended_button = ui_edit_toggle_new(vbox, _("Extended"));
	pd->sizeable_button = ui_edit_toggle_new(vbox, _("Dynamic size"));
	pd->length_spin = ui_edit_spin_new(vbox, _("Length:"), 1, 200, NULL);

	pd->font_id_entry = ui_edit_entry_new(vbox, _("Font key:"));
	gtk_signal_connect(GTK_OBJECT(pd->font_id_entry), "changed", text_edit_page_font_cb, pd);
	pd->image_entry = ui_edit_path_entry_new(vbox, _("Image:"), "SLIK_text_image");
	pd->pixmap = ui_edit_pixmap_new(vbox);
	ui_edit_path_entry_connect_pixmap(pd->image_entry, pd->pixmap);

	button = gtk_button_new_with_label(_("Add"));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", text_edit_page_add_cb, pd);
	gtk_widget_show(button);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 5);
	gtk_widget_show(sep);

	frame = ui_edit_frame_new(vbox, TRUE, _("Add font only"));
	pd->font_key_entry = ui_edit_entry_new(frame, _("Key:"));

	button = gtk_button_new_with_label(_("Add"));
	gtk_box_pack_end(GTK_BOX(frame), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", text_edit_page_font_add_cb, pd);
	gtk_widget_show(button);

	pd->clist = ui_edit_clist_new(hbox, titles, 3);
	gtk_signal_connect(GTK_OBJECT(pd->clist), "select_row", text_edit_page_clist_cb, pd);

	gtk_widget_show(hbox);

	return hbox;
}

static void text_edit_page_add(GtkWidget *widget, gpointer data)
{
	FontListData *fd = data;
	TextPage *pd;
	gchar *buf[4];

	pd = gtk_object_get_data(GTK_OBJECT(widget), "page");

	buf[0] = "";
	buf[1] = fd->key;
	buf[2] = ui_edit_bool_to_text(fd->extended);
	buf[3] = NULL;

	ui_edit_clist_append(pd->clist, buf, fd->image, fd);
}

void text_type_init_edit(WidgetObjectData *od)
{
	od->func_get_pixbuf = text_get_pixbuf;

	od->func_edit_write = text_edit_write;

	od->func_edit_read = text_edit_read;
	od->func_edit_free = font_edit_free;

	od->func_edit_props = text_edit_props;

	od->func_edit_page_new = text_edit_page_new;
	od->func_edit_page_add = text_edit_page_add;
}

