/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>

#include "ui2_typedefs.h"
#include "ui2_editor.h"

#include "ui2_button.h"
#include "ui2_default.h"
#include "ui2_display.h"
#include "ui2_main.h"
#include "ui2_parse.h"
#include "ui2_skin.h"
#include "ui2_util.h"
#include "ui2_widget.h"
#include "ui_utildlg.h"
#include "ui_fileops.h"
#include "ui_pixbuf_ops.h"
#include "ui_pixbuf_save.h"
#include "ui_tabcomp.h"


/*
 * well, the GtkText widget sucks, and I hate line wrapping,
 * so if you use less than 640x480 you've got problems.
 * these defaults are my preferences, they should be ok for most.
 */

#define HELP_WINDOW_WIDTH 600
#define HELP_WINDOW_HEIGHT 350

#define HELP_WINDOW_FONT "-adobe-courier-medium-r-normal--*-120-*-*-*-*-*-*"

#define CLIST_IMG_MAX_W 64
#define CLIST_IMG_MAX_H 32


typedef struct _EditTypeData EditTypeData;
struct _EditTypeData
{
	WidgetType type;
	GList *list;		/* list for read in widgets (data from ui_edit_widget_read) */
	GtkWidget *page;	/* base parent widget for adding new widget (inserted to notebook) */
};


static GList *editor_resource_list = NULL;

static gint dummy_type_id = -1;	/* widget type id of the dummy widget */


static void widget_list_set_widget(EditData *ed, WidgetData *wd);
static void widget_list_update(EditData *ed, WidgetData *wd, gint col, const gchar *text);
static void widget_list_reset(EditData *ed);

static void edit_widget_set_coord(EditData *ed, WidgetData *wd, gint x, gint y);

static void prop_update(EditData *ed, WidgetData *wd, gint force);

void ui_edit_widget_draw_highlight(UIData *ui, WidgetData *wd, gint enable);
static void ui_edit_widget_active_set(EditData *ed, WidgetData *wd);

static void help_window_scroll(EditData *ed, WidgetType type);


/*
 *-----------------------------------------------------------------------------
 * widget editor utils
 *-----------------------------------------------------------------------------
 */

void ui_edit_frame_sensitive(GtkWidget *widget, gint state, gint stop_at_box)
{
	GtkWidget *parent;

	parent = widget;
	while(parent)
		{
		if ((stop_at_box && GTK_IS_BOX(parent) && !GTK_IS_COMBO(parent)) ||
		    (!stop_at_box && GTK_IS_FRAME(parent)) )
			{
			gtk_widget_set_sensitive(parent, state);
			return;
			}
		parent = parent->parent;
		}
}

static GtkWidget *ui_edit_real_frame_new(GtkWidget *box, gint has_frame, const gchar *text, gint vert)
{
	GtkWidget *hbox;
	GtkWidget *frame = NULL;

	if (has_frame)
		{
		frame = gtk_frame_new(text);
		gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
		gtk_widget_show(frame);
		}

	if (vert)
		{
		hbox = gtk_vbox_new(FALSE, 0);
		}
	else
		{
		hbox = gtk_hbox_new(FALSE, 5);
		}
	if (frame)
		{
		gtk_container_add(GTK_CONTAINER(frame), hbox);
		}
	else
		{
		gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
		}
	gtk_widget_show(hbox);

	return hbox;
}

GtkWidget *ui_edit_frame_new(GtkWidget *box, gint has_frame, const gchar *text)
{
	return ui_edit_real_frame_new(box, has_frame, text, FALSE);
}

GtkWidget *ui_edit_vframe_new(GtkWidget *box, gint has_frame, const gchar *text)
{
	return ui_edit_real_frame_new(box, has_frame, text, TRUE);
}

GtkWidget *ui_edit_label_new(GtkWidget *box, const gchar *text)
{
	GtkWidget *label;

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	return label;
}

GtkWidget *ui_edit_pixmap_new(GtkWidget *box)
{
	GtkWidget *scrolled;
	GtkWidget *viewport;
	GtkWidget *pmap;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkPixbuf *pixbuf;

	pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, 1, 1);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pixbuf);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(box), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	viewport = gtk_viewport_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(scrolled), viewport);
	gtk_widget_show(viewport);

	pmap = gtk_pixmap_new(pixmap, mask);
	gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	gtk_container_add(GTK_CONTAINER(viewport), pmap);
	gtk_widget_show(pmap);

	return pmap;
}

void ui_edit_pixmap_blank(GtkWidget *pmap)
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkPixbuf *pixbuf;

	pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, 1, 1);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pixbuf);

	gtk_pixmap_set(GTK_PIXMAP(pmap), pixmap, mask);

	gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);
}

GtkWidget *ui_edit_toggle_new(GtkWidget *box, const gchar *text)
{
	GtkWidget *button;

	button = gtk_check_button_new_with_label(text);
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
        gtk_widget_show(button);

	return button;
}

void ui_edit_toggle_set(GtkWidget *button, gint active)
{
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), active);
}

void ui_edit_toggle_set_blocking(GtkWidget *button, gint active, gpointer block_data)
{
	gtk_signal_handler_block_by_data(GTK_OBJECT(button), block_data);
	ui_edit_toggle_set(button, active);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(button), block_data);
}

gint ui_edit_toggle_get(GtkWidget *button)
{
	return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
}

GtkWidget *ui_edit_path_entry_new(GtkWidget *box, const gchar *text, const gchar *history_key)
{
	GtkWidget *hbox;
	GtkWidget *combo;
	GtkWidget *entry;

	hbox = ui_edit_frame_new(box, FALSE, NULL);
	ui_edit_label_new(hbox, text);

	combo = tab_completion_new_with_history(&entry, NULL,
						history_key, 16, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
        gtk_widget_show(combo);

	gtk_entry_set_text(GTK_ENTRY(entry), "");

	return entry;
}

static void ui_edit_path_pixmap_cb(GtkWidget *entry, gpointer data)
{
	GtkWidget *pmap = data;
	gchar *path;

	path = gtk_entry_get_text(GTK_ENTRY(entry));

	if (isfile(path))
		{
		GdkPixbuf *pb;

		pb = gdk_pixbuf_new_from_file(path);
		if (pb)
			{
			GdkPixmap *pixmap;
			GdkBitmap *mask;
			gdk_pixbuf_render_pixmap_and_mask(pb, &pixmap, &mask, 1);
			gtk_pixmap_set(GTK_PIXMAP(pmap), pixmap, mask);
			gdk_pixmap_unref(pixmap);
			if (mask) gdk_bitmap_unref(mask);
			gdk_pixbuf_unref(pb);
			return;
			}
		}

	ui_edit_pixmap_blank(pmap);
}

void ui_edit_path_entry_connect_pixmap(GtkWidget *entry, GtkWidget *pmap)
{
	gtk_signal_connect(GTK_OBJECT(entry), "changed", ui_edit_path_pixmap_cb, pmap);
}

static gint key_entry_sort_cb(gconstpointer a, gconstpointer b)
{
	return strcmp((gchar *)a, (gchar *)b);
}

static void ui_edit_key_entry_set_keys(GtkWidget *combo, UIData *ui, WidgetType type, gpointer block_data)
{
	GList *list = NULL;
	GList *work;
	gchar *buf;

	buf = g_strdup(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry)));

	work = ui->register_list;
	while(work && type >= 0)
		{
		RegisterData *rd = work->data;
		if (rd->type == type) list = g_list_append(list, rd->key);
		work = work->next;
		}

	list = g_list_sort(list, key_entry_sort_cb);

	if (block_data) gtk_signal_handler_block_by_data(GTK_OBJECT(GTK_COMBO(combo)->entry), block_data);

	if (!list) list = g_list_append(list, "");
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), buf);

	if (block_data) gtk_signal_handler_unblock_by_data(GTK_OBJECT(GTK_COMBO(combo)->entry), block_data);

	g_free(buf);
	g_list_free(list);
}

GtkWidget *ui_edit_key_entry_new(GtkWidget *box, UIData *ui, WidgetType type)
{
	GtkWidget *hbox;
	GtkWidget *combo;

	hbox = ui_edit_frame_new(box, FALSE, NULL);
	ui_edit_label_new(hbox, _("Key:"));

	combo = gtk_combo_new();
	ui_edit_key_entry_set_keys(combo, ui, type, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
        gtk_widget_show(combo);

	return GTK_COMBO(combo)->entry;
}

GtkWidget *ui_edit_entry_new(GtkWidget *box, const gchar *text)
{
	GtkWidget *hbox;
	GtkWidget *entry;

	hbox = ui_edit_frame_new(box, FALSE, NULL);
	ui_edit_label_new(hbox, text);

	entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        gtk_widget_show(entry);

	return entry;
}

void ui_edit_entry_set(GtkWidget *entry, const gchar *text)
{
	gtk_entry_set_text(GTK_ENTRY(entry), text ? text : "");
}

void ui_edit_entry_set_blocking(GtkWidget *entry, const gchar *text, gpointer block_data)
{
	gtk_signal_handler_block_by_data(GTK_OBJECT(entry), block_data);
	ui_edit_entry_set(entry, text);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(entry), block_data);
}

gchar *ui_edit_entry_get(GtkWidget *entry)
{
	gchar *buf;

	buf = gtk_entry_get_text(GTK_ENTRY(entry));
	if (strlen(buf) > 0) return buf;
	return NULL;
}

GtkWidget *ui_edit_spin_new(GtkWidget *box, const gchar *text, gint lower, gint upper, GtkObject **adjustment)
{
	GtkObject *adj;
	GtkWidget *spin;
	GtkWidget *hbox;

	hbox = ui_edit_frame_new(box, FALSE, NULL);
	ui_edit_label_new(hbox, text);

	adj = gtk_adjustment_new(lower > 0.0 ? (gfloat)lower : 0.0, (gfloat)lower, (gfloat)upper, 1.0, 1.0, 1.0);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, 0);
	gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 0);
	gtk_widget_show(spin);

	if (adjustment) *adjustment = adj;

	return spin;
}

void ui_edit_spin_set(GtkWidget *spin, gint value)
{
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), (gfloat)value);
}

void ui_edit_spin_set_blocking(GtkWidget *spin, gint val, gpointer block_data)
{
	GtkAdjustment *adj;

	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_signal_handler_block_by_data(GTK_OBJECT(adj), block_data);
	gtk_adjustment_set_value(adj, (gfloat)val);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(adj), block_data);
}

gint ui_edit_spin_get(GtkWidget *spin)
{
	return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
}

/*
 *-----------------------------------------------------------------------------
 * widget editor clist utils
 *-----------------------------------------------------------------------------
 */


GtkWidget *ui_edit_clist_new(GtkWidget *box, gchar *titles[], gint columns)
{
	GtkWidget *scrolled;
	GtkWidget *clist;
	gint i;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(box), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	clist = gtk_clist_new_with_titles(columns, titles);
	gtk_clist_set_row_height(GTK_CLIST(clist), CLIST_IMG_MAX_H);
	for (i = 0; i < columns; i++)
		{
	        gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
		}
	gtk_container_add(GTK_CONTAINER(scrolled), clist);
	gtk_widget_show(clist);

	return clist;
}

gint ui_edit_clist_append(GtkWidget *clist, gchar *buf[], const gchar *image, gpointer data)
{
	gint row;

	row = gtk_clist_append(GTK_CLIST(clist), buf);
	gtk_clist_set_row_data(GTK_CLIST(clist), row, data);

	if (image)
		{
		GdkPixbuf *pb;

		pb = gdk_pixbuf_new_from_file(image);
		if (pb)
			{
			GdkPixmap *pixmap;
			GdkBitmap *mask;
			gint w, h;

			w = gdk_pixbuf_get_width(pb);
			h = gdk_pixbuf_get_height(pb);
			if (w > CLIST_IMG_MAX_W || h > CLIST_IMG_MAX_H)
				{
				gfloat scale;
				GdkPixbuf *t;

				if ((float)CLIST_IMG_MAX_W / w < (float)CLIST_IMG_MAX_H / h)
					{
					scale = (float)CLIST_IMG_MAX_W / w;
					}
				else
					{
					scale = (float)CLIST_IMG_MAX_H / h;
					}
				w = (float)w * scale;
				h = (float)h * scale;
				if (w < 1) w = 1;
				if (h < 1) h = 1;

				t = pb;
				pb = gdk_pixbuf_scale_simple(t, w, h, GDK_INTERP_NEAREST);
				gdk_pixbuf_unref(t);
				}

			gdk_pixbuf_render_pixmap_and_mask(pb, &pixmap, &mask, 1);
			gdk_pixbuf_unref(pb);

			gtk_clist_set_pixmap(GTK_CLIST(clist), row, 0, pixmap, mask);
			gdk_pixmap_unref(pixmap);
			if (mask) gdk_bitmap_unref(mask);
			}
		}

	return row;
}

gchar *ui_edit_bool_to_text(gint enable)
{
	return enable ? "X" : "";
}

gchar *ui_edit_int_to_text(gint value)
{
	return g_strdup_printf("%d", value);
}

/*
 *-----------------------------------------------------------------------------
 * editor widget funcs
 *-----------------------------------------------------------------------------
 */

static GdkPixbuf *ui_edit_widget_get_pixbuf(WidgetData *wd)
{
	if (wd->od && wd->od->func_get_pixbuf)
		{
		return wd->od->func_get_pixbuf(wd->widget);
		}
	return NULL;
}

static void ui_edit_widget_write(FILE *f, WidgetData *wd, SkinData *skin, const gchar *dir)
{
	if (wd->od && wd->od->func_edit_write)
		{
		const gchar *data;

		/* this starts the section and writes widget specific data */
		wd->od->func_edit_write(f, wd, skin, dir);

		/* now write the general widget data */
		data = ui_widget_get_data(wd, "data");

		if (wd->text_id) ui_edit_write_key_char(f, "text_id", wd->text_id);
		if (data) ui_edit_write_key_char(f, "data", data);

		if (wd->anchor_right) ui_edit_write_key_bool(f, "anchor_right", wd->anchor_right);
		if (wd->anchor_bottom) ui_edit_write_key_bool(f, "anchor_bottom", wd->anchor_bottom);
		}
}

static gpointer ui_edit_widget_read(UIData *ui, WidgetData *wd, GList *list)
{
	if (!wd || !wd->od || !wd->od->func_edit_read) return NULL;

	return wd->od->func_edit_read(ui, wd, list);
}

static gint ui_edit_widget_is_removable(WidgetData *wd)
{
	if (!wd || !wd->od || !wd->od->func_edit_removable) return TRUE;

	return wd->od->func_edit_removable(wd->widget);
}

void ui_edit_widget_add_finish(EditData *ed, WidgetData *wd, const gchar *image, const gchar *data)
{
	ui_widget_set_data(wd, "data", data);
	ui_widget_set_data(wd, "image", image);

	if (wd->od && wd->od->func_init)
		{
		wd->od->func_init(wd->widget, wd->key, ed->ui);
		}
	if (wd->od && wd->od->is_visible)
		{
		gint x, y, w, h, dmy;

		if (ui_widget_get_geometry(wd, &dmy, &dmy, &w, &h))
			{
			if (w > ed->ui->skin->width || h > ed->ui->skin->height)
				{
				ed->ui->skin->widget_list = g_list_remove(ed->ui->skin->widget_list, wd);
				ui_widget_free(wd);
				warning_dialog(_("Widget too big"), _("The widget is too big, the width or height exceeds the background size."));
				return;
				}

			x = ((ed->ui->skin->width - w) / 2);
			y = ((ed->ui->skin->height - h) / 2);

			ui_widget_set_coord(ed->ui, wd, x, y, TRUE);
			}
		}

	widget_list_reset(ed);
	widget_list_set_widget(ed, wd);
}

/* This will completely redraw a widget, updating it's back and redraw all widgets that overlap it
 * add_w and add_h are for extending the redraw region, useful when sizing a widget down - to update
 * the old area it no longer inhabits.
 */
void ui_edit_widget_resync(UIData *ui, WidgetData *wd, gint highlight, gint add_w, gint add_h)
{
	gint x, y, w, h;

	if (!wd || !(ui_widget_get_geometry(wd, &x, &y, &w, &h))) return;

	/* do not allow negative add's */
	add_w = MAX(0, add_w);
	add_h = MAX(0, add_h);

	if (highlight) ui_edit_widget_draw_highlight(ui, wd, FALSE);
	pixbuf_copy_area(ui->skin->overlay, x, y,
                         ui->skin->pixbuf, x, y, w + add_w, h + add_h, FALSE);
	ui_widget_sync_back(ui, wd);
	ui_display_redraw_area(ui, x, y, w + add_w, h + add_h);
	if (highlight) ui_edit_widget_draw_highlight(ui, wd, TRUE);
}

/*
 *-----------------------------------------------------------------------------
 * skin saving utils
 *-----------------------------------------------------------------------------
 */

gchar *ui_edit_copy_unique_file(const gchar *src, const gchar *dir, GdkPixbuf *pb,
				const gchar *hint1, const gchar *hint2)
{
	gchar *dest;
	gchar *name;

	if (!dir || !isdir(dir)) return NULL;

	if (!src && !pb) return NULL;

	name = NULL;

	if (!src)
		{
		/* create one from the pixbuf */
		gchar *buf;
		gchar *new;

		buf = g_strconcat(dir, "/", hint1, hint2, NULL);
		new = unique_filename(buf, ".png", "_", FALSE);
		g_free(buf);

		if (new)
			{
			if (debug_mode) printf("Save file is %s\n", new);

			if (!pixbuf_to_file_as_png(pb, new))
				{
				printf("Failed to write png \"%s\"\n", new);
				}

			name = g_strdup(filename_from_path(new));

			g_free(new);
			}

		return name;
		}

	if (!isfile(src)) return NULL;

	dest = g_strconcat(dir, "/", filename_from_path(src), NULL);

	if (strcmp(src, dest) == 0)
		{
		/* same file */
		name = g_strdup(filename_from_path(dest));
		}
	else if (isfile(dest))
		{
		/* dest exists */

		if (filesize(src) == filesize(dest) && checksum_simple(src) == checksum_simple(dest))
			{
			/* identical files (i hope) */
			name = g_strdup(filename_from_path(dest));
			}
		else
			{
			gchar *new;
			new = unique_filename(dest, NULL, "_", FALSE);
			if (new)
				{
				if (copy_file(src, new))
					{
					name = g_strdup(filename_from_path(new));
					}
				else
					{
					printf("failed to copy \"%s\" to \"%s\"\n", src, dest);
					}
				g_free(new);
				}
			}
		}
	else if (copy_file(src, dest))
		{
		name = g_strdup(filename_from_path(dest));
		}
	else
		{
		printf("failed to copy \"%s\" to \"%s\"\n", src, dest);
		}

	g_free(dest);

	return name;
}

void ui_edit_write_section(FILE *f, const gchar *section, const gchar *subsection)
{
	if (subsection)
		fprintf(f, "\n[%s_%s]\n", section, subsection);
	else
		fprintf(f, "\n[%s]\n", section);
}

void ui_edit_write_key_char(FILE *f, const gchar *key, const gchar *value)
{
	fprintf(f, "%s = %s\n", key, value ? value : "");
}

void ui_edit_write_key_int(FILE *f, const gchar *key, gint value)
{
	fprintf(f, "%s = %d\n", key, value);
}

void ui_edit_write_key_bool(FILE *f, const gchar *key, gint value)
{
	fprintf(f, "%s = %s\n", key, value ? "TRUE" : "FALSE");
}

void ui_edit_widget_set_path_key(WidgetData *wd, const gchar *key, const gchar *base, const gchar *name)
{
	gchar *buf;

	buf = g_strconcat(base, "/", name, NULL);
	ui_widget_set_data(wd, key, buf);
	g_free(buf);
}

gint ui_edit_save(EditData *ed, const gchar *skin_dir, const gchar *data_filename)
{
	SkinData *skin = ed->ui->skin;
	gchar *data_path;
	FILE *f;
	gchar *image;
	gchar *mask = NULL;
	GList *work;
	time_t ts;

	if (!skin_dir) return FALSE;

	if (!data_filename) data_filename = "skindata";

	/* ugly, but anchored widgets need to properly save coordinates */
	skin_resize(ed->ui, skin->width_def, skin->height_def);

	/* try to create skin dir, if it doesn't exist */
	if (!isdir(skin_dir))
		{
		printf("creating dir \"%s\"...\n", skin_dir);
		if (mkdir (skin_dir, 0755) < 0)
			{
			printf("fail to create dir \"%s\"\n", skin_dir);
			return FALSE;
			}
		}

	/* create / open skindata */

	data_path = g_strconcat(skin_dir, "/", data_filename, NULL);
	if (isfile(data_path))
		{
		gchar *bak;
		gchar *new;

		printf("backing up data \"%s\"...\n", data_path);
		bak = g_strconcat(data_path, ".bak", NULL);

		new = unique_filename(bak, NULL, NULL, TRUE);
		if (new)
			{
			if (!move_file(data_path, new))
				{
				printf("Failed to backup \"%s\" to \"%s\"\n", data_path, new);
				}
			g_free(new);
			}

		g_free(bak);
		}

	f = fopen(data_path, "w");
	if (!f)
		{
		printf("Failed to open \"%s\"\n", data_path);
		g_free(data_path);
		return FALSE;
		}

	fprintf(f, "#SLIK skin data file\n\n");
	fprintf(f, "#file created by %s %s (SLIK %s)\n\n", PACKAGE, VERSION, SLIK_VERSION);

	fprintf(f, "#Title:\n");
	fprintf(f, "#Version:\n");
	ts = time(NULL);
	fprintf(f, "#Date stamp: %s", ctime(&ts));
	fprintf(f, "#Author:\n");
	fprintf(f, "#URL:\n");
	fprintf(f, "#Comments:\n");

	ui_edit_write_section(f, "main", NULL);

	image = ui_edit_copy_unique_file(skin->background_filename, skin_dir,
					 skin->real_overlay ? skin->real_overlay : skin->overlay,
					 "background", NULL);
	if (skin->background_mask_filename)
		{
		mask = ui_edit_copy_unique_file(ed->ui->skin->background_mask_filename,
						skin_dir, NULL, "background_mask", NULL);
		}

	ui_edit_write_key_char(f, "image", image);
	ui_edit_write_key_bool(f, "transparent", skin->transparent);

	if (mask) ui_edit_write_key_char(f, "mask", mask);

	/* extra skin stuff */
	ui_edit_write_key_bool(f, "border", skin->has_border);
	if (skin->has_border)
		{
		ui_edit_write_key_int(f, "border_left", skin->border_left);
		ui_edit_write_key_int(f, "border_right", skin->border_right);
		ui_edit_write_key_int(f, "border_top", skin->border_top);
		ui_edit_write_key_int(f, "border_bottom", skin->border_bottom);

		ui_edit_write_key_bool(f, "border_left_stretch", skin->border_left_stretch);
		ui_edit_write_key_bool(f, "border_right_stretch", skin->border_right_stretch);
		ui_edit_write_key_bool(f, "border_top_stretch", skin->border_top_stretch);
		ui_edit_write_key_bool(f, "border_bottom_stretch", skin->border_bottom_stretch);
		}

	if (skin->sizeable) ui_edit_write_key_bool(f, "sizeable", skin->sizeable);
	if (skin->stretch) ui_edit_write_key_bool(f, "stretch", skin->stretch);

	if (skin->sizeable || skin->stretch || skin->has_border ||
	    skin->width_def != gdk_pixbuf_get_width(skin->real_overlay) ||
	    skin->height_def != gdk_pixbuf_get_height(skin->real_overlay) )
		{
		ui_edit_write_key_int(f, "width", skin->width_def);
		ui_edit_write_key_int(f, "height", skin->height_def);
		}

	if (skin->sizeable)
		{
		ui_edit_write_key_int(f, "width_min", skin->width_min);
		ui_edit_write_key_int(f, "width_max", skin->width_max);
		ui_edit_write_key_int(f, "width_increment", skin->width_inc);

		ui_edit_write_key_int(f, "height_min", skin->height_min);
		ui_edit_write_key_int(f, "height_max", skin->height_max);
		ui_edit_write_key_int(f, "height_increment", skin->height_inc);
		}

	/* we must save digits first, so that numbers can always find them,
	 * for this we simply save all non-visible widgets first
	 */
	work = ed->ui->skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;

		if (wd->od && !wd->od->is_visible) ui_edit_widget_write(f, wd, ed->ui->skin, skin_dir);
		work = work->next;
		}

	/* now do everything else */
	work = ed->ui->skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;

		if (wd->od && wd->od->is_visible) ui_edit_widget_write(f, wd, ed->ui->skin, skin_dir);
		work = work->next;
		}

	fclose(f);

	g_free(image);
	g_free(mask);
	g_free(data_path);

	return TRUE;
}

/*
 *-----------------------------------------------------------------------------
 * widget id list utils
 *-----------------------------------------------------------------------------
 */

static GList *ui_edit_widget_id_list_new(void)
{
	WidgetObjectData *od;
	WidgetType t = 0;
	GList *list = NULL;

	while ((od = ui_widget_object_by_type(t)) != NULL)
		{
		EditTypeData *td;

		td = g_new0(EditTypeData, 1);
		td->type = t;
		td->list = NULL;

		list = g_list_append(list, td);

		t++;
		}
	return list;
}

static void ui_edit_widget_id_list_free(GList *list)
{
	GList *work;
	work = list;
	while (work)
		{
		EditTypeData *td;
		WidgetObjectData *od;

		td = work->data;
		work = work->next;

		od = ui_widget_object_by_type(td->type);

		if (od && od->func_edit_free)
			{
			GList *w = td->list;
			while(w)
				{
				od->func_edit_free(w->data);
				w = w->next;
				}
			}

		g_list_free(td->list);
		g_free(td);
		}

	g_list_free(list);
}

static void ui_edit_widget_id_list_read(UIData *ui, WidgetData *wd, GList *list)
{
	GList *work;
	work = list;
	while (work)
		{
		EditTypeData *td = work->data;

		if (wd->type == td->type)
			{
			gpointer data;

			data = ui_edit_widget_read(ui, wd, td->list);
			if (data)
				{
				WidgetObjectData *od;
				od = ui_widget_object_by_type(td->type);
				if (td->page && od && od->func_edit_page_add)
					{
					od->func_edit_page_add(td->page, data);
					}
				td->list = g_list_append(td->list, data);
				}
			return;
			}

		work = work->next;
		}
}

/* this is a very hacky function for use by widgets that know what they are doing (for numbers.) */
void ui_edit_widget_id_list_set_page(GList *list, WidgetType type, GtkWidget *page)
{
	GList *work;
	work = list;
	while (work)
		{
		EditTypeData *td = work->data;

		if (td->type == type)
			{
			td->page = page;
			return;
			}
		work = work->next;
		}
}

/*
 *-----------------------------------------------------------------------------
 * skin background/image (main) data similar to contents of a ui2_xxx_edit.c
 *-----------------------------------------------------------------------------
 */


typedef struct _MainListData MainListData;
struct _MainListData
{
	gchar *image;
	gint transparent;
	gchar *mask;

	gint stretch;

	gint has_border;
	gint border_left;
	gint border_right;
	gint border_top;
	gint border_bottom;
	gint border_left_stretch;
	gint border_right_stretch;
	gint border_top_stretch;
	gint border_bottom_stretch;

	gint sizeable;
	gint width_def;
	gint width_min;
	gint width_max;
	gint width_inc;
	gint height_def;
	gint height_min;
	gint height_max;
	gint height_inc;
};

typedef struct _MainPage MainPage;
struct _MainPage
{
	GtkWidget *image_entry;
	GtkWidget *mask_entry;
	GtkWidget *transparent_button;

	GtkWidget *stretch_button;

	GtkWidget *has_border_button;
	GtkWidget *border_left_spin;
	GtkWidget *border_right_spin;
	GtkWidget *border_top_spin;
	GtkWidget *border_bottom_spin;
	GtkWidget *border_left_stretch_button;
	GtkWidget *border_right_stretch_button;
	GtkWidget *border_top_stretch_button;
	GtkWidget *border_bottom_stretch_button;

	GtkWidget *sizeable_button;
	GtkWidget *width_def_spin;
	GtkWidget *width_min_spin;
	GtkWidget *width_max_spin;
	GtkWidget *width_inc_spin;
	GtkWidget *height_def_spin;
	GtkWidget *height_min_spin;
	GtkWidget *height_max_spin;
	GtkWidget *height_inc_spin;

	GtkWidget *reset_size_button;

	GtkWidget *clist;
	GtkWidget *pixmap;

	UIData *ui;
};

static void edit_main_page_add(EditData *ed, MainListData *md);

static MainListData *edit_main_find(GList *list, const gchar *image)
{
	GList *work;
	work = list;
	while(work)
		{
		MainListData *md = work->data;
		if (strcmp(image, md->image) == 0) return md;
		work = work->next;
		}
	return NULL;
}

static void edit_main_read(EditData *ed, SkinData *skin)
{
	MainListData *md;

	if (!skin->background_filename) return;
	if (edit_main_find(ed->main_list, skin->background_filename)) return;

	md = g_new0(MainListData, 1);
	md->image = g_strdup(skin->background_filename);
	md->transparent = skin->transparent;
	md->mask = g_strdup(skin->background_mask_filename);

	md->stretch = skin->stretch;

	md->has_border = skin->has_border;
	md->border_left = skin->border_left;
	md->border_right = skin->border_right;
	md->border_top = skin->border_top;
	md->border_bottom = skin->border_bottom;
	md->border_left_stretch = skin->border_left_stretch;
	md->border_right_stretch = skin->border_right_stretch;
	md->border_top_stretch = skin->border_top_stretch;
	md->border_bottom_stretch = skin->border_bottom_stretch;

	md->sizeable = skin->sizeable;
	md->width_def = skin->width_def;
	md->width_min = skin->width_min;
	md->width_max = skin->width_max;
	md->width_inc = skin->width_inc;
	md->height_def = skin->height_def;
	md->height_min = skin->height_min;
	md->height_max = skin->height_max;
	md->height_inc = skin->height_inc;

	ed->main_list = g_list_append(ed->main_list, md);
	edit_main_page_add(ed, md);
}

static void edit_main_list_free(GList *list)
{
	GList *work;
	work = list;
	while(work)
		{
		MainListData *md = work->data;
		work = work->next;

		g_free(md->image);
		g_free(md->mask);
		g_free(md);
		}
	g_list_free(list);
}

static void edit_main_page_add_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	MainPage *mp;
	const gchar *image;
	const gchar *mask;
	GdkPixbuf *pb = NULL;
	SkinData *skin;
	gint width, height;
	GList *work;

	mp = ed->main_page;

	image = ui_edit_entry_get(mp->image_entry);
	mask = ui_edit_entry_get(mp->mask_entry);

	if (mask && !isfile(mask))
		{
		warning_dialog(_("Background error"), _("Mask is not valid."));
		return;
		}

	if (image) pb = gdk_pixbuf_new_from_file(image);

	if (!pb)
		{
		warning_dialog(_("Background error"), _("Image is not valid."));
		return;
		}

	width = gdk_pixbuf_get_width(pb);
	height = gdk_pixbuf_get_height(pb);

	skin = ed->ui->skin;

	skin->stretch = ui_edit_toggle_get(mp->stretch_button);

	skin->has_border = ui_edit_toggle_get(mp->has_border_button);
	skin->border_left = ui_edit_spin_get(mp->border_left_spin);
	skin->border_right = ui_edit_spin_get(mp->border_right_spin);
	skin->border_top = ui_edit_spin_get(mp->border_top_spin);
	skin->border_bottom = ui_edit_spin_get(mp->border_bottom_spin);
	skin->border_left_stretch = ui_edit_toggle_get(mp->border_left_stretch_button);
	skin->border_right_stretch = ui_edit_toggle_get(mp->border_right_stretch_button);
	skin->border_top_stretch = ui_edit_toggle_get(mp->border_top_stretch_button);
	skin->border_bottom_stretch = ui_edit_toggle_get(mp->border_bottom_stretch_button);

	skin->sizeable = ui_edit_toggle_get(mp->sizeable_button);
	skin->width_def = ui_edit_spin_get(mp->width_def_spin);
	skin->width_min = ui_edit_spin_get(mp->width_min_spin);
	skin->width_max = ui_edit_spin_get(mp->width_max_spin);
	skin->width_inc = ui_edit_spin_get(mp->width_inc_spin);
	skin->height_def = ui_edit_spin_get(mp->height_def_spin);
	skin->height_min = ui_edit_spin_get(mp->height_min_spin);
	skin->height_max = ui_edit_spin_get(mp->height_max_spin);
	skin->height_inc = ui_edit_spin_get(mp->height_inc_spin);

	if (skin->overlay) gdk_pixbuf_unref(skin->overlay);
	if (skin->real_overlay) gdk_pixbuf_unref(skin->real_overlay);
	skin->overlay = NULL;
	skin->real_overlay = pb;

	if (skin->has_border || skin->sizeable || skin->stretch)
		{
		skin->width = skin->width_def;
		skin->height = skin->height_def;
		}
	else
		{
		skin->width = width;
		skin->height = height;
		}

	g_free(skin->background_filename);
	skin->background_filename = g_strdup(image);

	if (skin->mask) gdk_pixbuf_unref(skin->mask);
	skin->mask = NULL;
	if (mask) skin->mask = gdk_pixbuf_new_from_file(mask);

	g_free(skin->background_mask_filename);
	skin->background_mask_filename = g_strdup(mask);

	skin->transparent = ui_edit_toggle_get(mp->transparent_button);

	work = skin->widget_list;
	while (work)
		{
		WidgetData *wd = work->data;
		gint x, y, w, h;

		if (wd->od && wd->od->is_visible && ui_widget_get_geometry(wd, &x, &y, &w, &h))
			{
			gint nx, ny;

			nx = x;
			ny = y;
			if (wd->anchor_right && nx > skin->width - w) nx = MAX(0, skin->width - w);
			if (wd->anchor_bottom && ny > skin->height - h) ny = MAX(0, skin->height - h);
			if (nx != x || ny != y) ui_widget_set_coord(ed->ui, wd, nx, ny, FALSE);
			}
		work = work->next;
		}

	ui_display_sync_all(ed->ui);
	prop_update(ed, NULL, TRUE);

	tab_completion_append_to_history(mp->image_entry, image);
	tab_completion_append_to_history(mp->mask_entry, mask);
}

static void edit_main_page_sync(MainPage *mp, MainListData *md)
{
	if (!md) return;

	ui_edit_toggle_set(mp->stretch_button, md->stretch);

	ui_edit_toggle_set(mp->has_border_button, md->has_border);
	ui_edit_spin_set(mp->border_left_spin, md->border_left);
	ui_edit_spin_set(mp->border_right_spin, md->border_right);
	ui_edit_spin_set(mp->border_top_spin, md->border_top);
	ui_edit_spin_set(mp->border_bottom_spin, md->border_bottom);
	ui_edit_toggle_set(mp->border_left_stretch_button, md->border_left_stretch);
	ui_edit_toggle_set(mp->border_right_stretch_button, md->border_right_stretch);
	ui_edit_toggle_set(mp->border_top_stretch_button, md->border_top_stretch);
	ui_edit_toggle_set(mp->border_bottom_stretch_button, md->border_bottom_stretch);

	ui_edit_toggle_set(mp->sizeable_button, md->sizeable);
	ui_edit_spin_set(mp->width_def_spin, md->width_def);
	ui_edit_spin_set(mp->width_min_spin, md->width_min);
	ui_edit_spin_set(mp->width_max_spin, md->width_max);
	ui_edit_spin_set(mp->width_inc_spin, md->width_inc);
	ui_edit_spin_set(mp->height_def_spin, md->height_def);
	ui_edit_spin_set(mp->height_min_spin, md->height_min);
	ui_edit_spin_set(mp->height_max_spin, md->height_max);
	ui_edit_spin_set(mp->height_inc_spin, md->height_inc);

	ui_edit_entry_set(mp->image_entry, md->image);
	ui_edit_toggle_set(mp->transparent_button, md->transparent);
	ui_edit_entry_set(mp->mask_entry, md->mask);
}

static void edit_main_page_clist_cb(GtkWidget *clist, gint row, gint col, GdkEvent *event, gpointer data)
{
	EditData *ed = data;
	MainPage *mp;
	MainListData *md;

	mp = ed->main_page;

	md = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	edit_main_page_sync(mp, md);
}

static void edit_main_page_add(EditData *ed, MainListData *md)
{
	MainPage *mp = ed->main_page;

	gchar *buf[8];

	buf[0] = "";
	buf[1] = ui_edit_int_to_text(md->width_def);
	buf[2] = ui_edit_int_to_text(md->height_def);;
	buf[3] = ui_edit_bool_to_text(md->sizeable);
	buf[4] = ui_edit_bool_to_text(md->has_border);
	buf[5] = ui_edit_bool_to_text(md->transparent);
	buf[6] = NULL;

	ui_edit_clist_append(mp->clist, buf, md->image, md);

	g_free(buf[1]);
	g_free(buf[2]);
}

static void edit_main_page_sizer_cb(GtkWidget *widget, gpointer data)
{
	MainPage *mp = data;
	gint stretch;
	gint size;
	gint state;

	stretch = ui_edit_toggle_get(mp->stretch_button);
	size = ui_edit_toggle_get(mp->sizeable_button);
	state = ui_edit_toggle_get(mp->has_border_button);

	ui_edit_frame_sensitive(mp->border_left_stretch_button, state, FALSE);
	ui_edit_frame_sensitive(mp->border_right_stretch_button, state, FALSE);
	ui_edit_frame_sensitive(mp->border_top_stretch_button, state, FALSE);
	ui_edit_frame_sensitive(mp->border_bottom_stretch_button, state, FALSE);

#if 0
	/* we now allow sizing at any time */
	ui_edit_frame_sensitive(mp->width_def_spin, size || stretch || state, FALSE);
#endif
	ui_edit_frame_sensitive(mp->width_min_spin, size, FALSE);
	ui_edit_frame_sensitive(mp->height_min_spin, size, FALSE);
}

static void edit_main_page_pack(MainPage *mp, GtkWidget *vbox)
{
	GtkWidget *hbox2;
	GtkWidget *frame;

	mp->has_border_button = ui_edit_toggle_new(vbox, _("Has border"));

	hbox2 = ui_edit_frame_new(vbox, FALSE, NULL);

	frame = ui_edit_frame_new(hbox2, TRUE, _("Left"));
	mp->border_left_spin = ui_edit_spin_new(frame, _("Size:"), 1, 200, NULL);
	mp->border_left_stretch_button = ui_edit_toggle_new(frame, _("Stretch"));

	frame = ui_edit_frame_new(hbox2, TRUE, _("Right"));
	mp->border_right_spin = ui_edit_spin_new(frame, _("Size:"), 1, 200, NULL);
	mp->border_right_stretch_button = ui_edit_toggle_new(frame, _("Stretch"));

	hbox2 = ui_edit_frame_new(vbox, FALSE, NULL);

	frame = ui_edit_frame_new(hbox2, TRUE, _("Top"));
	mp->border_top_spin = ui_edit_spin_new(frame, _("Size:"), 1, 200, NULL);
	mp->border_top_stretch_button = ui_edit_toggle_new(frame, _("Stretch"));

	frame = ui_edit_frame_new(hbox2, TRUE, _("Bottom"));
	mp->border_bottom_spin = ui_edit_spin_new(frame, _("Size:"), 1, 200, NULL);
	mp->border_bottom_stretch_button = ui_edit_toggle_new(frame, _("Stretch"));

	frame = ui_edit_frame_new(vbox, FALSE, NULL);

	mp->stretch_button = ui_edit_toggle_new(frame, _("Stretch background"));
	mp->sizeable_button = ui_edit_toggle_new(frame, _("Allow resize"));

	frame = ui_edit_frame_new(vbox, TRUE, _("Default size"));
	mp->width_def_spin = ui_edit_spin_new(frame, _("Width:"), 8, 2000, NULL);
	mp->height_def_spin = ui_edit_spin_new(frame, _("Height:"), 8, 2000, NULL);

	mp->reset_size_button = gtk_button_new_with_label(_("Reset to image"));
	gtk_box_pack_end(GTK_BOX(frame), mp->reset_size_button, FALSE, FALSE, 0);
	gtk_widget_show(mp->reset_size_button);

	frame = ui_edit_frame_new(vbox, TRUE, _("Width attributes"));
	mp->width_min_spin = ui_edit_spin_new(frame, _("Min:"), 8, 2000, NULL);
	mp->width_max_spin = ui_edit_spin_new(frame, _("Max:"), 8, 2000, NULL);
	mp->width_inc_spin = ui_edit_spin_new(frame, _("Increment:"), 1, 2000, NULL);

	frame = ui_edit_frame_new(vbox, TRUE, _("Height attributes"));
	mp->height_min_spin = ui_edit_spin_new(frame, _("Min:"), 8, 2000, NULL);
	mp->height_max_spin = ui_edit_spin_new(frame, _("Max:"), 8, 2000, NULL);
	mp->height_inc_spin = ui_edit_spin_new(frame, _("Increment:"), 1, 2000, NULL);
}

static GtkWidget *edit_main_page_new(EditData *ed)
{
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *button;
	MainPage *mp;
	gchar *titles[] = { _("Image"), _("Width"), _("Height"), _("Sizeable"), _("Border"), _("Transparent"), NULL };

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
	gtk_widget_show(vbox);

	mp = g_new0(MainPage, 1);

	edit_main_page_pack(mp, vbox);

	gtk_signal_connect(GTK_OBJECT(mp->has_border_button), "clicked", edit_main_page_sizer_cb, mp);
	gtk_signal_connect(GTK_OBJECT(mp->stretch_button), "clicked", edit_main_page_sizer_cb, mp);
	gtk_signal_connect(GTK_OBJECT(mp->sizeable_button), "clicked", edit_main_page_sizer_cb, mp);

	gtk_widget_set_sensitive(mp->reset_size_button, FALSE);

	mp->image_entry = ui_edit_path_entry_new(vbox, _("Image:"), "SLIK_main_image");
	mp->pixmap = ui_edit_pixmap_new(vbox);
	ui_edit_path_entry_connect_pixmap(mp->image_entry, mp->pixmap);
	mp->mask_entry = ui_edit_path_entry_new(vbox, _("Mask:"), "SLIK_main_mask_image");
	mp->transparent_button = ui_edit_toggle_new(vbox, _("Transparent"));

	button = gtk_button_new_with_label(_("Change"));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", edit_main_page_add_cb, ed);
	gtk_widget_show(button);

	mp->clist = ui_edit_clist_new(hbox, titles, 6);
	gtk_signal_connect(GTK_OBJECT(mp->clist), "select_row", edit_main_page_clist_cb, ed);

	edit_main_page_sizer_cb(NULL, mp);

	gtk_widget_show(hbox);
	ed->main_page = mp;

	return hbox;
}

static void edit_main_page_free(MainPage *mp)
{
	g_free(mp);
}

static void edit_main_set_spin(MainPage *mp)
{
	SkinData *skin = mp->ui->skin;

	ui_edit_spin_set_blocking(mp->border_left_spin, skin->border_left, mp);
	ui_edit_spin_set_blocking(mp->border_right_spin, skin->border_right, mp);
	ui_edit_spin_set_blocking(mp->border_top_spin, skin->border_top, mp);
	ui_edit_spin_set_blocking(mp->border_bottom_spin, skin->border_bottom, mp);

	ui_edit_spin_set_blocking(mp->width_def_spin, skin->width_def, mp);
	ui_edit_spin_set_blocking(mp->width_min_spin, skin->width_min, mp);
	ui_edit_spin_set_blocking(mp->width_max_spin, skin->width_max, mp);
	ui_edit_spin_set_blocking(mp->width_inc_spin, skin->width_inc, mp);
	ui_edit_spin_set_blocking(mp->height_def_spin, skin->height_def, mp);
	ui_edit_spin_set_blocking(mp->height_min_spin, skin->height_min, mp);
	ui_edit_spin_set_blocking(mp->height_max_spin, skin->height_max, mp);
	ui_edit_spin_set_blocking(mp->height_inc_spin, skin->height_inc, mp);
}

static void edit_main_props_force_redraw(MainPage *mp)
{
	SkinData *skin = mp->ui->skin;
	gint s;

	/* We must make it look sizeable so that toggling stretch/borders
	 * will still update the overlay.
         */

	s = skin->sizeable;
	skin->sizeable = TRUE;
	ui_display_sync_all(mp->ui);
	skin->sizeable = s;
}

static void edit_main_props_border_cb(GtkWidget *widget, gpointer data)
{
	MainPage *mp = data;
	SkinData *skin;

	skin = mp->ui->skin;

	skin->has_border = ui_edit_toggle_get(mp->has_border_button);

	edit_main_page_sizer_cb(NULL, mp);
	edit_main_props_force_redraw(mp);
}

static void edit_main_props_stretch_cb(GtkWidget *widget, gpointer data)
{
	MainPage *mp = data;
	SkinData *skin;

	skin = mp->ui->skin;

	skin->stretch = ui_edit_toggle_get(mp->stretch_button);

	edit_main_page_sizer_cb(NULL, mp);
	edit_main_props_force_redraw(mp);
}

static void edit_main_props_sizeable_cb(GtkWidget *widget, gpointer data)
{
	MainPage *mp = data;
	SkinData *skin;

	skin = mp->ui->skin;

	skin->sizeable = ui_edit_toggle_get(mp->sizeable_button);
	if (!skin->sizeable)
		{
		skin->width_def = skin->width;
		skin->height_def = skin->height;
		edit_main_set_spin(mp);
		}

	edit_main_page_sizer_cb(NULL, mp);
}

static void edit_main_props_bs_cb(GtkWidget *widget, gpointer data)
{
	MainPage *mp = data;
	SkinData *skin;

	skin = mp->ui->skin;

	skin->border_left_stretch = ui_edit_toggle_get(mp->border_left_stretch_button);
	skin->border_right_stretch = ui_edit_toggle_get(mp->border_right_stretch_button);
	skin->border_top_stretch = ui_edit_toggle_get(mp->border_top_stretch_button);
	skin->border_bottom_stretch = ui_edit_toggle_get(mp->border_bottom_stretch_button);

	ui_display_sync_all(mp->ui);
}

static void edit_main_props_spin_cb(GtkAdjustment *adj, gpointer data)
{
	MainPage *mp = data;
	SkinData *skin;
	gint w, h;

	skin = mp->ui->skin;

	skin->border_left = ui_edit_spin_get(mp->border_left_spin);
	skin->border_right = ui_edit_spin_get(mp->border_right_spin);
	skin->border_top = ui_edit_spin_get(mp->border_top_spin);
	skin->border_bottom = ui_edit_spin_get(mp->border_bottom_spin);

	skin->width_def = ui_edit_spin_get(mp->width_def_spin);
	skin->width_min = ui_edit_spin_get(mp->width_min_spin);
	skin->width_max = ui_edit_spin_get(mp->width_max_spin);
	skin->width_inc = ui_edit_spin_get(mp->width_inc_spin);
	skin->height_def = ui_edit_spin_get(mp->height_def_spin);
	skin->height_min = ui_edit_spin_get(mp->height_min_spin);
	skin->height_max = ui_edit_spin_get(mp->height_max_spin);
	skin->height_inc = ui_edit_spin_get(mp->height_inc_spin);

	w = gdk_pixbuf_get_width(skin->real_overlay) - 1;
	h = gdk_pixbuf_get_height(skin->real_overlay) - 1;

	if (adj == GTK_SPIN_BUTTON(mp->border_left_spin)->adjustment)
		{
		if (skin->border_right > w - skin->border_left) skin->border_right = w - skin->border_left;
		ui_display_sync_all(mp->ui);
		}
	if (adj == GTK_SPIN_BUTTON(mp->border_right_spin)->adjustment)
		{
		if (skin->border_left > w - skin->border_right) skin->border_left = w - skin->border_right;
		ui_display_sync_all(mp->ui);
		}
	if (adj == GTK_SPIN_BUTTON(mp->border_top_spin)->adjustment)
		{
		if (skin->border_bottom > h - skin->border_top) skin->border_bottom = w - skin->border_top;
		ui_display_sync_all(mp->ui);
		}
	if (adj == GTK_SPIN_BUTTON(mp->border_bottom_spin)->adjustment)
		{
		if (skin->border_top > h - skin->border_bottom) skin->border_top = w - skin->border_bottom;
		ui_display_sync_all(mp->ui);
		}

	if (adj == GTK_SPIN_BUTTON(mp->width_def_spin)->adjustment)
		{
		if (skin->width_min > skin->width_def) skin->width_min = skin->width_def;
		if (skin->width_max < skin->width_def) skin->width_max = skin->width_def;
		skin_resize(mp->ui, skin->width_def, skin->height_def);
		}
	if (adj == GTK_SPIN_BUTTON(mp->height_def_spin)->adjustment)
		{
		if (skin->height_min > skin->height_def) skin->height_min = skin->height_def;
		if (skin->height_max < skin->height_def) skin->height_max = skin->height_def;
		skin_resize(mp->ui, skin->width_def, skin->height_def);
		}

	if (adj == GTK_SPIN_BUTTON(mp->width_min_spin)->adjustment)
		{
		if (skin->width_def < skin->width_min) skin->width_def = skin->width_min;
		if (skin->width_max < skin->width_min) skin->width_max = skin->width_min;
		skin_resize(mp->ui, skin->width_min, skin->height_min);
		}
	if (adj == GTK_SPIN_BUTTON(mp->height_min_spin)->adjustment)
		{
		if (skin->height_def < skin->height_min) skin->height_def = skin->height_min;
		if (skin->height_max < skin->height_min) skin->height_max = skin->height_min;
		skin_resize(mp->ui, skin->width_min, skin->height_min);
		}

	if (adj == GTK_SPIN_BUTTON(mp->width_max_spin)->adjustment)
		{
		if (skin->width_def > skin->width_max) skin->width_def = skin->width_max;
		if (skin->width_min > skin->width_max) skin->width_min = skin->width_max;
		skin_resize(mp->ui, skin->width_max, skin->height_max);
		}
	if (adj == GTK_SPIN_BUTTON(mp->height_max_spin)->adjustment)
		{
		if (skin->height_def > skin->height_max) skin->height_def = skin->height_max;
		if (skin->height_min > skin->height_max) skin->height_min = skin->height_max;
		skin_resize(mp->ui, skin->width_max, skin->height_max);
		}

	edit_main_set_spin(mp);
}

static void edit_main_connect_spin(GtkWidget *spin, MainPage *mp)
{
	GtkAdjustment *adj;

	adj =gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed", edit_main_props_spin_cb, mp);
}

static void edit_main_props_rest_cb(GtkWidget *widget, gpointer data)
{
	MainPage *mp = data;
	SkinData *skin;

	skin = mp->ui->skin;

	ui_edit_spin_set(mp->width_def_spin, gdk_pixbuf_get_width(skin->real_overlay));
	ui_edit_spin_set(mp->height_def_spin, gdk_pixbuf_get_height(skin->real_overlay));
}

static gpointer edit_main_props(UIData *ui, GtkWidget *vbox, gpointer detail)
{
	MainPage *mp = detail;
	SkinData *skin;

	skin = ui->skin;

	if (!mp)
                {
		GtkAdjustment *adj;
		gint w, h;

		mp = g_new0(MainPage, 1);
		mp->ui = ui;

		edit_main_page_pack(mp, vbox);

		w = gdk_pixbuf_get_width(skin->real_overlay) - 2;
		h = gdk_pixbuf_get_height(skin->real_overlay) - 2;

		adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(mp->border_left_spin));
		adj->upper = (gfloat)w;

		adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(mp->border_right_spin));
		adj->upper = (gfloat)w;

		adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(mp->border_top_spin));
		adj->upper = (gfloat)h;

		adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(mp->border_bottom_spin));
		adj->upper = (gfloat)h;

		gtk_signal_connect(GTK_OBJECT(mp->has_border_button), "clicked", edit_main_props_border_cb, mp);
		gtk_signal_connect(GTK_OBJECT(mp->stretch_button), "clicked", edit_main_props_stretch_cb, mp);
		gtk_signal_connect(GTK_OBJECT(mp->sizeable_button), "clicked", edit_main_props_sizeable_cb, mp);

		gtk_signal_connect(GTK_OBJECT(mp->border_left_stretch_button), "clicked", edit_main_props_bs_cb, mp);
		gtk_signal_connect(GTK_OBJECT(mp->border_right_stretch_button), "clicked", edit_main_props_bs_cb, mp);
		gtk_signal_connect(GTK_OBJECT(mp->border_top_stretch_button), "clicked", edit_main_props_bs_cb, mp);
		gtk_signal_connect(GTK_OBJECT(mp->border_bottom_stretch_button), "clicked", edit_main_props_bs_cb, mp);

		edit_main_connect_spin(mp->border_left_spin, mp);
		edit_main_connect_spin(mp->border_right_spin, mp);
		edit_main_connect_spin(mp->border_top_spin, mp);
		edit_main_connect_spin(mp->border_bottom_spin, mp);

		edit_main_connect_spin(mp->width_def_spin, mp);
		edit_main_connect_spin(mp->width_min_spin, mp);
		edit_main_connect_spin(mp->width_max_spin, mp);
		edit_main_connect_spin(mp->width_inc_spin, mp);
		edit_main_connect_spin(mp->height_def_spin, mp);
		edit_main_connect_spin(mp->height_min_spin, mp);
		edit_main_connect_spin(mp->height_max_spin, mp);
		edit_main_connect_spin(mp->height_inc_spin, mp);

		gtk_signal_connect(GTK_OBJECT(mp->reset_size_button), "clicked", edit_main_props_rest_cb, mp);
                }

	ui_edit_toggle_set_blocking(mp->stretch_button, skin->stretch, mp);

	ui_edit_toggle_set_blocking(mp->has_border_button, skin->has_border, mp);
	ui_edit_toggle_set_blocking(mp->border_left_stretch_button, skin->border_left_stretch, mp);
	ui_edit_toggle_set_blocking(mp->border_right_stretch_button, skin->border_right_stretch, mp);
	ui_edit_toggle_set_blocking(mp->border_top_stretch_button, skin->border_top_stretch, mp);
	ui_edit_toggle_set_blocking(mp->border_bottom_stretch_button, skin->border_bottom_stretch, mp);

	ui_edit_toggle_set_blocking(mp->sizeable_button, skin->sizeable, mp);

	edit_main_set_spin(mp);

	edit_main_page_sizer_cb(NULL, mp);

	return mp;
}

/*
 *-----------------------------------------------------------------------------
 * widget read/import utils
 *-----------------------------------------------------------------------------
 */

static gint ui_edit_import_skin_keys(EditData *ed, const gchar *path)
{
	SkinData *skin;
	GList *work;
	gchar *dir;

	if (debug_mode) printf("Editor parsing skin \"%s\"...\n", path);

	dir = remove_level_from_path(path);
	skin = skin_parse(dir, path, TRUE);
	g_free(dir);

	if (!skin) return FALSE;

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;
		ui_edit_widget_id_list_read(ed->ui, wd, ed->widget_id_list);
		work = work->next;
		}

	edit_main_read(ed, skin);

	skin_free(skin);

	return TRUE;
}

/*
 *-----------------------------------------------------------------------------
 * widget properties interface
 *-----------------------------------------------------------------------------
 */

/* old_wd is used to check that is is an update of the background props */
static void prop_sync(EditData *ed, WidgetData *wd, WidgetData *old_wd)
{
	if (ed->detail_data && ((!wd && old_wd != NULL) || (wd && ed->detail_type != wd->type)) )
		{
		g_free(ed->detail_data);
		ed->detail_data = NULL;
		ed->detail_type = -1;
		}

	if (!ed->detail_data)
		{
		if (ed->details) gtk_widget_destroy(ed->details);
		ed->details = NULL;
		}

	if (!ed->details)
		{
		ed->details = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(ed->detail_vbox), ed->details, FALSE, FALSE, 0);
		}

	if (wd && wd->od && wd->od->func_edit_props)
		{
		ed->detail_data = wd->od->func_edit_props(ed->ui, wd, ed->details, ed->detail_data);
		ed->detail_type = wd->type;
		}
	else if (!wd)
		{
		ed->detail_data = edit_main_props(ed->ui, ed->details, ed->detail_data);
		ed->detail_type = -1;
		}

	if (!GTK_WIDGET_VISIBLE(ed->details)) gtk_widget_show(ed->details);
}

static void prop_set_coord(EditData *ed, gint x, gint y, gint max_x, gint max_y)
{
	GtkAdjustment *xa;
	GtkAdjustment *ya;

	xa = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(ed->spin_x));
	ya = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(ed->spin_y));

	gtk_signal_handler_block_by_data(GTK_OBJECT(xa), ed);
	xa->upper = (gfloat)max_x;
	xa->value = (gfloat)x;
	gtk_adjustment_value_changed(xa);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(xa), ed);

	gtk_signal_handler_block_by_data(GTK_OBJECT(ya), ed);
	ya->upper = (gfloat)max_y;
	ya->value = (gfloat)y;
	gtk_adjustment_value_changed(ya);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(ya), ed);
}

static void prop_set_sensitive(EditData *ed, gint state)
{
	ui_edit_frame_sensitive(ed->key_combo_entry, state, TRUE);
	ui_edit_frame_sensitive(ed->data_entry, state, TRUE);
	ui_edit_frame_sensitive(ed->text_id_entry, state, TRUE);

	ui_edit_frame_sensitive(ed->spin_x, state, TRUE);
	ui_edit_frame_sensitive(ed->spin_y, state, TRUE);
	ui_edit_frame_sensitive(ed->anchor_right_button, state, FALSE);
}

static void prop_update(EditData *ed, WidgetData *wd, gint force)
{
	gchar *buf;
	const gchar *ckey;
	const gchar *cdata;
	WidgetType ctype;
	const gchar *cimage;
	const gchar *cmask;
	gint car, cab;
	gint x, y, w, h;
	WidgetData *old_wd;
	GdkPixbuf *pb;

	if (wd)
		{
		if (!ui_widget_get_geometry(wd, &x, &y, &w, &h))
			{
			x = y = w = h = 0;
			}
		}
	else
		{
		x = y = 0;
		w = ed->ui->skin->width;
		h = ed->ui->skin->height;
		}

	buf = g_strdup_printf(_("%-12s w: %-3d h: %-3d"), wd ? ui_widget_type_to_text(wd->type) : _("skin size"), w, h);
	gtk_label_set_text(GTK_LABEL(ed->description), buf);
	g_free(buf);

	prop_set_coord(ed, x, y, ed->ui->skin->width - w, ed->ui->skin->height - h);

	old_wd = ed->info_widget;

	ed->info_widget = wd;

	prop_sync(ed, wd, old_wd);

	if (old_wd == ed->info_widget && !force) return;

	if (old_wd != ed->info_widget &&
	    ( ((old_wd == NULL) != (ed->info_widget == NULL)) || (old_wd && ed->info_widget && old_wd->type != ed->info_widget->type)) )
		help_window_scroll(ed, ed->info_widget ? ed->info_widget->type : -2);

	if (wd)
		{
		pb = ui_edit_widget_get_pixbuf(wd);

		ckey = wd->key;
		cdata = ui_widget_get_data(wd, "data");
		ctype = wd->type;
		cimage = ui_widget_get_data(wd, "image");
		cmask = ui_widget_get_data(wd, "clip_mask");

		car = wd->anchor_right;
		cab = wd->anchor_bottom;
		}
	else
		{
		pb = ed->ui->skin->real_overlay;

		ckey = "";
		cdata = NULL;
		ctype = -1;
		cimage = ed->ui->skin->background_filename;
		cmask = ed->ui->skin->background_mask_filename;

		car = FALSE;
		cab = FALSE;
		}

	if (pb)
		{
		GdkPixmap *pixmap;
		GdkBitmap *mask;
		gdk_pixbuf_render_pixmap_and_mask(pb, &pixmap, &mask, 1);
		gtk_pixmap_set(GTK_PIXMAP(ed->widget_pixmap), pixmap, mask);
		gdk_pixmap_unref(pixmap);
		if (mask) gdk_bitmap_unref(mask);
		}
	else
		{
		ui_edit_pixmap_blank(ed->widget_pixmap);
		}

	ui_edit_entry_set(GTK_COMBO(ed->key_combo_entry)->entry, ckey);
	ui_edit_entry_set(ed->data_entry, cdata);
	ui_edit_entry_set(ed->text_id_entry, (wd && wd->text_id) ? wd->text_id : "");
	ui_edit_key_entry_set_keys(ed->key_combo_entry, ed->ui, ctype, ed);

	ui_edit_entry_set(ed->image_entry, cimage);
	ui_edit_entry_set(ed->clip_mask_entry, cmask);

	ui_edit_toggle_set(ed->anchor_right_button, car);
	ui_edit_toggle_set(ed->anchor_bottom_button, cab);

	prop_set_sensitive(ed, (wd != NULL));
}

static void prop_clear(EditData *ed)
{
	ed->info_widget = NULL;

	if (ed->detail_data)
		{
		g_free(ed->detail_data);
		ed->detail_data = NULL;
		ed->detail_type = -1;
		}

	if (ed->details)
		{
		gtk_widget_destroy(ed->details);
		ed->details = NULL;
		}
}


/*
 *-----------------------------------------------------------------------------
 * widget properties callbacks
 *-----------------------------------------------------------------------------
 */

static void prop_spin_x_cb(GtkAdjustment *adjustment, gpointer data)
{
	EditData *ed = data;
	gint dmy, ox, x, y;

	if (!ed->info_widget || !ed->info_widget->od || !ed->info_widget->od->is_visible ||
	    !ui_widget_get_geometry(ed->info_widget, &ox, &y, &dmy, &dmy)) return;

	x = (gint)GTK_ADJUSTMENT(adjustment)->value;
	if (ox == x) return;

	ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, FALSE);
	edit_widget_set_coord(ed, ed->info_widget, x, y);
	ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, TRUE);
}

static void prop_spin_y_cb(GtkAdjustment *adjustment, gpointer data)
{
	EditData *ed = data;
	gint dmy, x, oy, y;

	if (!ed->info_widget || !ed->info_widget->od || !ed->info_widget->od->is_visible ||
	    !ui_widget_get_geometry(ed->info_widget, &x, &oy, &dmy, &dmy)) return;

	y = (gint)GTK_ADJUSTMENT(adjustment)->value;
	if (oy == y) return;

	ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, FALSE);
	edit_widget_set_coord(ed, ed->info_widget, x, y);
	ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, TRUE);
}

static void prop_key_entry_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	WidgetData *wd;
	gchar *buf;

	if (!ed->info_widget) return;

	wd = ed->info_widget;

	buf = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(ed->key_combo_entry)->entry));
	if (buf && strlen(buf) > 0 && strcmp (buf, wd->key) != 0)
		{
		/* reset the od to original */
		wd->od = ui_widget_object_by_type(wd->type);

		/* update any privately registered keys
		 * (assumes standard format <desc>_<key>_<action>)
		 */
		if (wd->od && wd->od->description)
			{
			GList *work;
			gchar *needle;
			gint nl;

			needle = g_strdup_printf("%s_%s_", wd->od->description, wd->key);
			nl = strlen(needle);
			work = ed->ui->register_list;
			while(work)
				{
				RegisterData *rd = work->data;

				if (rd->private && rd->private_widget == wd->widget &&
				    strncmp(rd->key, needle, nl) == 0)
					{
					gchar *t = g_strdup_printf("%s_%s_%s", wd->od->description,
								   buf, rd->key + nl);
					g_free(rd->key);
					rd->key = t;
					}

				work = work->next;
				}
			g_free(needle);
			}

		/* set new key */
		g_free(wd->key);
		wd->key = g_strdup(buf);

		/* re-finalize, widget may now need it */
		skin_finalize(ed->ui->skin);

		widget_list_update(ed, ed->info_widget, 0, ed->info_widget->key);
		ui_widget_draw(ed->ui, ed->info_widget, TRUE, FALSE);
		}
}

static void prop_data_entry_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	gchar *buf;
	const gchar *d;

	if (!ed->info_widget) return;

	buf = gtk_entry_get_text(GTK_ENTRY(ed->data_entry));
	d = ui_widget_get_data(ed->info_widget, "data");

	if ((buf && strlen(buf) == 0 && !d) ||
	    (buf && d && strcmp(buf, d) == 0))
		{
		return;
		}

	if (buf && strlen(buf) > 0)
		{
		ui_widget_set_data(ed->info_widget, "data", buf);
		widget_list_update(ed, ed->info_widget, 2, buf);
		}
	else
		{
		ui_widget_set_data(ed->info_widget, "data", NULL);
		widget_list_update(ed, ed->info_widget, 2, "");
		}

	ui_widget_draw(ed->ui, ed->info_widget, TRUE, FALSE);
}

static void prop_text_id_entry_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	gchar *buf;
	WidgetData *wd;

	if (!ed->info_widget) return;
	wd = ed->info_widget;

	buf = gtk_entry_get_text(GTK_ENTRY(ed->text_id_entry));

	g_free(wd->text_id);
	wd->text_id = NULL;
	if (buf && strlen(buf) > 0)
		{
		wd->text_id = g_strdup(buf);
		}
}

static void prop_anchor_right_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;

	if (!ed->info_widget) return;

	ed->info_widget->anchor_right = ui_edit_toggle_get(ed->anchor_right_button);
	ui_edit_widget_draw_highlight(ed->ui, ed->info_widget, TRUE);
}

static void prop_anchor_bottom_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;

	if (!ed->info_widget) return;

	ed->info_widget->anchor_bottom = ui_edit_toggle_get(ed->anchor_bottom_button);
	ui_edit_widget_draw_highlight(ed->ui, ed->info_widget, TRUE);
}

/*
 *-----------------------------------------------------------------------------
 * widget list stuff
 *-----------------------------------------------------------------------------
 */

static void widget_list_set_widget(EditData *ed, WidgetData *wd)
{
	gint row;

	row = gtk_clist_find_row_from_data(GTK_CLIST(ed->widget_clist), wd);
	if (row < 0)
		{
		gtk_clist_unselect_all(GTK_CLIST(ed->widget_clist));
		return;
		}

	gtk_clist_select_row(GTK_CLIST(ed->widget_clist), row, 0);

	if (gtk_clist_row_is_visible(GTK_CLIST(ed->widget_clist), row) != GTK_VISIBILITY_FULL)
		{
		gtk_clist_moveto(GTK_CLIST(ed->widget_clist), row, 0, 0.5, 0.0);
		}
}

static void widget_list_update(EditData *ed, WidgetData *wd, gint col, const gchar *text)
{
	gint row;

	row = gtk_clist_find_row_from_data(GTK_CLIST(ed->widget_clist), wd);
	if (row < 0) return;

	gtk_clist_set_text(GTK_CLIST(ed->widget_clist), row, col, text);
}

static void widget_list_reset(EditData *ed)
{
	GList *work;
	gint row;

	gtk_clist_freeze(GTK_CLIST(ed->widget_clist));
	gtk_clist_clear(GTK_CLIST(ed->widget_clist));

	work = ed->ui->skin->widget_list;
	while(work)
		{
		WidgetData *wd;
		gchar *text[4];

		wd = work->data;
		work = work->next;

		/* do not display the dummy widget (so it can not be removed) */
		if (wd->type == dummy_type_id) continue;

		text[0] = wd->key;
		text[1] = (gchar *)ui_widget_type_to_text(wd->type);
		text[2] = (gchar *)ui_widget_get_data(wd, "data");
		text[3] = NULL;

		row = gtk_clist_append(GTK_CLIST(ed->widget_clist), text);
		gtk_clist_set_row_data(GTK_CLIST(ed->widget_clist), row, wd);
		}

	gtk_clist_thaw(GTK_CLIST(ed->widget_clist));
}

static void widget_list_select_cb(GtkWidget *widget, gint row, gint col, GdkEvent *event, gpointer data)
{
	EditData *ed = data;
	WidgetData *wd;

	wd = gtk_clist_get_row_data(GTK_CLIST(ed->widget_clist), row);
	ui_edit_widget_active_set(ed, wd);
	prop_update(ed, wd, FALSE);
}

static void widget_list_row_move_cb(GtkWidget *widget, gint source_row, gint dest_row, gpointer data)
{
	EditData *ed = data;
	WidgetData *wd;

	wd = gtk_clist_get_row_data(GTK_CLIST(ed->widget_clist), source_row);
	
	ed->ui->skin->widget_list = g_list_remove(ed->ui->skin->widget_list, wd);
	ed->ui->skin->widget_list = g_list_insert(ed->ui->skin->widget_list, wd, dest_row);
}

/*
 *-----------------------------------------------------------------------------
 * active widget stuff
 *-----------------------------------------------------------------------------
 */

void ui_edit_widget_draw_highlight(UIData *ui, WidgetData *wd, gint enable)
{
	gint x, y, w, h;
	gint wx, wy, ww, wh;

	if (!wd ||
	    !wd->od ||
	    !wd->od->is_visible ||
	    !ui_widget_get_geometry(wd, &wx, &wy, &ww, &wh) )
		{
		return;
		}

	x = wx - 2;
	y = wy - 2;
	w = ww + 4;
	h = wh + 4;

	if (x < 0)
		{
		w += x;
		x = 0;
		}
	if (y < 0)
		{
		h += y;
		y = 0;
		}
	if (x + w > ui->skin->width) w = ui->skin->width - x;
	if (y + h > ui->skin->height) h = ui->skin->height - y;
	if (w < 1  || h < 1) return;

	if (enable)
		{
		gint ax, ay, bx, by;

		/* bounding box */
		pixbuf_draw_rect_fill(ui->skin->pixbuf,
				      x, y, w, 2,
				      255, 0, 0, 255);
		pixbuf_draw_rect_fill(ui->skin->pixbuf,
				      x, y + h - 2, w, 2,
				      255, 0, 0, 255);
		pixbuf_draw_rect_fill(ui->skin->pixbuf,
				      x, y, 2, h,
				      255, 0, 0, 255);
		pixbuf_draw_rect_fill(ui->skin->pixbuf,
				      x + w - 2, y, 2, h,
				      255, 0, 0, 255);

		/* anchor indicator */
		ax = (wd->anchor_right) ? (x + w - 6) : (x);
		if (ax + 6 > ui->skin->width) ax = ui->skin->width - 6;
		if (ax < 0) ax = 0;

		ay = (wd->anchor_bottom) ? (y + h - 6) : (y);
		if (ay + 6 > ui->skin->height) ay = ui->skin->height - 6;
		if (ay < 0) ay = 0;

		bx = ax;
		by = ay;

		if (wd->anchor_right) ax += 4;
		if (wd->anchor_bottom) by += 4;

		pixbuf_draw_rect_fill(ui->skin->pixbuf,
				      ax, ay, 2, 6,
				      255, 255, 0, 255);
		pixbuf_draw_rect_fill(ui->skin->pixbuf,
				      bx, by, 6, 2,
				      255, 255, 0, 255);
		}
	else
		{
		pixbuf_copy_area(ui->skin->overlay, x, y,
			 	 ui->skin->pixbuf, x, y, w, h, FALSE);
		ui_display_redraw_area(ui, x, y, w, h);
		}
	ui_display_render_area(ui, x, y, w, h);
}

static void ui_edit_widget_active_set(EditData *ed, WidgetData *wd)
{
	if (ed->active_widget == wd) return;

	if (ed->active_widget)
		{
		ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, FALSE);
		}
	
	ed->active_widget = wd;

	if (ed->active_widget)
		{
		ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, TRUE);
		}

	widget_list_set_widget(ed, ed->active_widget);
	if (!ed->active_widget) prop_update(ed, NULL, FALSE);

}

static void edit_widget_set_coord(EditData *ed, WidgetData *wd, gint x, gint y)
{
	ui_widget_set_coord(ed->ui, wd, x, y, TRUE);

	if (wd == ed->active_widget) prop_update(ed, wd, FALSE);
}

/*
 *-----------------------------------------------------------------------------
 * mouse handling (mirrored from ui2_display)
 *-----------------------------------------------------------------------------
 */

static void edit_motion(GtkWidget *w, GdkEventMotion *event, gpointer data)
{
	EditData *ed = data;
	gint x = (gint)event->x;
	gint y = (gint)event->y;

	if (ed->ui->active_widget) return;

	if (ed->in_move && ed->active_widget)
		{
		/* widget drag */
		gint px, py;
		gint wx, wy, ww, wh;

		px = x - ed->press_x;
		py = y - ed->press_y;
		if (ed->active_widget->od && ed->active_widget->od->is_visible &&
		    ui_widget_get_geometry(ed->active_widget, &wx, &wy, &ww, &wh))
			{
			px += ed->press_widget_x;
			py += ed->press_widget_y;
			if (px < 0) px = 0;
			if (py < 0) py = 0;
			if (px > ed->ui->skin->width - ww) px = ed->ui->skin->width - ww;
			if (py > ed->ui->skin->height - wh) py = ed->ui->skin->height - wh;
			if (wx != px || wy != py) edit_widget_set_coord(ed, ed->active_widget, px, py);
			}
		return;
		}

	if (!ed->click_to_focus) ui_edit_widget_active_set(ed, ui_widget_find_by_coord(ed->ui, x, y));
}

static void edit_pressed(GtkWidget *w, GdkEventButton *event, gpointer data)
{
	EditData *ed = data;
	gint x = (gint)event->x;
	gint y = (gint)event->y;
	WidgetData *old_wd;

	if (ed->ui->active_widget) return;

	if (debug_mode) printf("pressed:%d x %d\n", x, y);

	if (event->button != 2 && event->button != 3) return;

	old_wd = ed->active_widget;

	if (ed->click_to_focus)
		{
		ui_edit_widget_active_set(ed, ui_widget_find_by_coord(ed->ui, x, y));
		}

	if ((!ed->click_to_focus && ed->active_widget) ||
	    (ed->click_to_focus && ed->active_widget && ed->active_widget == old_wd) )
		{
		gint dmy;
		/* widget movement */
		ed->in_move = TRUE;
		ed->press_x = x;
		ed->press_y = y;
		if (!ui_widget_get_geometry(ed->active_widget, &ed->press_widget_x, &ed->press_widget_y, &dmy, &dmy))
			{
			ed->press_widget_x = ed->press_widget_y = 0;
			}
		ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, FALSE);
		}

	ed->in_press = TRUE;
}

static void edit_released(GtkWidget *w, GdkEventButton *event, gpointer data)
{
	EditData *ed = data;

	if (ed->ui->active_widget) return;

	if (ed->active_widget)
		{
		ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, TRUE);
		if (debug_mode) printf("edit releasing widget: \"%s\" (%d)\n", ed->active_widget->key, ed->active_widget->type);
		}

	ed->in_press = FALSE;
	ed->in_move = FALSE;
}

static void edit_leave(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	EditData *ed = data;

	if (ed->active_widget)
		{
		/* do nothing ? */
		}
}

static void edit_events_init(EditData *ed)
{
	gtk_signal_connect(GTK_OBJECT(ed->ui->display),"motion_notify_event", edit_motion, ed);
	gtk_signal_connect(GTK_OBJECT(ed->ui->display),"button_press_event", edit_pressed, ed);
	gtk_signal_connect(GTK_OBJECT(ed->ui->display),"button_release_event", edit_released, ed);
	gtk_signal_connect(GTK_OBJECT(ed->ui->display),"leave_notify_event", edit_leave, ed);

	ed->active_widget = NULL;
	ed->in_press = FALSE;
	ed->in_move = FALSE;
}

/*
 *-----------------------------------------------------------------------------
 * overlay
 *-----------------------------------------------------------------------------
 */

static void edit_overlay_press_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	GList *work;
	SkinData *skin;

	skin = ed->ui->skin;

	if (ed->info_widget) ui_edit_widget_draw_highlight(ed->ui, ed->info_widget, FALSE);

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;
		gint x, y, w, h;
		if (wd->od && wd->od->is_visible && ui_widget_get_geometry(wd, &x, &y, &w, &h))
			{
			if (wd->in_bounds)
				{
				pixbuf_draw_rect_fill(skin->pixbuf,
						      x, y, w, h,
						      255, 0, 0, 128);
				}
			else
				{
				if (x + w > skin->width) w = skin->width - x;
				if (x > skin->width - 2)
					{
					w = 2;
					x = skin->width - 2;
					}
				if (x < 0)
					{
					w += x;
					x = 0;
					}
				if (y + h > skin->height) h = skin->height - y;
				if (y > skin->height - 2)
					{
					h = 2;
					y = skin->height - 2;
					}
				if (y < 0)
					{
					h+= y;
					y = 0;
					}
				if (w < 2) w = 2;
				if (h < 2) h = 2;
				pixbuf_draw_rect_fill(skin->pixbuf,
						      x, y, w, h,
						      255, 0, 255, 128);
				}
			if (wd->anchor_right) x += w - 2;
			if (wd->anchor_bottom) y += h - 2;
			pixbuf_draw_rect_fill(skin->pixbuf,
					      x, y, 2, 2,
					      255, 255, 0, 255);
			}
		work = work->next;
		}
	ui_display_render_area(ed->ui, 0, 0, skin->width, skin->height);
}

static void edit_overlay_release_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;

	ui_display_redraw_area(ed->ui, 0, 0, ed->ui->skin->width, ed->ui->skin->height);
	if (ed->info_widget) ui_edit_widget_draw_highlight(ed->ui, ed->info_widget, TRUE);
}

/*
 *-----------------------------------------------------------------------------
 * misc. callbacks
 *-----------------------------------------------------------------------------
 */

static void edit_redraw_cb(GtkWidget *button, gpointer data)
{
	EditData *ed = data;
	skin_sync_back(ed->ui->skin, ed->ui);
	ui_display_sync(ed->ui, FALSE);
	ui_edit_widget_draw_highlight(ed->ui, ed->active_widget, TRUE);
}

static void edit_load_cb(GtkWidget *button, gpointer data)
{
	EditData *ed = data;
	gchar *skin_path;
	const gchar *skin_key;
	SkinData *skin;
	gchar *buf;

	skin_path = remove_trailing_slash(gtk_entry_get_text(GTK_ENTRY(ed->save_path_entry)));
	skin_key = gtk_entry_get_text(GTK_ENTRY(ed->save_key_entry));

	if (!skin_key || strlen(skin_key) == 0) skin_key = NULL;

	buf = g_strconcat(skin_path, "/", skin_key ? skin_key : "skindata", NULL);
	skin = skin_parse(skin_path, buf, TRUE);
	g_free(buf);

	if (skin)
		{
		ui_skin_set(ed->ui, skin);
		g_free(ed->ui->skin_path);
		ed->ui->skin_path = g_strdup(skin_path);

		g_free(ed->ui->skin_mode_key);
		ed->ui->skin_mode_key = g_strdup(skin_key);

		ed->active_widget = NULL;
		prop_clear(ed);
		widget_list_reset(ed);
		prop_update(ed, NULL, TRUE);
		ui_display_sync_all(ed->ui);

		tab_completion_append_to_history(ed->save_path_entry, skin_path);
		}
	else
		{
		warning_dialog(_("Load error"), _("Failed to load skin from path and key specified."));
		}

	g_free(skin_path);
}

static void edit_click_focus_cb(GtkWidget *button, gpointer data)
{
	EditData *ed = data;
	ed->click_to_focus = GTK_TOGGLE_BUTTON(button)->active;
}

static void edit_delete_cb(GtkWidget *button, gpointer data)
{
	EditData *ed = data;
	GList *work;
	WidgetData *wd;
	WidgetData *new_wd;

	if (!ed->active_widget) return;

	wd = ed->active_widget;

	if (!ui_edit_widget_is_removable(wd))
		{
		warning_dialog(_("Widget still referenced"), _("This widget can not be removed,\nother widgets still reference it."));
		return;
		}

	work = g_list_find(ed->ui->skin->widget_list, wd);
	if (!work) return;

	if (work->next)
		{
		new_wd = work->next->data;
		}
	else if (work->prev)
		{
		new_wd = work->prev->data;
		}
	else
		{
		new_wd = NULL;
		}

	ui_edit_widget_active_set(ed, new_wd);
	prop_update(ed, new_wd, TRUE);

	ui_unregister_key_private_for_widget(ed->ui, wd->widget);

	ed->ui->skin->widget_list = g_list_remove(ed->ui->skin->widget_list, wd);

	ui_edit_widget_draw_highlight(ed->ui, wd, FALSE);

	widget_list_reset(ed);
	widget_list_set_widget(ed, new_wd);

	ui_widget_free(wd);
}

static void edit_save_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	gchar *skin_path;
	const gchar *skin_key;

	skin_path = remove_trailing_slash(gtk_entry_get_text(GTK_ENTRY(ed->save_path_entry)));
	skin_key = gtk_entry_get_text(GTK_ENTRY(ed->save_key_entry));

	if (!skin_key || strlen(skin_key) == 0) skin_key = NULL;

	if (ui_edit_save(ed, skin_path, skin_key))
		{
		gchar *buf;

		tab_completion_append_to_history(ed->save_path_entry, skin_path);

		buf = g_strdup_printf(_("Skin successfully saved to \"%s\"\nwith data file \"%s\"."),
				      skin_path, skin_key ? skin_key : "skindata");
		warning_dialog(_("Skin saved"), buf);
		g_free(buf);
		}
	else
		{
		gchar *buf;

		buf = g_strdup_printf(_("Failed to save skin to \"%s\"\nwith data file \"%s\"."),
				      skin_path, skin_key ? skin_key : "skindata");
		warning_dialog(_("Skin save failed"), buf);
		g_free(buf);
		}

	g_free(skin_path);
}

/*
 *-----------------------------------------------------------------------------
 * copy data from the 'parent'
 *-----------------------------------------------------------------------------
 */

static void ui_edit_copy_registry(UIData *ui, UIData *source)
{
	GList *work;

	work = source->register_list;
	while(work)
		{
		RegisterData *srd = work->data;
		RegisterData *rd;

		if (srd->private == FALSE)
			{
			rd = g_new(RegisterData, 1);
			rd->type = srd->type;
			rd->key = g_strdup(srd->key);
			rd->private = FALSE;
			if (srd->callbacks_l == 0 ||
			    (rd->type == button_type_id() &&
				(strcmp(srd->key, "exit") == 0 ||	/* disable keys that should 	*/
				strcmp(srd->key, "iconify") == 0 ||	/* not work in the editor	*/
				strcmp(srd->key, "skin_toggle") == 0) ) )
				{
				rd->callbacks = NULL;
				rd->callbacks_l = 0;
				}
			else
				{
				rd->callbacks = g_memdup(srd->callbacks, srd->callbacks_l);
				rd->callbacks_l = srd->callbacks_l;
				}
			if (debug_mode) printf("copying callbackdata for \"%s\", size = %d\n", rd->key, rd->callbacks_l);

			ui->register_list = g_list_prepend(ui->register_list, rd);
			}

		work = work->next;
		}

	ui->register_list = g_list_reverse(ui->register_list);
}

static SkinData *ui_edit_copy_skin(UIData *source)
{
	/* I'm an idiot, I thought this was gonna be hard, but knowing the skin_path,
	 * just load it instead of painfully trying to go through skin->widget_list
	 * and copying everything.
	 */
	SkinData *skin;

	if (source->skin_path)
		{
		gchar *datafile;

		datafile = g_strconcat(source->skin_path, "/", source->skin_mode_key ? source->skin_mode_key : "skindata", NULL);
		skin = skin_parse(source->skin_path, datafile, TRUE);
		g_free(datafile);
		}
	else
		{
		skin = skin_load_default(source->skin_mode_key);
		}

	if (!skin)
		{
		skin = skin_new();
		}

	return skin;
}

static void ui_edit_copy_ui(UIData *ui, UIData *source)
{
	if (!ui || !source || ui == source) return;

	ui_register_free_all(ui);
	ui_edit_copy_registry(ui, source);

	ui_skin_set(ui, ui_edit_copy_skin(source));

	g_free(ui->skin_path);
	ui->skin_path = g_strdup(source->skin_path);

	g_free(ui->skin_mode_key);
	ui->skin_mode_key = g_strdup(source->skin_mode_key);
}

/*
 *-----------------------------------------------------------------------------
 * 'help' window
 *-----------------------------------------------------------------------------
 */

#define SEARCH_BUFFER_LENGTH 1024

static gint text_buffer_count_lines(const gchar *buf)
{
	const gchar *ptr = buf;
	gint l = 0;

	while(*ptr != '\0')
		{
		if (*ptr == '\n') l++;
		ptr++;
		}
	return l;
}

/* NULL needle return 0, but counts the line total */
static gint text_find(GtkText *haystack, const gchar *needle, gint *line)
{
	gint l;
	gint i;
	gint n = 0;
	gint line_count = 1;

	if (needle) n = strlen(needle);

	l = gtk_text_get_length(haystack);

	for (i = 0; i < l; i += SEARCH_BUFFER_LENGTH)
		{
		gint end;
		gint found = -1;

		end = i + SEARCH_BUFFER_LENGTH + n;
		if (end > l - 1) end = l - 1;

		if (end - i > 0)
			{
			gchar *buf;
			gchar *s = NULL;

			buf = gtk_editable_get_chars(GTK_EDITABLE(haystack), i, end);
			if (buf)
				{
				if (needle) s = strstr(buf, needle);
				if (line) line_count += text_buffer_count_lines(buf);
				}
			if (s) found = i + (s - buf);

			g_free(buf);
			}
		if (found >= 0)
			{
			if (line) *line = line_count;
			return found;
			}
		}

	if (line) *line = line_count;
	return 0;
}

static void text_set_position_from_lines(GtkText *text, gint line, gint line_count)
{
	GtkAdjustment *adj;
        gfloat value;
        
	if (line_count < 2 || line >= line_count) return;

	adj = text->vadj;

	value = adj->upper * ((float)(line - 1) / (float)(line_count - 1));
	gtk_adjustment_set_value(adj, value);
}

static void help_window_scroll(EditData *ed, WidgetType type)
{
	gchar *needle;
	const gchar *key;
	gint p;
	gint line;

	if (!ed->help_follow ||
	    !GTK_WIDGET_VISIBLE(ed->help_window)) return;

	key = ui_widget_type_to_text(type);

	if (type == -2) key = "main";

	if (!key) return;

	needle = g_strdup_printf("[section:%s]", key);

	p = text_find(GTK_TEXT(ed->help_text), needle, &line);

	if (p > 0 && p != gtk_editable_get_position(GTK_EDITABLE(ed->help_text)))
		{
		if (debug_mode) printf("help section %s found at %d\n", needle, p);

		gtk_text_freeze(GTK_TEXT(ed->help_text));
		/* well, GtkText broken, freeze does not work for set_position :( */
		text_set_position_from_lines(GTK_TEXT(ed->help_text), line, ed->help_lines);

		/* above not entirely accurate :( do the set anyway */
		gtk_editable_set_position(GTK_EDITABLE(ed->help_text), p);

		gtk_text_thaw(GTK_TEXT(ed->help_text));
		}

	g_free(needle);
}

static void help_window_load_text(EditData *ed, const gchar *path)
{
	FILE *f;
	gchar s_buf[1024];
	gint l;
	GdkFont *font;

	if (!ed->help_text || !path) return;

	f = fopen(path, "r");
	if (!f)
		{
		printf("unable to open help file %s\n", path);
		return;
		}

	gtk_text_freeze(GTK_TEXT(ed->help_text));
	l = gtk_text_get_length(GTK_TEXT(ed->help_text));
	if (l > 0) gtk_editable_delete_text(GTK_EDITABLE(ed->help_text), 0, l);

	font = gdk_font_load(HELP_WINDOW_FONT);

	while (fgets(s_buf, sizeof(s_buf), f))
		{
		gtk_text_insert(GTK_TEXT(ed->help_text), font, NULL, NULL, s_buf, strlen(s_buf));
		}

	fclose(f);

	if (font) gdk_font_unref(font);

	text_find(GTK_TEXT(ed->help_text), NULL, &ed->help_lines);

	gtk_text_thaw(GTK_TEXT(ed->help_text));
}

static gint help_window_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	EditData *ed = data;
	if (GTK_WIDGET_VISIBLE(ed->help_window))
		{
		gtk_widget_hide(ed->help_window);
		}

	return TRUE;
}

static void help_window_show_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;

	if (GTK_WIDGET_VISIBLE(ed->help_window))
		{
		gdk_window_raise(ed->help_window->window);
		}
	else
		{
		gtk_widget_show(ed->help_window);
		/* add text */

		help_window_load_text(ed, ed->help_path);
		}
}

static void help_window_follow_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;

	ed->help_follow = ui_edit_toggle_get(widget);
	if (ed->help_follow) help_window_scroll(ed, ed->info_widget ? ed->info_widget->type : -1);
}

static void help_window_setup(EditData *ed)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *button;
	GtkWidget *scroll;

	if (ed->help_window) return;

	/* window */

	ed->help_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(ed->help_window), FALSE, TRUE, FALSE);
	gtk_container_border_width (GTK_CONTAINER(ed->help_window), 5);
	gtk_window_set_wmclass(GTK_WINDOW(ed->help_window), "editor_help", "SLIK");

	gtk_widget_set_usize(ed->help_window, HELP_WINDOW_WIDTH, HELP_WINDOW_HEIGHT);

	gtk_window_set_title(GTK_WINDOW(ed->help_window), _("Help - SLIK"));

	gtk_signal_connect(GTK_OBJECT(ed->help_window), "delete_event", (GtkSignalFunc)help_window_delete_cb, ed);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(ed->help_window), vbox);
	gtk_widget_show(vbox);

	/* text window */

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	ed->help_text = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(ed->help_text), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), ed->help_text, TRUE, TRUE, 0);
	gtk_widget_show(ed->help_text);

	scroll = gtk_vscrollbar_new(GTK_TEXT(ed->help_text)->vadj);
	gtk_box_pack_start(GTK_BOX(hbox), scroll, FALSE, FALSE, 0);
	gtk_widget_show(scroll);

	/* extra */

	button = ui_edit_toggle_new(vbox, _("Scroll to follow active widget"));
	ed->help_follow = FALSE;
	ui_edit_toggle_set(button, ed->help_follow);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", help_window_follow_cb, ed);

	window_set_icon(ed->widget_window, ui_slik_logo(), NULL);
}

/*
 *-----------------------------------------------------------------------------
 * 'add' window
 *-----------------------------------------------------------------------------
 */

static void widget_window_add_skin(EditData *ed, const gchar *skin_dir, GtkWidget *label, GtkWidget *progress)
{
	GList *list;
	GList *work;

	if (!path_list(skin_dir, &list, NULL)) return;

	work = list;
	while(work)
		{
		gchar *path;
		const gchar *name;

		path = work->data;
		work = work->next;

		name = filename_from_path(path);

		if (strncmp(name, "skindata", 8) == 0 &&
		    strstr(name, ".bak") == NULL) /* ignore backup files */
			{
			if (label) gtk_label_set_text(GTK_LABEL(label), path);
			if (progress)
				{
				gfloat val;
				val = gtk_progress_get_value(GTK_PROGRESS(progress));
				val += 1.0;
				if (val > 100.0) val = 0.0;
				gtk_progress_set_value(GTK_PROGRESS(progress), val);
				while(gtk_events_pending()) gtk_main_iteration_do(TRUE);
				}
			ui_edit_import_skin_keys(ed, path);
			}
		}

	path_list_free(list);
}

static void widget_window_add_dir(EditData *ed, const gchar *path, GtkWidget *label, GtkWidget *progress)
{
	GList *dir_list = NULL;
	GList *work;

	if (!path_list(path, NULL, &dir_list)) return;

	work = dir_list;
	while(work)
		{
		gchar *skin_dir = work->data;
		widget_window_add_skin(ed, skin_dir, label, progress);
		work = work->next;
		}

	path_list_free(dir_list);
}

static void widget_window_extract_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *progress;
	GtkWidget *label;
	GList *work;

	gtk_widget_set_sensitive(widget, FALSE);

	if (editor_resource_list == NULL)
		{
		printf("editor resource list is empty\n");
		return;
		}

	window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(window), _("Reading skins - SLIK"));

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(window), vbox);
	gtk_widget_show(vbox);

	ui_edit_label_new(vbox, _("Extracting widgets, please wait..."));
	label = ui_edit_label_new(vbox, "");

	progress = gtk_progress_bar_new();
	gtk_progress_set_activity_mode(GTK_PROGRESS(progress), TRUE);
	gtk_progress_set_show_text(GTK_PROGRESS(progress), TRUE);
	gtk_progress_bar_set_activity_step(GTK_PROGRESS_BAR(progress), 10);
	gtk_box_pack_start(GTK_BOX(vbox), progress, FALSE, FALSE, 0);
	gtk_widget_show(progress);
	
	window_set_icon(window, ui_slik_logo(), NULL);
	gtk_widget_show(window);

	work = editor_resource_list;
	while(work)
		{
		gchar *path = work->data;
		work = work->next;
		widget_window_add_dir(ed, path, label, progress);
		}

	gtk_widget_destroy(window);
}

static void widget_window_import_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;
	const gchar *path;

	path = ui_edit_entry_get(ed->import_path_entry);

	if (isdir(path))
		{
		widget_window_add_skin(ed, path, NULL, NULL);
		tab_completion_append_to_history(ed->import_path_entry, path);
		}
	else if (ui_edit_import_skin_keys(ed, path))
		{
		tab_completion_append_to_history(ed->import_path_entry, path);
		}
}

static void widget_window_setup_notebook(EditData *ed)
{
	GtkWidget *page;
	GtkWidget *label;
	GList *work;

	page = edit_main_page_new(ed);
	label = gtk_label_new(_("background"));
	gtk_notebook_append_page(GTK_NOTEBOOK(ed->notebook), page, label);

	work = ed->widget_id_list;
	while(work)
		{
		WidgetObjectData *od;
		EditTypeData *td;

		td = work->data;
		work = work->next;

		od = ui_widget_object_by_type(td->type);
		if (od && od->func_edit_page_new)
			{
			td->page = od->func_edit_page_new(ed);
			label = gtk_label_new(ui_widget_type_to_text(td->type));
			gtk_notebook_append_page(GTK_NOTEBOOK(ed->notebook), td->page, label);
			}
		}
}

static gint widget_window_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	EditData *ed = data;
	if (GTK_WIDGET_VISIBLE(ed->widget_window))
		{
		gtk_widget_hide(ed->widget_window);
		}

	return TRUE;
}

static void widget_window_show_cb(GtkWidget *widget, gpointer data)
{
	EditData *ed = data;

	if (GTK_WIDGET_VISIBLE(ed->widget_window))
		{
		gdk_window_raise(ed->widget_window->window);
		}
	else
		{
		gtk_widget_show(ed->widget_window);
		}
}

static void widget_window_setup(EditData *ed)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *button;
	GtkWidget *combo;
	GtkWidget *sep;

	if (ed->widget_window) return;

	/* window */

	ed->widget_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(ed->widget_window), FALSE, TRUE, FALSE);
	gtk_container_border_width (GTK_CONTAINER(ed->widget_window), 5);
	gtk_window_set_wmclass(GTK_WINDOW(ed->widget_window), "editor_add_widget", "SLIK");

	gtk_window_set_title(GTK_WINDOW(ed->widget_window), _("Add widget - SLIK"));

	gtk_signal_connect(GTK_OBJECT(ed->widget_window), "delete_event", (GtkSignalFunc)widget_window_delete_cb, ed);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(ed->widget_window), vbox);
	gtk_widget_show(vbox);

	/* new widget notebook */

	ed->notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(vbox), ed->notebook, TRUE, TRUE, 0);
	gtk_widget_show(ed->notebook);
	widget_window_setup_notebook(ed);

	hbox = ui_edit_frame_new(vbox, FALSE, NULL);

	combo = tab_completion_new_with_history(&ed->import_path_entry, NULL, "SLIK_import", 16, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
	gtk_widget_show(combo);

	button = gtk_button_new_with_label(_("import"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", widget_window_import_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	sep = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(hbox), sep, FALSE, FALSE, 0);
	gtk_widget_show(sep);

	button = gtk_button_new_with_label(_("generate list from installed skins"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", widget_window_extract_cb, ed);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	window_set_icon(ed->widget_window, ui_slik_logo(), NULL);
}

/*
 *-----------------------------------------------------------------------------
 * dummy widget
 *-----------------------------------------------------------------------------
 */

static void editor_dummy_back_set(gpointer widget, GdkPixbuf *pb)
{
	EditData *ed = widget;

	/* this is used to update widget properties 
	 * (size may have changed on dynamic widgets,
	 *  and this signal is definitely after any size updates)
	 */

	prop_update(ed, ed->info_widget, FALSE);
}

/* we actually need the editor to register a unique type (for the dummy widget) */
static void editor_type_init(void)
{
	WidgetObjectData *od;

	if (dummy_type_id != -1) return;

	od = ui_widget_type_new("editor");
        dummy_type_id = od->type;

	od->func_back = editor_dummy_back_set;
}

/*
 *-----------------------------------------------------------------------------
 * edit window new / close
 *-----------------------------------------------------------------------------
 */

void ui_edit_set_help_path(EditData *ed, const gchar *path)
{
	g_free(ed->help_path);
	ed->help_path = g_strdup(path);
}

void ui_edit_close(EditData *ed)
{
	if (ed->widget_window) gtk_widget_destroy(ed->widget_window);
	if (ed->help_window) gtk_widget_destroy(ed->help_window);
	gtk_widget_destroy(ed->window);

	ui_edit_widget_id_list_free(ed->widget_id_list);
	edit_main_list_free(ed->main_list);
	edit_main_page_free(ed->main_page);

	if (ed->parent) ed->parent->edit = NULL;
	ui_free(ed->ui);

	g_free(ed->detail_data);
	g_free(ed->help_path);

	g_free(ed);
}

static gint ui_edit_destroy_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	EditData *ed = data;
	ui_edit_close(ed);

	return TRUE;
}

EditData *ui_edit_new(UIData *parent, const gchar *class, const gchar *subclass, const gchar *title)
{
	EditData *ed;
	GtkWidget *main_vbox;
	GtkWidget *sep;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkObject *adj;
	GtkWidget *scrolled;
	GtkWidget *button;
	GtkWidget *combo;
	gchar *titles[] = { _("key"), _("type"), _("data"), NULL};
	GList *list;

	if (!parent) return NULL;
	if (parent->edit) return NULL;

	ed = g_new0(EditData, 1);

	ed->click_to_focus = TRUE;

	ed->active_widget = NULL;
	ed->info_widget = NULL;

	ed->details = NULL;
	ed->detail_data = NULL;

	ed->widget_id_list = ui_edit_widget_id_list_new();
	ed->main_list = NULL;

	ed->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(ed->window), FALSE, TRUE, FALSE);
	gtk_container_border_width (GTK_CONTAINER(ed->window), 5);
	gtk_window_set_wmclass(GTK_WINDOW(ed->window), subclass, class);

	if (title)
		{
		gtk_window_set_title(GTK_WINDOW(ed->window), title);
		}
	else
		{
		gtk_window_set_title(GTK_WINDOW(ed->window), _("Skin editor - SLIK"));
		}

	gtk_signal_connect(GTK_OBJECT(ed->window), "delete_event", GTK_SIGNAL_FUNC(ui_edit_destroy_cb), ed);

	main_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(ed->window), main_vbox);
	gtk_widget_show(main_vbox);

	ed->hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), ed->hbox, TRUE, TRUE, 0);
	gtk_widget_show(ed->hbox);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(ed->hbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show(vbox);

	ed->table = gtk_table_new(1, 1, TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), ed->table, TRUE, TRUE, 0);
	gtk_widget_show(ed->table);

	/* utils */

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 0);
	gtk_widget_show(sep);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("help"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", help_window_show_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("overlay"));
	gtk_signal_connect(GTK_OBJECT(button), "pressed", edit_overlay_press_cb, ed);
	gtk_signal_connect(GTK_OBJECT(button), "released", edit_overlay_release_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("redraw"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", edit_redraw_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_check_button_new_with_label(_("click to focus"));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), ed->click_to_focus);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", edit_click_focus_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("add..."));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", widget_window_show_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("remove"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", edit_delete_cb, ed);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	/* widget list */

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	ed->widget_clist = gtk_clist_new_with_titles(3, titles);
	gtk_clist_set_column_auto_resize(GTK_CLIST(ed->widget_clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(ed->widget_clist), 1, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(ed->widget_clist), 2, TRUE);
	gtk_signal_connect(GTK_OBJECT(ed->widget_clist), "select_row", widget_list_select_cb, ed);
	gtk_signal_connect(GTK_OBJECT(ed->widget_clist), "row_move", widget_list_row_move_cb, ed);
	gtk_clist_set_reorderable(GTK_CLIST(ed->widget_clist), TRUE);
	gtk_container_add(GTK_CONTAINER(scrolled), ed->widget_clist);
	gtk_widget_show(ed->widget_clist);

	sep = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(ed->hbox), sep, FALSE, FALSE, 5);
	gtk_widget_show(sep);

	/* set up edit ui */

	ed->ui = g_new0(UIData, 1);

	ed->ui->decorations = FALSE;
	ed->ui->root_win_idle = -1;
	ed->ui->edit = NULL;
	ed->ui->skin = NULL;
	ed->ui->allow_move = FALSE;

	ed->ui->window = NULL;

	ed->ui->display = gtk_drawing_area_new();
	ui_display_events_init(ed->ui);
	gtk_table_attach(GTK_TABLE(ed->table), ed->ui->display, 0, 1, 0, 1, GTK_EXPAND, GTK_EXPAND, 0, 0);
	gtk_widget_show(ed->ui->display);

	edit_events_init(ed);

        gtk_widget_realize(ed->ui->display);

	parent->edit = ed->ui;
	ed->parent = parent;

	/* copy ui data from parent */

	ui_edit_copy_ui(ed->ui, parent);

	/* register a dummy widget so that the editor info pane also updates on a resize */
	editor_type_init();
	skin_register_widget(ed->ui->skin, "editor_dummy", NULL, dummy_type_id, ed);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(ed->hbox), vbox, FALSE, FALSE, 0);
	gtk_widget_show(vbox);

	/* widget property area */

	hbox = ui_edit_frame_new(vbox, FALSE, NULL);

	ed->description = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox), ed->description, TRUE, TRUE, 0);
	gtk_widget_show(ed->description);

	ed->spin_x = ui_edit_spin_new(hbox, _("x:"), 0, 0, &adj);
	gtk_signal_connect(adj, "value_changed", prop_spin_x_cb, ed);

	ed->spin_y = ui_edit_spin_new(hbox, _("y:"), 0, 0, &adj);
	gtk_signal_connect(adj, "value_changed", prop_spin_y_cb, ed);

	hbox = ui_edit_frame_new(vbox, FALSE, NULL);
	ui_edit_label_new(hbox, _("Key:"));

	ed->key_combo_entry = gtk_combo_new();
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(ed->key_combo_entry)->entry), "changed", prop_key_entry_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), ed->key_combo_entry, TRUE, TRUE, 0);
	gtk_widget_show(ed->key_combo_entry);

	ed->data_entry = ui_edit_entry_new(vbox, _("Data:"));
	gtk_signal_connect(GTK_OBJECT(ed->data_entry), "changed", prop_data_entry_cb, ed);

	hbox = ui_edit_frame_new(vbox, FALSE, NULL);

	ed->text_id_entry = ui_edit_entry_new(vbox, _("Text id:"));
	gtk_signal_connect(GTK_OBJECT(ed->text_id_entry), "changed", prop_text_id_entry_cb, ed);

	ed->detail_vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), ed->detail_vbox, FALSE, FALSE, 0);
	gtk_widget_show(ed->detail_vbox);

	ed->widget_pixmap = ui_edit_pixmap_new(vbox);

	ed->image_entry = ui_edit_entry_new(vbox, _("Image:"));
	ui_edit_frame_sensitive(ed->image_entry, FALSE, TRUE);

	ed->clip_mask_entry = ui_edit_entry_new(vbox, _("Clip mask:"));
	ui_edit_frame_sensitive(ed->clip_mask_entry, FALSE, TRUE);

	hbox = ui_edit_frame_new(vbox, TRUE, _("Coordinate anchor"));

	ed->anchor_right_button = ui_edit_toggle_new(hbox, _("Right"));
	gtk_signal_connect(GTK_OBJECT(ed->anchor_right_button), "clicked", prop_anchor_right_cb, ed);

	ed->anchor_bottom_button = ui_edit_toggle_new(hbox, _("Bottom"));
	gtk_signal_connect(GTK_OBJECT(ed->anchor_bottom_button), "clicked", prop_anchor_bottom_cb, ed);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), sep, FALSE, FALSE, 0);
	gtk_widget_show(sep);

	/* path, save */

	hbox = ui_edit_frame_new(main_vbox, FALSE, NULL);

	button = gtk_button_new_with_label(_("Load"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", edit_load_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	label = gtk_label_new(_("Skin path:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	combo = tab_completion_new_with_history(&ed->save_path_entry, ed->ui->skin_path,
						"SLIK_save", 16, NULL, NULL);
	if (!ed->ui->skin_path && editor_resource_list)
		{
		gtk_entry_set_text(GTK_ENTRY(ed->save_path_entry), (gchar *)editor_resource_list->data);
		}
	gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
	gtk_widget_show(combo);

	label = gtk_label_new(_("Data filename:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
	gtk_widget_show(combo);
	ed->save_key_entry = GTK_COMBO(combo)->entry;

	/* example skindata keys */
	list = g_list_append(NULL, "skindata");
	list = g_list_append(list, "skindata_alt");
	list = g_list_append(list, "skindata_1");
	list = g_list_append(list, "skindata_2");
	list = g_list_append(list, "skindata_3");

	gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	g_list_free(list);

	gtk_entry_set_text(GTK_ENTRY(ed->save_key_entry), ed->ui->skin_mode_key ? ed->ui->skin_mode_key : "skindata");
	
	button = gtk_button_new_with_label(_("Save"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", edit_save_cb, ed);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	/* finish up */
	widget_window_setup(ed);
	help_window_setup(ed);

	widget_list_reset(ed);
	prop_update(ed, NULL, TRUE);

	window_set_icon(ed->window, ui_slik_logo(), NULL);
	gtk_widget_show(ed->window);

	return ed;
}

/*
 *-----------------------------------------------------------------------------
 * distort fun
 *-----------------------------------------------------------------------------
 */

static gint distort_cb(gpointer data)
{
	UIData *ui = data;
	GList *work;

	if (!ui->skin) return FALSE;

	work = g_list_nth(ui->skin->widget_list, (float)rand() / (float)RAND_MAX * (float)g_list_length(ui->skin->widget_list));

	if (work)
		{
		gint x, y, w, h;
		gint ox, oy;

		WidgetData *wd = work->data;

		if (!wd->od || !wd->od->is_visible) return TRUE;

		if (!ui_widget_get_geometry(wd, &x, &y, &w, &h)) return TRUE;

		ox = x;
		oy = y;

		x += (rand() > RAND_MAX / 2 ? 1 : -1);
		y += (rand() > RAND_MAX / 2 ? 1 : -1);

		if (x < 0) x = 0;
		if (y < 0) y = 0;
		if (x + w > ui->skin->width) x = ui->skin->width - w;
		if (y + h > ui->skin->height) y = ui->skin->height - h;

		ui_widget_set_coord(ui, wd, x, y, TRUE);

		pixbuf_copy_area(ui->skin->overlay, ox, oy,
				 ui->skin->pixbuf, ox, oy, w, h, FALSE);
		ui_widget_sync_back(ui, wd);

		ui_display_redraw_area(ui, ox, oy, w, h);
		ui_widget_draw(ui, wd, FALSE, TRUE);
		ui_display_render_area(ui, ox, oy, w, h);
		}

	return TRUE;
}

void ui_distort(UIData *ui)
{
	gtk_timeout_add(5, distort_cb, ui);
}

/*
 *-----------------------------------------------------------------------------
 * register system and user skin directories here (for use by editor)
 *-----------------------------------------------------------------------------
 */

void ui_edit_add_skin_resource(const gchar *path)
{
	GList *work;
	if (!path) return;
	work = editor_resource_list;
	while(work)
		{
		gchar *t = work->data;
		work = work->next;
		if (strcmp(t, path) == 0) return;
		}
	editor_resource_list = g_list_append(editor_resource_list, g_strdup(path));
}





