/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include "ui2_typedefs.h"
#include "ui2_text.h"
#include "ui2_text_edit.h"

#include "ui2_display.h"
#include "ui2_main.h"
#include "ui2_parse.h"
#include "ui2_skin.h"
#include "ui2_util.h"
#include "ui2_widget.h"
#include "ui_pixbuf_ops.h"


/* in cps (characters per second) */
#define TEXT_DEFAULT_DELAY 8
/* reverse is in characters */
#define TEXT_DEFAULT_AUTOSCROLL_REVERSE 8

typedef struct _TextCallbackData TextCallbackData;
struct _TextCallbackData
{
	const gchar *(*status_get_func)(TextData *text, const gchar *key, gpointer data);
	gpointer status_get_data;
};


/* table to convert iso_8859 chars to similar ascii counterpart
 * only used in text_draw()
 */
static const gchar iso_ascii[]=
{
	' ','|','c','L','c','Y','|','S','\"','c',' ','<','!','-','r','~',
	'o',' ','2','3','\'','u','p','.',',','1',' ','>',' ',' ',' ','?',
	'A','A','A','A','A','A','A','C','E','E','E','E','I','I','I','I',
	'D','N','O','O','O','O','O','x','0','U','U','U','U','Y','P','B',
	'a','a','a','a','a','a','a','c','e','e','e','e','i','i','i','i',
	'o','n','o','o','o','o','o','/','o','u','u','u','u','y','p','y'
};


static WidgetType type_id_font = -1;

/*
 *-----------------------------
 * font new / free
 *-----------------------------
 */

FontData *font_new(GdkPixbuf *pb, gint extended)
{
	FontData *font;
	gint width;
	gint height;

	if (!pb) return NULL;

	font = g_new0(FontData, 1);

	font->overlay = util_size_pixbuf(pb, TRUE);
	font->extended = extended;

	width = gdk_pixbuf_get_width(font->overlay);
	height = gdk_pixbuf_get_height(font->overlay);

	font->char_width = width / 32;

	if (extended)
		font->char_height = height / 6;
	else
		font->char_height = height / 3;

	font->ref = 1;

	return font;
}

FontData *font_new_from_data(gchar **data, gint extended)
{
	GdkPixbuf *pb;

	pb = gdk_pixbuf_new_from_xpm_data((const char **)data);

	return font_new(pb, extended);
}

FontData *font_new_from_file(const gchar *file, gint extended)
{
	GdkPixbuf *pb;

	pb = gdk_pixbuf_new_from_file(file);

	return font_new(pb, extended);
}

static void font_free(FontData *font)
{
	if (!font) return;
	if (font->overlay) gdk_pixbuf_unref(font->overlay);
	g_free(font);
}

void font_ref(FontData *font)
{
	if (font) font->ref++;
}

void font_unref(FontData *font)
{
	if (font)
		{
		font->ref--;
		if (font->ref < 1) font_free(font);
		}
}

static void font_unref_cb(gpointer data)
{
	font_unref((FontData *)data);
}

/*
 *-----------------------------
 * draw
 *-----------------------------
 */

static void font_draw(FontData *font, guint8 c, gint x, gint y, gint offset, GdkPixbuf *pb)
{
	gint px, py;

	if (!font) return;

	if (font->extended)
		{
		if (c >= 32 && c < 128)
			c -= 32;
		else if (c >= 160 && c < 256)
			c -= 64;
		else
			c = 0;
		}
	else
		{
		if (c >= 32 && c < 128)
			c -= 32;
		else if (c >= 160 && c < 256)
			/* translate to a limited set */
			c = iso_ascii[c - 160] - 32;
		else
			c = 0;
		}

	py = c / 32;
	px = c - (py * 32);

	px = px * font->char_width;
	py = py * font->char_height;
	
	pixbuf_copy_area_alpha(font->overlay, px, py,
			       pb, x + offset * font->char_width, y,
			       font->char_width, font->char_height, 255);
}

/*
 *-----------------------------
 * ui funcs
 *-----------------------------
 */

static gint font_get_geometry(gpointer widget, gint *x, gint *y, gint *w, gint *h)
{
	FontData *font = widget;

	/* fonts have no geometry, they are not directly displayed */
	*x = 0;
	*y = 0;
	*w = font->char_width;
	*h = font->char_height;

	return TRUE;
}

static WidgetData *font_parse(SkinData *skin, GList *list, const gchar *skin_dir, const gchar *key, gint edit)
{
	WidgetData *wd = NULL;
	FontData *font;
	const gchar *text_id;
	gchar *filename;
	gint extended;

	/* req */
	filename = key_list_read_path(list, "image", skin_dir);
	if (!filename) return NULL;

	/* opt */
	extended = key_list_read_bool(list, "extended");
	text_id = key_list_read_chars(list, "id", NULL);

	font = font_new_from_file(filename, extended);

	if (font)
		{
		wd = font_register(skin, font, key, text_id);
		font_unref(font);

		if (edit)
			{
			ui_widget_set_data(wd, "image", filename);
			}
		}
	g_free(filename);

	return wd;
}

/*
 *-----------------------------
 * register ui
 *-----------------------------
 */

WidgetData *font_register(SkinData *skin, FontData *font, const gchar *key, const gchar *text_id)
{
	font_ref(font);
	return skin_register_widget(skin, key, text_id, type_id_font, font);
}

/*
 *-----------------------------
 * init
 *-----------------------------
 */

WidgetType font_type_id(void)
{
	return type_id_font;
}

void font_type_init(void)
{
	WidgetObjectData *od;

	if (type_id_font != -1) return;

	od = ui_widget_type_new("font");
	type_id_font = od->type;
	od->is_visible = FALSE;

	od->func_free = font_unref_cb;

	od->func_get_geometry = font_get_geometry;

	od->func_parse = font_parse;

	font_type_init_edit(od);
}

/*
 *------------
 * text side
 *------------
 */


static WidgetType type_id = -1;

static void text_scroll_update_set(TextData *text, gint enable);

/*
 *-----------------------------
 * new / free
 *-----------------------------
 */

TextData *text_new(FontData *font, gint length, gint sizeable, gint x, gint y)
{
	TextData *text;

	if (!font || length < 1) return NULL;

	util_size(&x);
	util_size(&y);

	text = g_new0(TextData, 1);

	text->font = font;
	font_ref(text->font);
	text->length = length;

	text->width = text->font->char_width * text->length;
	text->height = text->font->char_height;
	text->x = x;
	text->y = y;
	text->text = NULL;
	text->sizeable = sizeable;
	text->size_width = text->width;

	text->autoscroll_delay = TEXT_DEFAULT_DELAY;
	text->autoscroll = FALSE;
	text->autoscroll_reverse_count = 0;

	text->scroll_cb_id = -1;

	text->pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, text->width, text->height);

	return text;
}

TextData *text_new_from_data(gchar **data, gint length, gint extended, gint sizeable, gint x, gint y)
{
	TextData *text;
	FontData *font;

	font = font_new_from_data(data, extended);

	text = text_new(font, length, sizeable, x, y);
	font_unref(font);
	return text;
}

TextData *text_new_from_file(const gchar *file, gint length, gint extended, gint sizeable, gint x, gint y)
{
	TextData *text;
	FontData *font;

	font = font_new_from_file(file, extended);

	text = text_new(font, length, sizeable, x, y);
	font_unref(font);
	return text;
}

void text_free(TextData *text)
{
	if (!text) return;

	text_scroll_update_set(text, FALSE);

	font_unref(text->font);
	if (text->pixbuf) gdk_pixbuf_unref(text->pixbuf);
	g_free(text->text);
	g_free(text);
}

static void text_free_cb(gpointer data)
{
	text_free((TextData *)data);
}

/*
 *-----------------------------
 * draw
 *-----------------------------
 */

static void text_draw_real(TextData *text, GdkPixbuf *pb, UIData *ui)
{
	gint i;
	guint8 c;
	gint l;
	gchar *start;

	if (!text) return;

	if (text->text)
		l = strlen(text->text);
	else
		l = 0;

	if (text->offset >= l) text->offset = l - 1;
	if (text->offset < 0) text->offset = 0;

	if (text->text)
		start = text->text + text->offset;
	else
		start = NULL;

	l -= text->offset;

	pixbuf_copy_area(text->pixbuf, 0, 0, pb, text->x, text->y, text->width, text->height, FALSE);

	for (i=0; i < text->length; i++)
		{
		if (i < l)
			{
			c = start[i];
			}
		else
			{
			c = ' ';
			}
		font_draw(text->font, c, text->x, text->y, i, pb);
		}

	ui_display_render_area(ui, text->x, text->y, text->width, text->height);
	text_scroll_update_set(text, TRUE);
}

static void text_set_text_real(TextData *text, const gchar *new_text, GdkPixbuf *pb, UIData *ui)
{
	g_free(text->text);
	
	if (new_text)
		{
		text->text = g_strdup(new_text);
		}
	else
		{
		text->text = NULL;
		}

	text->offset = 0;
	text->autoscroll_reverse_count = 0;
	text->scroll = -1;

	text_draw_real(text, pb, ui);
}

static gint text_scroll_update_cb(gpointer data)
{
	TextData *text = data;
	GdkPixbuf *pb;
	UIData *ui;
	gint old_offset;

	if (!text) return FALSE;
	if (!text->text || !text->skin) return TRUE;
	if (text->scroll_cb_id == -1) return FALSE;

	pb = skin_get_pixbuf(text->skin);
	ui = skin_get_ui(text->skin);

	if (!pb) return TRUE;

	old_offset = text->offset;

	if (text->scroll > 0)
		{
		gint l = strlen(text->text);
		if (text->offset < l - text->length && l > text->length)
			{
			text->offset++;
			}
		else if (text->autoscroll)
			{
			text->autoscroll_reverse_count++;
			if (text->autoscroll_reverse_count > TEXT_DEFAULT_AUTOSCROLL_REVERSE)
				{
				text->autoscroll_reverse_count = 0;
				text->scroll = -1;
				}
			}
		}
	else
		{
		if (text->offset > 0)
			{
			text->offset--;
			}
		else if (text->autoscroll)
			{
			text->autoscroll_reverse_count++;
			if (text->autoscroll_reverse_count > TEXT_DEFAULT_AUTOSCROLL_REVERSE)
				{
				text->autoscroll_reverse_count = 0;
				text->scroll = 1;
				}
			}
		}

	if (text->offset != old_offset) text_draw_real(text, pb, ui);

	return TRUE;
}

static void text_scroll_update_set(TextData *text, gint enable)
{
	if (enable && text->scroll_cb_id == -1)
		{
		text->scroll_cb_id = gtk_timeout_add(1000 / text->autoscroll_delay, text_scroll_update_cb, text);
		}
	else if (!enable && text->scroll_cb_id != -1)
		{
		gtk_timeout_remove(text->scroll_cb_id);
		text->scroll_cb_id = -1;
		}
}

/*
 *-----------------------------
 * ui funcs
 *-----------------------------
 */

static void text_draw(gpointer data, const gchar *key, gint update, gint force, GdkPixbuf *pb, UIData *ui)
{
	TextData *text = data;

	if (update)
		{
		TextCallbackData *cd;

		cd = ui_get_registered_callbacks(ui, key, type_id);
		if (cd && cd->status_get_func)
			{
			const gchar *new_text;
			new_text = cd->status_get_func(text, key, cd->status_get_data);
			text_set_text_real(text, new_text, pb, ui);
			}
		else
			{
			const gchar *new_text;
			new_text = ui_widget_get_data_by_widget(ui, text);
			text_set_text_real(text, new_text, pb, ui);
			}
		}
	else if (force)
		{
		text_draw_real(text, pb, ui);
		}
}

static void text_reset(gpointer data, const gchar *key, GdkPixbuf *pb, UIData *ui)
{
	TextData *text = data;

	if (!text) return;

	text->scroll = -1;
}

static void text_motion(gpointer data, const gchar *key, gint x, gint y, GdkPixbuf *pb, UIData *ui)
{
	TextData *text = data;

	if (!text) return;

	if (text->autoscroll) return;

	if (x >= text->x && x < text->x + text->width &&
	    y >= text->y && y < text->y + text->height)
		{
		text->scroll = 1;
		}
	else
		{
		text->scroll = -1;
		}
}

static void text_back_set(gpointer data, GdkPixbuf *pb)
{
	TextData *text = data;

	if (!text) return;

	pixbuf_copy_area(pb, text->x, text->y, text->pixbuf, 0, 0,
			 text->width, text->height, FALSE);
}

static gint text_get_geometry(gpointer widget, gint *x, gint *y, gint *w, gint *h)
{
	TextData *text = widget;

	*x = text->x;
	*y = text->y;
	*w = text->width;
	*h = text->height;

	return TRUE;
}

static void text_set_coord(gpointer widget, gint x, gint y)
{
	TextData *text = widget;

	if (text->x != x) text->size_width = text->width;
	text->x = x;
	text->y = y;
}

static void text_set_size(gpointer widget, gint dev_w, gint dev_h)
{
	TextData *text = widget;
	gint old_w;

	if (!text->sizeable) return;

	old_w = text->width;

	text->size_width += dev_w;
	if (text->size_width < text->font->char_width) text->size_width = text->font->char_width;
	text->length = text->size_width / text->font->char_width;
	text->width = text->length * text->font->char_width;

	if (old_w == text->width) return;

	if (text->pixbuf) gdk_pixbuf_unref(text->pixbuf);
	text->pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, text->width, text->height);
}

static WidgetData *text_parse(SkinData *skin, GList *list, const gchar *skin_dir, const gchar *key, gint edit)
{
	WidgetData *wd = NULL;
	TextData *text;
	gchar *filename;
	const gchar *font_id;
	gint x, y;
	gint length;
	gint extended;
	gint sizeable;

	/* req */
	if (!key_list_read_int(list, "x", &x)) return NULL;
	if (!key_list_read_int(list, "y", &y)) return NULL;
	if (!key_list_read_int(list, "length", &length)) return NULL;
	filename = key_list_read_path(list, "image", skin_dir);
	font_id = key_list_read_chars(list, "font", NULL);
	if (!filename && !font_id) return NULL;

	/* opt */
	extended = key_list_read_bool(list, "extended");
	sizeable = key_list_read_bool(list, "sizeable");

	if (filename)
		{
		text = text_new_from_file(filename, length, extended, sizeable, x, y);
		}
	else
		{
		WidgetData *font_wd;
		FontData *font;

		font_wd = skin_widget_get_by_key(skin, font_id, font_type_id());
		if (!font_wd)
			{
			printf("font not found registered as \"%s\"\n", font_id);
			return NULL;
			}

		font = font_wd->widget;
		text = text_new(font, length, sizeable, x, y);
		}

	if (text)
		{
		wd = text_register(skin, text, key, NULL);

		if (edit)
			{
			ui_widget_set_data(wd, "image", filename);
			ui_widget_set_data(wd, "font_id", font_id);
			}
		}
	g_free(filename);

	return wd;
}

/*
 *-----------------------------
 * register ui / app side
 *-----------------------------
 */

WidgetData *text_register(SkinData *skin, TextData *text, const gchar *key, const gchar *text_id)
{
	text->skin = skin;
	return skin_register_widget(skin, key, text_id, type_id, text);
}

RegisterData *text_register_key(const gchar *key, UIData *ui,
				const gchar *(*status_get_func)(TextData *text, const gchar *key, gpointer data), gpointer status_get_data)
{
	TextCallbackData *cd;

	cd = g_new0(TextCallbackData, 1);

	cd->status_get_func = status_get_func;
	cd->status_get_data = status_get_data;

	return ui_register_key(ui, key, type_id, cd, sizeof(TextCallbackData));
}

/*
 *-----------------------------
 * app funcs
 *-----------------------------
 */

static void text_set_text_cb(WidgetData *wd, gpointer data, GdkPixbuf *pb, UIData *ui)
{
	TextData *text;
	const gchar *new_text;

	text = wd->widget;
	new_text = (const gchar *)data;

	text_set_text_real(text, new_text, pb, ui);
}

gint text_set_text(const gchar *key, UIData *ui, const gchar *new_text)
{
	return skin_widget_for_each_key(ui, key, type_id, text_set_text_cb, (gchar *)new_text);
}

static void text_set_autoscroll_cb(WidgetData *wd, gpointer data, GdkPixbuf *pb, UIData *ui)
{
	TextData *text;

	text = wd->widget;
	text->autoscroll = GPOINTER_TO_INT(data);
}

gint text_set_autoscroll(const gchar *key, UIData *ui, gint enable)
{
	return skin_widget_for_each_key(ui, key, type_id, text_set_autoscroll_cb, GINT_TO_POINTER(enable));
}

static void text_set_scroll_speed_cb(WidgetData *wd, gpointer data, GdkPixbuf *pb, UIData *ui)
{
	TextData *text;
	gint val;

	text = wd->widget;
	val = GPOINTER_TO_INT(data);
	if (val < 0) val = 1;
	text->autoscroll_delay = val;

	text_scroll_update_set(text, FALSE);
	text_scroll_update_set(text, TRUE);
}

gint text_set_scroll_speed(const gchar *key, UIData *ui, gint delay)
{
	return skin_widget_for_each_key(ui, key, type_id, text_set_scroll_speed_cb, GINT_TO_POINTER(delay));
}

/*
 *-----------------------------
 * init
 *-----------------------------
 */

WidgetType text_type_id(void)
{
	return type_id;
}

void text_type_init(void)
{
	WidgetObjectData *od;

	if (type_id != -1) return;

	od = ui_widget_type_new("text");
	type_id = od->type;

	od->func_draw = text_draw;
	od->func_reset = text_reset;
	od->func_motion = text_motion;
	od->func_back = text_back_set;
	od->func_free = text_free_cb;

	od->func_get_geometry = text_get_geometry;
	od->func_set_coord = text_set_coord;
	od->func_set_size = text_set_size;

	od->func_parse = text_parse;

	text_type_init_edit(od);
}



