/*
 * GQmpeg
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqmpeg.h"
#include "io_mpg123.h"

#include "playlist.h"
#include "ui_fileops.h"

typedef struct _SongInfoDialog SongInfoDialog;
struct _SongInfoDialog
{
	GtkWidget *vbox;
        GtkWidget *title_entry;
        GtkWidget *artist_entry;
        GtkWidget *album_entry;
	GtkWidget *year_entry;
	GtkWidget *comment_entry;
	GtkWidget *genre_combo;
	GtkWidget *save_button;
	gchar *path;
};

static void close_song_info_window(GtkWidget *widget, gpointer data);
static void save_tag_pressed(GtkWidget *w, gpointer data);
static void data_changed_cb(GtkWidget *w, gpointer data);
static gint sort_genre_list_cb(void *a, void *b);
static GtkWidget *add_combo_info(GtkWidget *vbox, const gchar *text, gint data, GtkWidget *sb);
static GtkWidget *add_entry_info(GtkWidget *vbox, const gchar *text, const gchar *data, gint length, GtkWidget *sb);
static void add_label_info(GtkWidget *vbox, const gchar *text, const gchar *data);
static void add_label_info_bool(GtkWidget *vbox, const gchar *text, gint t);

static void close_song_info_window(GtkWidget *widget, gpointer data)
{
	SongInfoDialog *s = data;
	g_free(s->path);
	g_free(s);
}

static void save_tag_pressed(GtkWidget *w, gpointer data)
{
	SongInfoDialog *s = data;
        gchar *title;
        gchar *artist;
        gchar *album;
	gchar *year;
	gchar *comment;
	gchar *genre_text;
	guint8 genre = 0;

        title = gtk_entry_get_text(GTK_ENTRY(s->title_entry));
        artist = gtk_entry_get_text(GTK_ENTRY(s->artist_entry));
        album = gtk_entry_get_text(GTK_ENTRY(s->album_entry));
	year = gtk_entry_get_text(GTK_ENTRY(s->year_entry));
	comment = gtk_entry_get_text(GTK_ENTRY(s->comment_entry));
	genre_text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(s->genre_combo)->entry));

	while (get_id3_genre(genre) && strcmp(get_id3_genre(genre), genre_text) != 0) genre++;
	if (get_id3_genre(genre) == NULL) genre = 0;

	set_id3_tag(s->path, title, artist, album, year, comment, genre);
	
	playlist_update_by_path(s->path);

	gtk_widget_set_sensitive(s->save_button, FALSE);
}

static void data_changed_cb(GtkWidget *w, gpointer data)
{
	GtkWidget *sb = data;
	gtk_widget_set_sensitive(sb, TRUE);
}

static gint sort_genre_list_cb(void *a, void *b)
{
	return strcmp((gchar *)a, (gchar *)b);
}

static GtkWidget *add_combo_info(GtkWidget *vbox, const gchar *text, gint data, GtkWidget *sb)
{
	GtkWidget *combo;
	GtkWidget *label;
	GtkWidget *hbox;
	GList *list = NULL;
	gint i = 0;

	while(get_id3_genre(i) != NULL)
		{
		gchar *g = get_id3_genre(i);
		list = g_list_prepend(list, g);
		i++;
		}
	list = g_list_sort(list, (GCompareFunc) sort_genre_list_cb);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);	

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	combo = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	if (get_id3_genre(data) != NULL)
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), get_id3_genre(data));
	gtk_combo_set_value_in_list(GTK_COMBO(combo),
				g_list_index(list, get_id3_genre(data)), FALSE);
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed",
				GTK_SIGNAL_FUNC(data_changed_cb), sb);
	gtk_box_pack_end(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
	gtk_entry_set_editable (GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
	gtk_widget_show(combo);

	g_list_free(list);
	return combo;
}

static GtkWidget *add_entry_info(GtkWidget *vbox, const gchar *text, const gchar *data, gint length, GtkWidget *sb)
{
	GtkWidget *entry;
	GtkWidget *label;
	GtkWidget *hbox;

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);	

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	entry = gtk_entry_new_with_max_length(length);
	if (data) gtk_entry_set_text(GTK_ENTRY(entry),data);
	gtk_signal_connect(GTK_OBJECT(entry), "changed",
				GTK_SIGNAL_FUNC(data_changed_cb), sb);
	gtk_box_pack_end(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
	gtk_widget_show(entry);

	return entry;
}

static void add_label_info(GtkWidget *vbox, const gchar *text, const gchar *data)
{
	gchar *buf;
	GtkWidget *label;

	if (data)
		{
		buf = g_strconcat(text, data, NULL);
		label = gtk_label_new(buf);
		g_free(buf);
		}
	else
		label = gtk_label_new(text);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

}

static void add_label_info_bool(GtkWidget *vbox, const gchar *text, gint t)
{
	gchar *data;
	if (t)
		data = _("Yes");
	else
		data = _("No");

	add_label_info(vbox, text, data);
}

static GtkWidget *mpg_create_http_info_window(const gchar *file)
{
	GtkWidget *vbox;
	GtkWidget *vbox1;
	GtkWidget *label;
	GtkWidget *frame;
	gchar *buf;

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox), 5);

	buf = g_strconcat("URL: ", file, NULL);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	frame = gtk_frame_new(_("Information"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox1), 5);
        gtk_container_add(GTK_CONTAINER(frame),vbox1);
        gtk_widget_show(vbox1);

	if (strstr(file, "mp3") == NULL)
		label = gtk_label_new(_("Live stream."));
	else
		label = gtk_label_new(_("File across network."));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	return vbox;
}

GtkWidget *mpg_create_info_window(const gchar *file)
{
	SongInfoDialog *s;
	ID3_Data *d;
	Mpg_Data *h;
	GtkWidget *hbox;
	GtkWidget *vbox1;
	GtkWidget *label;
	GtkWidget *frame;
	gchar *buf;

	if (type_is_http(file)) return mpg_create_http_info_window(file);

	s = g_new0(SongInfoDialog, 1);
	d = get_id3_tag(file);
	h = get_mpg_header_info(file);

	s->path = g_strdup(file);

	s->vbox = gtk_vbox_new(FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (s->vbox), "destroy",(GtkSignalFunc) close_song_info_window, s);
	gtk_container_border_width (GTK_CONTAINER (s->vbox), 5);

	buf = g_strconcat(_("Filename: "), filename_from_path(file), NULL);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start(GTK_BOX(s->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(s->vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	s->save_button = gtk_button_new_with_label(_("Save Tag"));
	gtk_signal_connect (GTK_OBJECT (s->save_button), "clicked",(GtkSignalFunc) save_tag_pressed, s);
	gtk_box_pack_end(GTK_BOX(hbox), s->save_button, FALSE, FALSE, 5);
	gtk_widget_set_sensitive(s->save_button, FALSE);
	gtk_widget_show(s->save_button);

	frame = gtk_frame_new(_("Song data"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(s->vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox1), 5);
        gtk_container_add(GTK_CONTAINER(frame),vbox1);
        gtk_widget_show(vbox1);

	if (h)
		{
		gchar s[64];
		add_label_info(vbox1, _("Version : "), h->version_text);
		add_label_info(vbox1, _("Layer : "), h->layer_text);
		add_label_info(vbox1, _("Channel encoding : "), h->mode_text);

		sprintf(s, _("%d Kbits"), h->bit_rate);
		add_label_info(vbox1, _("Bit Rate : "), s);
		sprintf(s, _("%d Hz"), h->sample_rate);
		add_label_info(vbox1, _("Sample Rate : "), s);
		sprintf(s, _("%d bytes"), h->file_length);
		add_label_info(vbox1, _("File Size : "), s);
		sprintf(s, _("%d seconds"), h->length);
		add_label_info(vbox1, _("Length : "), s);

		add_label_info_bool(vbox1, _("Copyright : "), h->copyright);
		add_label_info_bool(vbox1, _("Original : "), h->original);
		add_label_info_bool(vbox1, _("Emphasis : "), h->emphasis);
		add_label_info_bool(vbox1, _("Extensions : "), h->mode_extension);
		}
	else
		{
		label = gtk_label_new(_("unable to find a valid header"));
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
		gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE, 5);
		gtk_widget_show(label);
		}

	frame = gtk_frame_new(_("ID3 data"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(s->vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox1), 5);
        gtk_container_add(GTK_CONTAINER(frame),vbox1);
        gtk_widget_show(vbox1);

	if (d)
		{
		s->title_entry = add_entry_info(vbox1, _("Title : "), d->title, 30, s->save_button);
		s->artist_entry = add_entry_info(vbox1, _("Artist : "), d->artist, 30, s->save_button);
		s->album_entry = add_entry_info(vbox1, _("Album : "), d->album, 30, s->save_button);
		s->year_entry = add_entry_info(vbox1, _("Year : "), d->year, 4, s->save_button);

		s->genre_combo = add_combo_info(vbox1, _("Genre : "), d->genre, s->save_button);

		s->comment_entry = add_entry_info(vbox1, _("Comment : "), d->comment, 30, s->save_button);
		}
	else if (h)
		{
		s->title_entry = add_entry_info(vbox1, _("Title : "), "", 30, s->save_button);
		s->artist_entry = add_entry_info(vbox1, _("Artist : "), "", 30, s->save_button);
		s->album_entry = add_entry_info(vbox1, _("Album : "), "", 30, s->save_button);
		s->year_entry = add_entry_info(vbox1, _("Year : "), "", 4, s->save_button);

		s->genre_combo = add_combo_info(vbox1, _("Genre : "), 0, s->save_button);

		s->comment_entry = add_entry_info(vbox1, _("Comment : "), "", 30, s->save_button);
		}
	else
		{
		label = gtk_label_new(_("disabled when no header is found"));
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
		gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE, 5);
		gtk_widget_show(label);
		}

	free_id3_tag(d);
	if (h) g_free(h);

	return s->vbox;
}

