/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2002 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <config.h>

#ifdef HAVE_LDAP_STR2OBJECTCLASS

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <lber.h>
#include <ldap.h>
#include <ldap_schema.h>

#include "mainwin.h"
#include "configfile.h"
#include "common.h"
#include "util.h"
#include "schema.h"
#include "schemabrowse.h"
#include "template.h"
#include "tdefault.h"
#include "errorchain.h"
#include "debug.h"


extern struct gq_config config;
extern GtkWidget *current_template_clist;


GtkWidget *templatewin;


void create_template_edit_window(struct ldapserver *server, char *templatename)
{
     GList *list;
     GString *gmessage;
     GtkWidget *vbox1, *hbox0, *hbox1, *vbox2, *vbox3, *vbox4;
     GtkWidget *hbox2, *hbox3, *hbox4, *scrwin;
     GtkWidget *oclist, *templatelist, *clist, *button, *arrow, *label, *entry;
     struct server_schema *ss;
     struct gq_template *tmpl;
     char *otitle[1] = { "Available objectclasses" };
     char *ttitle[1] = { "Template objectclasses" };
     char *rtitle[1] = { "Required attributes" };
     char *atitle[1] = { "Allowed attributes" };

     gmessage = NULL;
     ss = get_schema(server);
     if(ss == NULL) {
	  gmessage = g_string_sized_new(64);
	  g_string_sprintf(gmessage, "no schema information found for server %s"
			   , server->name);
     }
     else if(ss->oc == NULL) {
	  gmessage = g_string_sized_new(64);
	  g_string_sprintf(gmessage, "no objectclass information found for server %s",
			   server->name);
     }
     else if(ss->at == NULL) {
	  gmessage = g_string_sized_new(64);
	  g_string_sprintf(gmessage, "no attribute type information found for server %s",
			   server->name);
     }

     if(gmessage) {
	  error_popup("Error getting schema", gmessage->str);
	  g_string_free(gmessage, TRUE);
	  return;
     }

     templatewin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_object_set_data(GTK_OBJECT(templatewin), "server", server);
     gtk_container_border_width(GTK_CONTAINER(templatewin), 10);
     gtk_window_set_default_size(GTK_WINDOW(templatewin), 700, 450);
     gtk_signal_connect(GTK_OBJECT(templatewin), "key_press_event",
			(GtkSignalFunc) close_on_esc, (gpointer) templatewin);

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_container_add(GTK_CONTAINER(templatewin), vbox1);
     gtk_widget_show(vbox1);

     hbox0 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox0, FALSE, FALSE, 0);

     hbox1 = gtk_hbox_new(FALSE, 20);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, TRUE, TRUE, 5);

     /* Schema objectclasses */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scrwin);
     gtk_box_pack_start(GTK_BOX(hbox1), scrwin, TRUE, TRUE, 0);

     oclist = gtk_clist_new_with_titles(1, otitle);
     GTK_CLIST(oclist)->button_actions[1] = GTK_BUTTON_SELECTS;
     GTK_WIDGET_UNSET_FLAGS(oclist, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(oclist), "server", server);
     gtk_object_set_data(GTK_OBJECT(templatewin), "oclist", oclist);
     gtk_clist_set_compare_func(GTK_CLIST(oclist), clistcasecmp);
     gtk_clist_column_titles_passive(GTK_CLIST(oclist));
     gtk_signal_connect(GTK_OBJECT(oclist), "select_row",
                        (GtkSignalFunc) select_oc_from_clist, NULL);
     gtk_widget_show(oclist);
     gtk_container_add(GTK_CONTAINER(scrwin), oclist);

     /* the two vboxes take care of placing the arrows *just so* */
     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, FALSE, FALSE, 0);

     vbox3 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox3);
     gtk_box_pack_start(GTK_BOX(vbox2), vbox3, TRUE, FALSE, 0);

     /* right arrow */
     button = gtk_button_new();
     gtk_widget_set_usize(button, 20, 30);
     GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(button), "direction", "right");
     arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
     gtk_widget_show(arrow);
     gtk_container_add(GTK_CONTAINER(button), arrow);
     gtk_signal_connect(GTK_OBJECT(button), "pressed",
			(GtkSignalFunc) arrow_button_callback, templatewin);
     gtk_widget_show(button);
     gtk_box_pack_start(GTK_BOX(vbox3), button, TRUE, TRUE, 10);

     /* left arrow */
     button = gtk_button_new();
     gtk_widget_set_usize(button, 20, 30);
     GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(button), "direction", "left");
     arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
     gtk_widget_show(arrow);
     gtk_container_add(GTK_CONTAINER(button), arrow);
     gtk_signal_connect(GTK_OBJECT(button), "pressed",
			(GtkSignalFunc) arrow_button_callback, templatewin);
     gtk_widget_show(button);
     gtk_box_pack_start(GTK_BOX(vbox3), button, TRUE, TRUE, 10);

     /* Template objectclasses */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scrwin);
     gtk_box_pack_start(GTK_BOX(hbox1), scrwin, TRUE, TRUE, 0);

     templatelist = gtk_clist_new_with_titles(1, ttitle);
     GTK_CLIST(templatelist)->button_actions[1] = GTK_BUTTON_SELECTS;
     GTK_WIDGET_UNSET_FLAGS(templatelist, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(templatelist), "server", server);
     gtk_object_set_data(GTK_OBJECT(templatewin), "templatelist", templatelist);
     gtk_clist_column_titles_passive(GTK_CLIST(templatelist));
     gtk_signal_connect(GTK_OBJECT(templatelist), "select_row",
                        (GtkSignalFunc) select_oc_from_clist, NULL);
     gtk_widget_show(templatelist);
     gtk_container_add(GTK_CONTAINER(scrwin), templatelist);

     /* vbox to hold required and allowed attributes clists */
     vbox4 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox4);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox4, TRUE, TRUE, 0);

     /* Required attributes */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scrwin);
     clist = gtk_clist_new_with_titles(1, rtitle);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_object_set_data(GTK_OBJECT(templatewin), "reqdattr", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                        (GtkSignalFunc) select_at_from_clist, NULL);
     gtk_widget_show(clist);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_box_pack_start(GTK_BOX(vbox4), scrwin, TRUE, TRUE, 0);

     /* Allowed attributes */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scrwin);
     clist = gtk_clist_new_with_titles(1, atitle);
     GTK_CLIST(clist)->button_actions[1] = GTK_BUTTON_SELECTS;
     GTK_WIDGET_UNSET_FLAGS(clist, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(clist), "server", server);
     gtk_object_set_data(GTK_OBJECT(templatewin), "allowedattr", clist);
     gtk_clist_column_titles_passive(GTK_CLIST(clist));
     gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                        (GtkSignalFunc) select_at_from_clist, NULL);
     gtk_widget_show(clist);
     gtk_container_add(GTK_CONTAINER(scrwin), clist);
     gtk_box_pack_start(GTK_BOX(vbox4), scrwin, TRUE, TRUE, 0);

     hbox2 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox2);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox2, FALSE, FALSE, 10);

#if EXPERIMENTAL_TEMPLATE_DEFAULTS
     /* Input defaults */
     button = gtk_button_new_with_label("  Edit input defaults  ");
     GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_DEFAULT);
     GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
     gtk_signal_connect(GTK_OBJECT(button), "pressed",
			(GtkSignalFunc) create_tdefault_edit_window,
			(gpointer) templatewin);
     gtk_widget_show(button);
     gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
#endif

     hbox3 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox3);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox3, FALSE, FALSE, 10);

     /* Template name */
     label = gtk_label_new("Template name");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox3), label, FALSE, FALSE, 0);

     entry = gtk_entry_new();
     gtk_entry_set_max_length(GTK_ENTRY(entry), 127);
     gtk_object_set_data(GTK_OBJECT(templatewin), "templatenamebox", entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			(GtkSignalFunc) save_template_callback,
			(gpointer) templatewin);
     gtk_widget_show(entry);
     gtk_box_pack_start(GTK_BOX(hbox3), entry, FALSE, FALSE, 10);

     hbox4 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox4);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox4, FALSE, FALSE, 10);

     /* Save */
     button = gtk_button_new_with_label("  Save  ");
     GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
     gtk_signal_connect(GTK_OBJECT(button), "pressed",
			(GtkSignalFunc) save_template_callback,
			(gpointer) templatewin);
     gtk_widget_show(button);
     gtk_box_pack_start(GTK_BOX(hbox4), button, FALSE, FALSE, 0);
     gtk_widget_grab_default(button);

     /* Cancel */
     button = gtk_button_new_with_label("  Cancel  ");
     GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
     gtk_signal_connect_object(GTK_OBJECT(button), "pressed",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) templatewin);
     gtk_widget_show(button);
     gtk_box_pack_end(GTK_BOX(hbox4), button, FALSE, FALSE, 0);
     gtk_signal_connect_object(GTK_OBJECT(templatewin), "destroy",
			       (GtkSignalFunc) delete_edit_window,
			       (gpointer) templatewin);

     fill_new_template(templatewin);

     /* default title, in case editing doesn't work out */
     gtk_window_set_title(GTK_WINDOW(templatewin), "GQ: create new template");

     if(templatename) {
	  gtk_window_set_title(GTK_WINDOW(templatewin), "GQ: edit template");
	  gtk_object_set_data(GTK_OBJECT(templatewin), "templatename", g_strdup(templatename));
	  tmpl = find_template_by_name(templatename);
	  if(tmpl) {
	       gtk_entry_set_text(GTK_ENTRY(entry), tmpl->name);
	       list = tmpl->objectclasses;
	       while(list) {
		    /* a bit slow maybe */
		    move_objectclass(oclist, templatelist, (char *) list->data);
		    build_used_attrs_list(templatewin);
		    list = list->next;
	       }
	  }
     }
     else {
	  /* a new template has the "top" objectclass selected
	     by default (LDAP V3 requirement) */
	  move_objectclass(oclist, templatelist, "top");
	  build_used_attrs_list(templatewin);
     }

     gtk_widget_show(templatewin);

     gtk_widget_grab_focus(entry);

}


/*
 * templatename holds the name of the existing template the editor is using.
 * this is g_strdup'ed so need to free it before the window goes
 */
gboolean delete_edit_window(GtkWidget *window)
{
     char *templatename;

     if( (templatename = gtk_object_get_data(GTK_OBJECT(window), "templatename")))
	  g_free(templatename);

     return(FALSE);
}


/*
 * initialize template editor
 */
void fill_new_template(GtkWidget *window)
{
     GList *list;
     GtkWidget *oclist, *templatelist;
     LDAPObjectClass *oc;
     struct ldapserver *server;
     char *ocname[1];

     if( (oclist = gtk_object_get_data(GTK_OBJECT(window), "oclist")) == NULL)
	  return;

     if( (templatelist = gtk_object_get_data(GTK_OBJECT(window), "templatelist")) == NULL)
	  return;

     if( (server = gtk_object_get_data(GTK_OBJECT(window), "server")) == NULL)
	  return;

     gtk_clist_freeze(GTK_CLIST(oclist));
     gtk_clist_clear(GTK_CLIST(oclist));
     list = server->ss->oc;
     while(list) {
	  oc = list->data;
	  if(oc) {
	       if(oc->oc_names && oc->oc_names[0])
		    ocname[0] = oc->oc_names[0];
	       else
		    ocname[0] = oc->oc_oid;
	       gtk_clist_append(GTK_CLIST(oclist), ocname);
	  }
	  list = list->next;
     }
     gtk_clist_thaw(GTK_CLIST(oclist));

}


/*
 * move objectclass selected in "Objectclasses" clist to "Template" clist
 */
void arrow_button_callback(GtkWidget *widget, GtkWidget *window)
{
     GtkWidget *oclist, *templatelist;
     char *direction, *objectclass;

     if( (oclist = gtk_object_get_data(GTK_OBJECT(window), "oclist")) == NULL)
	  return;

     if( (templatelist = gtk_object_get_data(GTK_OBJECT(window), "templatelist")) == NULL)
	  return;

     if( (direction = gtk_object_get_data(GTK_OBJECT(widget), "direction")) == NULL)
	  return;

     if(!strcmp(direction, "right")) {
	  objectclass = get_clist_selection(oclist);
	  move_objectclass(oclist, templatelist, objectclass);
     }
     else {
	  objectclass = get_clist_selection(templatelist);
	  move_objectclass(templatelist, oclist, objectclass);
	  gtk_clist_sort(GTK_CLIST(oclist));
     }

     build_used_attrs_list(window);

}


/*
 * default clist sort function is case-sensitive, this does a case-insensitive
 * sort instead
 */
int clistcasecmp(GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2)
{
     GtkCListRow *row1, *row2;

     row1 = (GtkCListRow *) ptr1;
     row2 = (GtkCListRow *) ptr2;

     return(strcasecmp(GTK_CELL_TEXT(row1->cell[0])->text,
		       GTK_CELL_TEXT(row2->cell[0])->text));
}


/*
 * move item by name from source clist to destination clist
 */
void move_objectclass(GtkWidget *source, GtkWidget *destination, char *objectclass)
{
     int row;
     char *dummy[1];

     if(objectclass == NULL)
	  return;

     row = get_clist_row_by_text(source, objectclass);
     if(row == -1)
	  return;

     dummy[0] = g_strdup(objectclass);
     gtk_clist_remove(GTK_CLIST(source), row);

     gtk_clist_append(GTK_CLIST(destination), dummy);
     g_free(dummy[0]);

}


/*
 * build "Required attributes" and "Allowed attributes" clists from
 * current objectclasses in "Template" clist
 */
void build_used_attrs_list(GtkWidget *window)
{
     LDAPObjectClass *oc;
     GtkWidget *templatelist, *reqdlist, *allowedlist;
     struct ldapserver *server;
     struct server_schema *ss;
     int i, attr;
     char *objectclass, *dummy[1];

     if( (server = gtk_object_get_data(GTK_OBJECT(window), "server")) == NULL)
	  return;

     if( (templatelist = gtk_object_get_data(GTK_OBJECT(window), "templatelist")) == NULL)
	  return;

     if( (reqdlist = gtk_object_get_data(GTK_OBJECT(window), "reqdattr")) == NULL)
	  return;

     if( (allowedlist = gtk_object_get_data(GTK_OBJECT(window), "allowedattr")) == NULL)
	  return;

     ss = get_server_schema(server);

     gtk_clist_freeze(GTK_CLIST(reqdlist));
     gtk_clist_clear(GTK_CLIST(reqdlist));
     gtk_clist_freeze(GTK_CLIST(allowedlist));
     gtk_clist_clear(GTK_CLIST(allowedlist));
     for(i = 0; i < GTK_CLIST(templatelist)->rows; i++) {
	  gtk_clist_get_text(GTK_CLIST(templatelist), i, 0, &objectclass);
	  oc = find_oc_by_oc_name(ss, objectclass);
	  if(oc) {

	       /* Required attributes */
	       if(oc->oc_at_oids_must)
		    for(attr = 0; oc->oc_at_oids_must[attr]; attr++)
			 if(get_clist_row_by_text(reqdlist, oc->oc_at_oids_must[attr]) == -1) {
			      dummy[0] = oc->oc_at_oids_must[attr];
			      gtk_clist_append(GTK_CLIST(reqdlist), dummy);
			 }

	       /* Allowed attributes */
	       if(oc->oc_at_oids_may)
		    for(attr = 0; oc->oc_at_oids_may[attr]; attr++)
			 if(get_clist_row_by_text(allowedlist, oc->oc_at_oids_may[attr]) == -1) {
			      dummy[0] = oc->oc_at_oids_may[attr];
			      gtk_clist_append(GTK_CLIST(allowedlist), dummy);
			 }

	  }

     }
     gtk_clist_thaw(GTK_CLIST(reqdlist));
     gtk_clist_thaw(GTK_CLIST(allowedlist));

}


/*
 * save new template or update existing template. Also saves to .gq
 * and destroys window
 */
void save_template_callback(GtkWidget *widget, GtkWidget *window)
{
     GList *list;
     struct gq_template *old_tmpl, *tmpl;
     char *old_template_name;

     tmpl = window2template(window);
     if(tmpl == NULL) {
	  /* window2template() failed */
	  error_popup("Weirdness!", "Template was not saved!");
	  return;
     }

     if(strlen(tmpl->name) == 0) {
	  /* need a name for this template, warning popup */
	  list = g_list_append(NULL, "You need to enter a name for the new template.");
	  list = g_list_append(list, "Please enter one and try again.");
	  warning_popup(list);
	  return;
     }

     if( (old_template_name = gtk_object_get_data(GTK_OBJECT(window), "templatename")) == NULL) {
	  /* add new template to list, but make sure the template name is unique */
	  if( (find_template_by_name(tmpl->name)) == NULL) {
	       config.templates = g_list_append(config.templates, tmpl);
	  }
	  else {
	       list = g_list_append(NULL, "A template by that name already exists!");
	       list = g_list_append(list, "Please choose another name, or delete "
				    "the existing template");
	       warning_popup(list);
	       return;
	  }
     }
     else {

	  /* update existing template */
	  old_tmpl = find_template_by_name(old_template_name);
	  if(old_tmpl) {

	       g_list_free(old_tmpl->objectclasses);
	       old_tmpl->objectclasses = tmpl->objectclasses;

	       if(strcmp(old_template_name, tmpl->name)) {
		    /* template name was changed too */
		    strncpy(old_tmpl->name, tmpl->name, 128);
	       }

	       /* tmpl is not inserted into global config, so we don't need it any more */
	       FREE(tmpl, "struct gq_template");

	  }
	  else {
	       /* hmmmm... template was deleted while editing it. Just add it in then */
	       config.templates = g_list_append(config.templates, tmpl);
	  }

     }

     save_config();

     fill_clist_templates(current_template_clist);

     gtk_widget_destroy(window);

}


/*
 * extract template objectclasses clist from window, and build
 * a (malloc'ed) gq_template struct with it
 */
struct gq_template *window2template(GtkWidget *window)
{
     GList *list;
     GtkWidget *templatenamebox, *templatelist;
     struct gq_template *tmpl;
     int i;
     char *objectclass;

     if( (templatelist = gtk_object_get_data(GTK_OBJECT(window), "templatelist")) == NULL)
	  return(NULL);

     if( (templatenamebox = gtk_object_get_data(GTK_OBJECT(window), "templatenamebox")) == NULL)
	  return(NULL);

     if( (tmpl = MALLOC(sizeof(struct gq_template), "struct gq_template")) == NULL)
	  return(NULL);

     strncpy(tmpl->name, gtk_entry_get_text(GTK_ENTRY(templatenamebox)), 128);

     list = NULL;
     for(i = 0; i < GTK_CLIST(templatelist)->rows; i++) {
	  gtk_clist_get_text(GTK_CLIST(templatelist), i, 0, &objectclass);
	  list = g_list_append(list, g_strdup(objectclass));

     }
     tmpl->objectclasses = list;

     return(tmpl);
}


/*
 * fill clist with names of templates
 */
void fill_clist_templates(GtkWidget *clist)
{
     GList *list;
     struct gq_template *tmpl;
     char *dummy[1];

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     list = config.templates;
     while(list) {
	  if( (tmpl = (struct gq_template *) list->data)) {
	       dummy[0] = tmpl->name;
	       gtk_clist_append(GTK_CLIST(clist), dummy);
	  }
	  list = list->next;
     }

     gtk_clist_thaw(GTK_CLIST(clist));

}


/*
 * find selected item in clist by clist->data (char *)
 * return clist row string
 */
char *get_clist_selection(GtkWidget *clist)
{
     GList *list;
     char *objectclass;

     objectclass = NULL;
     if( (list = GTK_CLIST(clist)->selection))
	  gtk_clist_get_text(GTK_CLIST(clist), GPOINTER_TO_INT(list->data),
			     0, &objectclass);

     return(objectclass);
}


/*
 * find selected item in clist by row string
 * return row number
 */
int get_clist_row_by_text(GtkWidget *clist, char *text)
{
     int i, row;
     char *str;

     row = -1;
     for(i = 0; i < GTK_CLIST(clist)->rows; i++) {
	  gtk_clist_get_text(GTK_CLIST(clist), i, 0, &str);
	  if(!strcasecmp(str, text)) {
	       row = i;
	       break;
	  }
     }

     return(row);
}


#endif /* HAVE_LDAP_STR2OBJECTCLASS */
