/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <config.h>

#include "common.h"
#include "mainwin.h"
#include "search.h"
#include "configfile.h"
#include "browse.h"
#include "prefs.h"
#include "util.h"
#include "template.h"
#include "filter.h"
#include "COPYING.h"
#include "i18n.h"
#include "debug.h"

#include "../icons/logo.xpm"

#ifdef HAVE_LDAP_STR2OBJECTCLASS
#    include "schemabrowse.h"
#endif

GtkWidget *mainwin = NULL;
GtkWidget *mainbook = NULL;
GtkWidget *statusbar;
GList *tablist = NULL;

void fill_serverlist_combo(GtkWidget *combo)
{
     GList *serverlist;
     struct ldapserver *server;

     if(combo == NULL)
	  return;

     serverlist = NULL;
     server = config.ldapservers;
     while(server) {
	  serverlist = g_list_append(serverlist, server->name);
	  server = server->next;
     }

     if(!serverlist)
	  /* all servers were deleted -- pass an empty string to the combo */
	  serverlist = g_list_append(serverlist, "");

     gtk_combo_set_popdown_strings(GTK_COMBO(combo), serverlist);

     g_list_free(serverlist);

}


void gq_exit(GtkWidget *widget, gpointer *data)
{

     cleanup();
     gtk_main_quit();

}


void cleanup(void)
{
     GList *templatelist, *oclist, *filterlist;
     struct ldapserver *server, *nextserver;
     struct gq_template *tmpl;

     cleanup_all_tabs();

     /* free malloc'd ldapservers */
     server = config.ldapservers;
     while(server) {
	  nextserver = server->next;
	  close_connection(server, TRUE);
	  FREE(server, "struct ldapserver");
	  server = nextserver;
     }

     /* free templates */
     templatelist = config.templates;
     while(templatelist) {
	  tmpl = (struct gq_template *) templatelist->data;
	  oclist = tmpl->objectclasses;
	  while(oclist) {
	       g_free(oclist->data);
	       oclist = oclist->next;
	  }
	  g_list_free(tmpl->objectclasses);
	  FREE(tmpl, "struct gq_template");

	  templatelist = templatelist->next;
     }
     g_list_free(config.templates);

     /* free filters */
     filterlist = config.filters;
     while(filterlist) {
	  FREE(filterlist->data, "struct gq_filter");
	  filterlist = filterlist->next;
     }
     g_list_free(config.filters);

#ifdef DEBUG
     report_num_mallocs();
#endif

}


void create_mainwin(void)
{
     GtkWidget *outer_vbox, *main_vbox, *menubar, *menuitem, *submenu;
     GtkWidget *File, *menuFile, *New, *Close, *Quit;
     GtkWidget *Search, *Browse, *Schema;
     GtkWidget *menuHelp, *Help, *License, *About;
     GtkWidget *Filters, *menuFilters;
     GtkWidget *handlebox;
     GtkAccelGroup *accel_group;
     GList *filterlist;
     struct gq_filter *filter;

     mainwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_default_size(GTK_WINDOW(mainwin), 770, 478);
     gtk_container_border_width(GTK_CONTAINER(mainwin), 0);
     gtk_signal_connect(GTK_OBJECT(mainwin), "destroy",
			GTK_SIGNAL_FUNC(gq_exit),
			NULL);
     gtk_window_set_title(GTK_WINDOW(mainwin), _("GQ"));
     gtk_window_set_policy(GTK_WINDOW(mainwin), FALSE, TRUE, FALSE);

     outer_vbox = gtk_vbox_new(FALSE, 2);
     gtk_container_border_width(GTK_CONTAINER(outer_vbox), 0);
     gtk_widget_show(outer_vbox);
     gtk_container_add(GTK_CONTAINER(mainwin), outer_vbox);

     main_vbox = gtk_vbox_new(FALSE, 2);
     gtk_object_set_data(GTK_OBJECT(mainwin), "main_vbox", main_vbox);
     gtk_container_border_width(GTK_CONTAINER(main_vbox), 4);
     gtk_widget_show(main_vbox);
     gtk_box_pack_end(GTK_BOX(outer_vbox), main_vbox, TRUE, TRUE, 1);

     accel_group = gtk_accel_group_new();
     gtk_accel_group_attach(accel_group, GTK_OBJECT(mainwin));

     handlebox = gtk_handle_box_new();
     gtk_widget_show(handlebox);
     gtk_box_pack_start(GTK_BOX(outer_vbox), handlebox, FALSE, TRUE, 0);

     menubar = gtk_menu_bar_new();
     gtk_widget_show(menubar);
     gtk_container_add(GTK_CONTAINER(handlebox), menubar);

     /* File menu */
     File = gtk_menu_item_new_with_label(_("File"));
     gtk_widget_show(File);
     gtk_container_add(GTK_CONTAINER(menubar), File);

     menuFile = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(File), menuFile);

     /* File | New */
     New = gtk_menu_item_new_with_label(_("New tab"));
     gtk_widget_show(New);
     gtk_container_add(GTK_CONTAINER(menuFile), New);
     submenu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(New), submenu);

     /* File | New | Search */
     Search = gtk_menu_item_new_with_label(_("Search"));
     gtk_widget_show(Search);
     gtk_menu_append(GTK_MENU(submenu), Search);
     gtk_signal_connect_object(GTK_OBJECT(Search), "activate",
			       GTK_SIGNAL_FUNC(new_modetab),
			       GINT_TO_POINTER(SEARCH_MODE));
     gtk_widget_add_accelerator(Search, "activate", accel_group, 'S',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

     /* File | New | Browse */
     Browse = gtk_menu_item_new_with_label(_("Browse"));
     gtk_widget_show(Browse);
     gtk_menu_append(GTK_MENU(submenu), Browse);
     gtk_signal_connect_object(GTK_OBJECT(Browse), "activate",
			       GTK_SIGNAL_FUNC(new_modetab),
			       GINT_TO_POINTER(BROWSE_MODE));
     gtk_widget_add_accelerator(Browse, "activate", accel_group, 'B',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
     /* ctrl-b is used by text widgets, so the searchterm textbox that
	always has focus in search mode blocks the above accelerator...*/
     gtk_signal_connect(GTK_OBJECT(mainwin), "key_press_event",
			GTK_SIGNAL_FUNC(ctrl_b_hack),
			(gpointer) Browse);

     /* File | New | Schema */
     Schema = gtk_menu_item_new_with_label(_("Schema"));
     gtk_widget_show(Schema);
     gtk_menu_append(GTK_MENU(submenu), Schema);
     gtk_widget_add_accelerator(Schema, "activate", accel_group, 'Z',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     gtk_signal_connect_object(GTK_OBJECT(Schema), "activate",
			       GTK_SIGNAL_FUNC(new_modetab),
			       GINT_TO_POINTER(SCHEMA_MODE));
#else
     gtk_widget_set_sensitive(Schema, FALSE);
#endif

     /* File | Preferences */
     menuitem = gtk_menu_item_new_with_label(_("Preferences"));
     gtk_widget_show(menuitem);
     gtk_container_add(GTK_CONTAINER(menuFile), menuitem);
     gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			GTK_SIGNAL_FUNC(create_prefs_window),
			NULL);
     gtk_widget_add_accelerator(menuitem, "activate", accel_group, 'P',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

     /* File | Close */
     Close = gtk_menu_item_new_with_label(_("Close"));
     gtk_widget_show(Close);
     gtk_container_add(GTK_CONTAINER(menuFile), Close);
     gtk_signal_connect_object(GTK_OBJECT(Close), "activate",
			       GTK_SIGNAL_FUNC(close_current_tab), NULL);
     gtk_widget_add_accelerator(Close, "activate", accel_group, 'Q',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

     /* File | Quit */
     Quit = gtk_menu_item_new_with_label(_("Quit"));
     gtk_widget_show(Quit);
     gtk_container_add(GTK_CONTAINER(menuFile), Quit);
     gtk_signal_connect_object(GTK_OBJECT(Quit), "activate",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       GTK_OBJECT(mainwin));


     /* Filters menu */
     Filters = gtk_menu_item_new_with_label(_("Filters"));
     gtk_widget_show(Filters);
     gtk_container_add(GTK_CONTAINER(menubar), Filters);

     menuFilters = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(Filters), menuFilters);
     gtk_object_set_data(GTK_OBJECT(mainwin), "filtermenu", menuFilters);

     /* Filters | New */
     New = gtk_menu_item_new_with_label(_("New filter"));
     gtk_widget_show(New);
     gtk_container_add(GTK_CONTAINER(menuFilters), New);
     submenu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(New), submenu);

     /* Filters | New | From Search tab */
     menuitem = gtk_menu_item_new_with_label(_("From Search tab"));
     gtk_widget_show(menuitem);
     gtk_menu_append(GTK_MENU(submenu), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(add_filter),
			       NULL);

     /* Filters | New | Filter Editor */
     menuitem = gtk_menu_item_new_with_label(_("Filter editor"));
     gtk_widget_show(menuitem);
     gtk_menu_append(GTK_MENU(submenu), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(add_new_filter_callback),
			       NULL);

     /* Filters | Edit Filters */
     menuitem = gtk_menu_item_new_with_label(_("Edit Filters"));
     gtk_widget_show(menuitem);
     gtk_container_add(GTK_CONTAINER(menuFilters), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(show_filters),
			       NULL);

     /* Filters separator */
     menuitem = gtk_menu_item_new();
     gtk_widget_show(menuitem);
     gtk_container_add(GTK_CONTAINER(menuFilters), menuitem);

     /* Filters | list of filters */
     filterlist = config.filters;
     while(filterlist) {
	  filter = (struct gq_filter *) filterlist->data;
	  menuitem = gtk_menu_item_new_with_label(filter->name);
	  gtk_object_set_data(GTK_OBJECT(menuitem), "filter", filter);
	  gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
				    GTK_SIGNAL_FUNC(filter_selected), (GtkObject *) filter);
	  gtk_container_add(GTK_CONTAINER(menuFilters), menuitem);
	  gtk_widget_show(menuitem);

	  filterlist = filterlist->next;
     }


     /* Help menu */
     Help = gtk_menu_item_new_with_label(_("Help"));
     gtk_widget_show(Help);
     gtk_container_add(GTK_CONTAINER(menubar), Help);
     gtk_menu_item_right_justify(GTK_MENU_ITEM(Help));

     menuHelp = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(Help), menuHelp);

     /* Help | License */
     License = gtk_menu_item_new_with_label(_("License"));
     gtk_widget_show(License);
     gtk_container_add(GTK_CONTAINER(menuHelp), License);
     gtk_signal_connect(GTK_OBJECT(License), "activate",
			GTK_SIGNAL_FUNC(create_license_window), NULL);

     /* Help | About */
     About = gtk_menu_item_new_with_label(_("About"));
     gtk_widget_show(About);
     gtk_container_add(GTK_CONTAINER(menuHelp), About);
     gtk_signal_connect(GTK_OBJECT(About), "activate",
			GTK_SIGNAL_FUNC(create_about_window), NULL);

     mainbook = gtk_notebook_new();
     gtk_widget_show(mainbook);
     GTK_WIDGET_UNSET_FLAGS(GTK_NOTEBOOK(mainbook), GTK_CAN_FOCUS);
     gtk_box_pack_start(GTK_BOX(main_vbox), mainbook, TRUE, TRUE, 0);

     statusbar = gtk_statusbar_new();
     gtk_object_set_data(GTK_OBJECT(mainwin), "statusbar", statusbar);
     gtk_widget_show(statusbar);
     gtk_box_pack_end(GTK_BOX(main_vbox), statusbar, FALSE, FALSE, 5);
     gtk_widget_set_sensitive(statusbar, TRUE);

     new_modetab(SEARCH_MODE);
     new_modetab(BROWSE_MODE | 32768);
     new_modetab(SCHEMA_MODE | 32768);

     gtk_signal_connect(GTK_OBJECT(mainbook), "switch-page",
			GTK_SIGNAL_FUNC(switchpage_refocus), NULL);

     gtk_widget_show(mainwin);

}


void new_modetab(int mode)
{
     GHashTable *hash;
     GtkWidget *vbox, *label, *focusbox;
     int focus;

     /* hack, hack */
     focus = !(mode & 32768);
     mode &= 32767;

     hash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(hash, "mode", GINT_TO_POINTER(mode));

     switch(mode) {
     case SEARCH_MODE:
	  label = gtk_label_new(_("Search"));
	  new_searchmode(hash);
	  break;
     case BROWSE_MODE:
	  label = gtk_label_new(_("Browse"));
	  new_browsemode(hash);
	  break;
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     case SCHEMA_MODE:
	  label = gtk_label_new(_("Schema"));
	  new_schemamode(hash);
	  break;
#endif
     default:
	  g_hash_table_destroy(hash);
	  return;
     }

     gtk_widget_show(label);
     vbox = g_hash_table_lookup(hash, "vbox");
     gtk_notebook_append_page(GTK_NOTEBOOK(mainbook), vbox, label);

     if(focus) {
	  gtk_notebook_set_page(GTK_NOTEBOOK(mainbook), -1);

	  focusbox = g_hash_table_lookup(hash, "focus");
	  if(focusbox)
	       gtk_widget_grab_focus(focusbox);
     }

     tablist = g_list_append(tablist, hash);

}


void switchpage_refocus(GtkNotebook *notebook, GtkNotebookPage *page,
			int pagenum, gpointer data) {
     GHashTable *hash;
     GtkWidget *focusbox;

     hash = g_list_nth_data(tablist, pagenum);
     if(!hash)
	  return;

     focusbox = g_hash_table_lookup(hash, "focus");
     if(focusbox) {
	  gtk_widget_grab_focus(focusbox);
	  gtk_widget_grab_default(focusbox);
	  gtk_editable_select_region(GTK_EDITABLE(focusbox), 0, -1);
     }

}


void cleanup_all_tabs(void)
{
     GList *tabs;
     GHashTable *hash;

     /* don't waste time refocusing on disappearing tabs */
     gtk_signal_disconnect_by_func(GTK_OBJECT(mainbook),
				   GTK_SIGNAL_FUNC(switchpage_refocus), NULL);

     tabs = tablist;
     while(tabs) {
	  hash = (GHashTable *) tabs->data;
	  cleanup_tab(hash);

	  tabs = tabs->next;
     }

}


void close_current_tab(void)
{
     GHashTable *hash;
     int tab;

     tab = gtk_notebook_get_current_page(GTK_NOTEBOOK(mainbook));
     hash = g_list_nth_data(tablist, tab);
     cleanup_tab(hash);
     g_hash_table_destroy(hash);
     gtk_notebook_remove_page(GTK_NOTEBOOK(mainbook), tab);
     tablist = g_list_remove(tablist, hash);
     if(tablist == NULL)
	  gtk_widget_destroy(mainwin);

}


void cleanup_tab(GHashTable *hash)
{
     int mode;

     mode = GPOINTER_TO_INT(g_hash_table_lookup(hash, "mode"));
     switch(mode) {
     case SEARCH_MODE:
	  cleanup_search_mode(hash);
	  break;
     case BROWSE_MODE:
	  cleanup_browse_mode(hash);
	  break;
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     case SCHEMA_MODE:
	  cleanup_schema_mode();
	  break;
#endif
     }

}


void update_serverlist(void)
{
     GList *tabs;
     GHashTable *hash;
     GtkWidget *servcombo;
     int mode;

     tabs = tablist;
     while(tabs) {
	  hash = (GHashTable *) tabs->data;
	  mode = GPOINTER_TO_INT(g_hash_table_lookup(hash, "mode"));
	  switch(mode) {
	  case SEARCH_MODE:
	       servcombo = g_hash_table_lookup(hash, "serverlist_combo");
	       fill_serverlist_combo(servcombo);
	       break;
	  case BROWSE_MODE:
	       /* XXX add browse tabs here */
	       break;
	  case SCHEMA_MODE:
	       /* XXX add schema tabs here */
	       break;

	  }

	  tabs = tabs->next;
     }


}


void create_license_window(void)
{
     GtkWidget *window, *vbox1, *hbox, *vbox2, *text, *hbox0;
     GtkWidget *vscroll, *ok_button;
     GtkStyle *license_style;

     window = gtk_dialog_new();
     gtk_object_set_data(GTK_OBJECT(window), "close_func",
			 close_license_window);
     gtk_widget_set_usize(window, 540, 400);
     gtk_window_set_title(GTK_WINDOW(window), _("License"));

     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_container_border_width(GTK_CONTAINER(vbox1), 20);
     gtk_widget_show(vbox1);

     hbox = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox, TRUE, TRUE, 0);

     license_style = gtk_style_new();
     license_style->font = gdk_font_load("fixed");
     gtk_object_set_data(GTK_OBJECT(window), "style", license_style);

     text = gtk_text_new(NULL, NULL);
     gtk_widget_set_style(text, license_style);
     gtk_text_set_point(GTK_TEXT(text), 0);
     gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL,
		     license, strlen(license));
     gtk_widget_show(text);
     gtk_text_set_editable(GTK_TEXT(text), FALSE);
     gtk_box_pack_start(GTK_BOX(hbox), text, TRUE, TRUE, 00);

     vscroll = gtk_vscrollbar_new(GTK_TEXT(text)->vadj);
     gtk_box_pack_start(GTK_BOX(hbox), vscroll, FALSE, FALSE, 0);
     gtk_widget_show(vscroll);

     vbox2 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox2);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(vbox2), hbox0, TRUE, FALSE, 0);

     ok_button = gtk_button_new_with_label(_("OK"));
     gtk_widget_show(ok_button);
     gtk_box_pack_start(GTK_BOX(hbox0), ok_button,
			FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(ok_button);
     gtk_signal_connect_object(GTK_OBJECT(window), "delete_event",
			       (GtkSignalFunc) close_license_window,
			       GTK_OBJECT(window));
     gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			       (GtkSignalFunc) close_license_window,
			       GTK_OBJECT(window));
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
			       (GtkSignalFunc) func_on_esc,
			       (gpointer) window);

     gtk_widget_show(window);

}


void close_license_window(GtkWidget *window)
{
     GtkStyle *style;

     style = gtk_object_get_data(GTK_OBJECT(window), "style");
     gtk_style_unref(style);
     gtk_widget_destroy(window);

}


GtkWidget *create_about_window(void)
{
     GtkWidget *about_window;
     GtkWidget *dialog_vbox2, *hbox0, *hbox;
     GtkWidget *about_label;
     GtkWidget *dialog_action_area2;
     GtkWidget *about_ok_button;
     GtkWidget *pixmap;
     GdkPixmap *gq_icon;
     GdkBitmap *gq_icon_mask;
     char title[32], about_text[512];

     about_window = gtk_dialog_new();
     gtk_widget_realize(GTK_WIDGET(about_window));
     gtk_container_border_width(GTK_CONTAINER(about_window), 12);
     snprintf(title, sizeof(title), _("About GQ"));
     gtk_window_set_title(GTK_WINDOW(about_window), title);
     gtk_window_set_policy(GTK_WINDOW(about_window), FALSE, FALSE, FALSE);

     dialog_vbox2 = GTK_DIALOG(about_window)->vbox;
     gtk_widget_show(dialog_vbox2);

     hbox = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox);
     gtk_box_pack_start(GTK_BOX(dialog_vbox2), hbox, TRUE, TRUE, 10);

     gq_icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(about_window)->window,
					    &gq_icon_mask,
					    &about_window->style->white,
					    logo_xpm);
     pixmap = gtk_pixmap_new(gq_icon, gq_icon_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox), pixmap, TRUE, TRUE, 10);

     snprintf(about_text, sizeof(about_text),
	      "GQ %s\n\n\n%s", VERSION, about_blurb);
     about_label = gtk_label_new(about_text);
     gtk_widget_show(about_label);
     gtk_box_pack_start(GTK_BOX(hbox), about_label, FALSE, FALSE, 4);

     dialog_action_area2 = GTK_DIALOG(about_window)->action_area;
     gtk_container_border_width(GTK_CONTAINER(dialog_action_area2), 10);
     gtk_widget_show(dialog_action_area2);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(dialog_action_area2), hbox0,
			TRUE, FALSE, 0);

     about_ok_button = gtk_button_new_with_label(_("OK"));
     gtk_widget_show(about_ok_button);
     gtk_box_pack_start(GTK_BOX(hbox0), about_ok_button,
			TRUE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(about_ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(about_ok_button);
     gtk_container_border_width(GTK_CONTAINER(about_ok_button), 7);
     gtk_signal_connect_object(GTK_OBJECT(about_ok_button), "clicked",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       GTK_OBJECT(about_window));

     gtk_signal_connect_object(GTK_OBJECT(about_window), "key_press_event",
			       GTK_SIGNAL_FUNC(close_on_esc), (gpointer) about_window);

     gtk_widget_show(about_window);

     return(about_window);
}


gboolean ctrl_b_hack(GtkWidget *widget, GdkEventKey *event, gpointer obj)
{

     if(event && event->type == GDK_KEY_PRESS &&
	event->state & GDK_CONTROL_MASK && event->keyval == GDK_b) {
	 /*	(event->keyval == GDK_b || event->keyval == GDK_c)) { */
	  gtk_signal_emit_by_name(GTK_OBJECT(obj), "activate");
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}

