/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen
    Parts: Copyright (C) 2002 Peter Stamfest <peter@stamfest.at>

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: browse.c,v 1.52 2002/06/21 00:34:32 stamfest Exp $ */

/*  #define DEBUG_DND */
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <config.h>

#include "common.h"
#include "configfile.h"
#include "browse.h"
#include "mainwin.h"
#include "template.h"
#include "util.h"
#include "tinput.h"
#include "errorchain.h"
#include "ldif.h"
#include "formfill.h"
#include "encode.h"
#include "input.h"
#include "debug.h"
#include "ldapops.h"
#include "i18n.h"

#ifdef BROWSER_DND
#include "browse-dnd.h"
#endif

/*  #include "../icons/warning.xpm" */

static gboolean button_press_on_tree_item(GtkWidget *tree,
					  GdkEventButton *event,
					  GHashTable *hash);

/* A GtkDestroyNotify callback to be used as a destroy function for
   browse_entry objects attached to gtk objects */
void destroy_browse_entry(browse_entry *entry) 
{
     if (!entry) return;
     if (entry->destroy) {
	  entry->destroy(entry);
     } else {
	  free(entry);
     }
}

/* RTTI info variable */
int dn_browse_entry_rtti = 0;

/* 
 * Destructor for dn_browse_entry objects
 */
static void destroy_dn_browse_entry(dn_browse_entry *entry) 
{
     if (!entry) return;
     if (entry->dn) g_free(entry->dn);
     free(entry);
}

static void dn_browse_entry_expand(dn_browse_entry *entry,
				   GtkCTree *ctree,
				   GtkCTreeNode *node,
				   GHashTable *hash)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     struct ldapserver *server;
     int msg, rc, num_children, update_counter;
     char *dn, message[MAX_DN_LEN + 21];
     char *dummy[] = { "dummy", NULL };
     char *labels[] = { NULL, NULL };
     char *c;
     dn_browse_entry *new_entry;
     GtkCTreeNode *new_item;
     char **exploded_dn = 0;

     if (!entry->seen) {
	  server = server_from_node(ctree, node);
/*  	  printf("server=%08lx host=%s dn=%s\n", (long) server, */
/*  		 server->ldaphost, */
/*  		 entry->dn); */
	  
	  if( (ld = open_connection(server)) == NULL) {
	       return;
	  }

#if HAVE_LDAP_CLIENT_CACHE
	  if (entry->uncache) {
	       ldap_uncache_entry(ld, entry->dn);
	       entry->uncache = FALSE;
	  }
#endif

	  gtk_clist_freeze(GTK_CLIST(ctree));
	  
	  while (GTK_CTREE_ROW(node)->children) {
	       gtk_ctree_remove_node(ctree, GTK_CTREE_ROW(node)->children);
	  }

	  snprintf(message, sizeof(message),
		   _("onelevel search on %s"), 
		   c = decoded_string(entry->dn));

	  if (c) free(c);

	  statusbar_msg(message);
	  
	  msg = ldap_search(ld, entry->dn, LDAP_SCOPE_ONELEVEL,
			    "objectclass=*", dummy, 1);
	  
	  if(msg == -1) {
	       statusbar_msg(ldap_err2string(msg));
	       close_connection(server, FALSE);
	       gtk_clist_thaw(GTK_CLIST(ctree));
	       return;
	  }
	  
	  num_children = update_counter = 0;

	  while( (rc = ldap_result(ld, msg, 0,
				   NULL, &res)) == LDAP_RES_SEARCH_ENTRY) {
	       for(e = ldap_first_entry(ld, res) ; e != NULL ;
		   e=ldap_next_entry(ld, e)) {
		    dn = ldap_get_dn(ld, e);

		    /* explode DN */
		    exploded_dn = gq_ldap_explode_dn(dn, FALSE);
		    labels[0] = decoded_string(exploded_dn[0]);

		    new_entry = (dn_browse_entry *) new_dn_browse_entry(dn);

		    new_item = gtk_ctree_insert_node(ctree,
						     node, NULL,
						     labels, 
						     0,
						     NULL, NULL, NULL, NULL,
						     FALSE, FALSE);
		    
		    gtk_ctree_node_set_row_data_full(ctree,
						     new_item,
						     new_entry,
						     (GtkDestroyNotify) destroy_browse_entry);

		    /* add dummy node */
		    new_item = gtk_ctree_insert_node(ctree,
						     new_item, NULL,
						     dummy, 
						     0,
						     NULL, NULL, NULL, NULL,
						     TRUE, FALSE);
		    
		    free(labels[0]);
		    gq_exploded_free(exploded_dn);
		    free(dn);

		    num_children++;
		    update_counter++;
		    if(update_counter >= 100) {
			 make_message(message, sizeof(message), num_children,
				      _("entry"), _("entries"),
				      _("found (running)"));
			 statusbar_msg(message);
			 update_counter = 0;
		    }
	       }
	       ldap_msgfree(res);
	  }
	  if (res) ldap_msgfree(res);
	  entry->leaf = (num_children == 0);

	  make_message(message, sizeof(message), num_children,
		       _("entry"), _("entries"), _("found (finished)"));
	  statusbar_msg(message);

	  gtk_clist_thaw(GTK_CLIST(ctree));

	  close_connection(server, FALSE);
	  
	  entry->seen = TRUE;
     }

     if (g_hash_table_lookup(hash, "expand-all")) {
	  GtkCTreeNode *n;
	  gtk_clist_freeze(GTK_CLIST(ctree));
	  for (n = GTK_CTREE_ROW(node)->children ; n ;
	       n = GTK_CTREE_NODE_NEXT(n)) {
	       gtk_ctree_expand(ctree, n);
	  }
	  gtk_clist_thaw(GTK_CLIST(ctree));
     }


}

static void browse_edit_from_entry(dn_browse_entry *entry,
				   GtkCTree *ctreeroot,
				   GtkCTreeNode *ctreenode,
				   GHashTable *hash)
{
     GList *oldlist, *newlist, *tmplist;
     GHashTable *browsehash;
     GString *cur_dn;
     GtkWidget *pane2_scrwin, *pane2_vbox;
     struct ldapserver *server;
     char *dn;

     if (ctreenode == NULL)
	  return;

     if( (server = server_from_node(ctreeroot, ctreenode)) == NULL)
	  return;

     dn = entry->dn;
     cur_dn = g_hash_table_lookup(hash, "cur_dn");
     g_string_assign(cur_dn, dn);
     g_hash_table_insert(hash, "cur_dn", cur_dn);
     
     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");

     /* delete old browsehash (if any) */
     browsehash = g_hash_table_lookup(hash, "browsehash");
     if (browsehash) {
	  browsehash_free(browsehash);
	  g_hash_table_remove(hash, "browsehash");
     }

     browsehash = g_hash_table_new(g_str_hash, g_str_equal);
     /* store browsehash */
     g_hash_table_insert(hash, "browsehash", browsehash);

     g_hash_table_insert(browsehash, "server", server);
     g_hash_table_insert(browsehash, "edit", GINT_TO_POINTER(1));

     tmplist = NULL;
     oldlist = formlist_from_entry(server, dn, 0);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     oldlist = add_schema_attrs(server, oldlist);
#endif

     if(oldlist) {
	  g_hash_table_insert(browsehash, "oldlist", oldlist);
	  newlist = dup_formlist(oldlist);
	  g_hash_table_insert(browsehash, "formlist", newlist);
	  g_hash_table_insert(browsehash, "olddn", g_strdup(dn));
	  g_hash_table_insert(browsehash, "dn", g_strdup(dn));

	  if (ctreeroot) {
	      g_hash_table_insert(browsehash, "ctreeroot", ctreeroot);
	      g_hash_table_insert(browsehash, "ctree_refresh", GTK_CTREE_ROW(ctreenode)->parent);
	      g_hash_table_insert(browsehash, "selected_ctree_refresh", GTK_CTREE_ROW(ctreenode)->parent);
	  }


	  /* XXX should free etc first */
	  pane2_scrwin = g_hash_table_lookup(hash, "pane2_scrwin");
	  gtk_container_remove(GTK_CONTAINER(pane2_scrwin),
			       GTK_BIN(pane2_scrwin)->child);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  g_hash_table_insert(browsehash, "target_vbox", pane2_vbox);
	  gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
	  gtk_widget_show(pane2_vbox);
	  gtk_widget_set_parent_window(pane2_vbox, (GdkWindow *) mainwin);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin), pane2_vbox);

	  create_form_content(browsehash);
	  build_inputform(browsehash);
     }
     else {
	  browsehash_free(browsehash);
	  g_hash_table_remove(hash, "browsehash");
     }

}

static void dn_browse_entry_refresh(dn_browse_entry *entry,
				    GtkCTree *ctree,
				    GtkCTreeNode *node,
				    GHashTable *hash)
{
     refresh_subtree(ctree, node);
     entry->base_methods.select((browse_entry *) entry, ctree, node, hash);
}

char* dn_browse_entry_get_name(dn_browse_entry *entry,
			       gboolean long_form)
{
     char **exploded_dn = gq_ldap_explode_dn(entry->dn, FALSE);
     char *l = decoded_string(exploded_dn[0]);
     char *g;

     gq_exploded_free(exploded_dn);

     /* impedance match -> malloc to g_malloc */
     g = g_strdup(l);
     free(l);
     return g;
}

/* 
 * Constructor for dn_browse_entry objects taking the dn
 */
browse_entry *new_dn_browse_entry(const char *dn) 
{
     dn_browse_entry *e;
     e = calloc(sizeof(dn_browse_entry), 1);
     e->base_methods.rtti_info = &dn_browse_entry_rtti;

     if (dn != NULL) e->dn = g_strdup(dn);
     e->seen = FALSE;
     e->leaf = FALSE;

     e->base_methods.destroy = 
	  (browse_entry_destructor) destroy_dn_browse_entry;
     e->base_methods.expand = 
	  (browse_entry_expand) dn_browse_entry_expand;
     e->base_methods.select = 
	  (browse_entry_select) browse_edit_from_entry;
     e->base_methods.refresh = 
	  (browse_entry_refresh) dn_browse_entry_refresh;
     e->base_methods.get_name = 
	  (browse_entry_get_name) dn_browse_entry_get_name;

     return (browse_entry *) e;
}

/* RTTI info variable */
int server_browse_entry_rtti = 0;

/*
 *  Really add a single suffix to the tree 
 */
static void add_suffix(server_browse_entry *entry, 
		       GtkCTree *ctreeroot, GtkCTreeNode *node,
		       char *suffix)
{
     GtkCTreeNode *new_item;
     char *labels[] = { suffix, NULL };
     browse_entry *new_entry = new_dn_browse_entry(suffix);
     
     new_item = gtk_ctree_insert_node(ctreeroot,
				      node, NULL,
				      labels, 
				      0,
				      NULL, NULL, NULL, NULL,
				      FALSE, FALSE);
     
     gtk_ctree_node_set_row_data_full(ctreeroot,
				      new_item,
				      new_entry,
				      (GtkDestroyNotify) destroy_browse_entry);

     /* add dummy node to have something to expand */
     labels[0] = "DUMMY";
     
     new_item = gtk_ctree_insert_node(ctreeroot,
				      new_item, NULL,
				      labels, 
				      0,
				      NULL, NULL, NULL, NULL,
				      TRUE, FALSE);
}

static void server_browse_entry_expand(server_browse_entry *entry, 
				       GtkCTree *ctree, GtkCTreeNode *node,
				       GHashTable *hash)
{
     GList *suffixes = NULL, *next;
     if (!entry->once_expanded) {
/*  	  printf("expanding %s\n", entry->server->name); */
	
	  while (GTK_CTREE_ROW(node)->children) {
	       gtk_ctree_remove_node(ctree, GTK_CTREE_ROW(node)->children);
	  }
	  entry->once_expanded = 1;
	  
	  suffixes = get_suffixes(entry->server);
	  
	  gtk_clist_freeze(GTK_CLIST(ctree));

	  for (next = suffixes ; next ; next = g_list_next(next) ) {
	       add_suffix(entry, ctree, node, next->data);
	       g_free(next->data);
	       next->data = NULL;
	  }

	  gtk_clist_thaw(GTK_CLIST(ctree));

	  g_list_free(suffixes);
     }
}

/*
 * a server was selected in the tree widget.
 *
 * FIXME: put up the server's editable info here? Easy to do, but
 *        does it belong here?
 */
static void server_browse_entry_selected(server_browse_entry *entry,
					 GtkCTree *ctree,
					 GtkCTreeNode *node,
					 GHashTable *hash)
{
     GString *cur_dn;
     GtkWidget *pane2_scrwin, *pane2_vbox, *label;
     char *server_name;

     server_name = entry->server->name; /* dn_by_node(node); */

     cur_dn = g_hash_table_lookup(hash, "cur_dn");
     if(strncmp(cur_dn->str, server_name, cur_dn->len)) {
	  g_string_assign(cur_dn, server_name);
	  g_hash_table_insert(hash, "cur_dn", cur_dn);

	  pane2_scrwin = g_hash_table_lookup(hash, "pane2_scrwin");
	  pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
/*  	  gtk_widget_destroy(pane2_vbox); */
	  /* remove the viewport of the scrolled window. This should
             _really_ destroy the widgets below it. The pane2_scrwin
             is a GtkBin Object and thus has only one child, use this
             to obtain the viewport */

	  gtk_container_remove(GTK_CONTAINER(pane2_scrwin),
			       GTK_BIN(pane2_scrwin)->child);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
	  gtk_widget_show(pane2_vbox);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
						pane2_vbox);

	  label = gtk_label_new(server_name);
	  gtk_widget_show(label);
	  gtk_box_pack_start(GTK_BOX(pane2_vbox), label, FALSE, FALSE, 0);
     }
}






static void server_browse_entry_refresh(server_browse_entry *entry,
					GtkCTree *ctree,
					GtkCTreeNode *node,
					GHashTable *hash)
{
     entry->once_expanded = 0;

     gtk_clist_freeze(GTK_CLIST(ctree));

     /* toggle expansion twice to fire the expand callback */
     gtk_ctree_toggle_expansion(ctree, node);
     gtk_ctree_toggle_expansion(ctree, node);

     gtk_clist_thaw(GTK_CLIST(ctree));
}

char* server_browse_entry_get_name(server_browse_entry *entry,
				   gboolean long_form)
{
     return g_strdup(entry->server->name);
}

browse_entry *new_server_browse_entry(struct ldapserver *server) 
{
     server_browse_entry *e;
     e = calloc(sizeof(server_browse_entry), 1);
     e->base_methods.rtti_info = &server_browse_entry_rtti;
    
     e->base_methods.expand = 
	  (browse_entry_expand) server_browse_entry_expand;
     e->base_methods.select =
	  (browse_entry_select) server_browse_entry_selected;
     e->base_methods.refresh =
	  (browse_entry_refresh) server_browse_entry_refresh;
     e->base_methods.get_name =
	  (browse_entry_get_name) server_browse_entry_get_name;

     e->server = server;

     return (browse_entry *) e;
}

/*
 * add all configured servers to the root node
 */
void add_all_servers(GtkCTree *ctree)
{
     struct ldapserver *server;
     browse_entry *entry;
     int server_cnt;
     char message[128];
     GtkCTreeNode *new_item;

     char *labels[2] = { NULL, NULL };

     server_cnt = 0;
     server = config.ldapservers;

     while(server) {
	  labels[0] = server->name;
	  entry = new_server_browse_entry(server);

	  new_item = gtk_ctree_insert_node(ctree,
					   NULL, NULL,
					   labels, 
					   0,
					   NULL, NULL, NULL, NULL,
					   FALSE, FALSE);

	  gtk_ctree_node_set_row_data_full(ctree,
					   new_item,
					   entry,
					   (GtkDestroyNotify) destroy_browse_entry);

	  /* add dummy nodes to get expansion capability  */

	  labels[0] = "DUMMY";

	  new_item = gtk_ctree_insert_node(ctree,
					   new_item, NULL,
					   labels, 
					   0,
					   NULL, NULL, NULL, NULL,
					   TRUE, FALSE);

	  server = server->next;
	  server_cnt++;
     }

     make_message(message, sizeof(message), 
		  server_cnt, _("server"), _("servers"), _("found"));
     statusbar_msg(message);
}

/* GTK callbacks translating to the correct entry object method calls */
void tree_row_expanded(GtkCTree *ctree, 
		       GtkCTreeNode *ctree_node, GHashTable *hash)
{
     browse_entry *entry;

     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree,
							  ctree_node);

     if (!entry) return;
     if (entry->expand) {
	  entry->expand(entry, ctree, ctree_node, hash);
     }
}


void tree_row_selected(GtkCTree *ctree, 
		       GtkCTreeNode *node,
		       int column, GHashTable *hash)
{
     browse_entry *entry;

     /* avoid recursive calls to this handler - this causes crashes!!! */
     if (gtk_signal_n_emissions_by_name(GTK_OBJECT(ctree), 
					"tree-select-row") > 1) {
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(ctree),
				       "tree-select-row");
	  return;
     }
     
     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     
     if (!entry) return;
     if (entry->select) {
	  entry->select(entry, ctree, node, hash);
     }
     g_hash_table_insert(hash, "tree-row-selected", node);
}


void tree_row_refresh(GtkMenuItem *menuitem, 
		      GHashTable *hash)
{
     GtkCTree *ctree = g_hash_table_lookup(hash, "ctreeroot");
     GtkCTreeNode *node = g_hash_table_lookup(hash, "tree-row-popped-up");

     browse_entry *entry;

     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree,
							  node);

     if (!entry) return;
     if (entry->refresh) {
	  entry->refresh(entry, ctree, node, hash);
     }
}

void tree_row_expand_all(GtkMenuItem *menuitem, 
			 GHashTable *hash)
{
     GtkCTree *ctree = g_hash_table_lookup(hash, "ctreeroot");
     GtkCTreeNode *node = g_hash_table_lookup(hash, "tree-row-popped-up");

     if (!ctree) return;
     if (!node) return;

     g_hash_table_insert(hash, "expand-all", "");
     gtk_ctree_expand(ctree, node);
/*       gtk_ctree_pre_recursive(ctree, node, gtk_ctree_expand, NULL); */
     g_hash_table_remove(hash, "expand-all");
}


/* free the entry related hash and data it refers to */
void browsehash_free(GHashTable *browsehash) 
{
     GList *oldlist, *formlist;
     char *c;

     c = g_hash_table_lookup(browsehash, "olddn");
     if (c) {
	  g_free(c);
	  g_hash_table_remove(browsehash, "olddn");
     }

     c = g_hash_table_lookup(browsehash, "dn");
     if (c) {
	  g_free(c);
	  g_hash_table_remove(browsehash, "dn");
     }
     
     oldlist = g_hash_table_lookup(browsehash, "oldlist");
     if (oldlist) {
	  free_formlist(oldlist);
	  g_hash_table_remove(browsehash, "oldlist");
     }
     formlist = g_hash_table_lookup(browsehash, "formlist");
     if (formlist) {
	  free_formlist(formlist);
	  g_hash_table_remove(browsehash, "formlist");
     }

     g_hash_table_destroy(browsehash);
}


void new_browsemode(GHashTable *hash)
{
     GString *cur_dn;
     GtkWidget *ctreeroot, *browsemode_vbox, *spacer;
     GtkWidget *mainpane, *pane2_vbox, *pane1_scrwin, *pane2_scrwin;

     browsemode_vbox = gtk_vbox_new(FALSE, 0);
     g_hash_table_insert(hash, "vbox", browsemode_vbox);

     spacer = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(spacer);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), spacer, FALSE, FALSE, 3);

     mainpane = gtk_hpaned_new();
     gtk_container_border_width(GTK_CONTAINER(mainpane), 2);
     gtk_widget_show(mainpane);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), mainpane, TRUE, TRUE, 0);

     ctreeroot = gtk_ctree_new(1, 0);
     g_hash_table_insert(hash, "ctreeroot", ctreeroot);

     gtk_clist_set_selection_mode(GTK_CLIST(ctreeroot), GTK_SELECTION_BROWSE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(ctreeroot), 0, TRUE);
     if (config.sort_browse) {
	  gtk_clist_set_auto_sort(GTK_CLIST(ctreeroot), TRUE);
     }

     gtk_signal_connect(GTK_OBJECT(ctreeroot), "tree-select-row",
      			GTK_SIGNAL_FUNC(tree_row_selected), hash);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "tree-expand",
			GTK_SIGNAL_FUNC(tree_row_expanded), hash);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "button_press_event",
			GTK_SIGNAL_FUNC(button_press_on_tree_item), hash);

#ifdef BROWSER_DND
     browse_dnd_setup(ctreeroot, hash);
#endif /* BROWSER_DND */

     add_all_servers(GTK_CTREE(ctreeroot));
     gtk_widget_show(ctreeroot);

     pane1_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane1_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane1_scrwin);
     gtk_widget_set_usize(pane1_scrwin, 300, -1);
     gtk_paned_set_position(GTK_PANED(mainpane), 300);
     gtk_paned_add1(GTK_PANED(mainpane), pane1_scrwin);
     gtk_container_add(GTK_CONTAINER(pane1_scrwin), ctreeroot);

     pane2_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane2_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane2_scrwin);
     g_hash_table_insert(hash, "pane2_scrwin", pane2_scrwin);

     pane2_vbox = gtk_vbox_new(FALSE, 5);
     gtk_widget_show(pane2_vbox);
     gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin), pane2_vbox);

     gtk_paned_add2(GTK_PANED(mainpane), pane2_scrwin);

     gtk_widget_show(browsemode_vbox);

     cur_dn = g_string_sized_new(128);
     g_string_assign(cur_dn, "some dummy string");
     g_hash_table_insert(hash, "cur_dn", cur_dn);
}

struct ldapserver *server_from_node(GtkCTree *ctreeroot, 
				    GtkCTreeNode *node)
{
     GtkCTreeRow *row = NULL;
     browse_entry *entry;

     for ( ; node ; node = row->parent ) {
	  row = GTK_CTREE_ROW(node);
	  entry = (browse_entry *) gtk_ctree_node_get_row_data(ctreeroot,
							       node);
	  if (IS_SERVER_ENTRY(entry)) {
	       return ((server_browse_entry*) entry)->server;
	  }
     }
     return NULL;
}

static int dn_row_compare_func(dn_browse_entry *row_data, char *dn)
{
     if (row_data == NULL || !IS_DN_ENTRY(row_data)) return 1;

     return strcasecmp(dn, row_data->dn);
}

/*
 * returns this DN's GtkCTreeNode
 */

GtkCTreeNode *node_from_dn(GtkCTree *ctreeroot, 
			   GtkCTreeNode *top,
			   char *dn)
{
     return gtk_ctree_find_by_row_data_custom(GTK_CTREE(ctreeroot),
					      top, dn, 
					      (GCompareFunc) dn_row_compare_func);
}

struct server_dn {
     const struct ldapserver *server;
     const char *dn;
     struct ldapserver *currserver;
     GtkCTreeNode *found;
};

static void tree_node_from_server_dn_check_func(GtkCTree *ctree,
						GtkCTreeNode *node,
						struct server_dn *sd)
{
     browse_entry *e;

     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     
     if (e == NULL) return;
     if (IS_SERVER_ENTRY(e)) {
	  sd->currserver = ((server_browse_entry *)e)->server;
	  return;
     }
     if (IS_DN_ENTRY(e)) {
	  if (strcasecmp(sd->dn, ((dn_browse_entry *)e)->dn) == 0 &&
	      sd->currserver && sd->currserver == sd->server) {
	       sd->found = node;
	  }
	  return;
     }
}

GtkCTreeNode *tree_node_from_server_dn(GtkCTree *ctree, 
				       const struct ldapserver *server,
				       const char *dn)
{
     GtkCTreeNode *thenode;

     struct server_dn *sd = g_malloc(sizeof(struct server_dn));
     sd->server = server;
     sd->dn = dn;
     sd->currserver = NULL;
     sd->found = NULL;

     gtk_ctree_pre_recursive(ctree,
			     NULL, /* root */
			     (GtkCTreeFunc) tree_node_from_server_dn_check_func,
			     sd);

     thenode = sd->found;

     g_free(sd);
     return thenode;
}

void cleanup_browse_mode(GHashTable *hash)
{
     GString *cur_dn;
     GHashTable *browsehash;

     cur_dn = g_hash_table_lookup(hash, "cur_dn");
     if(cur_dn) {
	  g_string_free(cur_dn, TRUE);
	  g_hash_table_remove(hash, "cur_dn");
     }

     browsehash = g_hash_table_lookup(hash, "browsehash");
     if (browsehash) {
	  browsehash_free(browsehash);
	  g_hash_table_remove(hash, "browsehash");
     }
}

void refresh_subtree_new_dn(GtkCTree *ctree, 
			    GtkCTreeNode *node,
			    const char *newdn, 
			    int options)
{
     GtkCTreeNode *new_node;
     dn_browse_entry *e = 
	  (dn_browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     if (IS_DN_ENTRY(e)) {
	  GtkCTreeNode *parent, *sibling;
	  char **exploded_dn;
	  char *labels[2] = { NULL, NULL };
	  gboolean is_leaf, is_expanded;
	  int n_expand = 2;

	  /* mark the entry to be uncached before we check for it again */
	  e->uncache = TRUE;
	  gtk_clist_freeze(GTK_CLIST(ctree));

	  gtk_ctree_get_node_info(ctree,
				  node,
				  NULL, /* text */
				  NULL, /* spacing */
				  NULL, /* pixmap_closed */
				  NULL, /* mask_closed */
				  NULL, /* pixmap_opened */
				  NULL, /* mask_opened */
				  &is_leaf, /* is_leaf */
				  &is_expanded);

	  if (newdn) {
	       parent = GTK_CTREE_ROW(node)->parent;
	       sibling = GTK_CTREE_ROW(node)->sibling;

	       /* disconnecting entry from row doesn't work without calling
		  the destroy notify function - thus copy the entry */
	       e = (dn_browse_entry *) new_dn_browse_entry(newdn ? newdn : e->dn);
	       
	       gtk_ctree_unselect(ctree, node);
	       
	       exploded_dn = gq_ldap_explode_dn(e->dn, FALSE);
	       labels[0] = decoded_string(exploded_dn[0]);
	       

	       /* add a new entry - alternatively we could just set
                  the new info (new rdn) for the existing node, but we
                  would nevertheless have to remove child nodes and
                  mark the node as unvisited. Therefore we just start
                  with a completely fresh tree */
	       new_node = gtk_ctree_insert_node(ctree,
						parent, sibling,
						labels, 
						0,
						NULL, NULL, NULL, NULL,
						FALSE, FALSE);
	       
	       gtk_ctree_node_set_row_data_full(ctree,
						new_node,
						e,
						(GtkDestroyNotify) destroy_browse_entry);
	       free(labels[0]);
	       
	       /* add dummy node to have something to expand */
	       labels[0] = "DUMMY";
	       
	       gtk_ctree_insert_node(ctree,
				     new_node, NULL,
				     labels, 0,
				     NULL, NULL, NULL, NULL,
				     TRUE, FALSE);
	       
	       /* select the newly added node */
	       gtk_ctree_select(ctree, new_node);
	       
	       /* delete old node */
	       gtk_ctree_remove_node(ctree, node);
	       node = new_node;

	       gq_exploded_free(exploded_dn);
	  } else {
	       /* ! newdn */
	       e->seen = FALSE;

	       /* make the node unexpanded */
	       if (is_expanded) 
		    gtk_ctree_toggle_expansion(ctree, node);
	  }
	  /* toggle expansion at least twice to fire the expand
	     callback (NOTE: do we need this anymore?) */
	  
	  if (options & REFRESH_FORCE_EXPAND) {
	       n_expand = 1;
	  } else if (options & REFRESH_FORCE_UNEXPAND) {
	       n_expand = 0;
	  } else if (is_expanded) {
	       n_expand = 1;
	  } else {
	       n_expand = 0;
	  }
	  
	  while (n_expand-- > 0) {
	       gtk_ctree_toggle_expansion(ctree, node);
	  }

	  show_dn(ctree, node, e->dn);


	  gtk_clist_thaw(GTK_CLIST(ctree));
     }
}

void refresh_subtree(GtkCTree *ctree, 
		     GtkCTreeNode *node)
{
     refresh_subtree_new_dn(ctree, node, NULL, 0);
}

void refresh_subtree_with_options(GtkCTree *ctree,
				  GtkCTreeNode *node,
				  int options)
{
     refresh_subtree_new_dn(ctree, node, NULL, options);
}

void show_dn(GtkCTree *tree, GtkCTreeNode *node, char *dn)
{
     char **dnparts;
     int i;
     GString *s;

     if (!dn) return;
     if (!tree) return;
     
     s = g_string_new("");
     dnparts = gq_ldap_explode_dn(dn, 0);
     
     for(i = 0 ; dnparts[i] ; i++) {
     }
     
     for(i-- ; i >= 0 ; i--) {
	  g_string_insert(s, 0, dnparts[i]);
	  
	  node = node_from_dn(tree,
			      node,
			      s->str);

	  if (node) gtk_ctree_expand(tree, node);
	  else break;

	  g_string_insert(s, 0, ",");
     }

     gq_exploded_free(dnparts);
     g_string_free(s, TRUE);
}


void _gtk_ctree_toggle_is_leaf(GtkCTree * ctree, GtkCTreeNode * node)
{
     gchar *text, *temp;
     guint8 spacing;
     GdkPixmap *pixmap_closed, *pixmap_opened;
     GdkBitmap *mask_closed, *mask_opened;
     gboolean is_leaf, expanded;

     g_return_if_fail(ctree != NULL);
     g_return_if_fail(GTK_IS_CTREE(ctree));

     gtk_ctree_get_node_info(ctree, node,
			     &text, &spacing,
			     &pixmap_closed, &mask_closed,
			     &pixmap_opened, &mask_opened,
			     &is_leaf, &expanded);
     temp = g_strdup(text);
     gtk_ctree_set_node_info(ctree, node,
			     temp, spacing,
			     pixmap_closed, mask_closed,
			     pixmap_opened, mask_opened,
			     is_leaf, expanded);
     g_free(temp);
}

static void browse_new_from_entry_callback(GtkMenuItem *widget,
					   dn_browse_entry *entry)
{
     if (IS_DN_ENTRY(entry)) {
	  char *dn = entry->dn;
	  
	  struct ldapserver *server =
	       (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(widget),
							 "server");

	  new_from_entry(server, dn);
     }
}

static void browse_new_from_template_callback(GtkWidget *widget,
					      struct gq_template *template)
{
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GHashTable *hash;
     GList *formlist;
     struct ldapserver *server;
     dn_browse_entry *entry;

     server = (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(widget),
							"server");
     entry = (dn_browse_entry *) gtk_object_get_data(GTK_OBJECT(widget),
						     "entry");
     if (!IS_DN_ENTRY(entry)) return;
     
     hash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(hash, "dn", NULL);
     g_hash_table_insert(hash, "server", server);

     formlist = formfill_from_template(server, template);
     if(formlist) {
	  g_hash_table_insert(hash, "formlist", formlist);
	  if (entry && entry->dn) {
	       /* don't need the RDN of the current entry */
	       char *newdn = g_malloc(strlen(entry->dn) + 2);
	       newdn[0] = ',';
	       newdn[1] = 0;
	       strcat(newdn, entry->dn);
	       g_hash_table_insert(hash, "dn", newdn);
	  }

	  create_form_window(hash);
	  create_form_content(hash);
	  build_inputform(hash);
     }
     else {
	  g_hash_table_destroy(hash);
     }
#endif /* HAVE_LDAP_STR2OBJECTCLASS */
}

void dump_subtree_ok_callback(GtkWidget *button, GtkWidget *filesel)
{

     LDAPMessage *res = NULL, *e;
     LDAP *ld = NULL;
     GList *bases, *I;
     struct ldapserver *server;
     browse_entry *entry;
     int msg, num_entries;
     char *filename;
     FILE *outfile = NULL;
     GString *out = NULL;
     GString *gmessage = NULL;
     GString *bigmessage = NULL;
     int written;
     char message[128];

     entry = gtk_object_get_data(GTK_OBJECT(filesel), "entry");
     server = gtk_object_get_data(GTK_OBJECT(filesel), "server");

     out = g_string_sized_new(2048);

     bases = NULL;

     if (IS_SERVER_ENTRY(entry)) {
	  bases = get_suffixes(((server_browse_entry *)entry)->server);
     } else {
	  bases = g_list_append(bases, 
				strdup(((dn_browse_entry *)entry)->dn));
     }

     if(g_list_length(bases) == 0) {
	  return;
     }

     set_busycursor();

     /* obtain filename and open file for reading */
     filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));

     if( (outfile = fopen(filename, "w")) == NULL) {
	  error_popup(_("Save failed"), strerror(errno));
     } else {
	  /* AFAIK, the UMich LDIF format doesn't take comments or a
             version string */
	  if (config.ldifformat != LDIF_UMICH) {
	       g_string_truncate(out, 0);
	       

	       prepend_ldif_header(out, server, bases);

	       written = fwrite(out->str, 1, out->len, outfile);
	       if(written != out->len) {
		    snprintf(message, sizeof(message),
			     _("%d of %d bytes written"),
			     written, out->len);
		    error_popup(_("Save failed"), message);
		    goto fail; /* sometimes goto is useful */
	       }
	  }
	  
	  if( (ld = open_connection(server)) == NULL) {
	       goto fail;
	  }
	  
	  num_entries = 0;
	  gmessage = g_string_sized_new(256);
	  for (I = g_list_first(bases) ; I ; I = g_list_next(I)) {
	       g_string_sprintf(gmessage, _("subtree search on %s"), (char *) I->data);
	       statusbar_msg(gmessage->str);

	       msg = ldap_search_s(ld, (char *) I->data, LDAP_SCOPE_SUBTREE,
				   "(objectclass=*)", NULL, 0, &res);
	       if(msg == LDAP_SUCCESS) {
		    for(e = ldap_first_entry(ld, res); e; e = ldap_next_entry(ld, e)) {
			 g_string_truncate(out, 0);
			 ldif_entry_out(out, ld, e);
			 num_entries++;
			 
			 written = fwrite(out->str, 1, out->len, outfile);
			 if(written != out->len) {
			      snprintf(message, sizeof(message),
				       _("%d of %d bytes written"),
				       written, out->len);
			      error_popup(_("Save failed"), message);

			      ldap_msgfree(res);
			      close_connection(server, FALSE);
			      goto fail;
			 }

		    }
		    ldap_msgfree(res);
	       } else if (msg == LDAP_SERVER_DOWN) {
		    server->server_down++;
		    ldap_msgfree(res); /* is this correct? */
	       } else {
		    /* report error */
		    statusbar_msg(ldap_err2string(ldap_result2error(ld, 
								    res, 1)));
		    goto fail;
	       }
	  }

	  make_message(message, sizeof(message), num_entries,
		       _("entry"), _("entries"), _("exported"));

	  bigmessage = g_string_new("");
	  g_string_sprintf(bigmessage, _("%s to %s"), message, filename);
	  statusbar_msg(bigmessage->str);
	  g_string_free(bigmessage, TRUE);
     }

 fail:
     if (outfile) fclose(outfile);

     for (I = g_list_first(bases) ; I ; I = g_list_next(I)) {
	  free(I->data);
     }
     if (bases) g_list_free(bases);
     
     set_normalcursor();
     if (out) g_string_free(out, TRUE);
     if (gmessage) g_string_free(gmessage, TRUE);
     if (ld) close_connection(server, FALSE);

     gtk_widget_destroy(filesel);
}


void dump_subtree_filesel_destroy(GtkWidget *button, GtkWidget *filesel)
{
     gtk_widget_destroy(filesel);
}


void dump_subtree(GtkWidget *widget, GHashTable *hash)
{
     GtkWidget *filesel;

     GtkCTree *ctree;
     GtkCTreeNode *ctree_node;

     browse_entry *entry;
     struct ldapserver *server;

     ctree = g_hash_table_lookup(hash, "ctreeroot");
     ctree_node = g_hash_table_lookup(hash, "selected_ctree_node");

     entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree,
							  ctree_node);
     if (entry == NULL) return;

     if ((server = server_from_node(ctree, ctree_node)) == NULL)
	  return;

     filesel = gtk_file_selection_new(_("Save LDIF"));
     gtk_object_set_data(GTK_OBJECT(filesel), "server", server);
     gtk_object_set_data(GTK_OBJECT(filesel), "entry", entry);

     gtk_signal_connect(GTK_OBJECT(filesel),
			"destroy",
			(GtkSignalFunc) dump_subtree_filesel_destroy,
			filesel);
     gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			"clicked",
			(GtkSignalFunc) dump_subtree_ok_callback,
			filesel);
     gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
			       "clicked",
			       (GtkSignalFunc) gtk_widget_destroy,
			       GTK_OBJECT(filesel));
     gtk_signal_connect_object(GTK_OBJECT(filesel), "key_press_event",
			       (GtkSignalFunc) close_on_esc, 
			       (gpointer) filesel);
     gtk_widget_show(filesel);

}

void delete_browse_entry(GtkWidget *widget, GHashTable *hash)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     struct ldapserver *server;
     dn_browse_entry *entry;
     int do_delete;

     ctree = g_hash_table_lookup(hash, "ctreeroot");
     node = g_hash_table_lookup(hash, "selected_ctree_node");

     entry = (dn_browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     if (entry == NULL) return;
     if ((server = server_from_node(ctree, node)) == NULL)
	  return;

     do_delete = 0;

     gtk_clist_freeze(GTK_CLIST(ctree));
     
     if (!entry->seen) {
	  /* toggle expansion twice to fire the expand callback */
	  gtk_ctree_toggle_expansion(ctree, node);
	  gtk_ctree_toggle_expansion(ctree, node);
     }

     if (entry->leaf) {
	  /* item is a leaf node */
	  do_delete = 1;
     } else {
	  /* maybe delete everything in the subtree as well?
	     should do another LDAP_SCOPE_SUBTREE search after
	     each batch of deletes, in case the server is limiting
	     the number of entries returned per search. This could
	     get hairy...

	     For now, just pop up a dialog box with a warning
	  */

	  do_delete = 
	       question_popup(_("Warning"),
			      _("This entry has a subtree!\n"
				"Do you want to delete every entry under it as well?"));
     }


     if (do_delete) {
	  if (delete_entry_full(server, entry->dn, TRUE)) {
	       browse_entry *p_entry;
	       GtkCTreeNode *parent = GTK_CTREE_ROW(node)->parent;

	       gtk_ctree_remove_node(ctree,
				     node);

	       /* the only thing left to do is to refresh the parent
                  node in order to get the leaf flag of that entry
                  right again */
	       p_entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree,
								      parent);
	       if (p_entry) {
		    p_entry->refresh(p_entry, ctree, parent, hash);
	       }
	  }
     }
     gtk_clist_thaw(GTK_CLIST(ctree));
}



/*
 * Button pressed on a tree item. Button 3 gets intercepted and puts up
 * a popup menu, all other buttons get passed along to the default handler
 *
 * the return values here are related to handling of button events only.
 */
static gboolean button_press_on_tree_item(GtkWidget *tree,
					  GdkEventButton *event,
					  GHashTable *hash)
{
     GtkCTree *ctreeroot;
     GtkWidget *root_menu, *menu, *submenu, *menu_item, *label;
     GtkCTreeNode *ctree_node;
     dn_browse_entry *entry;
     struct ldapserver *server;
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GList *templatelist;
     struct gq_template *template;
#endif
     int is_dn = 0;

     if (event->type == GDK_BUTTON_PRESS && event->button == 3
	 && event->window == GTK_CLIST(tree)->clist_window) {
	  int row, column, rc;
	  char *name;
     
	  rc = gtk_clist_get_selection_info(GTK_CLIST(tree), event->x, event->y, &row, &column);
	  if (rc == 0)
	       return TRUE;
    
	  ctree_node = gtk_ctree_node_nth(GTK_CTREE(tree), row);
	  if (ctree_node == 0)
	       return TRUE;

	  ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
	  
	  
	  entry = (dn_browse_entry *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

  	  if (entry == NULL) return TRUE;
	  is_dn = IS_DN_ENTRY(entry);
	  
	  if ((server = server_from_node(GTK_CTREE(ctreeroot),
					 ctree_node)) == NULL)
	       return(TRUE);

	  name = entry->base_methods.get_name((browse_entry *) entry, FALSE);
	  
	  g_hash_table_insert(hash, "tree-row-popped-up", ctree_node);

	  /* hack, hack. Need to pass both the ctree_node and hash... */
	  g_hash_table_insert(hash, "selected_ctree_node", ctree_node);

	  root_menu = gtk_menu_item_new_with_label("Root");
	  gtk_widget_show(root_menu);
	  menu = gtk_menu_new();
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);

	  label = gtk_menu_item_new_with_label(name);
	  gtk_widget_set_sensitive(label, FALSE);
	  gtk_widget_show(label);

	  gtk_menu_append(GTK_MENU(menu), label);
	  gtk_menu_set_title(GTK_MENU(menu), name);

	  menu_item = gtk_tearoff_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);

	  /* New submenu */
	  menu_item = gtk_menu_item_new_with_label(_("New"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  submenu = gtk_menu_new();
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu);
	  gtk_widget_show(menu_item);

#ifdef HAVE_LDAP_STR2OBJECTCLASS

	  templatelist = config.templates;
	  while(templatelist) {
	       template = (struct gq_template *) templatelist->data;
	       menu_item = gtk_menu_item_new_with_label(template->name);
	       gtk_object_set_data(GTK_OBJECT(menu_item), "server", server);
	       gtk_object_set_data(GTK_OBJECT(menu_item), "entry", entry);
	       gtk_menu_append(GTK_MENU(submenu), menu_item);
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  GTK_SIGNAL_FUNC(browse_new_from_template_callback),
				  (gpointer) template);
	       gtk_widget_show(menu_item);

	       templatelist = templatelist->next;
	  }

#endif

	  menu_item = gtk_menu_item_new_with_label(_("Use current entry"));
	  gtk_object_set_data(GTK_OBJECT(menu_item), "server", server);
	  gtk_menu_append(GTK_MENU(submenu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(browse_new_from_entry_callback),
			     (gpointer) entry);
	  gtk_widget_show(menu_item);

	  /* Refresh */
	  menu_item = gtk_menu_item_new_with_label(_("Refresh"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);
	  
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(tree_row_refresh),
			     (gpointer) hash);

#if 0
	  /* Expand all */
	  menu_item = gtk_menu_item_new_with_label(_("Expand all"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);
	  
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(tree_row_expand_all),
			     (gpointer) hash);
#endif

	  /* Export to LDIF */
	  menu_item = gtk_menu_item_new_with_label(_("Export to LDIF"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) dump_subtree,
			     (gpointer) hash);
	  gtk_widget_show(menu_item);

	  menu_item = gtk_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);

#ifdef BROWSER_DND
	  /* Copy */
	  menu_item = gtk_menu_item_new_with_label(_("Copy"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) copy_entry,
			     (gpointer) hash);

	  if (!is_dn) {
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  }

	  gtk_widget_show(menu_item);

	  /* Copy all */
	  menu_item = gtk_menu_item_new_with_label(_("Copy all"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) copy_entry_all,
			     (gpointer) hash);

	  if (!is_dn) {
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  }
	  gtk_widget_show(menu_item);

	  /* Paste */
	  menu_item = gtk_menu_item_new_with_label(_("Paste"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) paste_entry,
			     (gpointer) hash);

	  if (!is_dn) {
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  }
	  gtk_widget_show(menu_item);

	  menu_item = gtk_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);
#endif
	  /* Delete */
	  menu_item = gtk_menu_item_new_with_label(_("Delete"));
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) delete_browse_entry,
			     (gpointer) hash);

	  if (!is_dn) {
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  }

	  gtk_widget_show(menu_item);

	  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			 event->button, event->time);

	  gtk_signal_emit_stop_by_name(GTK_OBJECT(ctreeroot), 
				       "button_press_event");

	  g_free(name);
     }

     return(TRUE);
}

/* 
   Local Variables:
   c-basic-offset: 5
   End:
*/
