/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>

#include <config.h>

#include <lber.h>
#include <ldap.h>
#ifdef HAVE_LDAP_STR2OBJECTCLASS
    #include <ldap_schema.h>
#endif


#include "common.h"
#include "util.h"
#include "configfile.h"
#include "errorchain.h"
#include "template.h"
#include "debug.h"

#include "../icons/warning.xpm"


extern GtkWidget *mainwin;
extern GtkWidget *statusbar;
extern struct gq_config config;


/*
 * open connection to LDAP server, and store connection for caching
 */
LDAP *open_connection(struct ldapserver *server)
{
     LDAP *ld;
     int open_context, msg;
     char message[128];
     char *binddn, *bindpw;

     /* reuse previous connection if available */
     if(server->connection)
	  return(server->connection);

     sprintf(message, "connecting to %s", server->ldaphost);
     if(server->ldapport != 389)
	  sprintf(message + strlen(message), " port %d", server->ldapport);
     statusbar_msg(message);

     ld = NULL;
     open_context = error_new_context("Error connecting to server");

     if(server) {
	  ld = ldap_init(server->ldaphost, server->ldapport); 

	  if(!ld) {
	       sprintf(message, "Failed to initialize LDAP structure.");
	       error_push(open_context, message);
	  }
	  else {

	       if (server->enabletls) {
#if defined(HAVE_TLS)
		    {
			 int version = LDAP_VERSION3;
			 if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version)
			     != LDAP_OPT_SUCCESS)
			 {
			      sprintf(message, "Couldn't set protocol version to LDAPv3.");
			      error_push(open_context, message);
			 }
		    }

		    /* Let's turn on TLS */
		    msg = ldap_start_tls_s(ld, NULL, NULL);
		    if(msg != LDAP_SUCCESS) {
			 sprintf(message, "Couldn't enable TLS on the LDAP connection: %s",
				 ldap_err2string(msg));
			 error_push(open_context, message);
			 error_flush(open_context);
			 ldap_unbind(ld);

			 return(NULL);

		    }
#else
		    sprintf(message,
			    "TLS was not found to be supported by your LDAP libraries.\n"
			    "See README.TLS for more information.\n");
		    error_push(open_context, message);
#endif /* defined(HAVE_TLS) */
	      } 

	       
	       if(server->binddn[0]) {
		    binddn = server->binddn;
		    bindpw = server->bindpw;
	       }
	       else {
		    binddn = NULL;
		    bindpw = NULL;
	       }

	       msg = LDAP_SUCCESS;
	       if(server->bindtype == BINDTYPE_KERBEROS) {
		    #ifdef HAVE_KERBEROS
		    msg = ldap_bind_s(ld, binddn, bindpw, LDAP_AUTH_KRBV4);
		    #else
		    error_push(open_context, "GQ was compiled without Kerberos support.\n"
			       "Run 'configure --help' for more information\n");
		    statusbar_msg("");
		    ldap_unbind(ld);
		    ld = NULL;
		    #endif
	       }
	       else
		    msg = ldap_simple_bind_s(ld, binddn, bindpw);

	       if( msg != LDAP_SUCCESS) {
		    sprintf(message, "Couldn't bind LDAP connection: %s",
			    ldap_err2string(msg));
		    error_push(open_context, message);
		    statusbar_msg("");
		    /* might as well clean this up */
		    ldap_unbind(ld);
                    ld = NULL;
	       }
	       else
		    /* always store connection handle, regardless of connection
		       caching -- call close_connection() after each operation
		       to do the caching thing or not */
		    server->connection = ld;

	  }

     }

     error_flush(open_context);

     return(ld);
}


/*
 * called after every set of LDAP operations. This preserves the connection
 * if caching is enabled for the server. Set always to TRUE if you want to
 * close the connection regardless of caching.
 */
void close_connection(struct ldapserver *server, int always)
{

     if(server->connection && (!server->cacheconn || always)) {

	  clear_server_schema(server);

	  /* definitely close this connection */
	  ldap_unbind(server->connection);
	  server->connection = NULL;

     }

}


/*
 * clear cached server schema
 */
void clear_server_schema(struct ldapserver *server)
{
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GList *list;
     struct server_schema *ss;

     if(server->ss) {
	  ss = server->ss;

	  /* objectclasses */
	  list = ss->oc;
	  if(list) {
	       while(list) {
		    ldap_objectclass_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->oc);
	  }

	  /* attribute types */
	  list = ss->at;
	  if(list) {
	       while(list) {
		    ldap_attributetype_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->at);
	  }

	  /* matching rules */
	  list = ss->mr;
	  if(list) {
	       while(list) {
		    ldap_matchingrule_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->mr);
	  }

	  /* syntaxes */
	  list = ss->s;
	  if(list) {
	       while(list) {
		    ldap_syntax_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->s);
	  }

	  FREE(server->ss, "struct schema");
	  server->ss = NULL;
     }
     else
	  server->flags &= ~SERVER_HAS_NO_SCHEMA;
#endif

}


/*
 * delete entry
 */
gboolean delete_entry(struct ldapserver *server, char *dn)
{
     LDAP *ld;
     int msg;
     char message[MAX_DN_LEN + 128];
     gboolean rc = TRUE;

     /* FIXME confirm-mod check here */

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  return(FALSE);
     }

     sprintf(message, "deleting %s", dn);
     statusbar_msg(message);

     msg = ldap_delete_s(ld, dn);
     if(msg != LDAP_SUCCESS) {
	  error_popup("Error deleting entry", ldap_err2string(msg));
	  rc = FALSE;
     }
     else {
	  sprintf(message, "deleted %s", dn);
	  statusbar_msg(message);
     }

     set_normalcursor();
     close_connection(server, FALSE);

     return(rc);
}


/*
 * display hourglass cursor on mainwin
 */
void set_busycursor(void)
{
     GdkCursor *busycursor;

     busycursor = gdk_cursor_new(GDK_WATCH);
     gdk_window_set_cursor(mainwin->window, busycursor);
     gdk_cursor_destroy(busycursor);

}


/*
 * set mainwin cursor to default
 */
void set_normalcursor(void)
{

     gdk_window_set_cursor(mainwin->window, NULL);

}


void make_message(char *buffer, int cnt, char *singular, char *plural, char *suffix)
{

     switch(cnt) {
     case 0:
	  sprintf(buffer, "no %s %s", plural, suffix);
	  break;
     case 1:
	  sprintf(buffer, "1 %s %s", singular, suffix);
	  break;
     default:
	  sprintf(buffer, "%d %s %s", cnt, plural, suffix);
	  break;
     }

}


/*
 * callback for key_press_event on a widget, destroys obj if key was esc
 */
int close_on_esc(GtkWidget *widget, GdkEventKey *event, gpointer obj)
{

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape) {
	  gtk_widget_destroy(GTK_WIDGET(obj));
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}


/*
 * callback for key_press_event on a widget, calls func if key was esc
 */
int func_on_esc(GtkWidget *widget, GdkEventKey *event, GtkWidget *window)
{
     void (*func)(GtkWidget *);

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape) {
	  func = gtk_object_get_data(GTK_OBJECT(window), "close_func");
	  func(widget);
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}


int tokenize(struct tokenlist *list, char *keyword)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(!strcasecmp(list[i].keyword, keyword))
	       return(list[i].token);

     return(0);
}


char *detokenize(struct tokenlist *list, int token)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(list[i].token == token)
	       return(list[i].keyword);

     return(list[0].keyword);
}


/*
 * return pointer to username (must be freed)
 */
char *get_username(void)
{
     struct passwd *pwent;
     char *username;

     username = NULL;
     pwent = getpwuid(getuid());
     if(pwent && pwent->pw_name)
	  username = strdup(pwent->pw_name);
     endpwent();

     return(username);
}


/*
 * display message in main window's statusbar, and flushes the
 * GTK event queue
 */
void statusbar_msg(char *message)
{
     static guint context, msgid = 0;

     if(!context)
	  context = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar),
						 "mainwin");
     if(msgid)
	  gtk_statusbar_remove(GTK_STATUSBAR(statusbar), context, msgid);

     gtk_statusbar_push(GTK_STATUSBAR(statusbar), context, message);

     /* make sure statusbar gets updated right away */
     while(gtk_events_pending())
	  gtk_main_iteration();

}


/*
 * return pointer to (struct ldapserver *) matching name
 */
struct ldapserver *server_by_name(char *name)
{
     struct ldapserver *server;

     if(name == NULL || name[0] == '\0')
	  return(NULL);

     server = config.ldapservers;
     while(server) {
	  if(!strcmp(server->name, name))
	       return(server);
	  server = server->next;
     }

     /* XXX - This needs to be gotten rid of, and instead have the appropriate
               errors generated in the code that depends on server_by_name. 
	       -- a7r
     */
     error_popup("Error finding server", "Server not configured.");

     return(NULL);
}


/*
 * check if entry has a subtree
 */
int is_leaf_entry(struct ldapserver *server, char *dn)
{
     LDAP *ld;
     LDAPMessage *res;
     GString *gmessage;
     int msg, is_leaf;

     is_leaf = 0;

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  return(-1);
     }

     gmessage = g_string_sized_new(128);
     g_string_sprintf(gmessage, "checking subtree for %s", dn);
     statusbar_msg(gmessage->str);
     g_string_free(gmessage, TRUE);

     msg = ldap_search(ld, dn, LDAP_SCOPE_ONELEVEL, "(objectclass=*)",
		       NULL, 0);
     if(msg != -1) {
	  if( (ldap_result(ld, msg, 0, NULL, &res) != LDAP_RES_SEARCH_ENTRY))
	       is_leaf = 1;
	  ldap_msgfree(res);
     }

     close_connection(server, FALSE);
     set_normalcursor();
     statusbar_msg("");

     return(is_leaf);
}


GList *ar2glist(char *ar[])
{
     GList *tmp;
     int i;

     if(ar == NULL) {
	  /* gtk_combo_set_popdown_strings() can't handle an
	     empty list, so give it a list with an empty entry */
	  tmp = g_list_append(NULL, "");
	  return(tmp);
     }

     tmp = NULL;
     i = 0;
     while(ar[i])
	  tmp = g_list_append(tmp, ar[i++]);

     return(tmp);
}


/*
 * pops up a warning dialog (with hand icon), and displays all messages in
 * the GList , one per line. The GList is freed here.
 */
void warning_popup(GList *messages)
{
     GList *list;
     GtkWidget *window, *vbox1, *vbox2, *vbox3, *label;
     GtkWidget *hbox1, *ok_button;
     GtkWidget *pixmap;
     GdkPixmap *warning;
     GdkBitmap *warning_mask;

     window = gtk_dialog_new();
     gtk_container_border_width(GTK_CONTAINER(window), 12);
     gtk_window_set_title(GTK_WINDOW(window), "Warning");
     gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_widget_show(vbox1);

     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 10);
     warning = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(mainwin)->window,
					    &warning_mask,
					    &mainwin->style->white,
					    warning_xpm);
     pixmap = gtk_pixmap_new(warning, warning_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox1), pixmap, TRUE, TRUE, 10);

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 10);

     list = messages;
     while(list) {
	  label = gtk_label_new((char *) list->data);
	  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	  gtk_widget_show(label);
	  gtk_box_pack_start(GTK_BOX(vbox2), label, TRUE, TRUE, 0);
	  list = list->next;
     }

     /* OK button */
     vbox3 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox3);

     ok_button = gtk_button_new_with_label("   OK   ");
     gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) window);
     gtk_box_pack_start(GTK_BOX(vbox3), ok_button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(ok_button);
     gtk_widget_show(ok_button);

     gtk_signal_connect_object(GTK_OBJECT(window), "destroy",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) window);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) window);

     gtk_widget_show(window);

     g_list_free(messages);

}


#ifdef HAVE_LDAP_STR2OBJECTCLASS
GList *find_at_by_mr_oid(struct ldapserver *server, char *oid)
{
     GList *list, *srvlist;
     LDAPAttributeType *at;

     list = NULL;
     srvlist = server->ss->at;
     while(srvlist) {
	  at = (LDAPAttributeType *) srvlist->data;

	  if( (at->at_equality_oid && !strcasecmp(oid, at->at_equality_oid)) ||
	      (at->at_ordering_oid && !strcasecmp(oid, at->at_ordering_oid)) ||
	      (at->at_substr_oid && !strcasecmp(oid, at->at_substr_oid)) ) {
	       if(at->at_names && at->at_names[0])
		    list = g_list_append(list, at->at_names[0]);
	       else
		    list = g_list_append(list, at->at_oid);
	  }
	  srvlist = srvlist->next;
     }

     return(list);
}


GList *find_at_by_s_oid(struct ldapserver *server, char *oid)
{
     GList *list, *srvlist;
     LDAPAttributeType *at;

     list = NULL;
     srvlist = server->ss->at;
     while(srvlist) {
	  at = (LDAPAttributeType *) srvlist->data;

	  if(at->at_syntax_oid && !strcasecmp(oid, at->at_syntax_oid)) {
	       if(at->at_names && at->at_names[0])
		    list = g_list_append(list, at->at_names[0]);
	       else
		    list = g_list_append(list, at->at_oid);
	  }
	  srvlist = srvlist->next;
     }

     return(list);
}


GList *find_mr_by_s_oid(struct ldapserver *server, char *oid)
{
     GList *list, *srvlist;
     LDAPMatchingRule *mr;

     list = NULL;
     srvlist = server->ss->mr;
     while(srvlist) {
	  mr = (LDAPMatchingRule *) srvlist->data;

	  if(mr->mr_syntax_oid && !strcasecmp(oid, mr->mr_syntax_oid)) {
	       if(mr->mr_names && mr->mr_names[0])
		    list = g_list_append(list, mr->mr_names[0]);
	       else
		    list = g_list_append(list, mr->mr_oid);
	  }
	  srvlist = srvlist->next;
     }

     return(list);
}


GList *find_oc_by_at(struct ldapserver *server, char *atname)
{
     GList *list, *srvlist;
     LDAPObjectClass *oc;
     int i, found;

     list = NULL;
     srvlist = server->ss->oc;
     while(srvlist) {
	  oc = (LDAPObjectClass *) srvlist->data;

	  found = 0;

	  if(oc->oc_at_oids_must) {
	       i = 0;
	       while(oc->oc_at_oids_must[i] && !found) {
		    if(!strcasecmp(atname, oc->oc_at_oids_must[i])) {
			 found = 1;
			 if(oc->oc_names && oc->oc_names[0])
			      list = g_list_append(list, oc->oc_names[0]);
			 else
			      list = g_list_append(list, oc->oc_oid);
		    }
		    i++;
	       }
	  }

	  if(oc->oc_at_oids_may) {
	       i = 0;
	       while(oc->oc_at_oids_may[i] && !found) {
		    if(!strcasecmp(atname, oc->oc_at_oids_may[i])) {
			 found = 1;
			 if(oc->oc_names && oc->oc_names[0])
			      list = g_list_append(list, oc->oc_names[0]);
			 else
			      list = g_list_append(list, oc->oc_oid);
		    }
		    i++;
	       }
	  }

	  srvlist = srvlist->next;
     }

     return(list);
}
#endif /* HAVE_LDAP_STR2OBJECTCLASS */

struct gq_template *find_template_by_name(char *templatename)
{
     GList *templatelist;
     struct gq_template *template;

     template = NULL;
     templatelist = config.templates;
     while(templatelist) {
	  template = (struct gq_template *) templatelist->data;
	  if(!strcasecmp(templatename, template->name))
	       break;
	  else
	       template = NULL;
	  templatelist = templatelist->next;
     }

     return(template);
}



void dump_mods(LDAPMod **mods)
{
     LDAPMod *mod;
     int cmod, cval;

     cmod = 0;
     while(mods[cmod]) {
	  mod = mods[cmod];
	  switch(mod->mod_op) {
	  case LDAP_MOD_ADD: printf("LDAP_MOD_ADD"); break;
	  case LDAP_MOD_DELETE: printf("LDAP_MOD_DELETE"); break;
	  case LDAP_MOD_REPLACE: printf("LDAP_MOD_REPLACE"); break;
	  case LDAP_MOD_BVALUES: printf("LDAP_MOD_BVALUES"); break;
	  }
	  printf(" %s\n", mod->mod_type);
	  cval = 0;
	  while(mod->mod_values && mod->mod_values[cval]) {
	       printf("\t%s\n", mod->mod_values[cval]);
	       cval++;
	  }

	  cmod++;
     }


}

