/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef _WORKSPACE_H
#define _WORKSPACE_H

/* forward decl from client.h */
struct client;

/* forward decl */
struct desktop;

/* a workspace; a focus listing group, essentially */
typedef struct workspace {
	struct desktop	*desktop;	/* desktop the workspace is on */

	struct client	*focused;	/* ptr to the focused client in this ws's list */
	struct client	*focus_list;	/* focused client list for this workspace */
	struct client	*focus_last;	/* end of the list, to save time from traversing it */
} workspace_t;

/*
 * a desktop; a collection width * height workspaces.  workspaces exist to provide the
 * user with independant focus lists for seperate screens of a large virtual desktop;
 * which is a _lot_ nicer if you use click-focus modes: the pointer people probably
 * wont notice.
 *
 * when switching either desktops or just workspaces on the same desktop, a client
 * with flags.sticky set to true will "stick" to the glass.  this works by not moving
 * it/unmaping it, and we maintain it's status in terms of focus after adding it to
 * the other focuslist.  so if it is focused it will stay focused, and if not, it will
 * not become focused, even if there are no windows on the destination workspace.
 *
 * desktops work by moving windows in the opposite direction as the
 * viewport is moved; the viewport _must_ be moved in screen-sized icrements (otherwise
 * maintaining independant focus lists would be really really odd..). if a window is
 * visible across two of the screen-viewport boundries, it will be added to the newly
 * visible workspace's focus list, and removed from the old one.  every screen has
 * at least one desktop, which can be any size, as long as it has at least
 * one workspace (1x1).
 */
typedef struct desktop {
	int		num;		/* unique (on each screen) desktop number */

	int		width, height;	/* dimensions */
	int		viewx, viewy;	/* current viewport location */
	workspace_t	**workspaces;	/* grid of workspaces on this desktop */
	workspace_t	*current_space;	/* in-use workspace */

	struct desktop *prev;	
	struct desktop *next;
} desktop_t;

void workspace_shutdown();
int workspace_add_desks(screen_t *screen, int width, int height, int count);
void workspace_rm_desks(screen_t *screen);
void workspace_switch_desk(screen_t *screen, int num);
void workspace_add_client(workspace_t *workspace, struct client *client);
void workspace_rm_client(struct client *client);
void workspace_add_bypos(struct desktop *desktop, struct client *client);
int workspace_viewport_move(screen_t *screen, desktop_t *desktop, int xmove, int ymove);

#endif
