/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef _PLUGIN_H
#define _PLUGIN_H

/*
 * return values for plugin callbacks.  if I plugin callback
 * returns plugin unload that plugin is immediatly unloaded.
 * certain plugin callbacks can return plugin using, which means
 * that they are using the event that caused the callback,
 * and the main golem code shouldn't call any more plugin
 * callbacks for it, or do any normal processing on it.
 */
#define PLUGIN_OK		0
#define PLUGIN_UNLOAD		1
#define PLUGIN_USING		2

/* forward decl (see below) */
struct param;

/* this is seperated so it can be used in the plugin struct */
typedef struct subparams {
	int		count;		/* number of sub params */
	struct param 	**params;	/* the sub parameters */
} subparams_t;

/*
 * a tree of plugin parameters that are passed in for use by the plugin from
 * the user's rcfile.  there is a pointer to one of these in the plugin_t
 * structure who's varname will
 */
typedef struct param {
	char	*name;		/* name of the parameter */
	char	*value;		/* value of the parameter */

	subparams_t subparams;
} param_t;

/* 
 * a plugin is a dlopen()'d object that has a bunch of callback functions
 * the main code will call these functions for plugins that implement them.
 * 
 * currently a plugin should only be doing stuff while it's being used from
 * a callback function.  in the future there will be an method to allow independant
 * plugin processing, most likely using pthreads.
 *
 * plugins are able to create both managed and unmanaged windows, and recieve
 * events for those windows.  to create an unmanaged window the client uses
 * Xlib calls to create an override_redirect window, and then pass it to
 * plugin_setcontext (please note, this window _must_ be passed to
 * plugin_rmcontext before it's unloaded).
 *
 * the same callbacks are used for managed and unmanaged plugin windows; for
 * unmanaged ones the client_t * param is just set to NULL, and the plugin
 * must get all the information it needs from the XEvent that is passed in.
 */
typedef struct plugin {
	void	*hnd;					/* handle from dlopen */
	char	*name;					/* name the plugin was loaded as */

	/*
	 * these are functions that are called to alert
	 * the plugin of the stages during startup/shutdown.
	 * init is called during the rcfile parsing and
	 * as such it shouldn't use _any_ of the internal
	 * golem structures (doing so == breaks).  shutdown
	 * can be called at any time and should be prepared
	 * to release memory; it shouldn't use any of the
	 * internal structures unless it's sure that we are
	 * past start.  start == right after rcfile parsing
	 */
	int	(*init)(struct plugin *);		/* plugin initialization routine */
	void	(*shutdown)();				/* plugin death routine */
	int	(*start)(struct plugin *);		/* post-init plugin starting */

	/*
	 * the following callbacks correspond to things
	 * that happen in the windowmanager; such as when
	 * a window is zoomed or iconified, etc.
	 */
	int	(*init_hints)(struct plugin *,
			client_t *, dgroup_t *);	/* handle hints to the windowmanager */
	int	(*window_birth)(struct plugin *,
			client_t *);			/* a client window just got displayed */
	int	(*window_death)(struct plugin *,
			client_t *);			/* a client window was unmapped */
	int	(*geometry_change)(struct plugin *,
			client_t *);			/* resizes or moves, zooming, etc */
	int	(*iconify_notify)(struct plugin *,
			client_t *);			/* a window was iconified */
	int	(*restore_notify)(struct plugin *,
			client_t *);			/* window restored from iconified state */
	int	(*zoom_notify)(struct plugin *,
			client_t *);			/* a client is zoomed */
	int	(*unzoom_notify)(struct plugin *,
			client_t *);			/* a client is unzoomed */
	int	(*workspace_change)(struct plugin *,
			screen_t *, desktop_t *);	/* workspace switch on screen, desktop */
	int	(*desktop_change)(struct plugin *,
			screen_t *, desktop_t *);	/* desktop change, w/ ptr to old desktop */

	/*
	 * these callbacks are somewhat hackishly specific
	 * to the needs of a few plugins; they are located for
	 * specific animation callbacks, and the like.
	 */
	int	(*anim_birth)(struct plugin *,
			client_t *client);		/* window birthing animations */

	/*
	 * the following callbacks are special because
	 * they may return PLUGIN_USING to cease
	 * processing of the event in question by other
	 * callbacks and by golem
	 */
	int	(*map_request)(struct plugin *,
			screen_t *, XMapRequestEvent *);/* map requests */

	/*
	 * the following correspond to X events, and are
	 * recieved by plugin windows for either managed
	 * or unmanaged client windows they create (for
	 * unmanaged ones they must call plugin_setcontext
	 * on the window). some of these take client_t arguments;
	 * they will be NULL if the window isn't managed.
	 *
	 * XXX: clean this up some by perhaps making a general
	 * XEvent callback.
	 */
	int	(*map_notify)(struct plugin *,
			client_t *, XMapEvent *);
	int	(*unmap_notify)(struct plugin *,
			XUnmapEvent *);			/* window mappage */
	int	(*visibility_notify)(struct plugin *,
			client_t *, XVisibilityEvent *);/* changes in visibility */
	int	(*root_button_press)(struct plugin *,
			screen_t *, XButtonEvent *);
	int	(*root_button_release)(struct plugin *,
			screen_t *, XButtonEvent *);	/* clicks on the root window */
	int	(*button_press)(struct plugin *,
			client_t *, XButtonEvent *);
	int	(*button_release)(struct plugin *,
			client_t *, XButtonEvent *);	/* button presses */
	int	(*pointer_motion)(struct plugin *,
			client_t *, XMotionEvent *);	/* pointer motion */
	int	(*key_press)(struct plugin *,
			client_t *, XKeyEvent *);
	int	(*key_release)(struct plugin *,
			client_t *, XKeyEvent *);	/* key presses */
	int	(*expose)(struct plugin *,
			client_t *, XExposeEvent *);	/* exposures */
	int	(*enter_notify)(struct plugin *,
			client_t *, XCrossingEvent *);	/* window entries */
	int	(*leave_notify)(struct plugin *,
			client_t *, XCrossingEvent *);	/* window leaving */

	subparams_t	params;				/* parameters for the plugin */
	
	struct plugin	*next;
	struct plugin	*prev;
} plugin_t;

/* used to call plugin callbacks */
#define PLUGIN_CALLBACK(plugin, callback, args...) do {		\
	if (plugin->callback) {					\
		switch (plugin->callback(plugin, args)) {	\
		case PLUGIN_UNLOAD:				\
			plugin_unload(plugin);			\
			break;					\
		case PLUGIN_OK:					\
		default:					\
			break;					\
		}						\
	}							\
} while (0)

/* the list of loaded plugins */
extern plugin_t *plugin_list;

void plugin_shutdown();

/* loading and unloading of plugins */
plugin_t *plugin_load(char *name, subparams_t *params, int doinit);
void plugin_unload(plugin_t *plugin);

/* parameter structure functions */
param_t	*plugin_param(char *name, char *value);
int plugin_subparams_add(subparams_t *subparams, param_t *param);
int plugin_subparams_merge(subparams_t *subparams, subparams_t *addition);
void plugin_subparams_free(subparams_t *subparams);
void plugin_param_free(param_t *param);

/* 
 * these are called from plugins to get param values.  Some of
 * them need to make memory for the returned values; primarily
 * string.  They all return their param in the type-specific ret
 * as an argument in the function.  The reason to do this is because
 * returning, say, -1 on errors, is a valid value for ints and doubles.
 */
param_t	*plugin_find_param(subparams_t *subparams, char *name);
int plugin_string_param(subparams_t *subparams, char *name, char **ret);
int plugin_pixmap_param(subparams_t *subparams, char *name, pixmap_t **ret);
int plugin_int_param(subparams_t *subparams, char *name, int *ret);
int plugin_double_param(subparams_t *subparams, char *name, double *ret);
int plugin_bool_param(subparams_t *subparams, char *name, int *ret);

/* plugins call this to ask that events for a window find it's way to it */
void plugin_setcontext(plugin_t *plugin, Window wnd);
void plugin_rmcontext(Window wnd);

/* funcs that call a callback for every loaded plugin */
void plugin_start();
void plugin_init_hints(client_t *client, dgroup_t *dgroup);
void plugin_window_birth(client_t *client);
void plugin_anim_birth(client_t *client);
void plugin_window_death(client_t *client);
void plugin_geometry_change(client_t *client);
void plugin_zoom_notify(client_t *client);
void plugin_unzoom_notify(client_t *client);
void plugin_workspace_change(screen_t *screen, desktop_t *desktop);
void plugin_desktop_change(screen_t *screen, desktop_t *olddesk);
void plugin_iconify_notify(client_t *client);
void plugin_restore_notify(client_t *client);
int plugin_map_request(screen_t *screen, XMapRequestEvent *e);
void plugin_root_button_press(screen_t *screen, XButtonEvent *e);
void plugin_root_button_release(screen_t *screen, XButtonEvent *e);

#endif
