/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include <dlfcn.h>

/* the list of loaded plugins */
plugin_t	*plugin_list	= NULL;

/* unload all plugins */
void plugin_shutdown() {
	plugin_t *plugin, *tmp;

	plugin = plugin_list;
	while (plugin) {
		tmp = plugin->next;
		plugin_unload(plugin);
		plugin = tmp;
	}
}

/* load a plugin, call it's init */
plugin_t *plugin_load(char *name, subparams_t *params, int doinit) {
	char path[MAXPATHLEN];
	char *pluginpath, *tmp, *c, *plugindir;
	char *usedplugdir = NULL;
	plugin_t *plugin;
	void *hnd;

	/* get memory for the plug struct */
	plugin = calloc(1, sizeof(plugin_t));
	if (!plugin)
		return NULL;

	pluginpath = strdup(PLUGINPATH);
	tmp = pluginpath;

	/* we loop through the : deliminated path for plugins */
	while ((plugindir = strsep(&tmp, ":")) != NULL) {
		/* replace ~ w/ the home directory */
		if ((c = strchr(plugindir, '~')) != NULL) {
			char *newdir;
			int olen, hplen;

			olen = strlen(plugindir);
			newdir = malloc(olen + 2);
			if (!newdir) {
				warn("couldn't get memory in plugin.c");
				continue;
			}

			*c = '\0';
			strcpy(newdir, plugindir);
			strcat(&newdir[c - plugindir], "%s");
			strcat(&newdir[c - plugindir + 2], &c[1]);

			hplen = strlen(homedir_path);
			usedplugdir = malloc(hplen + olen + 1);
			snprintf(usedplugdir, hplen + olen + 1, newdir, homedir_path);
			free(newdir);

			plugindir = usedplugdir;
		}

		/* build the full path */
		if (snprintf(path, MAXPATHLEN, "%s/%s.so", plugindir, name) >= MAXPATHLEN) {
			if (usedplugdir) { free(usedplugdir); usedplugdir = NULL; }
			warnx("path to plugin exceeds MAXPATHLEN");
			continue;
		}

		if (usedplugdir) { free(usedplugdir); usedplugdir = NULL; }
		if (access(path, R_OK) != 0)
			continue;

		hnd = dlopen(path, RTLD_LAZY);
		if (hnd == NULL)
			warnx("failed dlopen for %s: %s", path, dlerror());
		else
			goto loaded;
	}

	/* we couldn't load the plugin anywhere on the path */
	warnx("unable to load plugin %s", name);
	free(pluginpath);
	free(plugin);
	return NULL;

loaded:
	warnx("%s: loaded", name);

	/* fill in our plugin structure */
	plugin->hnd			= hnd;
	plugin->name			= name;
	plugin->params.count		= params ? params->count : 0;
	plugin->params.params		= params ? params->params : NULL;
	plugin->init			= dlsym(hnd, SYM_PREFIX "init");
	plugin->shutdown		= dlsym(hnd, SYM_PREFIX "shutdown");
	plugin->start			= dlsym(hnd, SYM_PREFIX "start");
	plugin->init_hints		= dlsym(hnd, SYM_PREFIX "init_hints");
	plugin->window_birth		= dlsym(hnd, SYM_PREFIX "window_birth");
	plugin->window_death		= dlsym(hnd, SYM_PREFIX "window_death");
	plugin->geometry_change		= dlsym(hnd, SYM_PREFIX "geometry_change");
	plugin->zoom_notify		= dlsym(hnd, SYM_PREFIX "zoom_notify");
	plugin->unzoom_notify		= dlsym(hnd, SYM_PREFIX "unzoom_notify");
	plugin->iconify_notify		= dlsym(hnd, SYM_PREFIX "iconify_notify");
	plugin->restore_notify		= dlsym(hnd, SYM_PREFIX "restore_notify");
	plugin->workspace_change	= dlsym(hnd, SYM_PREFIX "workspace_change");
	plugin->desktop_change		= dlsym(hnd, SYM_PREFIX "desktop_change");
	plugin->anim_birth		= dlsym(hnd, SYM_PREFIX "anim_birth");
	plugin->map_request		= dlsym(hnd, SYM_PREFIX "map_request");
	plugin->map_notify		= dlsym(hnd, SYM_PREFIX "map_notify");
	plugin->unmap_notify		= dlsym(hnd, SYM_PREFIX "unmap_notify");
	plugin->visibility_notify	= dlsym(hnd, SYM_PREFIX "visibility_notify");
	plugin->root_button_press	= dlsym(hnd, SYM_PREFIX "root_button_press");
	plugin->root_button_release	= dlsym(hnd, SYM_PREFIX "root_button_release");
	plugin->button_press		= dlsym(hnd, SYM_PREFIX "button_press");
	plugin->button_release		= dlsym(hnd, SYM_PREFIX "button_release");
	plugin->pointer_motion		= dlsym(hnd, SYM_PREFIX "pointer_motion");
	plugin->key_press		= dlsym(hnd, SYM_PREFIX "key_press");
	plugin->key_release		= dlsym(hnd, SYM_PREFIX "key_release");
	plugin->expose			= dlsym(hnd, SYM_PREFIX "expose");
	plugin->enter_notify		= dlsym(hnd, SYM_PREFIX "enter_notify");
	plugin->leave_notify		= dlsym(hnd, SYM_PREFIX "leave_notify");

	/* link it up */
	plugin->next = plugin_list;
	plugin_list = plugin;
	if (plugin->next)
		plugin->next->prev = plugin;
	plugin->prev = NULL;

	/* call init if it's there */
	if (doinit) {
		if (plugin->init)
			if (plugin->init(plugin) != PLUGIN_OK)
				plugin_unload(plugin);
		plugin_subparams_free(&plugin->params);
	}

	free(pluginpath);
	return plugin;
}

/* unload a plugin, call it's shutdown */
void plugin_unload(plugin_t *plugin) {
	/* call shutdown if it's there */
	if (plugin->shutdown)
		plugin->shutdown(plugin);

	/* unlink it */
	if (plugin_list == plugin)
		plugin_list = plugin->next;
	if (plugin->prev)
		plugin->prev->next = plugin->next;
	if (plugin->next)
		plugin->next->prev = plugin->prev;

	/* get rid of the params (they still exist if init() unloaded) */
	plugin_subparams_free(&plugin->params);

#ifndef USE_DMALLOC
	/* don't do this while dmallocing or dmalloc freaks out */
	if (dlclose(plugin->hnd))
		warnx("failed dlclose: %s", dlerror());
#endif
	warnx("%s: unloaded", plugin->name);
	free(plugin->name);
	free(plugin);
}

/* make a new plugin parameter */
param_t *plugin_param(char *name, char *value) {
	param_t *param;

	param = malloc(sizeof(param_t));
	if (!param)
		return NULL;
	param->name = name;
	param->value = value;

	return param;
}

/* add a parameter to a subparams */
int plugin_subparams_add(subparams_t *subparams, param_t *param) {
	param_t **tmp;

	tmp = realloc(subparams->params, (subparams->count + 1) * sizeof(param_t *));
	if (!tmp)
		return -1;
	subparams->params = tmp;
	subparams->params[subparams->count++] = param;
	
	return 0;
}

/* merge the second param of subparams onto the first */
int plugin_subparams_merge(subparams_t *subparams, subparams_t *addition) {
	param_t **tmp;

	tmp = realloc(subparams->params,
		(subparams->count + addition->count) * sizeof(param_t *));
	if (!tmp)
		return -1;
	subparams->params = tmp;
	memcpy(&subparams->params[subparams->count], addition->params, addition->count * sizeof(param_t *));
	subparams->count += addition->count;
	free(addition->params);

	return 0;
}

/* free up a tree of subparams */
void plugin_subparams_free(subparams_t *subparams) {
	int i;

	if (subparams->count) {
		for (i = 0; i < subparams->count; i++)
			plugin_param_free(subparams->params[i]);
		free(subparams->params);
		subparams->count = 0;
	}
}

/* free up a param and it's subparams */
void plugin_param_free(param_t *param) {
	plugin_subparams_free(&param->subparams);
	free(param->name);
	free(param->value);
}

/* try to find a plugin parameter in the first depth of a tree */
param_t *plugin_find_param(subparams_t *subparams, char *name) {
	int i;

	for (i = 0; i < subparams->count; i++)
		if (strcmp(subparams->params[i]->name, name) == 0)
			return subparams->params[i];

	return NULL;
}

/* string parameters */
int plugin_string_param(subparams_t *subparams, char *name, char **ret) {
	param_t *param;

	param = plugin_find_param(subparams, name);
	if (!param)
		return -1;

	*ret = strdup(param->value);
	return *ret ? 0 : -1;
}

/* pixmap parameters */
int plugin_pixmap_param(subparams_t *subparams, char *name, pixmap_t **ret) {
	param_t *param;

	param = plugin_find_param(subparams, name);
	if (!param)
		return -1;

	*ret = pixmap_ident(param->value);
	return *ret ? 0 : -1;
}

/* integer parameters; we default to 0 if it can't be found */
int plugin_int_param(subparams_t *subparams, char *name, int *ret) {
	param_t *param;

	param = plugin_find_param(subparams, name);
	if (!param)
		return -1;

	*ret = atoi(param->value);
	return 0;
}

/* decimal parameters */
int plugin_double_param(subparams_t *subparams, char *name, double *ret) {
	param_t *param;

	param = plugin_find_param(subparams, name);
	if (!param)
		return -1;

	*ret = atof(param->value);
	return 0;
}

/* boolean parameters */
int plugin_bool_param(subparams_t *subparams, char *name, int *ret) {
	param_t *param;

	param = plugin_find_param(subparams, name);
	if (!param)
		return -1;

	/* "true", "false", or just ret -1 */
	if (strcmp(param->value, "true") == 0)
		return *ret = 1, 0;
	else if (strcmp(param->value, "false") == 0)
		return *ret = 0, 0;
	return -1;
}

/* set the plugin_context of a window so plugin will get events */
void plugin_setcontext(plugin_t *plugin, Window wnd) {
	XSaveContext(display, wnd, plugin_context, (XPointer) plugin);
}

/*
 * this _must_ be done for all windows that a plugin called plugin_setcontext
 * on before the plugin exits, otherwise an event may cause an attempted
 * callback on a allready-unloaded plugin, which is bad.
 */
void plugin_rmcontext(Window wnd) {
	XDeleteContext(display, wnd, plugin_context);
}

/* for making functions that callback all plugins */
#define PLUGIN_LOOP(callback, args...) do {			\
	plugin_t *plugin;					\
								\
	plugin = plugin_list;					\
	while (plugin) {					\
		PLUGIN_CALLBACK(plugin, callback, args);	\
		plugin = plugin->next;				\
	}							\
} while (0)

/* called post-init for all plugins */
void plugin_start() {
	plugin_t *plugin;

	plugin = plugin_list;
	while (plugin) {
		if (plugin->start) {
			switch (plugin->start(plugin)) {
			case PLUGIN_UNLOAD:
				plugin_unload(plugin);
				break;
			case PLUGIN_OK:
			default:
				break;
			}
		}
		plugin = plugin->next;
	}
}

/* handle window manager hints */
void plugin_init_hints(client_t *client, dgroup_t *dgroup) {
	PLUGIN_LOOP(init_hints, client, dgroup);
}

/* client window gets mapped */
void plugin_window_birth(client_t *client) {
	PLUGIN_LOOP(window_birth, client);
}

/* animation for window birth */
void plugin_anim_birth(client_t *client) {
	PLUGIN_LOOP(anim_birth, client);
}

/* client window gets unmapped */
void plugin_window_death(client_t *client) {
	PLUGIN_LOOP(window_death, client);
}

/* changes in client sizes */
void plugin_geometry_change(client_t *client) {
	PLUGIN_LOOP(geometry_change, client);
}

/* client zoomage */
void plugin_zoom_notify(client_t *client) {
	PLUGIN_LOOP(zoom_notify, client);
}

/* client unzoomage */
void plugin_unzoom_notify(client_t *client) {
	PLUGIN_LOOP(unzoom_notify, client);
}

/* workspace switching */
void plugin_workspace_change(screen_t *screen, desktop_t *desktop) {
	PLUGIN_LOOP(workspace_change, screen, desktop);
}

/* desktop switching */
void plugin_desktop_change(screen_t *screen, desktop_t *olddesk) {
	PLUGIN_LOOP(desktop_change, screen, olddesk);
}

/* client iconification */
void plugin_iconify_notify(client_t *client) {
	PLUGIN_LOOP(iconify_notify, client);
}

/* restoration of iconified clients */
void plugin_restore_notify(client_t *client) {
	PLUGIN_LOOP(restore_notify, client);
}

/* map requests, returns 1 if got a PLUGIN_USING */
int plugin_map_request(screen_t *screen, XMapRequestEvent *e) {
	plugin_t *plugin;
	int using = 0;

	plugin = plugin_list;
	while (plugin) {
		if (plugin->map_request) {
			switch (plugin->map_request(plugin, screen, e)) {
			case PLUGIN_UNLOAD:
				plugin_unload(plugin);
				break;
			case PLUGIN_USING:
				using = 1;
				goto done;
			case PLUGIN_OK:
			default:
				break;
			}
		}
		plugin = plugin->next;
	}

done:
	return using;
}

/* button presses on the root window */
void plugin_root_button_press(screen_t *screen, XButtonEvent *e) {
	PLUGIN_LOOP(root_button_press, screen, e);
}

/* button releases on the root window */
void plugin_root_button_release(screen_t *screen, XButtonEvent *e) {
	PLUGIN_LOOP(root_button_release, screen, e);
}
