/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include <sys/time.h>

/* 
 * basically a similar thing as action_move, w/ some added stuffs.
 * allow user to move the window if they want to; else the window places itself.  This
 * can be used as a sort of hybrid manual placement.
 */
static void placement_interact(client_t *client) {
	struct timeval tv;
	fd_set fdset;
	XEvent event;
	Window dumwin;
	long pointmask;
	int oldx, oldy;
	int winx, winy;
	int savx, savy;
	int dumint, mask;
	int mousing, xfd;

	pointmask = PointerMotionMask | ButtonMotionMask | ButtonReleaseMask;

	if (XGrabPointer(display, client->screen->root, 0, 0, GrabModeAsync, GrabModeAsync, 
			client->screen->root, 0, CurrentTime) != GrabSuccess)
		return;
	if (XGrabKeyboard(display, client->screen->root, 0, GrabModeAsync, GrabModeAsync,
			CurrentTime) != GrabSuccess) {
		XUngrabPointer(display, CurrentTime);
		return;
	}
	XGrabServer(display);

	oldx = winx = client->x;
	oldy = winy = client->y;
	mousing = 0;
	xfd = ConnectionNumber(display);

	XQueryPointer(display, client->screen->root, &dumwin, &dumwin, &savx, &savy, &dumint, &dumint, &mask);
	XWarpPointer(display, None, client->screen->root, 0, 0, 1, 1, winx, winy);
	draw_winbox(client->screen, client, winx, winy, client->width, client->height);
	XChangeActivePointerGrab(display, pointmask, cursor_place, CurrentTime);

	while (1) {
		while (1) {
			if (XCheckMaskEvent(display, pointmask 
					| KeyPressMask | KeyReleaseMask, &event) == 0)
				break;

			switch (event.type) {
			case MotionNotify:
				mousing = 1;
				winx = event.xmotion.x_root;
				winy = event.xmotion.y_root;

				draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);
				draw_winbox(client->screen, client, winx, winy, client->width, client->height);
				oldx = winx; oldy = winy;
				break;
			case KeyPress:
				event.xkey.window = client->window;
				event.xkey.subwindow = client->window;
				event.xkey.x = 0;
				event.xkey.y = 0;
				XSendEvent(display, client->window, 0, KeyPressMask, &event);
			case ButtonRelease:
				goto done;
			/* throw away keyreleases, buttonpresses */
			}
		}

		tv.tv_usec = options.interact_timeout;
		tv.tv_sec = 0;
		FD_ZERO(&fdset);
		FD_SET(xfd, &fdset);
		if (select(xfd + 1, &fdset, NULL, NULL, tv.tv_usec ? &tv : NULL) == 0)
			break;
	}

done:
	draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);
	XUngrabPointer(display, CurrentTime);
	XUngrabKeyboard(display, CurrentTime);
	XUngrabServer(display);
	if (mousing || event.type == ButtonRelease) {
		if (winx != client->x || winy != client->y) {
			client->x = winx;
			client->y = winy;
			action_send_config(client);
		}
		XWarpPointer(display, None, client->screen->root, 0, 0, 1, 1,
			client->x + ((client->left_space + client->width + client->right_space) / 2),
			client->y + ((client->top_space + client->height + client->bottom_space) / 2));
	} else {
		XWarpPointer(display, None, client->screen->root, 0, 0, 1, 1, savx, savy);
	}
}

/* random placement */
static void placement_random(client_t *client) {
	client->x = random() % (client->screen->width
		- (client->left_space + client->width + client->right_space));
	client->y = random() % (client->screen->height
		- (client->top_space + client->height + client->bottom_space));
}

/* place the client where the mouse cursor is */
static void placement_pointer(client_t *client) {
	Window dumwin;
	int dumint, mask;
	
	XQueryPointer(display, client->screen->root, &dumwin, &dumwin, &client->x, &client->y, &dumint, &dumint, &mask);
}

/* helper for smart; calculate the intersection length of two lines */
static int intersect_length(int p1, int l1, int p2, int l2) {
	int isect, tmp;

	if (p1 > p2) {
		tmp = p1;
		p1 = p2;
		p2 = tmp;
		tmp = l1;
		l1 = l2;
		l2 = tmp;
	}

	if (p1 + l1 < p2)
		isect = 0;
	else if (p2 + l2 < p1 + l1)
		isect = l2;
	else
		isect = p1 + l1 - p2;

	return isect;
}

/* helper for smart; calculate the amount of window area covered by a rect */
static int area_covered(client_t *client, int x, int y, int width, int height) {
	client_t *client_test;
	int area = 0;
	int twidth, theight;

	client_test = client_list;
	while (client_test) {
		if (client_test != client && client_test->screen == client->screen 
				&& client_test->workspace && client_test->workspace->desktop == client->screen->desktop
				&& client_test->x > 0 && client_test->y > 0 && client_test->x < client->screen->width
				&& client_test->y < client->screen->width 
				&& client_test->state == NormalState) {
			twidth = client_test->left_space + client_test->width + client_test->right_space;
			theight = client_test->top_space + client_test->height + client_test->bottom_space;
			area += intersect_length(client_test->x, twidth, x, width)
				* intersect_length(client_test->y, theight, y, height);
		}
		client_test = client_test->next;
	}

	return area;
}

/* smart placement; based on wmaker's smart placement routine, but start at the center */
static void placement_smart(client_t *client) {
	int width, height;
	int x, y;
	int midx, midy;
	int sum, low;
	int lowx = 0, lowy = 0;
	int xdir = 1;
	int ydir = 1;

	width = client->left_space + client->width + client->right_space;
	height = client->top_space + client->height + client->bottom_space;
	low = INT_MAX;

	if (width < client->screen->width && height < client->screen->height) {
		midx = (client->screen->width / 2) - (width / 2);
		midy = (client->screen->height / 2) - (height / 2);
	} else {
		/*
		 * XXX: panic cuz none of this works w/ stuff that is
		 * too big.
		 */
		client->x = 0;
		client->y = 0;
		return;
	}


	/* find the place that the window covers the least amount of other windows */	
	for (y = midy; y > 0; ) {
		for (x = midx; x > 0; ) {
			if (x < client->screen->width - width && y < client->screen->height - height) {
				if ((sum = area_covered(client, x, y, width, height)) < low) {
					low = sum;
					lowx = x;
					lowy = y;
				}
			}

			if (xdir)
				x = midx + (midx - x) + PLACETEST_XINC;
			else
				x = midx - (x - midx) - PLACETEST_XINC;
			xdir = !xdir;
		}

		if (ydir)
			y = midy + (midy - y) + PLACETEST_YINC;
		else
			y = midy - (y - midy) - PLACETEST_YINC;
		ydir = !ydir;
	}

	client->x = lowx;
	client->y = lowy;
}

/* exported client placement function */
void placement_place(client_t *client) {
	int interact;

	if (!options.place_transients && client->flags.transient) {
		interact = 0;
	} else if (!options.place_nonzeros && (client->x > 0 || client->y > 0)) {
		interact = 1;
	} else {
		interact = 1;

		switch (options.placement) {
		case PLACEMENT_NONE:					break;
		case PLACEMENT_RANDOM:	placement_random(client);	break;
		case PLACEMENT_POINTER:	placement_pointer(client);	break;
		case PLACEMENT_SMART:	placement_smart(client);	break;
		}
	}

	/* allow plugins to draw an animation */
	plugin_anim_birth(client);

	if (interact && options.place_interactive)
		placement_interact(client);
}
