/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* list of pixmap structs */
static pixmap_t *pixmap_list	= NULL;

/* free all pixmaps in preperation for a shutdown/restart */
void pixmap_shutdown() {
	pixmap_t *pixmap, *tmp;
	int i;

	pixmap = pixmap_list;
	while (pixmap) {
		/*
		 * free all the memory associated with the pixmap;
		 * pixmap->xpmfn is freed when pixmap_getpixmaps is
		 * called, so it is not neccessary to free it here.
		 */
		for (i = 0; i < screen_count; i++) {
			XFreePixmap(display, pixmap->pixmaps[i]);
			XFreePixmap(display, pixmap->shapemasks[i]);
		}
		free(pixmap->ident);
		free(pixmap->pixmaps);
		free(pixmap->shapemasks);

		/* now free the pixmap_t */
		tmp = pixmap->next;
		free(pixmap);
		pixmap = tmp;
	}
}

/* add a pixmap to our list */
pixmap_t *pixmap_add(char *ident, char *xpmfn) {
	pixmap_t *pixmap;

	pixmap = calloc(1, sizeof(pixmap_t));
	if (!pixmap)
		return NULL;
	pixmap->ident = ident;
	pixmap->xpmfn = xpmfn;

	pixmap->next = pixmap_list;
	pixmap_list = pixmap;

	return pixmap;
}

/* find a pixmap by identifier; used during rcfile parsing */
pixmap_t *pixmap_ident(char *ident) {
	pixmap_t *pixmap;

	pixmap = pixmap_list;
	while (pixmap) {
		if (strcmp(pixmap->ident, ident) == 0)
			return pixmap;
	
		pixmap = pixmap->next;
	}

	return NULL;
}

/*
 * get the actual X pixmaps; screen should point to the first
 * screen_t * in the list of screens (which is also the highest
 * numbered screen in the list; new screens are added onto the
 * front of the screen list).  return a -1 on error.
 */
int pixmap_getpixmaps(screen_t *screen_list) {
	pixmap_t *pixmap;
	screen_t *screen;
	Window dumwin;
	int dumint;

	pixmap = pixmap_list;
	while (pixmap) {
		/* get memory */
		pixmap->pixmaps = malloc(sizeof(Pixmap) * screen_count);
		if (!pixmap->pixmaps)
			return -1;
		pixmap->shapemasks = malloc(sizeof(Pixmap) * screen_count);
		if (!pixmap->shapemasks)
			return -1;

		/* get the pixmaps */
		screen = screen_list;
		while (screen) {
			XpmReadFileToPixmap(display, screen->root, pixmap->xpmfn, &pixmap->pixmaps[screen->num],
				&pixmap->shapemasks[screen->num], NULL);
			screen = screen->next;
		}

		/* get the dimensions */
		XGetGeometry(display, pixmap->pixmaps[0], &dumwin, &dumint, &dumint, &pixmap->width,
			&pixmap->height, &dumint, &dumint);

		/* free the pixmap filename; it's no longer needed */
		free(pixmap->xpmfn);

		pixmap = pixmap->next;
	}

	return 0;
}
