/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef _CLIENT_H
#define _CLIENT_H

/* forwards */
struct dgroup;		/* dgroup.h */
struct plugin;		/* plugin.h */

/* various flags set on clients */
typedef struct {
	unsigned int nofocus:1;			/* don't allow input focus: don't go in the focus_list */
	unsigned int noresize:1;		/* don't allow resizing */
	unsigned int nomove:1;			/* don't allow moving */
	unsigned int noiconify:1;		/* don't allow iconification */
	unsigned int nodelete:1;		/* don't allow window deletion */

	unsigned int zoomed:1;			/* is this window zoomed (maximized, etc) */
	unsigned int internal:1;		/* this managed window is one of ours (not a client's) */
	unsigned int transient:1;		/* this is a transient window */
	unsigned int shaped:1;			/* have shaped frame window cuz client window is shaped */
	unsigned int sticky:1;			/* is this window sticky: stay open on workspace switches */

	unsigned int unmap_from_reparent:1;	/* the unmap_notify event we get is from reparenting */
} clientflags_t;

/*
 * structure describing a managed client.  this is the place where we keep all of
 * the data relating to individually managed windows; the windows may or may not
 * be made by other wmanager clients; if flags.internal it means it is one of our
 * internal windows, and the client->plugin will point to the plugin that created it.
 */
typedef struct client {
	Window		window;			/* the client window */
	screen_t	*screen;		/* screen this client is on */
	workspace_t	*workspace;		/* workspace on the screen for this client */
	int		state;			/* the window's state (X states) */

	int		x, y;			/* geometry information */
	int		width, height;
	int		save_x, save_y;		/* saved geometry (while zoomed, etc) */
	int		save_width, save_height;
	int		last_width, last_height;/* last width and height for hinting reshapes */

	/*
	 * XXX: right now these members only serve
	 * to increase the overhead of a client_t (doh).
	 * this information is acessable inside of
	 * the client's dgroup structure, and I
	 * have not yet decided if there's going
	 * to be a need to keep them here also.
	 */
	int		top_space;		/* space made by the decoration on all sides */
	int		left_space;
	int		right_space;
	int		bottom_space;

	XWMHints	*wmhints;		/* hints from the client */
	XSizeHints	normalhints;
	XClassHint	classhint;

	char		*store_name;		/* the window's name */

	Window	 	frame;			/* frame window (reparent of window) */
	Window		*decorwin;		/* decoration windows */

	struct dgroup	*dgroup;		/* decoration group this client uses */
	struct plugin	*plugin;		/* if created by a plugin, this lets us know which */
	clientflags_t	flags;			/* flags for this client */

	struct client	*focus_next;		/* list pointers for focusing list */
	struct client	*focus_prev;

	struct client	*next;			/* list pointers for the client list */
	struct client	*prev;
} client_t;

extern client_t *client_list;
extern client_t *client_focused;

void client_shutdown();
client_t *client_add(screen_t *screen, Window w, clientflags_t *flags, struct plugin *plugin, struct dgroup *dgroup);
void client_map(client_t *client, int allowplace);
void client_rm(client_t *client);
void client_shape(client_t *client);
void client_sizeframe(client_t *client);
void client_setstate(client_t *client, int state);

#endif
