/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* list of clients */
client_t 	*client_list 		= NULL;
client_t 	*client_focused		= NULL;

/*
 * shutdown client stuff; unload all client_t's and reparent all clients
 * back to the root window.
 */
void client_shutdown() {
	client_t *client, *tmp;

	/* is probably not neccsary, just safe */
	XGrabServer(display);

	client = client_list;
	while (client) {
		if (!client->flags.internal) {
			XReparentWindow(display, client->window, client->screen->root,
				client->x, client->y);
			/* XXX: should save their initial border */
			XSetWindowBorderWidth(display, client->window, 1);
		}
		tmp = client->next;
		client_rm(client);
		client = tmp;
	}

	XUngrabServer(display);
}

/* XWMHints structure gets handled here (the part of it that we use, that is) */
static void client_init_wmhints(client_t *client, dgroup_t *dgroup) {
	Atom type;
	int dumint;
	long dumlong, *state;

	/*
	 * start it up in the state it thinks it's in, or use the state hint
	 * in the XWMHints structure.
	 */
	XGetWindowProperty(display, client->window, WM_STATE, 0, 1, 0, WM_STATE,
		&type, &dumint, &dumlong, &dumlong, (unsigned char **) &state);
	if (type == WM_STATE) {
		client->state = *state;
		XFree(state);
	} else {
		if (client->wmhints && client->wmhints->flags & StateHint)
			client->state = client->wmhints->initial_state;
		else
			client->state = WithdrawnState;
	}
}

/* add decoration for a client during client_add */
static void client_decorate(client_t *client, dgroup_t *dgroup) {
	XSetWindowAttributes attr;

	/* figure out what decoration group to use, unless a hints handler set it already */
	if (!client->dgroup) {
		if (dgroup) {
			client->dgroup = dgroup;
		} else {
			if (client->flags.transient)
				client->dgroup = options.dgroup_trans;
			else if (client->flags.internal)
				client->dgroup = options.dgroup_internal;
			else
				client->dgroup = options.dgroup_default;
		}
	}

	dgroup_set_space(client);

	attr.override_redirect = 1;
	attr.background_pixel = BlackPixel(display, client->screen->num);
	client->frame = XCreateWindow(display, client->screen->root, client->x, client->y, 
		client->width + client->left_space + client->right_space, 
		client->height + client->top_space + client->bottom_space, 0, CopyFromParent, 
		CopyFromParent, CopyFromParent, CWOverrideRedirect | CWBackPixel, &attr);
	XSetWindowBorderWidth(display, client->window, 0);
	decor_decorate(client);
	client_shape(client);
}

/* grab input on our decoration for the client */
static void client_dograbs(client_t *client) {
	XSetWindowAttributes attr;
	long mask;

	mask = SubstructureNotifyMask | SubstructureRedirectMask;
	switch (options.focus) {
	case FOCUS_CLICK:
		if (!client->flags.nofocus)
			XGrabButton(display, AnyButton, AnyModifier, client->frame,
				1, ButtonPressMask, GrabModeSync, GrabModeAsync, None, None);
		else
			XGrabButton(display, AnyButton, options.mouse_modifier, client->frame,
				1, ButtonPressMask, GrabModeSync, GrabModeAsync, None, None);
		break;
	case FOCUS_POINTER:
		mask |= LeaveWindowMask;
	case FOCUS_SLOPPY:
		mask |= EnterWindowMask;
		XGrabButton(display, AnyButton, options.mouse_modifier, client->frame,
			1, ButtonPressMask, GrabModeSync, GrabModeAsync, None, None);
		break;
	}
	XSelectInput(display, client->frame, mask);
	XShapeSelectInput(display, client->window, ShapeNotifyMask);

	/* prevent us from recieving some unneccessary events */
	attr.do_not_propagate_mask = ButtonPressMask | ButtonReleaseMask;
	mask = CWDontPropagate;
	XChangeWindowAttributes(display, client->window, mask, &attr);
}

/*
 * add a client.
 * flags is the flags for the client, or NULL if we use defaults.
 * not all of the flags passed in will be honored: we decide here
 * if it's a transient, etc.  if plugin is not NULL and it's an internal
 * it means this should be associated with that plugin as an internal
 * window.  dgroup is the dgroup to use, otherwise we will try to figure
 * out for ourselves and use dgroups based on if it is a transient/internal,
 * and so forth.
 */
client_t *client_add(screen_t *screen, Window w, clientflags_t *flags, plugin_t *plugin, dgroup_t *dgroup) {
	XWindowAttributes win_attr;
	Window dumwin;
	client_t *client = NULL;
	long supplied;

	/*
	 * we need to do a grab so that it's not possible for the window we are
	 * about to manage to disappear halfway through the process of us
	 * preparing to manage it.
	 */
	XGrabServer(display);

	/* if we can't get window attributes, give up */
	if (!XGetWindowAttributes(display, w, &win_attr))
		goto done;

	/* get mem and set flags */
	client = calloc(1, sizeof(client_t));
	if (!client)
		goto done;
	if (flags)
		memcpy(&client->flags, flags, sizeof(clientflags_t));

	/* fill in client_t stuff */
	client->window = w;
	client->screen = screen;
	client->x = win_attr.x;
	client->y = win_attr.y;
	client->last_width = client->width = win_attr.width;
	client->last_height = client->height = win_attr.height;

	/* get PropertyChange information */
	if (!client->flags.internal)
		XSelectInput(display, client->window, PropertyChangeMask);

	/* get hints and do stuff that they require */
	client->wmhints = XGetWMHints(display, client->window);
	XGetWMNormalHints(display, client->window, &client->normalhints, &supplied);
	XGetClassHint(display, client->window, &client->classhint);
	XFetchName(display, client->window, &client->store_name);
	if (XGetTransientForHint(display, client->window, &dumwin))
		client->flags.transient = 1;

	/* do window manager hints, and decorate */
	client_init_wmhints(client, dgroup);
	plugin_init_hints(client, dgroup);
	client_decorate(client, dgroup);

	/* put this client_t into the x context for this window, and also in it's decorations */
	XSaveContext(display, client->window, client_context, (XPointer) client);
	XSaveContext(display, client->frame, client_context, (XPointer) client);

	/* select input and then reparent the client to it's frame window */
	client_dograbs(client);
	XReparentWindow(display, client->window, client->frame, client->left_space, client->top_space);

	/* link it up */
	client->next = client_list;
	client_list = client;
	if (client->next)
		client->next->prev = client;
	client->prev = NULL;

	/* save non internals */
	if (!client->flags.internal)
		XAddToSaveSet(display, client->window);

	/* clients may be associated with a plugin, NULL if it's not */
	client->plugin = plugin;

done:
	/* ungrab the server */
	XUngrabServer(display);

	return client;
}

/* remove a client */
void client_rm(client_t *client) {
	/* unlink */
	if (client_list == client)
		client_list = client->next;
	if (client->prev)
		client->prev->next = client->next;
	if (client->next)
		client->next->prev = client->prev;

	if (client->wmhints) 			XFree(client->wmhints);
	if (client->classhint.res_name) 	XFree(client->classhint.res_name);
	if (client->classhint.res_class)	XFree(client->classhint.res_class);
	if (client->store_name)			XFree(client->store_name);

	XDeleteContext(display, client->frame, client_context);
	XDeleteContext(display, client->window, client_context);
	XDestroyWindow(display, client->frame);

	decor_undecor(client);

	/* if it's on a workspace get it off */
	if (client->workspace) workspace_rm_client(client);
	free(client);
}

/* called by map_request and by manage_existing_windows (screen.c) */
void client_map(client_t *client, int allowplace) {
	/* no matter what state, we map it's top window */
	XMapWindow(display, client->window);

	/* to prevent losing clients if you restart after changing desktop dims */
	if (client->x + client->width + client->left_space + client->right_space < 0 
			|| client->x >= (client->screen->desktop->width - client->screen->desktop->viewx)
			* client->screen->width || client->y + client->height + client->top_space
			+ client->bottom_space < 0
			|| client->y >= (client->screen->desktop->height - client->screen->desktop->viewy)
			* client->screen->width) {
		client->y = client->x = 0;
		client_sizeframe(client);
	}

	/* put it into the state it wants to be in */
	switch (client->state) {
	case IconicState:
		/* client starts as an icon */
		client_setstate(client, IconicState);
		plugin_iconify_notify(client);
		break;
	case WithdrawnState:
	case NormalState:
	default:
		/* window placement, and birth anim */
		if (allowplace) {
			placement_place(client);
			client_sizeframe(client);
		} else {
			/*
			 * if placement is not allowed, it means that we are probably
			 * running on a window that existed before the window manager
			 * was started; this means the unmap_from_reparent flag will
			 * still be set.  If the flag isn't set, we still do a birth
			 * animation.
			 */
			if (!client->flags.unmap_from_reparent)
				plugin_anim_birth(client);
		}

		/* the window gets focus in map_notify */
		XMapRaised(display, client->frame);
		client_setstate(client, NormalState);

		/* send configure to the client to make sure it knows where it is */
		action_send_config(client);

		/* add to the workspace appropriate for the client's position on the screen */
		workspace_add_bypos(client->screen->desktop, client);

		/* plugin notification */
		plugin_window_birth(client);
		break;
	}
}

/* shape a client */
void client_shape(client_t *client) {
	XRectangle *rect = NULL;
	XRectangle r1;
	int n, order;

	rect = XShapeGetRectangles(display, client->window, ShapeBounding, &n, &order);
	if (n > 1) {
		r1.x = 0;
		r1.y = 0;
		r1.width = client->width;
		r1.height = client->height;
		XShapeCombineRectangles(display, client->frame, ShapeBounding, client->left_space,
			client->top_space, &r1, 1, ShapeSubtract, Unsorted);

		XShapeCombineShape(display, client->frame, ShapeBounding, client->left_space,
			client->top_space, client->window, ShapeBounding, ShapeUnion);
		client->flags.shaped = 1;
	} else if (client->flags.shaped) {
		r1.x = 0;
		r1.y = 0;
		r1.width = client->width;
		r1.height = client->height;
		XShapeCombineRectangles(display, client->frame, ShapeBounding, client->left_space,
			client->top_space, &r1, 1, ShapeUnion, Unsorted);
		client->flags.shaped = 0;
	}

	XFree(rect);
}

/* resize the decoration and such for a client: reshape, etc. */
void client_sizeframe(client_t *client) {
	if (client->width == client->last_width && client->height == client->last_height) {
		XMoveWindow(display, client->frame, client->x, client->y);
	} else {
		XMoveResizeWindow(display, client->frame, client->x, client->y,
			client->width + client->left_space + client->right_space,
			client->height + client->top_space + client->bottom_space);
		XResizeWindow(display, client->window, client->width, client->height);

		decor_shapesize(client);
		if (client->flags.shaped)
			client_shape(client);

		client->last_width = client->width;
		client->last_height = client->height;
	}
}

/* set the state for a window (our var and the WM_STATE) */
void client_setstate(client_t *client, int state) {
	unsigned long data[2];

	data[0] = state;
	data[1] = None;

	XChangeProperty(display, client->window, WM_STATE, WM_STATE, 32,
		PropModeReplace, (unsigned char *) data, 2);
	client->state = state;

	XSync(display, 0);
}
