/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* spawn a child */
pid_t action_exec(int screen, char *cmd) {
	char buff[256];
	char number[10];
	char *ptr;
	pid_t pid;

	if ((pid = fork()) == 0) {
		snprintf(buff, sizeof(buff), "DISPLAY=%s", XDisplayName(NULL));
		ptr = strchr(strchr(buff, ':'), '.');
		if (ptr) *ptr = '\0';
		snprintf(number, sizeof(number), "%i", screen);
		strncat(buff, ".", sizeof(buff) - strlen(buff));
		strncat(buff, number, sizeof(buff) - strlen(buff));
		putenv(buff);

		execl(_PATH_BSHELL, "sh", "-c", cmd, NULL);
		exit(1);
	}

	return pid;
}

/* send a synthetic ConfigureNotify event to a client (as on a move, etc) */
void action_send_config(client_t *client) {
	XEvent event;

	event.type = ConfigureNotify;
	event.xconfigure.display = display;
	event.xconfigure.event = client->window;
	event.xconfigure.window = client->window;
	event.xconfigure.x = client->x + client->left_space;
	event.xconfigure.y = client->y + client->top_space;
	event.xconfigure.width = client->width;
	event.xconfigure.height = client->height;
	event.xconfigure.border_width = 0;
	event.xconfigure.above = client->frame;
	event.xconfigure.override_redirect = 0;
	XSendEvent(display, client->window, 0, StructureNotifyMask, &event);
}

/* send a ICCCM Client Message */
void action_sendcmesg(Window w, Atom a, Time timestamp) {
	XClientMessageEvent e;

	e.type = ClientMessage;
	e.window = w;
	e.message_type = WM_PROTOCOLS;
	e.format = 32;
	e.data.l[0] = a;
	e.data.l[1] = timestamp;
	XSendEvent(display, w, 0, 0, (XEvent *) &e);
}

/* interactivly move a window */
void action_move(client_t *client) {
	XEvent event;
	Window dumwin;
	long eventmask;
	int oldx, oldy;
	int winx, winy;
	int ptx, pty;
	int moving;
	int dumint;

	/* ungrab the grab we have from the passive grab */
	if (client->flags.nomove) {
		XUngrabPointer(display, CurrentTime);
		return;
	}

	eventmask = PointerMotionMask | ButtonMotionMask | ButtonReleaseMask;
	if (XGrabPointer(display, client->screen->root, 0, eventmask, GrabModeAsync, GrabModeAsync,
			client->screen->root, None, CurrentTime) != GrabSuccess)
		return;
	if (options.opaquemove) {
		XRaiseWindow(display, client->frame);
		eventmask |= ExposureMask;
	} else {
		XGrabServer(display);
	}

	XQueryPointer(display, client->screen->root, &dumwin, &dumwin, &ptx, &pty, &dumint, &dumint, &dumint);
	oldx = winx = client->x;
	oldy = winy = client->y;
	moving = 0;

	while (1) {
		XMaskEvent(display, eventmask, &event);
		
		switch (event.type) {
		case MotionNotify:
			winy += (event.xmotion.y_root - pty);
			pty = event.xmotion.y_root;
			winx += (event.xmotion.x_root - ptx);
			ptx = event.xmotion.x_root;

			if (!options.opaquemove) {
				if (moving)
					draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);
				draw_winbox(client->screen, client, winx, winy, client->width, client->height);
			} else
				XMoveWindow(display, client->frame, winx, winy);

			if (!moving) {
				XChangeActivePointerGrab(display, eventmask & ~ExposureMask, cursor_move, CurrentTime);
				moving = 1;
			}
			oldx = winx;
			oldy = winy;
			break;
		case ButtonRelease:
			XUngrabPointer(display, CurrentTime);
			if (!options.opaquemove && moving)
				draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);
			if (!options.opaquemove) {
				XRaiseWindow(display, client->frame);
				XUngrabServer(display);
			}
			if (winx != client->x || winy != client->y) {
				client->x = winx;
				client->y = winy;
				XMoveWindow(display, client->frame, winx, winy);
				action_send_config(client);

				/* let plugins know about geometry change */
				plugin_geometry_change(client);
			}
			return;
		case Expose:
			event_expose(&event);
			break;
		}
	}
}

/* 
 * interactively size a window (this is twm-style)
 * works as follows: once you move your mouse across a particular edge, that edge
 * becomes bound to the mouse movements, moving the mouse across the opposite edge will
 * unbind the currently bound edge in favor of the new one (using the dx and dy ptrs).
 */
void action_resize(client_t *client) {
	XEvent event;
	Window dumroot, dumchild;
	int x1, y1, x2, y2;
	int oldx1, oldy1, oldx2, oldy2;
	int *dx, *dy;
	int ptx, pty;
	int mask;
	int minw, minh, basew, baseh;

	/* ungrab the grab we have from the passive grab */
	if (client->flags.noresize) {
		XUngrabPointer(display, CurrentTime);
		return;
	}

	if (XGrabPointer(display, client->screen->root, 0, ButtonReleaseMask | PointerMotionMask | 
			ButtonMotionMask, GrabModeAsync, GrabModeAsync, client->screen->root, 
			cursor_move, CurrentTime) != GrabSuccess)
		return;
	XGrabServer(display);
	XRaiseWindow(display, client->frame);

	XQueryPointer(display, client->screen->root, &dumroot, &dumchild, &ptx, &pty, &x1, &y1, &mask);
	oldx1 = x1 = client->x;
	oldy1 = y1 = client->y;
	oldx2 = x2 = x1 + client->left_space + client->right_space + client->width;
	oldy2 = y2 = y1 + client->top_space + client->bottom_space + client->height;
	dx = dy = NULL;

	draw_winbox(client->screen, client, x1, y1, x2 - x1 - client->left_space - client->right_space,
		y2 - y1 - client->top_space - client->bottom_space);

	/* figure out min & base heights for normal hints */		
	if (client->normalhints.flags & PBaseSize) {
		basew = client->normalhints.base_width;
		baseh = client->normalhints.base_height;
		if (client->normalhints.flags & PMinSize) {
			minw = client->normalhints.min_width;
			minh = client->normalhints.min_height;
		} else {
			minw = basew;
			minh = baseh;
		}
	} else {
		/* ICCCM says to use PMinSize if we can't use PBaseSize, etc */
		if (client->normalhints.flags & PMinSize) {
			minw = basew = client->normalhints.min_width;
			minh = baseh = client->normalhints.min_height;
		} else {
			minw = basew = 0;
			minh = baseh = 0;
		}
	}

	while (1) {
		XMaskEvent(display, PointerMotionMask | ButtonMotionMask | ButtonReleaseMask, 
			&event);

		switch (event.type) {
		case MotionNotify:
			ptx = event.xmotion.x_root;
			pty = event.xmotion.y_root;

			if (ptx > client->x + client->left_space + client->right_space + client->width) {
				x1 = client->x;
				dx = &x2;
			} else if (ptx < client->x) {
				x2 = client->x + client->left_space + client->right_space + client->width;
				dx = &x1;
			}

			if (pty > client->y + client->top_space + client->bottom_space + client->height) {
				y1 = client->y;
				dy = &y2;
			} else if (pty < client->y) {
				y2 = client->y + client->top_space + client->bottom_space + client->height;
				dy = &y1;
			}

			if (dx) *dx = ptx;
			if (dy) *dy = pty;

			/* this mess checks constraints imposed from our normalhints */
#define SETWIDTH(w) do { 							\
	if (dx == &x1)								\
		x1 = x2 - (w) - client->left_space - client->right_space;	\
	else if (dx == &x2)							\
		x2 = x1 + (w) + client->left_space + client->right_space;	\
} while (0)
#define SETHEIGHT(h) do {							\
	if (dy == &y1)								\
		y1 = y2 - (h) - client->top_space - client->bottom_space;	\
	else if (dy == &y2)							\
		y2 = y1 + client->top_space + client->bottom_space + (h);	\
} while (0)
			if (client->normalhints.flags & PResizeInc) {
				int wid, hgt;

				wid = x2 - x1 - client->left_space - client->right_space - basew;
				hgt = y2 - y1 - client->top_space - client->bottom_space - baseh;

				SETWIDTH((wid + client->normalhints.width_inc 
					- (wid % client->normalhints.width_inc) 
					+ basew) - client->normalhints.width_inc);
				SETHEIGHT((hgt + client->normalhints.height_inc 
					- (hgt % client->normalhints.height_inc) 
					+ baseh) - client->normalhints.height_inc);
			}
			if (client->normalhints.flags & PMaxSize) {
				if (x2 - x1 - client->left_space - client->right_space > client->normalhints.max_width)
					SETWIDTH(client->normalhints.max_width);
				if (y2 - y1 - client->top_space - client->bottom_space > client->normalhints.max_height)
					SETHEIGHT(client->normalhints.max_height);
			}
			if (client->normalhints.flags & PMinSize) {
				if (x2 - x1 - client->left_space - client->right_space < client->normalhints.min_width)
					SETWIDTH(minw);
				if (y2 - y1 - client->top_space - client->bottom_space < client->normalhints.min_height)
					SETHEIGHT(minh);
			} else {
				if (x2 - x1 - client->left_space - client->right_space < 1)
					SETWIDTH(1);
				if (y2 - y1 - client->top_space - client->bottom_space < 1)
					SETHEIGHT(1);
			}
#undef SETWIDTH
#undef SETHEIGHT
			if (oldx1 != x1 || oldy1 != y1 || oldx2 != x2 || oldy2 != y2) {
				draw_winbox(client->screen, client, oldx1, oldy1,
					oldx2 - oldx1 - client->left_space - client->right_space,
					oldy2 - oldy1 - client->top_space - client->bottom_space);
				draw_winbox(client->screen, client, x1, y1,
					x2 - x1 - client->left_space - client->right_space,
					y2 - y1 - client->top_space - client->bottom_space);
				oldx1 = x1; oldy1 = y1; oldx2 = x2; oldy2 = y2;
			}
			break;
		case ButtonRelease:
			draw_winbox(client->screen, client, oldx1, oldy1,
				oldx2 - oldx1 - client->left_space - client->right_space,
				oldy2 - oldy1 - client->top_space - client->bottom_space);
			XUngrabPointer(display, CurrentTime);
			XUngrabServer(display);
			if (x1 != client->x || y1 != client->y 
					|| x2 != client->x + client->width
					|| y2 != client->y + client->top_space + client->height) {
				client->x = x1;
				client->y = y1;
				client->width = x2 - x1 - client->left_space - client->right_space;
				client->height = y2 - y1 - client->top_space - client->bottom_space;
				client_sizeframe(client);

				/* let plugins know about geometry change */
				plugin_geometry_change(client);
			}
			return;
		}
	}
}

/* move a client window to the icon'd window list */
void action_iconify(client_t *client) {
	if (client->flags.noiconify)
		return;

	/* let plugs know, and then unmap it to iconic */
	XUnmapWindow(display, client->frame);
	client_setstate(client, IconicState);
	plugin_iconify_notify(client);

	/* get it out of the workspace */
	workspace_rm_client(client);
}

/* restore an iconified client window */
void action_restore(client_t *client) {
	if (client->state != IconicState)
		return;

	/* set state to normal, and add it to the current workspace */
	client_setstate(client, NormalState);
	workspace_add_client(client->screen->desktop->current_space, client);
	plugin_restore_notify(client);
	XMapRaised(display, client->frame);
	focus_setfocused(client);
} 

/* zoom (maximize) or unzoom a window */
void action_zoom(client_t *client) {
	/* zooming counts as resizing and as moving */
	if (client->flags.noresize || client->flags.nomove)
		return;

	/* zoom or unzoom it */
	if (client->flags.zoomed) {
		client->flags.zoomed = 0;
		client->x = client->save_x;
		client->y = client->save_y;
		client->width = client->save_width;
		client->height = client->save_height;
	} else {
		client->flags.zoomed = 1;
		client->save_x = client->x;
		client->save_y = client->y;
		client->save_width = client->width;
		client->save_height = client->height;
		if (options.fullscreen_zoom) {
			client->x = -client->left_space;
			client->y = -client->top_space;
			client->width = client->screen->width;
			client->height = client->screen->height;
		} else {
			client->x = 0;
			client->y = 0;
			client->width = client->screen->width - client->left_space - client->right_space;
			client->height = client->screen->height - client->top_space - client->bottom_space;
		}
		XRaiseWindow(display, client->frame);
		focus_setfocused(client);
	}

	/* notify plugins */
	if (client->flags.zoomed)
		plugin_zoom_notify(client);
	else
		plugin_unzoom_notify(client);

	/*
	 * It is neccesary to send a synth configure notify if we didn't
	 * resize the client, but only moved it, because it wont recieve
	 * a real one in that case.
	 */
	client_sizeframe(client);
	if (client->width == client->save_width && client->height == client->save_height)
		if (client->x != client->save_x || client->y != client->save_y)
			action_send_config(client);
}
