/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "plugutil.h"

/* default flags and the like for a WM_CLASS */
typedef struct wcdefs {
	char		*res_name;	/* resource name */
	char		*res_class;	/* pointer into res_name, don't free() it */

	/* dgroup to use, or null */
	dgroup_t	*dgroup;

	/* these correspond to clientflags_t, read client.h */
	int		nofocus;
	int		noresize;
	int		nomove;
	int		noiconify;
	int		nodelete;
	int		sticky;

	/* list pointer */
	struct wcdefs *next;
} wcdefs_t;

/* list of wcdefs */
static wcdefs_t	*wcdefs_list	= NULL;

/* find a wcdefs based on a res_name and res_class */
static wcdefs_t *wcdefs_find(char *res_name, char *res_class) {
	wcdefs_t *wcdefs;

	/* make sure it has res_name and res_class */
	if (!res_name || !res_class)
		return NULL;

	/* loop through and cmp the strings */
	wcdefs = wcdefs_list;
	while (wcdefs) {
		if (strcmp(wcdefs->res_class, res_class) == 0)
			if (strcmp(wcdefs->res_name, res_name) == 0)
				goto found;
		wcdefs = wcdefs->next;
	}

	return NULL;
found:
	return wcdefs;
}

/* build our defaults information structures */
int init(plugin_t *plugin) {
	wcdefs_t *wcdefs;
	param_t *param;
	char *res_name, *res_class;
	char *dgroup_name;
	int i;

	/* get all the wcdefs_t's built */
	SUBPARAMS_FOREACH(i, param, &plugin->params) {
		if (strcmp(param->name, "def") != 0)
			continue;

		/* get the res_name.res_class pair from the value */
		res_name = strdup(param->value);
		if (!res_name)
			PERR(plugin, "no memory to strdup res_name.res_class");
		res_class = strrchr(res_name, '.');
		if (*res_class != '\0')
			*res_class++ = '\0';

		/* see if a entry exists, or make one */
		wcdefs = wcdefs_find(res_name, res_class);
		if (!wcdefs) {
			wcdefs = malloc(sizeof(wcdefs_t));
			if (!wcdefs) {
				free(res_name);
				PERR(plugin, "no memory for wcdefs");
			}

			/* init the structure */
			wcdefs->res_name = res_name;
			wcdefs->res_class = res_class;
			wcdefs->dgroup = NULL;
			wcdefs->nofocus = -1;
			wcdefs->noresize = -1;
			wcdefs->nomove = -1;
			wcdefs->noiconify = -1;
			wcdefs->nodelete = -1;
			wcdefs->sticky = -1;

			/* place it onto the list */
			wcdefs->next = wcdefs_list;
			wcdefs_list = wcdefs;
		} else
			free(res_name);
		
		/* now read for anything they set */
		OPTIONAL_PARAM(wcdefs->nofocus, &param->subparams, bool, "nofocus", wcdefs->nofocus);
		OPTIONAL_PARAM(wcdefs->noresize, &param->subparams, bool, "noresize", wcdefs->noresize);
		OPTIONAL_PARAM(wcdefs->nomove, &param->subparams, bool, "nomove", wcdefs->nomove);
		OPTIONAL_PARAM(wcdefs->noiconify, &param->subparams, bool, "noiconify", wcdefs->noiconify);
		OPTIONAL_PARAM(wcdefs->nodelete, &param->subparams, bool, "nodelete", wcdefs->nodelete);
		OPTIONAL_PARAM(wcdefs->sticky, &param->subparams, bool, "sticky", wcdefs->sticky);

		/* dgroup */
		OPTIONAL_PARAM(dgroup_name, &param->subparams, string, "dgroup", NULL);
		if (dgroup_name) {
			wcdefs->dgroup = dgroup_ident(dgroup_name);
			free(dgroup_name);
		}
	}

	return PLUGIN_OK;
}

/* free our wcdefs */
void shutdown() {
	wcdefs_t *wcdefs, *tmp;

	wcdefs = wcdefs_list;
	while (wcdefs) {
		/* don't free res_class, it's just a ptr into res_name */
		free(wcdefs->res_name);
		tmp = wcdefs->next;
		free(wcdefs);
		wcdefs = tmp;
	}
}

/* handle default setting on newly arriving clients */
int init_hints(plugin_t *plugin, client_t *client, dgroup_t *dgroup) {
	wcdefs_t *wcdefs;

	/* find an appropriate wcdefs */
	wcdefs = wcdefs_find(client->classhint.res_name, client->classhint.res_class);
	if (!wcdefs)
		return PLUGIN_OK;

	/*
	 * set flags to our wcdefs flags.  don't touch flags that the user didn't
	 * say anything about in their rc.
	 */
	if (wcdefs->nofocus != -1)
		client->flags.nofocus = wcdefs->nofocus;
	if (wcdefs->noresize != -1)
		client->flags.noresize = wcdefs->noresize;
	if (wcdefs->nomove != -1)
		client->flags.nomove = wcdefs->nomove;
	if (wcdefs->noiconify != -1)
		client->flags.noiconify = wcdefs->noiconify;
	if (wcdefs->nodelete != -1)
		client->flags.nodelete = wcdefs->nodelete;
	if (wcdefs->sticky != -1)
		client->flags.sticky = wcdefs->sticky;

	/* set dgroup if we got one specified */
	if (wcdefs->dgroup)
		client->dgroup = wcdefs->dgroup;

	return PLUGIN_OK;
}
