/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "plugutil.h"
#include "pier.h"
#include "parseparams.h"

/* items that do nothing other than pixmap like something */
static void handle_nothing(plugin_t *plugin, pier_t *pier, param_t *param, int type) {
	char *pixmap;

	OPTIONAL_PARAM(pixmap, &param->subparams, string, "pixmap", NULL);

	if (pier_additem(pier, type, NULL, NULL, NULL, pixmap) == NULL)
		if (pixmap)
			free(pixmap);
}

/* handle launcher items */
static void handle_launch(plugin_t *plugin, pier_t *pier, param_t *param, int type) {
	char *cmd;
	char *pixmap;

	OPTIONAL_PARAM(cmd, &param->subparams, string, "cmd", NULL);
	if (!cmd) {
		PWARN(plugin, "cmd subparam is required for pier launch items");
		return;
	}
	OPTIONAL_PARAM(pixmap, &param->subparams, string, "pixmap", NULL);

	if (pier_additem(pier, type, cmd, NULL, NULL, pixmap) == NULL) {
		if (pixmap)
			free(pixmap);
		free(cmd);
	}
}

/* handle afterstep-style swallowed apps */
static void handle_swallow(plugin_t *plugin, pier_t *pier, param_t *param, int type) {
	char *cmd;
	char *classstr, *c;
	char *res_name, *res_class;

	OPTIONAL_PARAM(cmd, &param->subparams, string, "cmd", NULL);
	if (!cmd) {
		PWARN(plugin, "cmd subparam is required for pier swallowed/docked apps");
		return;
	}
	OPTIONAL_PARAM(classstr, &param->subparams, string, "class", NULL);
	if (!classstr) {
		PWARN(plugin, "class subparam is required for swallowed/docked apps");
		goto free1;
	}

	/* divide the classstr at the '.', for res_name.res_class */
	c = strchr(classstr, '.');
	*c = '\0';
	c++;
	res_name = strdup(classstr);
	if (!res_name) {
		free(classstr);
		goto free1;
	}
	res_class = strdup(c);
	if (!res_class) {
		free(classstr);
		goto free2;
	}

	free(classstr);
	if (pier_additem(pier, type, cmd, res_name, res_class, NULL) != NULL)
		return;

	free(res_class);
free2:
	free(res_name);
free1:
	free(cmd);
}

/* table of handler functions for item types */
static struct paramhndlr {
	char	*text;
	int	type;
	void	(*handler)(plugin_t *plugin, pier_t *pier, param_t *param, int type);
} handlers[] = {
	{"nothing",	PI_NOTHING,	handle_nothing},
	{"launch",	PI_LAUNCH,	handle_launch},
	{"swallow",	PI_SWALLOWED,	handle_swallow},
	{"dock",	PI_WMDOCK,	handle_swallow}
};

/* read parameters to create piers */
void parseparams(plugin_t *plugin) {
	pier_t *pier;
	param_t *param;
	param_t *itemparam;
	int screen, type, x, y;
	int i, n, t;

	/* loop through all subparams and find the "pier" params */
	SUBPARAMS_FOREACH(i, param, &plugin->params) {
		if (strcmp(param->name, "pier") != 0)
			continue;

		/* get screen and type for the pier */
		OPTIONAL_PARAM(screen, &param->subparams, int, "screen", 0);
		if (strcmp(param->value, "horizontal") == 0)
			type = PT_HORIZ;
		else if (strcmp(param->value, "vertical") == 0)
			type = PT_VERT;
		else {
			PWARNF(plugin, "unknown pier type: %s", param->value);
			continue;
		}

		/* x and y position params; -1 means all the way far */
		OPTIONAL_PARAM(x, &param->subparams, int, "x", 0);
		OPTIONAL_PARAM(y, &param->subparams, int, "y", 0);

		/* make the pier itself, and then loop on it's items */
		pier = pier_create(screen, type, x, y);

		/* handle all the items */
		SUBPARAMS_FOREACH(t, itemparam, &param->subparams) {
			if (strcmp(itemparam->name, "item") != 0)
				continue;

			for (n = 0; n < sizeof(handlers) / sizeof(struct paramhndlr); n++) {
				if (strcmp(handlers[n].text, itemparam->value) == 0) {
					handlers[n].handler(plugin, pier, itemparam, handlers[n].type);
					goto found;
				}
			}
			PWARNF(plugin, "ignoring unknown pier item type %s", itemparam->value);
found:
		}
	}
}
