/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "pager.h"

/* the size ratio of a pager's screen to the real thing */
double		pager_ratio	= 0.04;

/* decoration group for pager windows */
static dgroup_t	*pager_dgroup	= NULL;

/* context for pagers */
XContext	pager_context;

/* setup the pager system */
void pager_init(double ratio, dgroup_t *dgroup) {
	if (ratio > 0)
		pager_ratio = ratio;
	if (dgroup)
		pager_dgroup = dgroup;
	else
		pager_dgroup = options.dgroup_internal;
	pager_context = XUniqueContext();
}

/* create a pager */
pager_t *pager_create(plugin_t *plugin, screen_t *screen, desktop_t *desktop) {
	XSetWindowAttributes attr;
	clientflags_t flags;
	pager_t *pager;
	int wid, hei;

	pager = calloc(1, sizeof(pager_t));
	if (!pager)
		return NULL;
	pager->desktop = desktop;

	/* create pager window */
	wid = pager_ratio * screen->width * desktop->width + (desktop->width - 1);
	hei = pager_ratio * screen->height * desktop->height + (desktop->height - 1);
	attr.background_pixel = BlackPixel(display, screen->num);
	attr.background_pixmap = ParentRelative;
	pager->win = XCreateWindow(display, screen->root,
		0, desktop->num * (hei + pager_dgroup->top_space + pager_dgroup->bottom_space),
		wid, hei, 1, CopyFromParent, CopyFromParent, CopyFromParent, CWBackPixmap, &attr);
	XSelectInput(display, pager->win, ExposureMask | ButtonReleaseMask | ButtonPressMask);
	XSaveContext(display, pager->win, pager_context, (XPointer) pager);

	/* create client_t for our pager */
	bzero(&flags, sizeof(clientflags_t));
	flags.internal = 1;
	flags.nofocus = 1;
	flags.noresize = 1;
	flags.noiconify = 1;
	flags.nodelete = 1;
	pager->client = client_add(screen, pager->win, &flags, plugin, pager_dgroup);
	if (!pager->client)
		goto free1;

	XSetWindowBackgroundPixmap(display, pager->client->frame, ParentRelative);

	/* map windows */
	pager->client->state = NormalState;
	XMapWindow(display, pager->win);
	XMapWindow(display, pager->client->frame);

	return pager;

free1:
	XDestroyWindow(display, pager->win);
	free(pager);
	return NULL;
}

/* clean up after a pager */
void pager_delete(pager_t *pager) {
	paged_t *paged, *tmp;

	/* get rid of paged_t's */
	paged = pager->paged_list;
	while (paged) {
		tmp = paged->next;
		pager_rmpaged(pager, paged, paged->client);
		paged = tmp;
	}

	/* get rid of our window */
	XDeleteContext(display, pager->win, pager_context);
	XDestroyWindow(display, pager->win);
	client_rm(pager->client);
	free(pager);
}

/* add a paged client to a pager */
void pager_addpaged(plugin_t *plugin, pager_t *pager, client_t *client) {
	paged_t *paged;
	int x, y, wid, hei;

	paged = calloc(1, sizeof(paged_t));
	if (!paged)
		return;
	paged->client = client;

	/* make our mini window */
	x = pager_ratio * (client->x + pager->desktop->viewx * client->screen->width);
	x += x / (pager_ratio * client->screen->width);
	y = pager_ratio * (client->y + pager->desktop->viewy * client->screen->height);
	y += y / (pager_ratio * client->screen->height);
	wid = (client->width * pager_ratio) - 2;
	hei = (client->height * pager_ratio) - 2;
	paged->win = XCreateSimpleWindow(display, pager->win,
		x, y, wid > 0 ? wid : 1, hei > 0 ? hei : 1, 1,
		WhitePixel(display, pager->client->screen->num), BlackPixel(display, pager->client->screen->num));
	XMapRaised(display, paged->win);

	/* link it up */
	paged->next = pager->paged_list;
	if (paged->next)
		paged->next->prev = paged;
	paged->prev = NULL;
	pager->paged_list = paged;
}

/*
 * remove a paged client from a pager; if paged is null we find
 * paged ourselves.
 */
void pager_rmpaged(pager_t *pager, paged_t *paged, client_t *client) {
	/* if they supplied a paged, use it */
	if (paged)
		goto found;

	/* find the paged for this client */
	paged = pager->paged_list;
	while (paged) {
		if (paged->client == client)
			goto found;
		paged = paged->next;
	}
	return;

found:
	/* get rid of this paged item */
	XDestroyWindow(display, paged->win);

	/* unlink */
	if (paged->prev)
		paged->prev->next = paged->next;
	if (paged->next)
		paged->next->prev = paged->prev;
	if (pager->paged_list == paged)
		pager->paged_list = paged->next;
	free(paged);
}

/* move a paged_t to a new pager */
void pager_movepaged(pager_t *pager, paged_t *paged, pager_t *newpager) {
	/* unlink the paged from the old pager */
	if (paged->prev)
		paged->prev->next = paged->next;
	if (paged->next)
		paged->next->prev = paged->prev;
	if (pager->paged_list  == paged)
		pager->paged_list = paged->next;

	/* link it up on the new pager */
	paged->next = newpager->paged_list;
	if (paged->next)
		paged->next->prev = paged;
	paged->prev = NULL;
	newpager->paged_list = paged;
}

/*
 * resize/move our pager representation of a client; if paged is null
 * we find the paged_t ourselves, using client.
 */
void pager_sizepaged(pager_t *pager, paged_t *paged, client_t *client) {
	int x, y, wid, hei;

	/* if they gave us a paged_t, use it */
	if (paged)
		goto found;

	/* get the paged_t */
	paged = pager->paged_list;
	while (paged) {
		if (paged->client == client)
			goto found;
		paged = paged->next;
	}
	return;

found:
	x = pager_ratio * (client->x + pager->desktop->viewx * client->screen->width);
	x += x / (pager_ratio * client->screen->width);
	y = pager_ratio * (client->y + pager->desktop->viewy * client->screen->height);
	y += y / (pager_ratio * client->screen->height);
	wid = (client->width * pager_ratio) - 2;
	hei = (client->height * pager_ratio) - 2;
	XMoveResizeWindow(display, paged->win, x, y, wid > 0 ? wid : 1, hei > 0 ? hei : 1);
}

/* clicks on the pager window */
void pager_click(pager_t *pager, int x, int y) {
	desktop_t *desktop;

	desktop = pager->desktop;
	x /= pager->client->screen->width * pager_ratio;
	y /= pager->client->screen->height * pager_ratio;

	/* we switch workspaces, and desktops if it's on a differnt desk */
	workspace_viewport_move(pager->client->screen, desktop, x - desktop->viewx,
		y - desktop->viewy);
	if (desktop != pager->client->screen->desktop)
		workspace_switch_desk(pager->client->screen, desktop->num);
}

/* handle exeposures */
void pager_expose(pager_t *pager, GC gc, XExposeEvent *e) {
	int ratwid, rathei;
	int x, y, wid, hei;
	int i, tmp;
	int vx, vy;

	/* set up some tmp vars */
	ratwid = pager_ratio * pager->client->screen->width;
	rathei = pager_ratio * pager->client->screen->height;

	/* if e is null we do the whole thing */
	if (e) {
		x = e->x;
		y = e->y;
		wid = e->width;
		hei = e->height;
	} else {
		x = y = 0;
		wid = pager->client->width;
		hei = pager->client->height;
	}

	/* draw using the grid color */
	XSetForeground(display, gc, pagerscr[pager->client->screen->num].grid_color);

	/* draw the desktop grid */
	for (i = 1; i < pager->desktop->width; i++) {
		tmp = i * ratwid + (i - 1);
		if (tmp >= x && tmp <= x + wid)
			XDrawLine(display, pager->win, gc, tmp, y, tmp, y + hei);
	}
	for (i = 1; i < pager->desktop->height; i++) {
		tmp = i * rathei + (i - 1);
		if (tmp >= y && tmp <= y + hei)
			XDrawLine(display, pager->win, gc, x, tmp, x + wid, tmp);
	}

	/* draw selected portion */
	XSetForeground(display, gc, pagerscr[pager->client->screen->num].sel_color);

	/* get information about the current workspace */
	if (pager->client->screen->desktop != pager->desktop)
		return;
	vx = pager->desktop->viewx + 1;
	vy = pager->desktop->viewy + 1;
	vx = (vx - 1) * ratwid + (vx - 1);
	vy = (vy - 1) * rathei + (vy - 1);

	/* get constraints from the exposure */
	if (y <= vy)
		y = vy - 1;
	if (y + hei > vy + rathei)
		hei = vy + rathei - y;
	if (x <= vx)
		x = vx - 1;
	if (x + wid > vx + ratwid)
		wid = vx + ratwid - x;

	/* draw them */
	if (y < vy + rathei) {
		if (pager->desktop->viewy + 1 == pager->desktop->height)
			hei++;
		if (pager->desktop->viewx + 1 < pager->desktop->width)
			XDrawLine(display, pager->win, gc, vx + ratwid, y, vx + ratwid, y + hei);
		if (pager->desktop->viewx)
			XDrawLine(display, pager->win, gc, vx - 1, y, vx - 1, y + hei);
	}
	if (x < vx + ratwid) {
		if (pager->desktop->viewy)
			XDrawLine(display, pager->win, gc, x, vy - 1, x + wid, vy - 1);
		if (pager->desktop->viewy + 1 < pager->desktop->height)
			XDrawLine(display, pager->win, gc, x, vy + rathei, x + wid, vy + rathei);
	}
}

/* translate coordinates from paged_t space to client_t space */
#define GETWINDOWCOORDS(x, y, tmpx, tmpy) do {							\
	(tmpx) = (x); (tmpy) = (y);								\
	(tmpx) -= (tmpx) / (pager_ratio * client->screen->width);				\
	(tmpy) -= (tmpy) / (pager_ratio * client->screen->height);				\
	(tmpx) = ((tmpx) / pager_ratio) - pager->desktop->viewx * client->screen->width;	\
	(tmpy) = ((tmpy) / pager_ratio) - pager->desktop->viewy * client->screen->height;	\
} while (0)

/*
 * handle a dragged paged_t; we need to move the client_t frame to the new
 * location and put the client_t into it's new workspace.
 */
static void pager_dragged(pager_t *pager, paged_t *paged, int x, int y, int newx, int newy, int usenewxy) {
	client_t *client;

	/* save some typing */
	client = paged->client;

	/* find the new spot to put our client's frame, if it wasn't told to us */
	if (!usenewxy)
		GETWINDOWCOORDS(x, y, newx, newy);

	/*
	 * don't actually do anything if it didn't move, but we need to do a
	 * workspace_add_bypos still if it changed desktops.
	 */
	if (newx == client->x && newy == client->y) {
		if (client->workspace->desktop != pager->desktop)
			workspace_add_bypos(pager->desktop, client);
		return;
	}

	/* sizeframe will move the client, and we send a ICCCM synth-config */
	client->x = newx;
	client->y = newy;
	client_sizeframe(client);
	action_send_config(client);
	workspace_add_bypos(pager->desktop, client);
}

/*
 * main dragging routine for moving paged_t's.  paged_t's need to be
 * dragable between desktops, as well as workspaces.
 */
void pager_drag(pager_t *origpager, paged_t *paged, XButtonEvent *ev) {
	XEvent event;
	pager_t *pager;
	client_t *client;
	Window dumwin;
	long eventmask;
	int ptx, pty;
	int tmpx, tmpy;
	int oldx, oldy;
	int haveold = 0;
	int i;

	/* don't move nomove clients */
	if (paged->client->flags.nomove)
		return;

	/* set up some vars */
	pager = origpager;
	client = paged->client;
	eventmask = PointerMotionMask | ButtonMotionMask | ButtonReleaseMask;
	if (!XTranslateCoordinates(display, pager->win, paged->win, ev->x, ev->y,
			&ptx, &pty, &dumwin))
		return;

	/* obtain pointer grab */
	if (XGrabPointer(display, client->screen->root, 0, eventmask, GrabModeAsync, GrabModeAsync,
			client->screen->root, cursor_move, CurrentTime) != GrabSuccess)
		return;
	eventmask |= ExposureMask;

	/* setup for the opaque or nonopaque movement */
	if (options.opaquemove) {
		XRaiseWindow(display, client->frame);

		/* raise pager windows so the client doesn't ever obstruct the pager */
		for (i = 0; i < pagerscr[client->screen->num].pager_count; i++)
			XRaiseWindow(display, pagerscr[client->screen->num].pagers[i]->client->frame);
	} else {
		if (pager->desktop == client->screen->desktop) {
			XUnmapWindow(display, client->frame);
			XGrabServer(display);
			GETWINDOWCOORDS(ev->x - ptx, ev->y - pty, oldx, oldy);
			draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);
			haveold = 1;
		}
	}

loop:
	/* dragging event loop */
	while (1) {
		XMaskEvent(display, eventmask, &event);

		switch (event.type) {
		case MotionNotify:
			/* handle dragging in a pager, else on the root */
			if (pager) {
				/* translate coordinates to relative to the pager window */
				tmpx = event.xmotion.x_root - pager->client->x - pager->client->left_space;
				tmpy = event.xmotion.y_root - pager->client->y - pager->client->top_space;

				/* erase the old winbox before moving the paged_t window  */
				if (!options.opaquemove && haveold && pager->desktop == client->screen->desktop)
					draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);

				/* see if they are dragging it out of a pager now */
				if (tmpx > pager->client->width || tmpx < 0
						|| tmpy > pager->client->height || tmpy < 0) {
					XReparentWindow(display, paged->win, client->screen->root,
						event.xmotion.x_root - ptx, event.xmotion.y_root - pty);
					pager = NULL;
					if (options.opaquemove)
						XUnmapWindow(display, client->frame);
					else
						XUngrabServer(display);
					continue;
				}

				/* else we move it inside of the current pager */
				XMoveWindow(display, paged->win, tmpx - ptx, tmpy - pty);

				/* draw the winbox or move the client window */
				if (pager->desktop != client->screen->desktop)
					continue;
				GETWINDOWCOORDS(tmpx - ptx, tmpy - pty, tmpx, tmpy);
				if (options.opaquemove) {
					XMoveWindow(display, client->frame, tmpx, tmpy);
				} else {
					draw_winbox(client->screen, client, tmpx, tmpy, client->width, client->height);

					oldx = tmpx; oldy = tmpy;
					haveold = 1;
				}
			} else {
				/*
				 * check if it is being dragged into a pager
				 * XXX: unfortunatly because I'm moving the paged->win around
				 * with the cursor, subwindow always is the paged->win, so I have
				 * to do this mess.
				 */
				for (i = 0; i < pagerscr[client->screen->num].pager_count; i++) {
					/* get coords relative to the pager window */
					tmpx = event.xmotion.x_root - pagerscr[client->screen->num].pagers[i]->client->x
						- pagerscr[client->screen->num].pagers[i]->client->left_space;
					tmpy = event.xmotion.y_root - pagerscr[client->screen->num].pagers[i]->client->y
						- pagerscr[client->screen->num].pagers[i]->client->top_space;

					/* now check if it's inside */
					if (tmpx <= pagerscr[client->screen->num].pagers[i]->client->width
							&& tmpy <= pagerscr[client->screen->num].pagers[i]->client->height
							&& tmpx >= 0 && tmpy >= 0) {
						pager = pagerscr[client->screen->num].pagers[i];
						XReparentWindow(display, paged->win, pager->win, tmpx - ptx, tmpy - pty);
						if (pager->desktop == client->screen->desktop) {
							if (options.opaquemove) {
								XMapWindow(display, client->frame);
							} else {
								XGrabServer(display);
								haveold = 0;
							}
						}
						goto loop;
					}
				}

				/* move it to follow cursor */
				XMoveWindow(display, paged->win, event.xmotion.x_root - ptx, event.xmotion.y_root - pty);
			}
			break;
		case Expose:
			event_expose(&event);
			break;
		case ButtonRelease:
			goto done;
		}
	}

done:
	/* release grab */
	XUngrabPointer(display, CurrentTime);

	/*
	 * if they aren't in a pager, we drop the client where the cursor,
	 * otherwise we place the client based on the paged_t win in the pager.
	 * the client may need to be moved across desktops.
	 */
	if (!pager) {
		/* find which pager to be using, and if neccesary move it there */
		pager = pagerscr[client->screen->num].pagers[client->screen->desktop->num];
		if (origpager != pager)
			pager_movepaged(origpager, paged, pager);

		/* we need to put the paged_t window back, and map the client frame */
		XReparentWindow(display, paged->win, pager->win, 0, 0);

		/* drag the stuff and size the paged */
		pager_dragged(pager, paged, tmpx - ptx, tmpy - pty, event.xbutton.x_root,
			event.xbutton.y_root, 1);
		XMapWindow(display, client->frame);
		pager_sizepaged(pager, paged, client);
	} else {
		/* translate coordinates to relative to the pager window */
		tmpx = event.xmotion.x_root - pager->client->x - pager->client->left_space;
		tmpy = event.xmotion.y_root - pager->client->y - pager->client->top_space;

		/* move paged around */
		if (pager != origpager)
			pager_movepaged(origpager, paged, pager);
		if (pager->desktop == client->screen->desktop && !options.opaquemove) {
			draw_winbox(client->screen, client, oldx, oldy, client->width, client->height);
			XUngrabServer(display);
		}

		/* drag the stuff */
		pager_dragged(pager, paged, tmpx - ptx, tmpy - pty, 0, 0, 0);

		/* map after moving it properly if not opaque */
		if (pager->desktop == client->screen->desktop && !options.opaquemove)
			XMapRaised(display, client->frame);
	}

	/* focus the client_t they were dragging */
	focus_setfocused(client);
}

#undef GETWINDOWCOORDS
