/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "pager.h"
#include "plugutil.h"

/* per screen structure */
struct pagerscr		*pagerscr;

/* names of the colors to use for the grid */
static char		*gridclr;
static char		*selclr;

/* get parameters, init pager system */
int init(plugin_t *plugin) {
	double ratio;
	dgroup_t *dgroup;
	char *dgident;

	/* get our parameters */
	OPTIONAL_PARAM(ratio, &plugin->params, double, "size_ratio", 0.04);
	OPTIONAL_PARAM(gridclr, &plugin->params, string, "grid_color", NULL);
	OPTIONAL_PARAM(selclr, &plugin->params, string, "select_color", NULL);

	/* dgroup to use for the pagers */
	OPTIONAL_PARAM(dgident, &plugin->params, string, "pager_dgroup", NULL);
	if (dgident) {
		dgroup = dgroup_ident(dgident);
		if (!dgroup)
			PWARN(plugin, "undeclared decoration group identifier");
		free(dgident);
	} else {
		dgroup = NULL;
	}

	/* set up the pager system */
	pager_init(ratio, dgroup);

	return PLUGIN_OK;
}

/* handle plugin shutdowns */
void shutdown() {
	int i, n;

	/* clear up the per-screen data */
	if (pagerscr) {
		for (i = 0; i < ScreenCount(display); i++) {
			for (n = 0; n < pagerscr[i].pager_count; n++)
				pager_delete(pagerscr[i].pagers[n]);
			XFreeGC(display, pagerscr[i].drawgc);
			free(pagerscr[i].pagers);
		}
		free(pagerscr);
	} else {
		/* grid colors aren't yet free */
		if (gridclr) free(gridclr);
		if (selclr) free(selclr);
	}
}

/* start up our pagers, make our screen structures, etc */
int start(plugin_t *plugin) {
	XGCValues gcvalues;
	XColor clr;
	screen_t *screen;
	desktop_t *desktop;
	int i;

	/* get the per-screen structure */
	pagerscr = malloc(sizeof(struct pagerscr) * screen_count);
	if (!pagerscr)
		return PLUGIN_UNLOAD;

	/* fill in each screen */
	screen = screen_list;
	while (screen) {
		/* get grid and selected colors */
		if (!gridclr || !XParseColor(display, DefaultColormap(display, screen->num),
				gridclr, &clr)) {
			pagerscr[screen->num].grid_color = BlackPixel(display, screen->num);
		} else {
			XAllocColor(display, DefaultColormap(display, screen->num), &clr);
			pagerscr[screen->num].grid_color = clr.pixel;
		}
		if (!selclr || !XParseColor(display, DefaultColormap(display, screen->num),
				selclr, &clr)) {
			pagerscr[screen->num].sel_color = WhitePixel(display, screen->num);
		} else {
			XAllocColor(display, DefaultColormap(display, screen->num), &clr);
			pagerscr[screen->num].sel_color = clr.pixel;
		}
		
		/* get drawing graphics context */
		gcvalues.foreground = pagerscr[screen->num].grid_color;
		pagerscr[screen->num].drawgc = XCreateGC(display, RootWindow(display, screen->num),
			GCForeground, &gcvalues);

		/* make one pager per desktop */
		pagerscr[screen->num].pagers = calloc(screen->desktop_count, sizeof(pager_t *));
		pagerscr[screen->num].pager_count = screen->desktop_count;
		desktop = screen->desktop_list;
		for (i = 0; i < screen->desktop_count; i++) {
			pagerscr[screen->num].pagers[i] = pager_create(plugin,
				screen, desktop);
			desktop = desktop->next;
		}

		screen = screen->next;
	}

	/* free grid color names */
	if (gridclr) free(gridclr);
	if (selclr) free(selclr);

	return PLUGIN_OK;
}

/* new clients arriving */
int window_birth(plugin_t *plugin, client_t *client) {
	if (client->state == NormalState && !client->flags.internal) {
		pager_addpaged(plugin, pagerscr[client->screen->num].pagers[client->workspace->desktop->num],
			client);
	}

	return PLUGIN_OK;
}

/* old clients departing */
int window_death(plugin_t *plugin, client_t *client) {
	if (client->state == NormalState && !client->flags.internal) {
		pager_rmpaged(pagerscr[client->screen->num].pagers[client->workspace->desktop->num],
			NULL, client);
	}

	return PLUGIN_OK;
}

/* iconification and restoration just remove/add to the pager like death/birth */
int iconify_notify(plugin_t *plugin, client_t *client) {
	if (client->workspace && !client->flags.internal) {
		pager_rmpaged(pagerscr[client->screen->num].pagers[client->workspace->desktop->num],
			NULL, client);
	}

	return PLUGIN_OK;
}

int restore_notify(plugin_t *plugin, client_t *client) {
	return window_birth(plugin, client);
}

/* changes in a client's geometry */
int geometry_change(plugin_t *plugin, client_t *client) {
	pager_t *pager;

	/*
	 * we need to clear the window of moved pagers for our temporary
	 * ParentRelative hack.
	 */
	if (client->flags.internal) {
		if (XFindContext(display, client->window, pager_context, (XPointer *) &pager) != 0)
			return PLUGIN_OK;

		XClearWindow(display, pager->win);
		pager_expose(pager, pagerscr[client->screen->num].drawgc, NULL);
	} else if (client->state == NormalState) {
		pager_sizepaged(pagerscr[client->screen->num].pagers[client->workspace->desktop->num],
			NULL, client);
	}

	return PLUGIN_OK;
}

/* zooming of clients; just a change in geom for our purposes */
int zoom_notify(plugin_t *plugin, client_t *client) {
	return geometry_change(plugin, client);
}

/* unzooming of clients; also just a geom change for our purposes */
int unzoom_notify(plugin_t *plugin, client_t *client) {
	return geometry_change(plugin, client);
}

/* desktop changing */
int desktop_change(plugin_t *plugin, screen_t *screen, desktop_t *olddesk) {
	client_t *client;
	int i;

	/* move sticky client paged_t's to the new pager */
	client = client_list;
	while (client) {
		if (client->workspace && client->workspace->desktop == screen->desktop) {
			if (client->flags.sticky) {
				pager_rmpaged(pagerscr[screen->num].pagers[olddesk->num], NULL, client);
				pager_addpaged(plugin, pagerscr[screen->num].pagers[screen->desktop->num], client);
			}
		}

		client = client->next;
	}

	/* find the olddesk pager, expose it and the newdesk pager */
	for (i = 0; i < pagerscr[screen->num].pager_count; i++) {
		if (pagerscr[screen->num].pagers[i]->desktop == olddesk
				|| pagerscr[screen->num].pagers[i]->desktop == screen->desktop) {
			pager_expose(pagerscr[screen->num].pagers[i], pagerscr[screen->num].drawgc, NULL);
		}
	}

	return PLUGIN_OK;
}

/* workspace changes; we need to check for sticky windows */
int workspace_change(plugin_t *plugin, screen_t *screen, desktop_t *desktop) {
	pager_t *pager;
	client_t *client;

	/* pager */
	pager = pagerscr[screen->num].pagers[desktop->num];

	/* find sticky windows and size their paged windows */
	client = client_list;
	while (client) {
		if (client->workspace && client->workspace == desktop->current_space) {
			if (client->flags.sticky)
				pager_sizepaged(pager, NULL, client);
		}
		client = client->next;
	}

	/* expose the pager */
	pager_expose(pager, pagerscr[screen->num].drawgc, NULL);

	return PLUGIN_OK;
}

/* exposures on our pager windows */
int expose(plugin_t *plugin, client_t *client, XExposeEvent *e) {
	pager_t *pager;

	if (XFindContext(display, e->window, pager_context, (XPointer *) &pager) != 0)
		return PLUGIN_OK;

	pager_expose(pager, pagerscr[client->screen->num].drawgc, e);

	return PLUGIN_OK;
}

/* presses */
int button_press(plugin_t *plugin, client_t *client, XButtonEvent *e) {	
	pager_t *pager;
	paged_t *paged;

	/* button 2 drags windows */
	if (e->button != Button2 || e->subwindow == None)
		return PLUGIN_OK;

	/* get the pager_t */
	if (XFindContext(display, e->window, pager_context, (XPointer *) &pager) != 0)
		return PLUGIN_OK;

	/* figure out which paged_t to deal with */
	paged = pager->paged_list;
	while (paged) {
		if (paged->win == e->subwindow)
			break;
		paged = paged->next;
	}

	/* pass on to the pager drag routine */
	if (paged)
		pager_drag(pager, paged, e);

	return PLUGIN_OK;
}

/* releases */
int button_release(plugin_t *plugin, client_t *client, XButtonEvent *e) {
	pager_t *pager;

	/* pass button1 clicks on to pager_click */
	if (e->button == Button1) {
		if (XFindContext(display, e->window, pager_context, (XPointer *) &pager) == 0)
			pager_click(pager, e->x, e->y);
	}

	return PLUGIN_OK;
}
