/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "plugutil.h"

/* per-screen structure for gnome compliance plugin */
static struct gnomescr {
	Window		wm_check;	/* gnome compliance check window, also used for button proxy */
} *gnomescr = NULL;

/* atoms */
#define NUM_PROTOCOLS			6
#define WIN_CLIENT_LIST			win_protocols_list[0]
#define WIN_DESKTOP_BUTTON_PROXY	win_protocols_list[1]
#define WIN_WORKSPACE			win_protocols_list[2]
#define WIN_WORKSPACE_COUNT		win_protocols_list[3]
#define WIN_AREA_COUNT			win_protocols_list[4]
#define WIN_AREA			win_protocols_list[5]

Atom win_supporting_wm_check;
Atom win_protocols;
Atom win_protocols_list[NUM_PROTOCOLS];
char *win_protocols_names[NUM_PROTOCOLS]	= { "_WIN_CLIENT_LIST",
						    "_WIN_DESKTOP_BUTTON_PROXY",
						    "_WIN_WORKSPACE",
						    "_WIN_WORKSPACE_COUNT",
						    "_WIN_AREA_COUNT",
						    "_WIN_AREA" };

/* set the list of clients for screen */
static int set_client_list(screen_t *screen) {
	Window *list, *tmp;
	client_t *client;
	int size, count;

	count = 0;
	size = 10;
	list = malloc(sizeof(Window) * size);
	if (!list)
		return -1;

	client = client_list;
	while (client) {
		if (!client->flags.internal && client->screen == screen) {
			if (++count > size) {
				tmp = realloc(list, sizeof(Window) * (size + size));
				if (!tmp) {
					free(list);
					return -1;
				}
				size += size;
			}
			list[count - 1] = client->window;
		}
		client = client->next;
	}

	XChangeProperty(display, screen->root, WIN_CLIENT_LIST, XA_CARDINAL, 32,
		PropModeReplace, (unsigned char *) list, count);
	free(list);
	return 0;
}

/* set the count of workspaces for screen */
static void set_workspace_count(screen_t *screen) {
	XChangeProperty(display, screen->root, WIN_WORKSPACE_COUNT, XA_CARDINAL, 32,
		PropModeReplace, (unsigned char *) &screen->desktop_count, 1);
}

/* plugin death */
void shutdown() {
	int i, cnt;

	/* kill our per-screen structure */
	if (gnomescr) {
		cnt = ScreenCount(display);
		for (i = 0; i < cnt; i++) {
			if (win_supporting_wm_check)
				XDeleteProperty(display, RootWindow(display, i), win_supporting_wm_check);
			if (win_protocols)
				XDeleteProperty(display, RootWindow(display, i), win_protocols);
			if (WIN_DESKTOP_BUTTON_PROXY)
				XDeleteProperty(display, RootWindow(display, i), WIN_DESKTOP_BUTTON_PROXY);
			if (WIN_CLIENT_LIST)
				XDeleteProperty(display, RootWindow(display, i), WIN_CLIENT_LIST);
			if (gnomescr[i].wm_check)
				XDestroyWindow(display, gnomescr[i].wm_check);
		}
		free(gnomescr);
	}
}

/* alloc stuff; create the gnome window check window */
int start(plugin_t *plugin) {
	XSetWindowAttributes attr;
	screen_t *screen;
	long val[2];

	/* get memory for the per-screen structure */
	gnomescr = calloc(screen_count, sizeof(struct gnomescr));
	if (!gnomescr)
		return PLUGIN_UNLOAD;

	/* get atoms */
	win_supporting_wm_check = XInternAtom(display, "_WIN_SUPPORTING_WM_CHECK", 0);
	win_protocols = XInternAtom(display, "_WIN_PROTOCOLS", 0);
	XInternAtoms(display, win_protocols_names, NUM_PROTOCOLS, 0, win_protocols_list);

	/* fill in the per-screen structure */
	attr.override_redirect = 1;
	screen = screen_list;
	while (screen) {
		/* create the gnome-compliance check window */
		gnomescr[screen->num].wm_check = XCreateWindow(display, screen->root, -30, -30,
			2, 2, 0, CopyFromParent, CopyFromParent, CopyFromParent,
			CWOverrideRedirect, &attr);
		XChangeProperty(display, screen->root, win_supporting_wm_check,  XA_CARDINAL,
			32, PropModeReplace, (unsigned char *) &gnomescr[screen->num].wm_check, 1);
		XChangeProperty(display, gnomescr[screen->num].wm_check, win_supporting_wm_check, XA_CARDINAL,
			32, PropModeReplace, (unsigned char *) &gnomescr[screen->num].wm_check, 1);

		/* set the win_protocols list onto the root window */
		XChangeProperty(display, screen->root, win_protocols, XA_ATOM, 32, PropModeReplace,
			(unsigned char *) win_protocols_list, NUM_PROTOCOLS);

		/* reuse the check window as the button proxy window */
		XChangeProperty(display, screen->root, WIN_DESKTOP_BUTTON_PROXY, XA_CARDINAL,
			32, PropModeReplace, (unsigned char *) &gnomescr[screen->num].wm_check, 1);
		XChangeProperty(display, gnomescr[screen->num].wm_check, WIN_DESKTOP_BUTTON_PROXY, XA_CARDINAL,
			32, PropModeReplace, (unsigned char *) &gnomescr[screen->num].wm_check, 1);

		/* setup the CLIENT_LIST on the screen*/
		if (set_client_list(screen) == -1)
			return PLUGIN_UNLOAD;

		/* setup workspace count, current workspace */
		set_workspace_count(screen);
		XChangeProperty(display, screen->root, WIN_WORKSPACE, XA_CARDINAL,
			32, PropModeReplace, (unsigned char *) &screen->desktop->num, 1);

		/* set the workspace area properties */
		val[0] = screen->desktop->width;
		val[1] = screen->desktop->height;
		XChangeProperty(display, screen->root, WIN_AREA_COUNT, XA_CARDINAL, 32, PropModeReplace,
			(unsigned char *) val, 2);
		val[0] = screen->desktop->viewx;
		val[1] = screen->desktop->viewy;
		XChangeProperty(display, screen->root, WIN_AREA, XA_CARDINAL, 32, PropModeReplace,
			(unsigned char *) val, 2);

		screen = screen->next;
	}

	return PLUGIN_OK;
}

/* when new windows arrive */
int window_birth(plugin_t *plugin, client_t *client) {
	if (set_client_list(client->screen) == -1)
		return PLUGIN_UNLOAD;
	return PLUGIN_OK;
}

/* when clients leave */
int window_death(plugin_t *plugin, client_t *client) {
	if (set_client_list(client->screen) == -1)
		return PLUGIN_UNLOAD;
	return PLUGIN_OK;
}

/* workspace changes */
int workspace_change(plugin_t *plugin, screen_t *screen, desktop_t *desktop) {
	long val[2];

	if (desktop == screen->desktop) {
		val[0] = screen->desktop->viewx;
		val[1] = screen->desktop->viewy;
		XChangeProperty(display, screen->root, WIN_AREA, XA_CARDINAL, 32, PropModeReplace,
			(unsigned char *) val, 2);
	}

	return PLUGIN_OK;
}

/* when the user switches desktops */
int desktop_change(plugin_t *plugin, screen_t *screen, desktop_t *olddesk) {
	long val[2];

	/* set the workspace area properties */
	val[0] = screen->desktop->width;
	val[1] = screen->desktop->height;
	XChangeProperty(display, screen->root, WIN_AREA_COUNT, XA_CARDINAL, 32, PropModeReplace,
		(unsigned char *) val, 2);

	/* now set the current desktop */
	XChangeProperty(display, screen->root, WIN_WORKSPACE, XA_CARDINAL,
		32, PropModeReplace, (unsigned char *) &screen->desktop->num, 1);

	/* desktop change also changes workspace */
	return workspace_change(plugin, screen, screen->desktop);
}

/* button press on the root window */
int root_button_press(plugin_t *plugin, screen_t *screen, XButtonEvent *e) {
	XUngrabPointer(display, CurrentTime);
	XSendEvent(display, gnomescr[screen->num].wm_check, 0, SubstructureNotifyMask, (XEvent *) e);
	return PLUGIN_OK;
}

/* button release on the root window */
int root_button_release(plugin_t *plugin, screen_t *screen, XButtonEvent *e) {
	XSendEvent(display, gnomescr[screen->num].wm_check, 0, SubstructureNotifyMask, (XEvent *) e);
	return PLUGIN_OK;
}
